/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;

namespace Unity {

public enum SearchType
{
  DEFAULT,
  GLOBAL,

  N_TYPES
}

public class Scope : GLib.Object
{
  public bool active { get; set; default = false; }
  public bool exported { get; private set; default = false; }
  public bool search_in_global { get; set; default = true; }
  public string dbus_path { get; construct; }
  public OptionsFilter sources { get; internal set; }
  public List<Filter> filters { get { return _filters; } }

  public Dee.SerializableModel results_model {
    get { return _pimpl._results_model; }
  }

  public Dee.SerializableModel global_results_model {
    get { return _pimpl._global_results_model; }
  }

  public signal ActivationResponse? activate_uri (string uri);
  public signal Preview? preview_uri (string uri);
  public signal void filters_changed ();
  public signal void active_sources_changed (string[] active_ids);
  public signal void search_changed (LensSearch search, SearchType search_type,
                                     Cancellable cancellable);
  [Signal (detailed = true)]
  public signal string generate_search_key (LensSearch search);
  
  private LensSearch last_search[2];
  private ScopeImpl _pimpl;
  private bool is_local = false;
  internal List<Filter> _filters = new List<Filter>();
   
  public Scope (string dbus_path_)
  {
    Object (dbus_path: dbus_path_);
  }

  construct
  {
    sources = new CheckOptionFilter ("sources", "Sources", null, true);
    _pimpl = new ScopeImpl (this);
    // make sure the length of last_search matches
    static_assert (SearchType.N_TYPES == 2);
  }

  public void export () throws IOError
  {
    if (!exported && !is_local)
    {
      _pimpl.export();
      exported = true;
      prepare_search ();
    }
  }

  public Filter? get_filter (string id)
  {
    foreach (Filter filter in _filters)
    {
      if (filter.id == id)
        return filter;
    }
    return null;
  }

  /**
   * Invalidates current search and queues new search.
   *
   * This method will invalidate (and cancel) last search and queue a new
   * search (with the same search_string). The {@link Scope.search_changed}
   * signal will be emitted immediately in case the Lens managing this scope
   * is active, or as soon as it becomes active.
   *
   * @param search_type: Type of search to queue.
   */
  public void queue_search_changed (SearchType search_type)
    requires (search_type < SearchType.N_TYPES)
  {
    bool is_global_search = search_type == SearchType.GLOBAL;
    LensSearch? search = last_search[search_type];
    invalidate_search (search_type);
    if (search == null)
    {
      var ht = new HashTable<string, Variant> (str_hash, str_equal);
      var model = is_global_search ? global_results_model : results_model;
      search = new LensSearch ("", ht, model);
    }

    _pimpl.update_search_key (search, search_type);
    _pimpl.schedule_search_changed.begin (search, search_type, true);
  }

  /**
   * Invalidates last search.
   *
   * Invalidate last search, so that the next search request will trigger
   * the {@link Scope.search_changed} signal even if the search would be
   * otherwise discarded because of unchanged search key.
   *
   * @param search_type: Type of search to invalidate.
   * @see Scope.generate_search_key
   */
  public void invalidate_search (SearchType search_type)
    requires (search_type < SearchType.N_TYPES)
  {
    _pimpl.invalidate_search (search_type);
    // we need to also invalidate last LensSearch, in case generate_search_key
    // is not used
    last_search[search_type] = null;
  }

  /*
   * For our private implmentation
   */
  internal unowned LensSearch? get_last_search (SearchType search_type)
  {
    return last_search[search_type];
  }

  internal void set_last_search (LensSearch search, SearchType search_type)
  {
    last_search[search_type] = search;
  }

  internal void set_view_type_internal (ViewType view_type)
  {
    _pimpl.view_type = view_type;
  }

  internal void set_active_sources_internal (string[] active_sources_)
  {
    foreach (var filter_option in sources.options)
    {
      filter_option.active = filter_option.id in active_sources_;
    }

    this.active_sources_changed (active_sources_);
  }

  internal void set_local (bool is_local_)
  {
    is_local = is_local_;
    if (is_local) prepare_search ();
  }

  private void prepare_search ()
  {
    /* because of the way unity works right now, we need to queue a global 
     * search ourselves, unity won't initially send an empty string search
     * (but will set proper view type) */
    if (search_in_global) queue_search_changed (SearchType.GLOBAL);
  }

  /*
   * For local Scope implementations, only used internally
   */
  internal unowned Dee.SerializableModel get_filter_model ()
  {
    return _pimpl._filters_model;
  }

  internal async ActivationReplyRaw activate (string uri, uint action_type)
  {
    try {
      var reply = yield _pimpl.activate (uri, action_type);
      return reply;
    } catch (Error e) {
      warning (@"Unable to activate scope: $(e.message)");
    }
    var res = ActivationReplyRaw ();
    res.uri = uri;
    res.handled = HandledType.NOT_HANDLED;
    res.hints = new HashTable<string, Variant> (null, null);
    return res;
  }

  internal async HashTable<string, Variant> search (
      string search_string, HashTable<string, Variant> hints) throws Error
  {
    var result = yield _pimpl.search (search_string, hints);
    return result;
  }

  internal async HashTable<string, Variant> global_search (
      string search_string, HashTable<string, Variant> hints) throws Error
  {
    var result = yield _pimpl.global_search (search_string, hints);
    return result;
  }

  internal async PreviewReplyRaw preview (string uri)
  {
    try {
      var reply = yield _pimpl.preview (uri);
      return reply;
    } catch (Error e) {
      warning (@"Unable to preview scope: $(e.message)");
    }
    return PreviewReplyRaw ();
  }
}

} /* namespace */
