/*
 * Copyright (C) 2011 Canonical, Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

using GLib;
using Dee;

namespace Unity {

private class ScopeProxyRemote : GLib.Object, ScopeProxy
{
  public string dbus_name { get; construct; }
  public string dbus_path { get; construct; }

  public string[] sources { get; set; }
  public bool search_in_global { get; set; }

  public Dee.SharedModel results_model {
    get { return _results_model; }
    private set {
      _results_model = value;
    }
  }

  public Dee.SharedModel global_results_model {
    get { return _global_results_model; }
    private set {
      _global_results_model = value;
    }
  }

  public Dee.SharedModel filters_model {
    get { return _filters_model; }
    private set {
      _filters_model = value;
    }
  }

  private DBusConnection _bus;
  private uint _watcher;

  private ScopeService? _service = null;

  private Dee.SharedModel? _results_model;
  private Dee.SharedModel? _global_results_model;
  private Dee.SharedModel? _filters_model;
  private uint _reconnection_id = 0;

  public ScopeProxyRemote (string dbus_name_, string dbus_path_)
  {
    Object (dbus_name:dbus_name_, dbus_path:dbus_path_);
  }

  construct
  {
    try {
      _bus = Bus.get_sync (BusType.SESSION);
      _watcher = Bus.watch_name_on_connection (_bus,
                                               dbus_name,
                                               BusNameWatcherFlags.NONE,
                                               on_scope_appeared,
                                               on_scope_vanished);

      /* If it doesn't exist when we load up, start it */
      start_reconnection_timeout ();
    } catch (Error e) {
      critical (@"Unable to connect to session bus: $(e.message)");
    }
  }

  private async void connect_to_scope ()
  {
    try {
      _service = yield _bus.get_proxy (dbus_name, dbus_path);
      _service.changed.connect (on_changed);
      yield _service.info_request ();

    } catch (Error e) {
      warning (@"Unable to connect to Scope ($dbus_path @ $dbus_name): $(e.message)");
    }
  }
 
  private void on_scope_appeared (DBusConnection conn,
                                  string         name,
                                  string         name_owner)
  {
    debug (@"Scope appeared: $dbus_name");
    if (_reconnection_id != 0)
      Source.remove (_reconnection_id);
    connect_to_scope.begin ();
  }

  private void on_scope_vanished (DBusConnection conn,
                                  string         name)
  {
    /* FIXME: Add some protection against Scopes that keep restarting */
    debug (@"Scope vanished: $dbus_name");
    
    search_in_global = false;
    sources = {};
    if (_results_model != null)
      _results_model.clear ();	

    _results_model = new Dee.SharedModel ("where.in.the.world.is.kamstrup");;
    _results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");

    if (_global_results_model != null)
      _global_results_model.clear ();

    _global_results_model = new Dee.SharedModel ("where.in.the.world.is.kamstrup");
    _global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");
     
    if (_filters_model != null)
      _filters_model.clear ();	

    _filters_model = new Dee.SharedModel ("where.in.the.world.is.kamstrup");
    _filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");

    start_reconnection_timeout ();
  }

  private void start_reconnection_timeout ()
  {
    if (_reconnection_id != 0)
      Source.remove (_reconnection_id);

    _reconnection_id = Timeout.add_seconds (1, () =>
    {
      if (_service == null)
        connect_to_scope ();

      _reconnection_id = 0;
      return false;
    });
  }

  /*
   * Implementation of the ScopeService interface
   */
  public async ActivationReplyRaw activate (string uri, uint action_type)
  {
    if (_service is ScopeService)
    {
      try {
        var raw = yield _service.activate (uri, action_type);
        return raw;
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path): $(e.message)");
      }
    }
    return ActivationReplyRaw ();
  }

  public async void search (string search_string,
                            HashTable<string, Variant> hints)
  {
    if (_service is ScopeService)
    {
      try {
        yield _service.search (search_string, hints);
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path): $(e.message)");
      }
    }
  }

  public async void global_search (string search_string,
                                   HashTable<string, Variant> hints)
  {
    if (_service is ScopeService)
    {
      try {
        yield _service.global_search (search_string, hints);
      } catch (Error e) {
        warning (@"Unable to global_search scope ($dbus_path): $(e.message)");
      } 
    }
  }

  public async PreviewReplyRaw preview (string uri)
  {
    if (_service is ScopeService)
    {
      try {
        var raw = yield _service.preview (uri);
        return raw;
      } catch (Error e) {
        warning (@"Unable to search scope ($dbus_path): $(e.message)");
      }
    }
    return PreviewReplyRaw();
  }

  public async void set_active (bool active)
  {
    if (_service is ScopeService)
    {
      try {
        yield _service.set_active (active);
      } catch (Error e) {
        warning (@"Unable to set_active ($dbus_path): $(e.message)");
      } 
    }
  }

  public async void set_active_sources (string[] sources)
  {
    if (_service is ScopeService)
    {
      try {
        yield _service.set_active_sources (sources);
      } catch (Error e) {
        warning (@"Unable to set_sources ($dbus_path): $(e.message)");
      } 
    }
  }
  
  /* This is where we figure out the diff between states */
  public void on_changed (ScopeInfo scope_info)
  {
    if (dbus_path != scope_info.dbus_path)
    {
      warning (@"Unable to handle Scope changed signal: dbus_path mismatch. Expected $dbus_path got $(scope_info.dbus_path)");
      return;
    }
    debug (@"Processing changed signal for $dbus_path");
    sources = scope_info.sources;
    search_in_global = scope_info.search_in_global;

    if (results_model == null ||
        results_model.get_swarm_name () != scope_info.results_model_name)
    {
      results_model = new Dee.SharedModel (scope_info.results_model_name);
      results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");
    }

    if (global_results_model == null||
        global_results_model.get_swarm_name () != scope_info.global_results_model_name)
    {
      global_results_model = new Dee.SharedModel (scope_info.global_results_model_name);
      global_results_model.set_schema ("s", "s", "u", "s", "s", "s", "s");
    }

    if (filters_model == null ||
        filters_model.get_swarm_name () != scope_info.filters_model_name)
    {
      filters_model = new Dee.SharedModel (scope_info.filters_model_name);
      filters_model.set_schema ("s", "s", "s", "s", "a{sv}", "b", "b", "b");
    }
  }
  
  public void on_search_finished (string search_string,
                                  HashTable<string, Variant> hints)
  {
    search_finished (search_string, hints);
  }
  
  public void on_global_search_finished (string search_string,
                                         HashTable<string, Variant> hints)
  {
    global_search_finished (search_string, hints);
  }
}

} /* Namespace */
