/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include "syncdaemon.h"

static DBusGConnection *bus = NULL;
static DBusGProxy *daemon_proxy = NULL;
static DBusGProxy *status_proxy = NULL;

void
syncdaemon_init (void)
{
	GError *error;

	/* Initialize connection to DBus if not done already */
	if (bus == NULL) {
		error = NULL;
		bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
		if (error) {
			g_warning ("Couldn't get session bus: %s", error->message);
			g_error_free (error);
			return;
		}
	}

	/* Initialize connection to syncdaemon */
	if (daemon_proxy == NULL) {
		daemon_proxy = dbus_g_proxy_new_for_name (bus, "com.ubuntuone.SyncDaemon",
							  "/", "com.ubuntuone.SyncDaemon.SyncDaemon");
		if (daemon_proxy == NULL)
			g_warning ("Could not get proxy for com.ubuntuone.SyncDaemon.SyncDaemon");
	}

	/* Initialize com.ubuntuone.SyncDaemon.Status proxy */
	if (status_proxy == NULL) {
		status_proxy = dbus_g_proxy_new_for_name (bus, "com.ubuntuone.SyncDaemon",
							  "/status", "com.ubuntuone.SyncDaemon.Status");
		if (status_proxy == NULL)
			g_warning ("Could not get proxy for com.ubuntuone.SyncDaemon.Status");
	}
}

gboolean
syncdaemon_is_connected (void)
{
	GHashTable *return_value;
	gboolean is_connected = FALSE;
	GError *error = NULL;

	syncdaemon_init ();

	if (dbus_g_proxy_call (status_proxy, "current_status", &error,
			       G_TYPE_INVALID,
			       dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING), &return_value,
			       G_TYPE_INVALID)) {
		is_connected = g_strcmp0 (g_hash_table_lookup (return_value, "is_connected"), "True") == 0;

		g_hash_table_destroy (return_value);
	} else {
		g_warning ("Error calling current_status: %s", error->message);
		g_error_free (error);
	}

	return is_connected;
}

void
syncdaemon_connect (void)
{
	GError *error;

	syncdaemon_init ();

	if (syncdaemon_is_connected ()) {
		g_warning ("Syncdaemon already connected, not connecting again");
		return;
	}

	/* Make the syncdaemon connect if it's not connected */
	error = NULL;

	if (!dbus_g_proxy_call (daemon_proxy, "connect", &error,
				G_TYPE_INVALID,
				G_TYPE_INVALID)) {
		g_warning ("Could not connect to Ubuntu One: %s", error->message);
		g_error_free (error);
	}
}

void
syncdaemon_finalize (void)
{
	g_object_unref (G_OBJECT (daemon_proxy));
	g_object_unref (G_OBJECT (status_proxy));
	dbus_g_connection_unref (bus);
}

static void
free_syncdaemon_download (SyncdaemonDownload *item_data)
{
	g_free (item_data->path);
	g_free (item_data->share_id);
	g_free (item_data->node_id);

	g_free (item_data);
}

GHashTable *
syncdaemon_get_current_downloads (void)
{
	GError *error;
	GPtrArray *return_value;
	GHashTable *downloads = NULL;

	/* Make sure we are connected when we get here */
	syncdaemon_connect ();

	/* Ask the syncdaemon about current downloads */
	error = NULL;
	if (dbus_g_proxy_call (status_proxy, "current_downloads", &error,
			       G_TYPE_INVALID,
			       dbus_g_type_get_collection (
				       "GPtrArray",
				       dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_STRING)), &return_value,
			       G_TYPE_INVALID)) {
		gint i;

		downloads = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, (GDestroyNotify) free_syncdaemon_download);
		for (i = 0; i < return_value->len; i++) {
			SyncdaemonDownload *item_data;
			gchar *str;
			GHashTable *values = (GHashTable *) g_ptr_array_index (return_value, i);

			item_data = g_new0 (SyncdaemonDownload, 1);
			item_data->path = g_strdup (g_hash_table_lookup (values, "path"));
			item_data->share_id = g_strdup (g_hash_table_lookup (values, "share_id"));
			item_data->node_id = g_strdup (g_hash_table_lookup (values, "node_id"));

			str = g_hash_table_lookup (values, "n_bytes_read");
			if (str != NULL)
				item_data->n_bytes_read = atol (str);
			else
				item_data->n_bytes_read = 0;

			str = g_hash_table_lookup (values, "deflated_size");
			if (str != NULL)
				item_data->deflated_size = atol (str);
			else
				item_data->deflated_size = 0;

			g_hash_table_insert (downloads, g_strdup (item_data->path), item_data);
		}

		g_ptr_array_free (return_value, TRUE);
	} else {
		g_warning ("Error calling current_downloads: %s", error->message);
		g_error_free (error);
	}

	return downloads;
}

