/**
 * ===========================================================
 * LibRepository : a free Java content repository access layer
 * ===========================================================
 *
 * Project Info:  http://reporting.pentaho.org/librepository/
 *
 * (C) Copyright 2006-2008, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 * ------------
 * ZipEntryOutputStream.java
 * ------------
 */

package org.pentaho.reporting.libraries.repository.zipwriter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.zip.CRC32;
import java.util.zip.Deflater;
import java.util.zip.DeflaterOutputStream;
import java.util.zip.InflaterInputStream;
import java.util.zip.ZipEntry;

import org.pentaho.reporting.libraries.repository.LibRepositoryBoot;
import org.pentaho.reporting.libraries.repository.RepositoryUtilities;

/**
 * A fully buffered stream.
 *
 * @author Thomas Morgner
 */
public class ZipEntryOutputStream extends OutputStream
{
  private ByteArrayOutputStream outputStream;
  private DeflaterOutputStream deflaterOutputStream;
  private boolean closed;
  private ZipContentItem item;
  private CRC32 crc32;
  private long size;

  public ZipEntryOutputStream(final ZipContentItem item)
  {
    if (item == null)
    {
      throw new NullPointerException();
    }
    this.item = item;
    this.outputStream = new ByteArrayOutputStream();
    this.deflaterOutputStream = new DeflaterOutputStream
        (outputStream, new Deflater(RepositoryUtilities.getZipLevel(item)));
    this.crc32 = new CRC32();
    this.size = 0;
  }

  public void write(final int b)
      throws IOException
  {
    if (closed)
    {
      throw new IOException("Already closed");
    }
    deflaterOutputStream.write(b);
    crc32.update(b);
    size += 1;
  }

  public void write(final byte[] b, final int off, final int len)
      throws IOException
  {
    if (closed)
    {
      throw new IOException("Already closed");
    }
    deflaterOutputStream.write(b, off, len);
    crc32.update(b, off, len);
    size += len;
  }

  public void close()
      throws IOException
  {
    if (closed)
    {
      // A duplicate close is just a NO-OP as with all other output streams.
      return;
    }

    deflaterOutputStream.close();
    final byte[] data = outputStream.toByteArray();
    final ByteArrayInputStream bin = new ByteArrayInputStream(data);
    final InflaterInputStream infi = new InflaterInputStream(bin);

    final ZipRepository repository = (ZipRepository) item.getRepository();

    final String contentId = (String) item.getContentId();
    final ZipEntry zipEntry = new ZipEntry(contentId);

    final Object comment = item.getAttribute(LibRepositoryBoot.ZIP_DOMAIN, LibRepositoryBoot.ZIP_COMMENT_ATTRIBUTE);
    if (comment != null)
    {
      zipEntry.setComment(String.valueOf(comment));
    }
    final Object version = item.getAttribute(LibRepositoryBoot.REPOSITORY_DOMAIN, LibRepositoryBoot.VERSION_ATTRIBUTE);
    if (version instanceof Date)
    {
      final Date date = (Date) version;
      zipEntry.setTime(date.getTime());
    }

    final int zipMethod = RepositoryUtilities.getZipMethod(item);
    zipEntry.setCrc(crc32.getValue());
    if (zipMethod == Deflater.NO_COMPRESSION)
    {
      zipEntry.setCompressedSize(size);
      zipEntry.setSize(size);
    }
    else
    {
      zipEntry.setSize(size);
    }
    repository.writeContent(zipEntry, infi, zipMethod, RepositoryUtilities.getZipLevel(item));
    infi.close();

    closed = true;
    outputStream = null;
    deflaterOutputStream = null;
  }


  public void write(final byte[] b)
      throws IOException
  {
    if (closed)
    {
      throw new IOException("Already closed");
    }
    deflaterOutputStream.write(b);
    crc32.update(b);
    size += b.length;
  }

  public void flush()
      throws IOException
  {
    if (closed)
    {
      throw new IOException("Already closed");
    }
    deflaterOutputStream.flush();
  }
}
