/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * version 2 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <QtGui>
#include "QtCTL.h"

#include <QTextEdit>
#include <QTextStream>
#include <QCloseEvent>
#include <QFileDialog>

#include "ui_MainWindow.h"

#include <OpenCTL/Program.h>
#include <OpenCTL/Module.h>
#include <GTLCore/PixelDescription.h>
#include <GTLCore/Buffer.h>
#include <GTLCore/Type.h>

#include "QtGTL/QImageBuffer.h"

QtCTL::QtCTL()
{
  m_mainWindow = new Ui_MainWindow;
  QWidget* widget = new QWidget;
  m_mainWindow->setupUi(widget);
  setCentralWidget(widget);
  connect(m_mainWindow->pushButtonOpen, SIGNAL(released()), SLOT(open()));
  connect(m_mainWindow->pushButtonReload, SIGNAL(released()), SLOT(reload()));
  connect(m_mainWindow->pushButtonApply, SIGNAL(released()), SLOT(apply()));
  connect(m_mainWindow->pushButtonRefresh, SIGNAL(released()), SLOT(refreshImageViewer()));
}

void QtCTL::loadFile(const QString &fileName)
{
  QImage img;
  if (not img.load(fileName)) {
    QMessageBox::warning(this, tr("QtCTL"),
                          tr("Cannot read image %1..")
                          .arg(fileName));
    return;
  }
  currentImage = img;
  originalImage = img;
  refreshImageViewer();
}

QtCTL::~QtCTL()
{

}

void QtCTL::refreshImageViewer()
{
  QPixmap pm = QPixmap::fromImage(currentImage);
  m_mainWindow->imageViewer->setPixmap(pm.scaled( m_mainWindow->imageViewer->size(), Qt::KeepAspectRatio));
}


void QtCTL::open()
{
  QString fileName = QFileDialog::getOpenFileName(this);
  if (!fileName.isEmpty())
  loadFile(fileName);
}

void QtCTL::reload()
{
  currentImage = originalImage;
  refreshImageViewer();
}

void QtCTL::apply()
{
  OpenCTL::Module p("testCall");
  p.setSource( m_mainWindow->ctlCodeEditor->toPlainText().toAscii ().data());
  p.compile();
  if(p.isCompiled())
  {
    OpenCTL::Program program( "apply", &p, GTLCore::PixelDescription( GTLCore::Type::UnsignedInteger8, 4));
    if(program.initialised())
    {
      QtGTL::QImageBuffer buffer(currentImage);
      program.apply(buffer,buffer);
      refreshImageViewer();
    } else {
      QMessageBox::critical(this, tr("QtCTL"), tr("Couldn't initialized the CTL Program, check the signature of the entry function 'apply'."));
    }
  } else {
    QMessageBox::critical(this, tr("QtCTL"), tr("Compilation error: %1").arg( p.compilationErrorsMessage().c_str() ));
  }
}

#include "QtCTL.moc"
