# -*- coding: UTF-8 -*-

'''handler tests'''

# (c) 2007 Canonical Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

# TODO: test a KernelModuleHandler with an external package/repo (non-avail module)

import unittest, sys, os.path, re

from jockey.oslib import OSLib
import jockey.handlers
from jockey.xorg_driver import XorgDriverHandler

import sandbox

class TestFirmwareHandler(jockey.handlers.FirmwareHandler):

    def __init__(self, ui, free=False):
        self.destfile = os.path.join(OSLib.inst.workdir, 'lib', 'firmware',
            'test.bin')
        jockey.handlers.FirmwareHandler.__init__(self, ui,
            'mint', self.destfile, url='file:///bin/cat', free=free)

    def enable(self):
        jockey.handlers.FirmwareHandler.enable(self)

        d = os.path.dirname(self.destfile)
        if not os.path.isdir(d):
            os.makedirs(d)
        f = open(self.destfile, 'w')
        f.write(open(self.firmware_file).read())
        f.close()

        jockey.handlers.KernelModuleHandler.rebind(self.module)

    def disable(self):
        if os.path.isfile(self.destfile):
            os.unlink(self.destfile)
        jockey.handlers.KernelModuleHandler.disable(self)


class HandlerTest(unittest.TestCase):

    def setUp(self):
        '''Provide an AbstractUI instance for getting _() etc. and save/restore
        /proc/modules, module blacklist, installed packages, and xorg.conf.
        '''
        orig_argv = sys.argv
        sys.argv = ['ui-test']
        self.ui = sandbox.TestUI()
        sys.argv = orig_argv

        self.orig_installed_packages = OSLib.inst.installed_packages
        try:
            self.orig_moduleblacklist = open(
                OSLib.inst.module_blacklist_file).read()
        except IOError:
            self.orig_moduleblacklist = None

    def tearDown(self):
        if self.orig_moduleblacklist is not None:
            open(OSLib.inst.module_blacklist_file,
                'w').write(self.orig_moduleblacklist)
        else:
            try:
                os.unlink(OSLib.inst.module_blacklist_file)
            except OSError:
                pass
        OSLib.inst._load_module_blacklist()

        OSLib.inst.installed_packages = self.orig_installed_packages
        OSLib.inst._make_proc_modules()
        OSLib.inst._make_xorg_conf()
        jockey.handlers.KernelModuleHandler.read_loaded_modules()

    def test_basic(self):
        '''abstract Handler'''

        # no description, no rationale
        h = jockey.handlers.Handler(self.ui, 'shiny driver')
        self.assertEqual(h.name(), 'shiny driver')
        self.assertEqual(h.description(), None)
        self.assertEqual(h.id(), 'Handler:shiny driver')
        # implicit rationale needs free()
        self.assertRaises(NotImplementedError, h.rationale)
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)

        # description and rationale
        h = jockey.handlers.Handler(self.ui, 'shiny driver', 
            'I provide bling', rationale='I am important!')
        self.assertEqual(h.rationale(), 'I am important!')
        self.assertEqual(h.description(), 'I provide bling')

    def test_kmod_free(self):
        '''KernelModuleHandler: free, no description, used'''

        h = jockey.handlers.KernelModuleHandler(self.ui, 'vanilla')
        self.assertEqual(h.name(), sandbox.fake_modinfo['vanilla']['description'])
        self.assertEqual(h.id(), 'kmod:vanilla')
        self.assertEqual(str(h), 'kmod:vanilla([KernelModuleHandler, free, enabled] '
            'free module with available hardware, graphics card)')
        self.assertEqual(h.description(), None)
        self.assertEqual(h.rationale(), None) # free
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)

        self.assert_(h.free())
        self.assert_(h.enabled())
        self.assert_(h.used())
        self.assertEqual(h.available(), None)
        self.assert_(hasattr(h.ui_category(), 'isspace'), 
            'UI category is a string')

    def test_kmod_nonfree(self):
        '''KernelModuleHandler: nonfree, custom description, unused'''

        h = jockey.handlers.KernelModuleHandler(self.ui, 'cherry',
            description='you want me')
        self.assertEqual(h.name(), sandbox.fake_modinfo['cherry']['description'])
        self.assertEqual(h.id(), 'kmod:cherry')
        self.assertEqual(str(h), 'kmod:cherry([KernelModuleHandler, nonfree, enabled] '
            'nonfree module with nonavailable hardware, wifi)')
        self.assertEqual(h.description(), 'you want me')
        self.assert_('support the hardware' in h.rationale())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)

        self.failIf(h.free())
        self.assert_(h.enabled())
        self.failIf(h.used())
        self.assertEqual(h.available(), None)

        # enabling it should modprobe
        h.enable()
        self.assert_(h.used())

    def test_kmod_customdesc(self):
        '''KernelModuleHandler: custom name and rationale'''

        h = jockey.handlers.KernelModuleHandler(self.ui, 'cherry',
            name='fruitz', rationale='goood')
        self.assertEqual(h.name(), 'fruitz')
        self.assertEqual(h.id(), 'kmod:cherry')
        self.assertEqual(h.rationale(), 'goood')

    def test_kmod_enabling(self):
        '''KernelModuleHandler: enabling and disabling'''

        self.failIf(OSLib.inst.module_blacklisted('vanilla'))

        h = jockey.handlers.KernelModuleHandler(self.ui, 'vanilla')
        h.disable()
        self.assert_(h.changed())
        self.assert_(OSLib.inst.module_blacklisted('vanilla'))
        h.disable()
        self.assert_(h.changed())
        self.assert_(OSLib.inst.module_blacklisted('vanilla'))
        h.enable()
        self.assert_(h.changed())
        self.failIf(OSLib.inst.module_blacklisted('vanilla'))
        driver_dir = os.path.join(OSLib.inst.sys_dir, 'module', 'vanilla',
            'drivers', 'pci:vanilla')
        self.assert_(os.path.isfile(os.path.join(driver_dir, 'unbind')))
        self.assert_(os.path.isfile(os.path.join(driver_dir, 'bind')))

    def test_group(self):
        '''HandlerGroup'''

        h = jockey.handlers.HandlerGroup(self.ui, 'graphics drivers', 'gfxdrv')

        # add a free driver
        h.add(jockey.handlers.KernelModuleHandler(self.ui, 'vanilla'))
        self.assertEqual(h.name(), 'graphics drivers')
        self.assertEqual(h.id(), 'gfxdrv')
        self.assertEqual(h.description(), None)
        self.assertEqual(h.rationale(), None)
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.assert_(h.free())
        self.assert_(h.enabled())
        self.assert_(h.used())
        self.assertEqual(h.available(), None)
        self.assert_(hasattr(h.ui_category(), 'isspace'), 
            'UI category is a string')

        # add a nonfree driver
        h.add(jockey.handlers.KernelModuleHandler(self.ui, 'vanilla3d'))
        self.assertEqual(h.name(), 'graphics drivers')

        self.assert_('support the hardware' in h.rationale())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.free()) # subset nonfree -> group nonfree
        self.assert_(h.enabled()) # both modules are enabled by default
        self.assert_(h.used()) # vanilla is used by default, ORed
        self.assertEqual(h.available(), None)
        self.assert_(hasattr(h.ui_category(), 'isspace'), 
            'UI category is a string')

        self.failIf(OSLib.inst.module_blacklisted('vanilla'))
        self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(jockey.handlers.KernelModuleHandler.module_loaded('vanilla'))
        self.failIf(jockey.handlers.KernelModuleHandler.module_loaded('vanilla3d'))

        h.enable()
        self.assert_(h.enabled())
        self.assert_(h.changed())
        self.assert_(jockey.handlers.KernelModuleHandler.module_loaded('vanilla'))
        self.assert_(jockey.handlers.KernelModuleHandler.module_loaded('vanilla3d'))

        h.disable()
        self.failIf(h.enabled())
        self.assert_(h.changed())
        self.assert_(OSLib.inst.module_blacklisted('vanilla'))
        self.assert_(OSLib.inst.module_blacklisted('vanilla3d'))


    def test_driverpackage(self):
        '''DriverPackageHandler: default name and rationale'''

        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        h = jockey.handlers.DriverPackageHandler(self.ui,
            'mesa-vanilla', rationale='Get full graphics acceleration speed')

        self.assertEqual(h.name(), sandbox.fake_pkginfo['mesa-vanilla']['description'][0])
        self.assertEqual(h.id(), 'pkg:mesa-vanilla')
        self.assertEqual(h.description(), sandbox.fake_pkginfo['mesa-vanilla']['description'][1])
        self.assertEqual(h.rationale(), 'Get full graphics acceleration speed')
        self.failIf(h.free())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(OSLib.inst.package_installed('mesa-vanilla')) 

        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 

    def test_driverpackage_customdesc(self):
        '''DriverPackageHandler: custom name and rationale'''

        self.failIf(OSLib.inst.package_installed('mesa-vanilla'))
        h = jockey.handlers.DriverPackageHandler(self.ui,
            'mesa-vanilla', 'VanillaName', 'VanillaDesc')

        self.assertEqual(h.name(), 'VanillaName')
        self.assertEqual(h.id(), 'pkg:mesa-vanilla')
        self.assertEqual(h.description(), 'VanillaDesc')
        self.assert_('support the hardware' in h.rationale())
        self.assertEqual(h.free(), False)
        self.failIf(h.free())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)

    def test_driverpackage_failedinstall(self):
        '''DriverPackageHandler: package installation fails'''

        self.failIf(OSLib.inst.package_installed('mesa-vanilla'))
        h = jockey.handlers.DriverPackageHandler(self.ui,
            'mesa-vanilla', 'VanillaName', 'VanillaDesc')

        OSLib.inst.next_package_install_fails = True
        h.enable()
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        self.failIf(h.enabled())
        self.failIf(h.changed())

    def test_modulepackage(self):
        '''ModulePackageHandler'''

        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        h = jockey.handlers.ModulePackageHandler(self.ui,
            'vanilla3d', 'mesa-vanilla', rationale='Get full graphics acceleration speed')

        self.assertEqual(h.name(), sandbox.fake_modinfo['vanilla3d']['description'])
        self.assertEqual(h.id(), 'kmod:vanilla3d')
        self.assertEqual(h.description(), None)
        self.assertEqual(h.rationale(), 'Get full graphics acceleration speed')
        self.failIf(h.free())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(OSLib.inst.package_installed('mesa-vanilla')) 
        self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))

        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        self.assert_(OSLib.inst.module_blacklisted('vanilla3d'))

    def test_xorg_driver_nomodules(self):
        '''XorgDriverHandler with no default Modules xorg.conf section'''
        
        h = XorgDriverHandler(self.ui, 'vanilla3d', 'mesa-vanilla', 'v3d',
            'vanilla', extra_conf_options={'SuperSpeed': 'true'}, 
            add_modules=['glx'], remove_modules=['dri'],
            name='Vanilla accelerated graphics driver')

        orig_xorg_conf = open(OSLib.inst.xorg_conf_path).read()

        self.assertEqual(h.name(), 'Vanilla accelerated graphics driver')
        self.assertEqual(h.id(), 'kmod:vanilla3d')
        self.assertEqual(h.description(), None)
        self.assert_('support the hardware' in h.rationale())
        self.failIf(h.free())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)
        self.failIf(h.supports_composite())

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(OSLib.inst.package_installed('mesa-vanilla')) 
        self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())

        conf = open(OSLib.inst.xorg_conf_path).read()
        self.assert_(re.search('^\s*driver\s*"v3d"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*option\s*"SuperSpeed"\s*"true"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.failIf(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))

        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        self.assert_(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())

        # restores original backup
        self.assertEqual(open(OSLib.inst.xorg_conf_path).read(), orig_xorg_conf)

    def test_xorg_driver_nobackup(self):
        '''XorgDriverHandler with no xorg.conf backup'''
        
        h = XorgDriverHandler(self.ui, 'vanilla3d', 'mesa-vanilla', 'v3d',
            'vanilla', extra_conf_options={'SuperSpeed': 'true'}, 
            add_modules=['glx'], remove_modules=['dri'],
            name='Vanilla accelerated graphics driver')

        h.enable()
        conf = open(OSLib.inst.xorg_conf_path).read()
        self.assert_(re.search('^\s*driver\s*"v3d"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*option\s*"SuperSpeed"\s*"true"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.failIf(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))

        # unlink backup file
        os.unlink(os.path.join(OSLib.inst.backup_dir, 'v3d.oldconf'))

        h.disable()
        conf = open(OSLib.inst.xorg_conf_path).read()
        self.assert_(re.search('^\s*driver\s*"vanilla"\s*$', conf, re.I|re.M))
        self.failIf('SuperSpeed' in conf)
        self.failIf(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))

    def test_xorg_driver_modules(self):
        '''XorgDriverHandler with default Modules xorg.conf section'''

        # append a Module section
        f = open(OSLib.inst.xorg_conf_path, 'a')
        print >> f, '''
Section "Module"
        Load            "dri"
        SubSection "extmod"
                Option          "omit xfree86-dga"
        EndSubSection
        Load            "foo"
EndSection
'''
        f.close()

        h = XorgDriverHandler(self.ui, 'vanilla3d', 'mesa-vanilla', 'v3d',
            'vanilla', extra_conf_options={'SuperSpeed': 'true'}, 
            add_modules=['glx'], remove_modules=['dri'],
            name='Vanilla accelerated graphics driver')

        self.assertEqual(h.name(), 'Vanilla accelerated graphics driver')
        self.assertEqual(h.description(), None)
        self.assert_('support the hardware' in h.rationale())
        self.failIf(h.free())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(OSLib.inst.package_installed('mesa-vanilla')) 
        self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())

        conf = open(OSLib.inst.xorg_conf_path).read()
        self.assert_(re.search('^\s*driver\s*"v3d"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*option\s*"SuperSpeed"\s*"true"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.failIf(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"foo"\s*$', conf, re.I|re.M))

        # unlink backup file
        os.unlink(os.path.join(OSLib.inst.backup_dir, 'v3d.oldconf'))

        # should restore module dri, and drop glx
        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        self.assert_(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())

        conf = open(OSLib.inst.xorg_conf_path).read()
        self.failIf(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"foo"\s*$', conf, re.I|re.M))

    def test_xorg_driver_no_xorgconf(self):
        '''XorgDriverHandler without a default xorg.conf'''

        os.unlink(OSLib.inst.xorg_conf_path)

        h = XorgDriverHandler(self.ui, 'vanilla3d', 'mesa-vanilla', 'v3d',
            'vanilla', extra_conf_options={'SuperSpeed': 'true'}, 
            add_modules=['glx'], remove_modules=['dri'],
            name='Vanilla accelerated graphics driver')

        self.failIf(h.free())
        self.failIf(h.changed())
        self.failIf(h.can_change()) # should create an xorg.conf from scratch
        self.failIf(h.enabled())
        self.assertEqual(h.available(), None)

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(OSLib.inst.package_installed('mesa-vanilla')) 
        self.failIf(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())

        conf = open(OSLib.inst.xorg_conf_path).read()
        self.assert_(re.search('^\s*driver\s*"v3d"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*option\s*"SuperSpeed"\s*"true"\s*$', conf, re.I|re.M))
        self.assert_(re.search('^\s*load\s*"glx"\s*$', conf, re.I|re.M))
        self.failIf(re.search('^\s*load\s*"dri"\s*$', conf, re.I|re.M))

        # should remove xorg.conf again
        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(OSLib.inst.package_installed('mesa-vanilla')) 
        self.assert_(OSLib.inst.module_blacklisted('vanilla3d'))
        self.assert_(OSLib.inst.pop_reboot_flag())
        self.failIf(os.path.exists(OSLib.inst.xorg_conf_path))

    def test_firmware_handler(self):
        '''standard FirmwareHandler'''

        h = jockey.handlers.FirmwareHandler(self.ui, 'vanilla', '/foo')

        self.failIf(h.free())
        self.assertEqual(h.name(), sandbox.fake_modinfo['vanilla']['description'])
        self.assertEqual(h.id(), 'kmod:vanilla')
        self.assertEqual(str(h), 'kmod:vanilla([FirmwareHandler, nonfree, disabled] '
            'free module with available hardware, graphics card)')
        self.assertEqual(h.description(), None)
        self.assert_('proprietary firmware' in h.rationale())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)

        self.failIf(h.enabled())
        self.failIf(h.used())
        self.assertEqual(h.available(), None)

    def test_testfirmware_handler(self):
        '''TestFirmwareHandler'''

        h = TestFirmwareHandler(self.ui)
        self.assertEqual(h.name(), sandbox.fake_modinfo['mint']['description'])
        self.assertEqual(h.id(), 'kmod:mint')
        self.assertEqual(str(h), 'kmod:mint([TestFirmwareHandler, nonfree, disabled] '
            'nonfree module with available hardware, wifi)')
        self.assertEqual(h.description(), None)
        self.assert_('proprietary firmware' in h.rationale())
        self.failIf(h.changed())
        self.assertEqual(h.can_change(), None)

        self.failIf(h.free())
        self.failIf(h.enabled())
        self.failIf(h.used())
        self.assertEqual(h.available(), None)

        h.enable()
        self.assert_(h.changed())
        self.assert_(h.enabled())
        self.assert_(h.used())

        h.disable()
        self.assert_(h.changed())
        self.failIf(h.enabled())
        self.failIf(h.used())

    def test_firmware_handler_free(self):
        '''standard FirmwareHandler, forced to free'''

        h = TestFirmwareHandler(self.ui, free='1')
        self.assertEqual(h.name(), sandbox.fake_modinfo['mint']['description'])
        self.assertEqual(h.id(), 'kmod:mint')
        self.assert_(h.free())
        self.assertEqual(str(h), 'kmod:mint([TestFirmwareHandler, free, disabled] '
            'nonfree module with available hardware, wifi)')
        self.assertEqual(h.description(), None)
        self.assert_('firmware' in h.rationale())
        self.failIf('proprietary' in h.rationale())

