/* libsane-hpoj -- SANE backend for hpoj-supported multi-function peripherals */

/* Copyright (C) 2001-2003 Hewlett-Packard Company
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * is provided AS IS, WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, and
 * NON-INFRINGEMENT.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 *
 * In addition, as a special exception, Hewlett-Packard Company
 * gives permission to link the code of this program with any
 * version of the OpenSSL library which is distributed under a
 * license identical to that listed in the included LICENSE.OpenSSL
 * file, and distribute linked combinations including the two.
 * You must obey the GNU General Public License in all respects
 * for all of the code used other than OpenSSL.  If you modify
 * this file, you may extend this exception to your version of the
 * file, but you are not obligated to do so.  If you do not wish to
 * do so, delete this exception statement from your version.
 */
 
/* Original author: David Paschal */

#ifndef HPOJ_H
#define HPOJ_H

#include "sane.h"
#include "saneopts.h"
#include "ptal.h"
#include "hpojip.h"

#define LEN_BUFFER		16384
#define LEN_DEVICE_ID_STRING	4096
#define LEN_STRING_OPTION_VALUE	20
#define LEN_SCL_BUFFER		256	/* Increase if reading binary data. */
#define LEN_MODEL_RESPONSE	20
#define MAX_LIST_SIZE		20

/* Timeout-related defines (see also ptalPmlRequestSetRetry): */
#define INFINITE_TIMEOUT				((struct timeval *)0)
#define SCL_SEND_COMMAND_START_TIMEOUT			0
#define SCL_SEND_COMMAND_CONTINUE_TIMEOUT		2
#define SCL_INQUIRE_START_TIMEOUT			30
#define SCL_INQUIRE_CONTINUE_TIMEOUT			5
#define SCL_DEVICE_LOCK_TIMEOUT				0
#define SCL_PREPARE_SCAN_DEVICE_LOCK_MAX_RETRIES	4
#define SCL_PREPARE_SCAN_DEVICE_LOCK_DELAY		1
#define PML_SELECT_POLL_TIMEOUT				1
#define PML_UPLOAD_TIMEOUT				45
#define PML_START_SCAN_WAIT_ACTIVE_MAX_RETRIES		40
#define PML_START_SCAN_WAIT_ACTIVE_DELAY		1
#define MFPDTF_EARLY_READ_TIMEOUT			60
#define MFPDTF_LATER_READ_TIMEOUT			20
#define NON_MFPDTF_READ_START_TIMEOUT			60
#define NON_MFPDTF_READ_CONTINUE_TIMEOUT		2

#if 1
	#define PAD_VALUE_LINEART		0
	#define PAD_VALUE_GRAYSCALE_COLOR	-1
#else
	#define PAD_VALUE_LINEART		1
	#define PAD_VALUE_GRAYSCALE_COLOR	0x80
#endif

enum hpojOption_e {
	OPTION_FIRST=0,
	OPTION_NUM_OPTIONS=0,

	GROUP_SCAN_MODE,
	// OPTION_IMAGE_SOURCE,	/* "scanner", "fax" */
	// OPTION_PREVIEW_MODE,	/* bool */
	OPTION_SCAN_MODE,
	// OPTION_BIT_DEPTH,	/* for >8bpc, res must be 300, 600, or 1200 */
	// OPTION_BIND_RESOLUTION,
	OPTION_SCAN_RESOLUTION,
	// OPTION_X_RESOLUTION,
	// OPTION_Y_RESOLUTION,

	// GROUP_ENHANCEMENT,	/* ranges */
	// OPTION_BRIGHTNESS,	/* Will have to be simulated. */
	// OPTION_CONTRAST,	/* Will have to be simulated. */
	// OPTION_SHARPENING,

	GROUP_ADVANCED,
	OPTION_CONTRAST,
	OPTION_COMPRESSION,
	OPTION_JPEG_COMPRESSION_FACTOR,
	OPTION_BATCH_SCAN,
	OPTION_ADF_MODE,	/* For >8bpc must be flatbed (i.e. D series) */
	OPTION_DUPLEX,
	// OPTION_DUPLEX_ORIENTATION,	/* "tablet", "book" */
	// OPTION_MANUAL_CHANGE_DOCUMENT,	/* button */
	// OPTION_MANUAL_DUPLEX_CHANGE_SIDE,
	// OPTION_MANUAL_UNLOAD_DOCUMENT,
	// OPTION_DEBUG_DUMP,	/* button */

	GROUP_GEOMETRY,
	OPTION_LENGTH_MEASUREMENT,
	OPTION_TL_X,
	OPTION_TL_Y,
	OPTION_BR_X,
	OPTION_BR_Y,
	// OPTION_MIRROR_HORIZ,
	// OPTION_MIRROR_VERT,

	OPTION_LAST
};

#define STR_SCAN_MODE_LINEART	"Lineart"
#define STR_SCAN_MODE_GRAYSCALE	"Grayscale"
#define STR_SCAN_MODE_COLOR	"Color"

enum hpojScanMode_e {
	SCAN_MODE_FIRST=0,
	SCAN_MODE_LINEART=0,
	SCAN_MODE_GRAYSCALE,
	SCAN_MODE_COLOR,
	SCAN_MODE_LAST
};

#define COMPRESSION_NONE	0x01
#define COMPRESSION_MH		0x02
#define COMPRESSION_MR		0x04
#define COMPRESSION_MMR		0x08
#define COMPRESSION_JPEG	0x10

#define STR_COMPRESSION_NONE	"None"
#define STR_COMPRESSION_MH	"MH"
#define STR_COMPRESSION_MR	"MR"
#define STR_COMPRESSION_MMR	"MMR"
#define STR_COMPRESSION_JPEG	"JPEG"

#define MIN_JPEG_COMPRESSION_FACTOR	0
#define MAX_JPEG_COMPRESSION_FACTOR	100
/* To prevent "2252" asserts on OfficeJet 600 series: */
#define SAFER_JPEG_COMPRESSION_FACTOR	10

#define ADF_MODE_AUTO		0x01
#define ADF_MODE_FLATBED	0x02
#define ADF_MODE_ADF		0x04

#define STR_ADF_MODE_AUTO	"Auto"
#define STR_ADF_MODE_FLATBED	"Flatbed"
#define STR_ADF_MODE_ADF	"ADF"

#define LENGTH_MEASUREMENT_UNKNOWN		0
#define LENGTH_MEASUREMENT_UNLIMITED		1
#define LENGTH_MEASUREMENT_APPROXIMATE		2
#define LENGTH_MEASUREMENT_PADDED		3
#define LENGTH_MEASUREMENT_EXACT		4

#define STR_LENGTH_MEASUREMENT_UNKNOWN		"Unknown"
#define STR_LENGTH_MEASUREMENT_UNLIMITED	"Unlimited"
#define STR_LENGTH_MEASUREMENT_APPROXIMATE	"Approximate"
#define STR_LENGTH_MEASUREMENT_PADDED		"Padded"
#define STR_LENGTH_MEASUREMENT_EXACT		"Exact"

#define STR_UNKNOWN		"???"

#define BYTES_PER_LINE(pixelsPerLine,bitsPerPixel) \
	((((pixelsPerLine)*(bitsPerPixel))+7)/8)

#if 1
	#define GEOMETRY_OPTION_TYPE		SANE_TYPE_FIXED
	#define MILLIMETER_SHIFT_FACTOR		SANE_FIXED_SCALE_SHIFT
#else
	#define GEOMETRY_OPTION_TYPE		SANE_TYPE_INT
	#define MILLIMETER_SHIFT_FACTOR		0
#endif

#define DECIPOINTS_PER_INCH		720
#define DEVPIXELS_PER_INCH		300
#define MILLIMETERS_PER_10_INCHES	254
#define INCHES_PER_254_MILLIMETERS	10

#define INCHES_TO_MILLIMETERS(inches) \
    hpojDivideAndShift(__LINE__, \
	(inches), \
	MILLIMETERS_PER_10_INCHES, \
	INCHES_PER_254_MILLIMETERS, \
	MILLIMETER_SHIFT_FACTOR)

#define DECIPIXELS_TO_MILLIMETERS(decipixels) \
    hpojDivideAndShift(__LINE__, \
	(decipixels), \
	MILLIMETERS_PER_10_INCHES, \
	INCHES_PER_254_MILLIMETERS*hpoj->decipixelsPerInch, \
	MILLIMETER_SHIFT_FACTOR)

#define MILLIMETERS_TO_DECIPIXELS(millimeters) \
    hpojDivideAndShift(__LINE__, \
	(millimeters), \
	INCHES_PER_254_MILLIMETERS*hpoj->decipixelsPerInch, \
	MILLIMETERS_PER_10_INCHES, \
	-MILLIMETER_SHIFT_FACTOR)

#define PIXELS_TO_MILLIMETERS(pixels,pixelsPerInch) \
    hpojDivideAndShift(__LINE__, \
	(pixels), \
	MILLIMETERS_PER_10_INCHES, \
	(pixelsPerInch)*INCHES_PER_254_MILLIMETERS, \
	MILLIMETER_SHIFT_FACTOR)

#define MILLIMETERS_TO_PIXELS(millimeters,pixelsPerInch) \
    hpojDivideAndShift(__LINE__, \
	(millimeters), \
	INCHES_PER_254_MILLIMETERS*(pixelsPerInch), \
	MILLIMETERS_PER_10_INCHES, \
	-MILLIMETER_SHIFT_FACTOR)


typedef struct hpojScanner_s {
	ptalDevice_t dev;
	ptalChannel_t chan;
	ptalChannel_t chanReserve;

	SANE_Device saneDevice;	/* "vendor", "model" dynamically allocated. */
	SANE_Parameters prescanParameters;
	SANE_Parameters scanParameters;

	enum {
		SCANNER_TYPE_SCL,
		SCANNER_TYPE_PML
	} scannerType;
	int decipixelsPerInch;

	/* These are bitfields of COMPRESSION_* values. */
	int supportsScanMode[SCAN_MODE_LAST];
	SANE_String_Const scanModeList[MAX_LIST_SIZE];
	enum hpojScanMode_e currentScanMode,effectiveScanMode;

	SANE_Range resolutionRange;
	SANE_Int resolutionList[MAX_LIST_SIZE];
	SANE_Int lineartResolutionList[MAX_LIST_SIZE];	/* 300 dpi. */
	SANE_Int currentResolution,effectiveResolution;

	SANE_Range contrastRange;
	SANE_Int defaultContrast,currentContrast;

	SANE_String_Const compressionList[MAX_LIST_SIZE];
	int defaultCompression[SCAN_MODE_LAST];
	SANE_Int currentCompression;	/* One of the COMPRESSION_* values. */

	SANE_Range jpegCompressionFactorRange;
	SANE_Int defaultJpegCompressionFactor;
	SANE_Int currentJpegCompressionFactor;

	SANE_Bool currentBatchScan;
	int beforeScan;
	int alreadyPreAdvancedDocument;
	int alreadyPostAdvancedDocument;
	int noDocsConditionPending;

	int supportedAdfModes;
	SANE_String_Const adfModeList[MAX_LIST_SIZE];
	int currentAdfMode;
	int currentPageNumber;

	int supportsDuplex;
	SANE_Bool currentDuplex;
	int currentSideNumber;

	SANE_Int currentLengthMeasurement;
	SANE_String_Const lengthMeasurementList[MAX_LIST_SIZE];

	SANE_Range tlxRange,tlyRange,brxRange,bryRange;
	SANE_Fixed currentTlx,currentTly,currentBrx,currentBry;
	SANE_Fixed effectiveTlx,effectiveTly,effectiveBrx,effectiveBry;

	SANE_Option_Descriptor option[OPTION_LAST];

	ptalMfpdtf_t mfpdtf;
	IP_HANDLE hJob;
	int fdSaveCompressedData;
	int fromDenali;
	int preDenali;
	unsigned char inBuffer[LEN_BUFFER];
	int bufferOffset;
	int bufferBytesRemaining;
	int totalBytesRemaining;
	int endOfData;

	struct {
		char compat1150[LEN_MODEL_RESPONSE+1];
		char compatPost1150[LEN_MODEL_RESPONSE+1];
		int compat;
		char decipixelChar;

		int minXRes,minYRes;
		int maxXRes,maxYRes;
		int maxXExtent,maxYExtent;

		int adfCapability;
		int unloadAfterScan;

		ptalPmlObject_t objSupportedFunctions;
	} scl;

	struct {
		ptalPmlObject_t
			objScannerStatus,
			objResolutionRange,
			objUploadTimeout,
			objContrast,
			objResolution,
			objPixelDataType,
			objCompression,
			objCompressionFactor,
			objUploadError,
			objUploadState,
			objAbcThresholds,
			objSharpeningCoefficient,
			objNeutralClipThresholds,
			objToneMap,
			objCopierReduction,
			objScanToken,
			objModularHardware;

		char scanToken[PTAL_PML_MAX_VALUE_LEN];
		char zeroScanToken[PTAL_PML_MAX_VALUE_LEN];
		int lenScanToken;
		int scanTokenIsSet;

		int openFirst;
		int dontResetBeforeNextNonBatchPage;
		int startNextBatchPageEarly;
		int flatbedCapability;

		int alreadyRestarted;
		int scanDone;
		int previousUploadState;
	} pml;

} *hpojScanner_t;


#define UNDEFINED_MODEL(hpoj) (!hpoj->saneDevice.model)
#define _SET_DEFAULT_MODEL(hpoj,s,len) \
	do { \
		if (UNDEFINED_MODEL(hpoj)) { \
			hpoj->saneDevice.model=malloc(len+1); \
			memcpy((char *)hpoj->saneDevice.model,s,len); \
			((char *)hpoj->saneDevice.model)[len]=0; \
		} \
	} while(0)
#define SET_DEFAULT_MODEL(hpoj,s) _SET_DEFAULT_MODEL(hpoj,s,strlen(s))

#define ADD_XFORM(x) \
	do { \
		pXform->eXform=x; \
		PTAL_LOG_DEBUG("hpoj:%s: sane_hpoj_start: added xform=%d.\n", \
			hpoj->saneDevice.name,pXform->eXform); \
		pXform++; \
	} while(0)

#define FIX_GEOMETRY(low,high,min,max) \
	do { \
		if (high<low) high=low; \
		if (high==low) { \
			if (high==max) { \
				low--; \
			} else { \
				high++; \
			} \
		} \
	} while(0)


/*                    10240              + (a)*32 - 3040    + (b) - 62
 *                    7138               + (a)*32           + (b)
 *                    0x1BE2             + (a)*0x20         + (b)      */
#define SCL_CMD(a,b) ( (('*'-'!'+1)<<10) + (((a)-'`'+1)<<5) + ((b)-'@'+1) )
#define SCL_CMD_PUNC(x)    ((((x)>>10)&0x1F)+'!'-1)
#define SCL_CMD_LETTER1(x) ((((x)>> 5)&0x1F)+'`'-1)
#define SCL_CMD_LETTER2(x) (( (x)     &0x1F)+'@'-1)

#define SCL_af					(hpoj->scl.decipixelChar)
#define SCL_CHAR_DECIPOINTS			'a'
#define SCL_CHAR_DEVPIXELS			'f'

#define SCL_CMD_RESET				SCL_CMD('z','E') /* No param! */
#define SCL_CMD_CLEAR_ERROR_STACK		SCL_CMD('o','E') /* No param! */
#define SCL_CMD_INQUIRE_PRESENT_VALUE		SCL_CMD('s','R')
#define SCL_CMD_INQUIRE_MINIMUM_VALUE		SCL_CMD('s','L')
#define SCL_CMD_INQUIRE_MAXIMUM_VALUE		SCL_CMD('s','H')
#define SCL_CMD_INQUIRE_DEVICE_PARAMETER	SCL_CMD('s','E')

#define SCL_CMD_SET_OUTPUT_DATA_TYPE		SCL_CMD('a','T')
#define SCL_CMD_SET_DATA_WIDTH			SCL_CMD('a','G')
#define SCL_CMD_SET_MFPDTF			SCL_CMD('m','S') /* No inq! */
#define SCL_CMD_SET_COMPRESSION			SCL_CMD('a','C') /* No inq! */
#define SCL_CMD_SET_JPEG_COMPRESSION_FACTOR	SCL_CMD('m','Q') /* No inq! */
#define SCL_CMD_SET_X_RESOLUTION		SCL_CMD('a','R')
#define SCL_CMD_SET_Y_RESOLUTION		SCL_CMD('a','S')
#define SCL_CMD_SET_X_POSITION			SCL_CMD(SCL_af,'X')
#define SCL_CMD_SET_Y_POSITION			SCL_CMD(SCL_af,'Y')
#define SCL_CMD_SET_X_EXTENT			SCL_CMD(SCL_af,'P')
#define SCL_CMD_SET_Y_EXTENT			SCL_CMD(SCL_af,'Q')
#define SCL_CMD_SET_DOWNLOAD_TYPE		SCL_CMD('a','D')
#define SCL_CMD_DOWNLOAD_BINARY_DATA		SCL_CMD('a','W')
#define SCL_CMD_SET_CCD_RESOLUTION		SCL_CMD('m','R')
#define SCL_CMD_CHANGE_DOCUMENT			SCL_CMD('u','X')
#define SCL_CMD_UNLOAD_DOCUMENT			SCL_CMD('u','U')
#define SCL_CMD_CHANGE_DOCUMENT_BACKGROUND	SCL_CMD('u','Y')
#define SCL_CMD_SCAN_WINDOW			SCL_CMD('f','S')
#define SCL_CMD_SET_DEVICE_LOCK			SCL_CMD('f','H')
#define SCL_CMD_SET_DEVICE_LOCK_TIMEOUT		SCL_CMD('f','I')

#if 0
#define SCL_CMD_SET_PRESCAN			SCL_CMD('m','B')
#define SCL_CMD_SET_NUMBER_OF_IMAGES_FOUND	SCL_CMD('m','P')
#define SCL_CMD_SET_SHARPENING			SCL_CMD('a','N')
#endif

/* Pseudo-commands for inquiring flatbed- and ADF-specific min/max values: */
#define SCL_PSEUDO_FLATBED_X_RESOLUTION		(SCL_CMD_SET_X_RESOLUTION+1000)
#define SCL_PSEUDO_FLATBED_Y_RESOLUTION		(SCL_CMD_SET_Y_RESOLUTION+1000)
#define SCL_PSEUDO_FLATBED_Y_EXTENT		(SCL_CMD_SET_Y_EXTENT    +1000)
#define SCL_PSEUDO_ADF_X_RESOLUTION		(SCL_CMD_SET_X_RESOLUTION+2000)
#define SCL_PSEUDO_ADF_Y_RESOLUTION		(SCL_CMD_SET_Y_RESOLUTION+2000)
#define SCL_PSEUDO_ADF_Y_EXTENT			(SCL_CMD_SET_Y_EXTENT    +2000)

#define SCL_DATA_TYPE_LINEART		0
#define SCL_DATA_TYPE_GRAYSCALE		4
#define SCL_DATA_TYPE_COLOR		5

#define SCL_DATA_WIDTH_LINEART		1
#define SCL_DATA_WIDTH_GRAYSCALE	8	/* or 10, 12, 14, 16? */
#define SCL_DATA_WIDTH_COLOR		24	/* or 30, 36, 42, 48? */

#define SCL_MFPDTF_OFF			0
#define SCL_MFPDTF_ON			2

#define SCL_COMPRESSION_NONE		0
#define SCL_COMPRESSION_JPEG		2

#define SCL_MIN_Y_RES_1150		50	/* 42 is absolute minimum. */
#define SCL_MAX_RES_1150_1170		300

#define SCL_DOWNLOAD_TYPE_COLORMAP	15

#define SCL_DEVICE_LOCK_RELEASED	0
#define SCL_DEVICE_LOCK_SET		1
#define SCL_DEVICE_LOCK_TIMEOUT		0

#define SCL_CHANGE_DOC_SIMPLEX		0
#define SCL_CHANGE_DOC_DUPLEX		2
#define SCL_CHANGE_DOC_DUPLEX_SIDE	12


#define SCL_INQ_HP_MODEL_11			18
#define SCL_INQ_HP_MODEL_12			19
#define SCL_INQ_ADF_FEED_STATUS			23
#define SCL_INQ_ADF_CAPABILITY			24
#define SCL_INQ_ADF_DOCUMENT_LOADED		25
#define SCL_INQ_ADF_READY_TO_UNLOAD		27
#define SCL_INQ_MAX_ERROR_STACK			256	/* always 1 */
#define SCL_INQ_CURRENT_ERROR_STACK		257	/* 0 or 1 errors */
#define SCL_INQ_CURRENT_ERROR			259	/* error number */
#define SCL_INQ_SESSION_ID			505
#define SCL_INQ_BULB_WARM_UP_STATUS		506
#define SCL_INQ_PIXELS_PER_SCAN_LINE		1024
#define SCL_INQ_BYTES_PER_SCAN_LINE		1025
#define SCL_INQ_NUMBER_OF_SCAN_LINES		1026
#define SCL_INQ_ADF_READY_TO_LOAD		1027
#define SCL_INQ_DEVICE_PIXELS_PER_INCH		1028	/* 300 */
#if 0
#define SCL_INQ_NATIVE_OPTICAL_RESOLUTION	1029
#endif

#define SCL_ADF_FEED_STATUS_OK			0
#define SCL_ADF_FEED_STATUS_BUSY		1000
#define SCL_ADF_FEED_STATUS_PAPER_JAM		1024
#define SCL_ADF_FEED_STATUS_ORIGINAL_ON_GLASS	1027
#define SCL_ADF_FEED_STATUS_PORTRAIT_FEED	1028


#define SCL_ERROR_COMMAND_FORMAT_ERROR		0
#define SCL_ERROR_UNRECOGNIZED_COMMAND		1
#define SCL_ERROR_PARAMETER_ERROR		2
#define SCL_ERROR_ILLEGAL_WINDOW		3
#define SCL_ERROR_SCALING_ERROR			4
#define SCL_ERROR_DITHER_ID_ERROR		5
#define SCL_ERROR_TONE_MAP_ID_ERROR		6
#define SCL_ERROR_LAMP_ERROR			7
#define SCL_ERROR_MATRIX_ID_ERROR		8
#define SCL_ERROR_CAL_STRIP_PARAM_ERROR		9
#define SCL_ERROR_GROSS_CALIBRATION_ERROR	10
#define SCL_ERROR_NO_MEMORY			500
#define SCL_ERROR_SCANNER_HEAD_LOCKED		501
#define SCL_ERROR_CANCELLED			502
#define SCL_ERROR_PEN_DOOR_OPEN			503
#define SCL_ERROR_ADF_PAPER_JAM			1024
#define SCL_ERROR_HOME_POSITION_MISSING		1025
#define SCL_ERROR_PAPER_NOT_LOADED		1026
#define SCL_ERROR_ORIGINAL_ON_GLASS		1027

#define SCL_COMPAT_1150			0x0001	/* model 11 "5300A", 12 null */
#define SCL_COMPAT_1170			0x0002	/* model 12 "5400A" */
#define SCL_COMPAT_R_SERIES		0x0004	/* model 12 "5500A" */
#define SCL_COMPAT_G_SERIES		0x0008	/* model 12 "5600A" */
#define SCL_COMPAT_K_SERIES		0x0010	/* model 12 "5700A" */
#define SCL_COMPAT_D_SERIES		0x0020	/* model 12 "5800A" */
#define SCL_COMPAT_6100_SERIES		0x0040	/* model 12 "5900A" */
#define SCL_COMPAT_OFFICEJET		0x1000	/* model 11 not null */
#define SCL_COMPAT_POST_1150		0x2000	/* model 12 not null */


#define PML_MAX_WIDTH_INCHES		9
#if 1
	#define PML_MAX_HEIGHT_INCHES	15
#else
	#define PML_MAX_HEIGHT_INCHES	4
#endif

#define PML_SCANNER_STATUS_UNKNOWN_ERROR	0x01
#define PML_SCANNER_STATUS_INVALID_MEDIA_SIZE	0x02
#define PML_SCANNER_STATUS_FEEDER_OPEN		0x04
#define PML_SCANNER_STATUS_FEEDER_JAM		0x08
#define PML_SCANNER_STATUS_FEEDER_EMPTY		0x10

#define PML_DATA_TYPE_LINEART		1
#define PML_DATA_TYPE_GRAYSCALE		8
#define PML_DATA_TYPE_COLOR		24

struct PmlResolution {
	unsigned char x[4];
	unsigned char y[4];
} __attribute__((packed));

#define PML_CONTRAST_MIN		0
#define PML_CONTRAST_MAX		100
#define PML_CONTRAST_DEFAULT		50

#define PML_COMPRESSION_NONE		1
#define PML_COMPRESSION_DEFAULT		2
#define PML_COMPRESSION_MH		3
#define PML_COMPRESSION_MR		4
#define PML_COMPRESSION_MMR		5
#define PML_COMPRESSION_JPEG		6

#define PML_MODHW_ADF			1

#define PML_UPLOAD_STATE_IDLE		1
#define PML_UPLOAD_STATE_START		2
#define PML_UPLOAD_STATE_ACTIVE		3
#define PML_UPLOAD_STATE_ABORTED	4
#define PML_UPLOAD_STATE_DONE		5
#define PML_UPLOAD_STATE_NEWPAGE	6

#define PML_UPLOAD_ERROR_SCANNER_JAM		207
#define PML_UPLOAD_ERROR_MLC_CHANNEL_CLOSED	208
#define PML_UPLOAD_ERROR_STOPPED_BY_HOST	209
#define PML_UPLOAD_ERROR_STOP_KEY_PRESSED	210
#define PML_UPLOAD_ERROR_NO_DOC_IN_ADF		211
#if 0
#define PML_UPLOAD_ERROR_JOB_SCHEDULE		212
#endif
#define PML_UPLOAD_ERROR_COVER_OPEN		213
#define PML_UPLOAD_ERROR_DOC_LOADED		214
#define PML_UPLOAD_ERROR_DEVICE_BUSY		216	/* What is 215? */

#define PML_SUPPFUNC_DUPLEX		0x00000008

#endif
