/*
 * Copyright (C) 2010 Neil Jagdish Patel
 * Copyright(C) 2010 Canonical Ltd.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 * Authored by Ken VaDine <ken.vandine@canonical.com>
 */

namespace GwibberGtk
{
    public static const int MAX_MESSAGE_LENGTH = 140;

    public class Entry : Gtk.Box
    {
        public Gwibber.Service service;

        public InputTextView text_view;
        private AccountTargetBar target_bar;

        public Entry ()
        {
            Object ();
        }

        construct
        {
            set_orientation (Gtk.Orientation.VERTICAL);
            service = new Gwibber.Service ();

            text_view = new InputTextView (service);
            target_bar = new AccountTargetBar ();
            add_with_properties (text_view,"expand", true);
            add_with_properties (target_bar,"expand", false);

            text_view.show ();
            target_bar.show ();
            target_bar.send.clicked.connect(() => {
                text_view.submit ();});
            // Stick a couple pixels at the bottom to keep the buttons from
            // hitting the edge
            var spacer = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
            add_with_properties (spacer, "padding", 2, "expand", false);

            text_view.get_buffer ().changed.connect (() => 
            {
              target_bar.set_counter(text_view.get_buffer ().get_char_count ());
            });
        }
    }

    public class SimpleEntry : Gtk.Box
    {
        public Gwibber.Service service;

        private InputTextView text_view;
        private SendBar send_bar;

        public SimpleEntry ()
        {
            Object ();
        }

        construct
        {
            service = new Gwibber.Service ();
            
            text_view = new InputTextView (service);
            send_bar = new SendBar (service);
            add (text_view);
            add_with_properties(send_bar,"expand", false);
            text_view.show ();
            send_bar.show ();
            send_bar.send.clicked.connect(() => {
                text_view.submit ();});
        }


    }

    public class InputTextView : Gtk.TextView
    {
        public Gwibber.Service service;
        public Gwibber.Connection conn_service;
        public Gwibber.URLShorten urlshorten;

        public Gdk.Color fg_color;
        public Gdk.Color error_color;

        bool last_was_shortened = false;
        private string _mid = null;
        private string _action = null;

        public InputTextView (Gwibber.Service service)
        {
            Object (border_width:5,
                    accepts_tab:true,
                    editable:true,
                    cursor_visible:true,
                    wrap_mode:Gtk.WrapMode.WORD_CHAR,
                    left_margin:2,
                    right_margin:2,
                    pixels_above_lines:2,
                    pixels_below_lines:2);
        }

        construct
        {
            service = new Gwibber.Service ();
            conn_service = new Gwibber.Connection ();
            urlshorten = new Gwibber.URLShorten ();

            unowned Gtk.BindingSet binding_set;
            binding_set = Gtk.BindingSet.by_class (typeof (InputTextView).class_ref ());
            Gtk.BindingEntry.add_signal (binding_set, Gdk.keyval_from_name ("Return"), 0, "submit", 0);
            Gtk.BindingEntry.add_signal (binding_set, Gdk.keyval_from_name ("KP_Enter"), 0, "submit", 0);
            Gtk.BindingEntry.add_signal (binding_set, Gdk.keyval_from_name ("Escape"), 0, "clear", 0);
            fg_color = get_style ().text[Gtk.StateType.NORMAL];

            Gdk.Color.parse ("indianred", out error_color);

            get_buffer ().changed.connect (on_text_changed);
            get_buffer ().insert_text.connect (on_text_inserted);

            set_sensitive (conn_service.is_connected ());
            conn_service.connection_changed.connect((source) => {
                    on_connection_changed (source);});

#if HAVE_GTKSPELL
            try
            {
              var spell = new Gtk.Spell.attach(this, null);
            }
            catch (Error e)
            {
            }
#endif
        }

        public void reset () {
          Timeout.add(100, () => {
            mid = null;
            action = null;
            clear ();
            return false;
          });
        }

        public string? mid {
            get { return _mid; }
            set {
                if (_mid != value)
                {
                    _mid = value;
                }
            }
        }

        public string? action {
            get { return _action; }
            set {
                if (_action != value)
                {
                    _action = value;
                }
            }
        }

        [Signal (action=true)]
        public virtual signal void clear() {
            buffer.set_text("");
	}

        [Signal (action=true)]
        public virtual signal void submit () {
            string msg = buffer.text;
            if (msg.length < 1) {
              debug ("SEND FAILED, empty post");
              return;
            }
            if ((mid != null) && (action != null)) {
              debug ("Sending %s for %s", action, mid);
              service.send_message(msg, mid, action, null);
              mid = null;
              action = null;
            } else {
              service.send_message(msg, null, null, null);
            }
            debug ("(submit) Message: %s posted", msg);
            reset();
        }

        private void on_connection_changed (bool is_connected)
	{
		set_sensitive(is_connected);
	}

        private void on_text_changed ()
        {
            var chars = get_buffer ().get_char_count ();
            modify_fg (Gtk.StateType.NORMAL,
                        chars > MAX_MESSAGE_LENGTH ? error_color : fg_color);
        }

        private void on_text_inserted (ref Gtk.TextIter iter, string text, int len)
        {
            if (last_was_shortened == false
                && len > 30
                && text != null
                && text[0:4] == "http")
            {
                var buf = get_buffer ();
                Signal.stop_emission_by_name (buf, "insert-text") ;
                var shrt = urlshorten.shorten (text);
                last_was_shortened = true;
                buf.insert (ref iter, shrt, -1);
            }
            else
            {
                last_was_shortened = false;
            }
        }


    }

    public class AccountToggleButton : Gtk.ToggleButton
    {
        public string color {  get; construct;}
        public AccountToggleButton (string color)
        {
            Object (color:color);
        }

        construct
        {
          set_size_request (24, 24);
        }

        public override bool draw (Cairo.Context context) 
        {
            double factor = 0.3;

            if (get_active())
            {
                context.translate(-1, -1);
                factor = 1.0;
            } else if (get_state () == 2) {
                factor = 0.3;
            } else if (get_state () == 1) {
                factor = 0.5;
            }

            context.push_group ();

            Gtk.Allocation a;
            get_allocation (out a);
            
            Gdk.RGBA c = Gdk.RGBA ();
            c.red = 1.0f;
            c.green = 1.0f;
            c.blue = 1.0f;
            c.alpha = 1.0f;

            c.parse (color);
            context.set_source_rgba (c.red, c.green, c.blue, 1.0f);

            context.rectangle (a.width - 2, a.height - 6, 2, 6);
            context.rectangle (a.width - 6, a.height - 2, 6, 2);
            context.fill ();
            
            propagate_draw (get_child (), context);
                        
            context.pop_group_to_source ();
            context.paint_with_alpha (factor);

            return true;
        }
    }

    public class AccountTargetBar : Gtk.Box
    {
        public Gwibber.Accounts accounts_service;
        public Gwibber.Connection conn_service;
        public Gtk.Button send;
        public Gtk.Label count;
	public Gee.HashMap<string,AccountToggleButton> accounts_buttons_map;
        private Gtk.Box abox;
        private string _selected = null;

        public AccountTargetBar ()
        {
            Object (spacing:0);
        }

        public string selected {
          get { return _selected; }
          set
            {
              if (_selected != value);
              {
                _selected = value;
                if (_selected != null)
                {
                  foreach (var k in accounts_buttons_map.keys)
                  {
                    if (_selected != k)
                      accounts_buttons_map[k].hide ();
                    else
                    {
                      accounts_buttons_map[k].active = true;
                      accounts_buttons_map[k].sensitive = false;
                    }
                    accounts_buttons_map[_selected].show_all ();
                  }
                } else 
                {
                  foreach (var k in accounts_buttons_map.keys)
                    accounts_buttons_map[k].sensitive = true;
                  show_all ();
                }
              }
            }
        }

        construct
        {
            var icon_theme = Gtk.IconTheme.get_default ();
            icon_theme.prepend_search_path (Config.PKGDATADIR + "/ui/icons");
            // Prepend local icon path if running from a source checkout
            var local_icon_path = GLib.Path.build_path (Path.DIR_SEPARATOR_S, Environment.get_current_dir (), "data/icons");
            if (GLib.FileUtils.test (local_icon_path, GLib.FileTest.IS_DIR))
              icon_theme.prepend_search_path (local_icon_path);
            local_icon_path = GLib.Path.build_path (Path.DIR_SEPARATOR_S, Environment.get_current_dir (), "../data/icons");
            if (GLib.FileUtils.test (local_icon_path, GLib.FileTest.IS_DIR))
              icon_theme.prepend_search_path (local_icon_path);

            accounts_service = new Gwibber.Accounts();
            conn_service = new Gwibber.Connection ();
            // Add buttons to button area at the bottom
            var box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 12);

            var send_with_label = new Gtk.Label(_("Send with:"));
            //box.pack_start(send_with_label, false, false, 0);

            accounts_buttons_map = new Gee.HashMap<string,AccountToggleButton> ();

            var accounts_list = accounts_service.list ();
            
            abox = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 6);
            abox.set_homogeneous (true);
            box.pack_start (abox, false, false, 0);

            for (int i = 0; i < accounts_list.length(); i++) {
                Gwibber.Account account = accounts_list.nth_data(i);
                if (account == null) {
                    continue;
                }

                // If there is no send_enabled key, don't display an icon
                if (account.send_enabled == null) {
                    continue;
                }

                var account_button = create_button (account);

                accounts_buttons_map[account.id] = account_button;
            }


            send = new Gtk.Button.with_label(_("Send"));
            box.pack_end(send, false, false, 0);

            count = new Gtk.Label ("");
            count.set_markup ("<b>" + MAX_MESSAGE_LENGTH.to_string () + "</b>");
            box.pack_end(count, false, false, 0);
            pack_start (box, true, true, 0);
            send.set_sensitive (conn_service.is_connected ());
            conn_service.connection_changed.connect((source) => {
                    on_connection_changed (source);});
            accounts_service.created.connect((source) => {
                account_created(accounts_buttons_map, source);});
        }

        private AccountToggleButton create_button (Gwibber.Account account)
        { 
            Gtk.Image icon_service = new Gtk.Image.from_icon_name(account.service, Gtk.IconSize.MENU);
  
            icon_service.show();
            AccountToggleButton account_button = new AccountToggleButton(account.color);
            account_button.set_active((account.send_enabled == "1")?true:false);
            if (account.send_enabled != "1") {
                account_button.tooltip_text = (account.service + " (" + account.username + ") - " + _("Disabled"));
            } else {
                account_button.tooltip_text = (account.service + " (" + account.username + ")");
            }
            account_button.set_image(icon_service);

            account_button.clicked.connect((source) => {
                on_account_toggled(account_button, account);});

            account.updated.connect(() => {
                account_updated(accounts_buttons_map, account);});
            //account.notify["send_enabled"].connect (()=> {
            //    account_updated(accounts_buttons_map, account);});

            account.deleted.connect(() => {
                account_deleted(accounts_buttons_map, account.id);});

            abox.pack_start(account_button, false, false, 0);
            account_button.show_all ();
            return account_button;
        }

        public void set_counter (int chars)
        {
            if (chars > MAX_MESSAGE_LENGTH)
            {
              count.set_markup ("<b> -" + (chars - MAX_MESSAGE_LENGTH).to_string () + "</b>");
            } else
            {
              count.set_markup ("<b>" + (MAX_MESSAGE_LENGTH - chars).to_string () + "</b>");
            }
        }

        public void on_account_toggled(AccountToggleButton account_button, Gwibber.Account account) {

	    if (((account.send_enabled == "1")?true:false) !=  (account_button.get_active()))
            {
                account.toggle_send_enabled();
            }
        }

        private void account_updated(Gee.HashMap<string,AccountToggleButton> accounts_buttons_map, Gwibber.Account account) 
        {
            if (!accounts_buttons_map.has_key(account.id))
                return;
            AccountToggleButton account_button = accounts_buttons_map[account.id];
            if (account.send_enabled != "1") {
                account_button.tooltip_text = (account.service + " (" + account.username + ") - " + _("Disabled"));
            } else {
                account_button.tooltip_text = (account.service + " (" + account.username + ")");
            }
	    if (((account.send_enabled == "1")?true:false) !=  (account_button.get_active())) {
                account_button.set_active((account.send_enabled == "1")?true:false);
            }
        }

        private void account_created(Gee.HashMap<string,AccountToggleButton> accounts_buttons_map, Gwibber.Account account)
        {
            if (accounts_buttons_map.has_key(account.id))
            {
                account_updated (accounts_buttons_map, account);
                return;
            }
            AccountToggleButton account_button = create_button (account);
            accounts_buttons_map[account.id] = account_button;
        }

        private void account_deleted(Gee.HashMap<string,AccountToggleButton> accounts_buttons_map, string id)
        {
            if (!accounts_buttons_map.has_key(id))
                return;
            AccountToggleButton account_button = accounts_buttons_map[id];
            account_button.destroy ();
            accounts_buttons_map.unset(id, out account_button);
        }


        private void on_connection_changed (bool is_connected) {
                send.set_sensitive(is_connected);
        } 
    }

    public class SendBar : Gtk.Box
    {
        public Gwibber.Service service { get; construct; }
        public Gtk.Button send;
        public SendBar (Gwibber.Service service)
        {
            Object (service:service,
                    spacing:5);
        }

        construct
        {
            // Add buttons to button area at the bottom
            var box = new Gtk.Box (Gtk.Orientation.HORIZONTAL, 0);
            send = new Gtk.Button.with_label(_("Send"));
            box.pack_end(send, false, false, 0);
            add(box);
        }
    }
}
