from gwibber.microblog import network, util
from gwibber.microblog.util import resources
from gwibber.microblog.util.auth import Authentication

import json, re
from gettext import lgettext as _

import logging
logger = logging.getLogger("FourSquare")
logger.debug("Initializing.")

import json, urllib2

PROTOCOL_INFO = {
  "name": "Foursquare",
  "version": "5.0",
  
  "config": [
    "private:secret_token",
    "access_token",
    "receive_enabled",
    "username",
    "color",
  ],

  "authtype": "oauth2",
  "color": "#990099",

  "features": [
    "receive",
  ],

  "default_streams": [
    "receive",
  ],
}

URL_PREFIX = "https://api.foursquare.com/v2"

class Client:
  """Query Foursquare and converts data.
  
  The Client class is responsible for querying Foursquare and turning the data obtained
  into data that Gwibber can understand. Foursquare uses a version of OAuth for security.
  
  Tokens have already been obtained when the account was set up in Gwibber and are used to 
  authenticate when getting data.
  
  """
  def __init__(self, acct):
    self.service = util.getbus("Service")
    self.account = acct
    self._loop = None

  def _retrieve_user_id(self):
    #Make a request with our new token for the user's own data
    url = "https://api.foursquare.com/v2/users/self?oauth_token=" + self.account["access_token"]
    data = json.load(urllib2.urlopen(url))
    fullname = ""
    if isinstance(data, dict):
      if data["response"]["user"].has_key("firstName"):
         fullname += data["response"]["user"]["firstName"] + " "
      if data["response"]["user"].has_key("lastName"):
        fullname += data["response"]["user"]["lastName"]
      self.account["username"] = fullname.encode("utf-8")
      self.account["user_id"] = data["response"]["user"]["id"]
      self.account["uid"] = self.account["user_id"]
    else:
      logger.error("Couldn't get user ID")

  def _login(self):
    old_token = self.account.get("access_token", None)
    with self.account.login_lock:
      # Perform the login only if it wasn't already performed by another thread
      # while we were waiting to the lock
      if self.account.get("access_token", None) == old_token:
        self._locked_login(old_token)

    return "access_token" in self.account and \
        self.account["access_token"] != old_token

  def _locked_login(self, old_token):
    logger.debug("Re-authenticating" if old_token else "Logging in")

    auth = Authentication(self.account, logger)
    reply = auth.login()
    if reply and reply.has_key("AccessToken"):
      self.account["access_token"] = reply["AccessToken"]
      self._retrieve_user_id()
      logger.debug("User id is: %s" % self.account["uid"])
    else:
      logger.error("Didn't find token in session: %s", (reply,))

  def _message(self, data):
    """Parses messages into Gwibber compatible forms.
    
    Arguments: 
      data -- A data object obtained from Foursquare containing a complete checkin
    
    Returns: 
      m -- A data object compatible with inserting into the Gwibber database for that checkin
    
    """
    m = {}; 
    m["mid"] = str(data["id"])
    m["service"] = "foursquare"
    m["account"] = self.account["id"]
    m["time"] = data["createdAt"]

    shouttext = ""
    text = ""
    
    if data.has_key("shout"):
      shout = data["shout"]
      shout = shout.replace('& ', '&amp; ')
        
    if data.has_key("venue"):
      venuename = data["venue"]["name"]
      venuename = venuename.replace('& ', '&amp; ')
    
      if data.has_key("shout"):
        shouttext += shout + "\n\n"
        text += shout + "\n"
      if data["venue"].has_key("id"):
        m["url"] = "https://foursquare.com/venue/%s" % data["venue"]["id"]
      else:
        m["url"] = "https://foursquare.com"
      shouttext += "Checked in at <a href='" + m["url"] + "'>" + venuename + "</a>"
      text += "Checked in at " + venuename
      if data["venue"]["location"].has_key("address"):
        shouttext += ", " + data["venue"]["location"]["address"]
        text += ", " + data["venue"]["location"]["address"]
      if data["venue"]["location"].has_key("crossstreet"):
        shouttext += " and " + data["venue"]["location"]["crossstreet"]
        text += " and " + data["venue"]["location"]["crossstreet"]
      if data["venue"]["location"].has_key("city"):
        shouttext += ", " + data["venue"]["location"]["city"]
        text += ", " + data["venue"]["location"]["city"]
      if data["venue"]["location"].has_key("state"):
        shouttext += ", " + data["venue"]["location"]["state"]
        text += ", " + data["venue"]["location"]["state"]
      if data.has_key("event"):
        if data["event"].has_key("name"):
          shouttext += " for " + data["event"]["name"]
    else:
      if data.has_key("shout"):
          shouttext += shout + "\n\n"
          text += shout + "\n"
      else:
          text= "Checked in off the grid"
          shouttext= "Checked in off the grid"
    
    m["text"] = text
    m["content"] = shouttext
    m["html"] = shouttext
    
    m["sender"] = {}
    m["sender"]["id"] = data["user"]["id"]
    m["sender"]["image"] = data["user"]["photo"]
    m["sender"]["url"] = "https://www.foursquare.com/user/" + data["user"]["id"]
    if data["user"]["relationship"] == "self": 
        m["sender"]["is_me"] = True 
    else: 
        m["sender"]["is_me"] = False
    fullname = ""
    if data["user"].has_key("firstName"):
        fullname += data["user"]["firstName"] + " "
    if data["user"].has_key("lastName"):
        fullname += data["user"]["lastName"]

    if data.has_key("photos"):
      if data["photos"]["count"] > 0:
        m["photo"] = {}
        m["photo"]["url"] = ""
        m["photo"]["picture"] = data["photos"]["items"][0]["url"] 
        m["photo"]["name"] = ""
        m["type"] = "photo"

    if data.has_key("likes"):
      if data["likes"]["count"] > 0:
        m["likes"]["count"] = data["likes"]["count"]

    m["sender"]["name"] = fullname
    m["sender"]["nick"] = fullname

    if data.has_key("source"):
        m["source"] = "<a href='" + data["source"]["url"] + "'>" + data["source"]["name"] + "</a>"
    else:
        m["source"] = "<a href='https://foursquare.com/'>Foursquare</a>"
    
    if data.has_key("comments"):
      if data["comments"]["count"] > 0:

        m["comments"] = []
        comments = self._get_comments(data["id"])
        for comment in comments:
          # Get the commenter's name
          fullname = ""
          if comment["user"].has_key("firstName"):
            fullname += comment["user"]["firstName"] + " "
          if comment["user"].has_key("lastName"):
            fullname += comment["user"]["lastName"]
         
          # Create a sender
          sender = {
            "name" : fullname,
            "id"   : comment["user"]["id"],
            "is_me": False,
            "image": comment["user"]["photo"],
            "url"  : "https://www.foursquare.com/user/" + comment["user"]["id"]
          }
          # Create a comment
          m["comments"].append({
              "text"  : comment["text"],
              "time"  : comment["createdAt"],
              "sender": sender,
          })

    return m

  def _check_error(self, data):
    """Checks to ensure the data obtained by Foursquare is in the correct form.
    
    If it's not in the correct form, an error is logged in gwibber.log
    
    Arguments:
    data -- A data structure obtained from Foursquare
    
    Returns:
    True if data is valid (is a dictionary and contains a 'recent' parameter).
    If the data is not valid, then return False.
    
    """
    if isinstance(data, dict) and "recent" in data:
      return True
    else:
      logger.error("Foursquare error %s", data)
      return False
  
  def _get_comments(self, checkin_id):
    """Gets comments on a particular check in ID.
    
    Arguments: 
    checkin_id -- The checkin id of the checkin
    
    Returns: 
    A comment object
    
    """ 
    url = "/".join((URL_PREFIX, "checkins", checkin_id))
    url = url + "?oauth_token=" + self.token
    data = network.Download(url, None, False).get_json()["response"]
    return data["checkin"]["comments"]["items"]

  def _get(self, path, parse="message", post=False, single=False, **args):
    """Establishes a connection with Foursquare and gets the data requested.
    
    Arguments: 
      path -- The end of the URL to look up on Foursquare
      parse -- The function to use to parse the data returned (message by default)
      post -- True if using POST, for example the send operation. False if using GET, most operations other than send. (False by default)
      single -- True if a single checkin is requested, False if multiple (False by default)
      **args -- Arguments to be added to the URL when accessed
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by the parse function.
    
    """
    if "access_token" not in self.account and not self._login():
      logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Authentication failed"), "Auth needs updating")
      logger.error("%s", logstr)
      return [{"error": {"type": "auth", "account": self.account, "message": _("Authentication failed, please re-authorize")}}]

    url = "/".join((URL_PREFIX, path))
    
    url = url + "?oauth_token=" + self.account["access_token"]

    data = network.Download(url, None, post).get_json()["response"]

    resources.dump(self.account["service"], self.account["id"], data)

    if isinstance(data, dict) and data.get("errors", 0):
      if "authenticate" in data["errors"][0]["message"]:
        # Try again, if we get a new token
        if self._login():
          logger.debug("Authentication error, logging in again")
          return self._get(path, parse, post, single, args)
        else:
          logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Authentication failed"), data["errors"][0]["message"])
          logger.error("%s", logstr)
          return [{"error": {"type": "auth", "account": self.account, "message": data["errors"][0]["message"]}}]
      else:
        for error in data["errors"]:
          logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Unknown failure"), error["message"])
          return [{"error": {"type": "unknown", "account": self.account, "message": error["message"]}}]
    elif isinstance(data, dict) and data.get("error", 0):
      if "Incorrect signature" in data["error"]:
        # Try again, if we get a new token
        if self._login():
          logger.debug("Authentication error, logging in again")
          return self._get(path, parse, post, single, args)
        else:
          logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Request failed"), data["error"])
          logger.error("%s", logstr)
          return [{"error": {"type": "auth", "account": self.account, "message": data["error"]}}]
    elif isinstance(data, str):
      logstr = """%s: %s - %s""" % (PROTOCOL_INFO["name"], _("Request failed"), data)
      logger.error("%s", logstr)
      return [{"error": {"type": "request", "account": self.account, "message": data}}]
    if not self._check_error(data):
      return []

    checkins = data["recent"]
    if single: return [getattr(self, "_%s" % parse)(checkins)]
    if parse: return [getattr(self, "_%s" % parse)(m) for m in checkins]
    else: return []
    
  def __call__(self, opname, **args):
    return getattr(self, opname)(**args)

  def receive(self):
    """Gets a list of each friend's most recent check-ins.
    
    The list of each friend's recent check-ins is then
    saved to the database.
 
    Arguments: 
    None
    
    Returns:
    A list of Gwibber compatible objects which have been parsed by the parse function.
    
    """
    return self._get("checkins/recent", v=20120612)
