/*
 * Copyright (C) 2010 Neil Jagdish Patel
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License
 * version 3.0 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3.0 for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library. If not, see
 * <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <njpatel@gmail.com>
 */

using Gwibber;

namespace GwibberGtk
{
  public class UserViewTile : Gtk.EventBox
  {
    private const int our_padding = 6;

    private Gtk.Alignment align;
    private Gtk.HBox   main_box;
    private Gtk.Alignment lalignment;
    private Gtk.Alignment ralignment;
    private Gtk.EventBox  close_box;
    private Gtk.Image  close_icon;
    private Gtk.VBox   vbox;
    private new Gtk.Label  name;
    public Gwibber.Streams streams { get; construct set; }
    private ulong bp_handler_id;
    private ulong cb_handler_id;

    public UserViewTile (Gwibber.Streams streams)
    {
      Object (above_child:false, visible_window:false, streams:streams);
    }

    construct
    {
      draw.connect (on_draw);

      //set_no_show_all (true);

      align = new Gtk.Alignment (0.0f, 0.0f, 1.0f, 1.0f);
      align.set_padding (our_padding, our_padding, our_padding, our_padding);
      add (align);

      main_box = new Gtk.HBox (false, 12);
      align.add (main_box);

      lalignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      ralignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      main_box.pack_start (lalignment, false, false, 0);
      main_box.pack_end (ralignment, false, false, 0);

      vbox = new Gtk.VBox (false, 0);
      main_box.pack_start (vbox, true, true, 0);
      
      var hbox = new Gtk.HBox (false, 0);
      vbox.pack_start (hbox, false, false, 2);
      
      name = new Gtk.Label ("");
      name.set_markup ("<b></b>");
      name.set_alignment (0.0f, 0.5f);
      hbox.pack_start (name, true, true, 0);

      close_icon = new Gtk.Image.from_stock (Gtk.Stock.CLOSE, Gtk.IconSize.MENU);
      close_box = new Gtk.EventBox ();
      close_box.set_visible_window (false);

      close_box.add (close_icon);
      ralignment.add (close_box);
    }

    [Signal (action=true)]
    public virtual signal void show_transient (string id, string name)
    {
    }

    private bool on_draw (Cairo.Context cr)
    {
      Gtk.Allocation alloc;
      get_allocation (out alloc);

      cr.set_line_width (1.0);

      cr.set_source_rgba (1.0, 1.0, 1.0, 1.0);
      cr.move_to (alloc.x, 0.5);
      cr.line_to (alloc.x + alloc.width, 0.5);
      cr.stroke ();

      var pat = new Cairo.Pattern.linear (0, 0, 0, alloc.height);
      pat.add_color_stop_rgba (0.0f, 1.0f, 1.0f, 1.0f, 0.2f);
      pat.add_color_stop_rgba (1.0f, 1.0f, 1.0f, 1.0f, 0.0f);

      cr.rectangle (0, 0, alloc.width, alloc.height);
      cr.set_source (pat);
      cr.fill ();

      cr.set_source_rgba (0.0, 0.0, 0.0, 0.1);
      cr.move_to (alloc.x, alloc.height - 0.5);
      cr.line_to (alloc.x + alloc.width, alloc.height - 0.5);
      cr.stroke ();

      propagate_draw (align, cr);

      return true;
    }

    public void reset ()
    {
      hide ();
    }

    public void set_details (Gee.HashMap<string?, string?> stream)
    {
      name.set_markup ("<b>" + stream["name"] + "</b>");
       
      if (cb_handler_id > 0)
        close_box.disconnect (cb_handler_id);
      cb_handler_id = close_box.button_press_event.connect ((event) => {
        reset ();
        Idle.add (() => {
          streams.delete (stream["id"]);
          return false;
          });
        return true;
        });

      if (bp_handler_id > 0)
        disconnect (bp_handler_id);
      bp_handler_id = button_press_event.connect (() => {
        show_transient (stream["id"], stream["name"]);
        return true;
      });

      queue_resize ();
      show ();
    }                        
  }

  public class UserProfileTile : Gtk.EventBox
  {
    private const int our_padding = 6;

    private Gtk.Alignment align;
    private Gtk.HBox   main_box;
    private Gtk.Alignment lalignment;
    private Gtk.Alignment ralignment;
    private Gtk.EventBox  icon_box;
    private Gtk.Image  icon;
    private Gtk.VBox   vbox;
    private new Gtk.Label  name;
    private Gtk.Label  description;
    private Gtk.Label  website;
    private Gtk.Label  followers;
    private Gtk.Label  friends;
    private Gtk.Image  protected;
    private Gtk.Image  service;
    private Gtk.Label  statuses;
    private Gwibber.Utils utils;
    private Gtk.HBox follower_box;
    private Gtk.HBox follow_box;
    private Gtk.Label follow;
    private Gtk.CheckButton follow_but;
    private Gtk.Button _close;
    private Gtk.Image _close_icon;
    private bool _showing = false;

    public bool showing {
      get { return _showing; }
      set {
          _showing = value;
        if (_showing)
          show_all ();
        else
          hide ();
      }
    }

    public UserProfileTile ()
    {
      Object (above_child:false, visible_window:false);
    }

    construct
    {
      utils = new Gwibber.Utils ();

      align = new Gtk.Alignment (0.0f, 0.0f, 1.0f, 1.0f);
      align.set_padding (our_padding, our_padding, our_padding, our_padding);
      add (align);

      main_box = new Gtk.HBox (false, 12);
      align.add (main_box);

      lalignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      ralignment = new Gtk.Alignment (0.5f, 0.0f, 0.0f, 0.0f);
      main_box.pack_start (lalignment, false, false, 0);
      main_box.pack_end (ralignment, false, false, 0);

      icon_box = new Gtk.EventBox ();
      icon_box.set_visible_window (false);
      icon = new Gtk.Image.from_icon_name ("stock_person", Gtk.IconSize.DIALOG);
      icon_box.add(icon);
      lalignment.add (icon_box);

      _close_icon = new Gtk.Image.from_stock (Gtk.Stock.CLOSE, Gtk.IconSize.MENU);
      _close = new Gtk.Button ();
      _close.set_image (_close_icon);
      ralignment.add (_close);
      _close.clicked.connect (() => {
        showing = false;
      });

      vbox = new Gtk.VBox (false, 0);
      main_box.pack_start (vbox, true, true, 0);

      var hbox = new Gtk.HBox (false, 0);
      vbox.pack_start (hbox, false, false, 2);

      service = new Gtk.Image.from_icon_name ("gwibber", Gtk.IconSize.MENU);
      service.set_no_show_all (true);
      hbox.pack_start (service, false, false, 2);

      name = new Gtk.Label ("");
      name.set_markup ("<b></b>");
      name.set_single_line_mode (false);
      name.set_line_wrap (true);
      name.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
      name.set_alignment (0.0f, 0.5f);
      hbox.pack_start (name, false, false, 0);

      protected = new Gtk.Image.from_icon_name ("stock_lock", Gtk.IconSize.MENU);
      protected.set_no_show_all (true);
      hbox.pack_start (protected, false, false, 0);

      description = new Gtk.Label ("");
      description.set_selectable (false);
      description.set_single_line_mode (false);
      description.set_line_wrap (true);
      description.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
      description.set_alignment (0.0f, 0.0f);
      vbox.pack_start (description, false, false, 2);

      website = new Gtk.Label ("");
      website.set_selectable (false);
      website.set_single_line_mode (false);
      website.set_line_wrap (true);
      website.set_line_wrap_mode (Pango.WrapMode.WORD_CHAR);
      website.set_alignment (0.0f, 0.0f);
      vbox.pack_start (website, false, false, 2);

      follower_box = new Gtk.HBox (false, 2);
      vbox.pack_start (follower_box, false, false, 2);

      followers = new Gtk.Label ("");
      followers.set_markup (_("Followers") + ": ");
      followers.set_alignment (0.0f, 0.5f);
      follower_box.pack_start (followers, false, false, 2);

      friends = new Gtk.Label ("");
      friends.set_markup (_("Following") + ": ");
      friends.set_alignment (0.0f, 0.5f);
      follower_box.pack_start (friends, false, false, 2);

      statuses = new Gtk.Label ("");
      statuses.set_markup (_("Statuses") + ": ");
      statuses.set_alignment (0.0f, 0.5f);
      follower_box.pack_start (statuses, false, false, 2);

      follow_box = new Gtk.HBox (false, 0);
      follow_but = new Gtk.CheckButton ();
      follow = new Gtk.Label (_("Following"));
      follow.set_selectable (false);
      follow_box.pack_start (follow_but, false, false, 2);
      follow_box.pack_start (follow, false, false, 2);

      vbox.pack_start (follow_box, false, false, 2);

      show_all ();
    }

    public void reset ()
    {
      hide ();
    }

    public void set_details (string? _id,
                             string? _sender,
                             string? _sender_nick,
                             int _followers,
                             int _friends,
                             int _statuses,
                             bool _following,
                             bool _protected,
                             string? _website,
                             string? _description,
                             string? _account,
                             string? _service,
                             string? _icon_uri)
    {

      string sender_str = "";

      if (_sender != null && _sender_nick != null)
      {
        if (_sender.length > 0)
          sender_str = "<b>" + _sender + "</b> (" + _sender_nick + ")";
        else
          sender_str = "<b>" + _sender_nick + "</b>";

        name.set_markup (sender_str);
      } else
        name.hide ();

      if (_description != null)
        description.set_markup (_description);
      else
        description.hide ();

      if (_website != null)
      {
        if (_website.length > 0)
        {
          website.set_markup ("<a href='" + _website + "'>" + _website + "</a>");
          website.show ();
        } else
          website.hide ();
      } else
        website.hide ();
  
      friends.set_markup (_("Friends") + "\n" + _friends.to_string ());
      followers.set_markup (_("Followers") + "\n " + _followers.to_string ());
      statuses.set_markup (_("Statuses") + "\n " + _statuses.to_string ());

      var _avatar_cache_image = utils.avatar_path(_icon_uri);
      if (_avatar_cache_image != null)
        icon.set_from_file(_avatar_cache_image);
      else
        icon.set_from_icon_name ("stock_person", Gtk.IconSize.DIALOG);

      if (follow_but.get_active () != _following);
        follow_but.set_active (_following);
      follow_but.toggled.connect (() => {
        var service = new Gwibber.Service ();
        service.follow (_account, _sender_nick, follow_but.get_active ());
        // wait a moment so twitter records the follow/unfollow operation
        Timeout.add_seconds(10, () => {
          service.perform(_id);
          return false;
        });
      });

      if (_protected)
        protected.show ();
      else
        protected.hide ();

      var service_icon = GLib.Path.build_filename (Config.PKGDATADIR, "ui/icons/breakdance/16x16/" + _service + ".png");
      service.set_from_file(service_icon);
      service.show ();
    }
  }
}
