/*
 * Copyright 2012 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License version 3, as published 
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but 
 * WITHOUT ANY WARRANTY; without even the implied warranties of 
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along 
 * with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Alberto Mardegan <alberto.mardegan@canonical.com>
 */

public class TestPlugin : Ap.Plugin {
    public Gtk.Widget widget_to_build;

    public TestPlugin(Ag.Account account) {
        Object(account: account);
    }

    public override unowned Gtk.Widget build_widget () {
        return widget_to_build;
    }
}

public class TestOAuthPlugin : Ap.OAuthPlugin {
    public TestOAuthPlugin(Ag.Account account) {
        Object(account: account);
    }

    construct {
        var account_oauth_params =
            new HashTable<string, GLib.Value?> (str_hash, null);
        account_oauth_params.insert ("long", "short");
        account_oauth_params.insert ("wide", "narrow");
        set_account_oauth_parameters (account_oauth_params);

        var oauth_params = new HashTable<string, GLib.Value?> (str_hash, null);
        oauth_params.insert ("long", "not short");
        oauth_params.insert ("wide", "not narrow");
        set_oauth_parameters (oauth_params);
    }
}

public class TestApplicationPlugin : Ap.ApplicationPlugin {
    public Gtk.Widget widget_to_build;

    public TestApplicationPlugin(Ag.Application application,
                                 Ag.Account account) {
        Object(account: account, application: application);
    }

    public override unowned Gtk.Widget build_widget () {
        return widget_to_build;
    }
}

int main (string[] args)
{
    Gtk.test_init (ref args);

    Test.add_func ("/libaccount-plugin/plugin/create", accountplugin_create);
    Test.add_func ("/libaccount-plugin/plugin/create-headless",
                   accountplugin_create_headless);
    Test.add_func ("/libaccount-plugin/application-plugin/create",
                   applicationplugin_create);

    Test.run ();

    return Posix.EXIT_SUCCESS;
}

void accountplugin_create ()
{
    var manager = new Ag.Manager ();
    var account = manager.create_account ("MyProvider");

    var plugin = new TestPlugin (account);

    assert (plugin.account == account);
    assert (plugin.get_account () == account);

    plugin.need_authentication = true;
    assert (plugin.need_authentication == true);
    assert (plugin.get_need_authentication () == true);

    /* Make sure that the virtual method is called */
    plugin.widget_to_build = new Gtk.Label ("Hello world!");
    assert (plugin.build_widget () == plugin.widget_to_build);
}

void accountplugin_create_headless ()
{
    var manager = new Ag.Manager ();
    var account = manager.create_account ("MyProvider");

    var plugin = new TestOAuthPlugin (account);

    assert (plugin.account == account);
    assert (plugin.get_account () == account);

    string test_username = "Long John Silver";

    plugin.need_authentication = false;
    plugin.set_credentials (test_username, "irrelevant password");

    var main_loop = new GLib.MainLoop (null, false);

    plugin.finished.connect (() => {
        main_loop.quit ();
    });
    GLib.Idle.add (() => {
        plugin.act_headless ();
        return false;
    });

    main_loop.run ();

    /* check that the account was stored */
    assert (account.id != 0);

    assert (account.get_display_name () == test_username);

    /* The accounts created in headless mode must be disabled */
    account.select_service (null);
    assert (!account.get_enabled ());

    var service = manager.get_service ("MyService");
    var account_service = new Ag.AccountService (account, service);
    var auth_data = account_service.get_auth_data ();

    assert (auth_data.get_method () == "oauth2");
    assert (auth_data.get_mechanism () == "user_agent");

    var parameters = auth_data.get_parameters ();
    assert (parameters["long"].get_string () == "short");
    assert (parameters["wide"].get_string () == "narrow");

    /* delete the account */
    GLib.Idle.add (() => {
        plugin.delete_account.begin ((obj, res) => {
            try
            {
                plugin.delete_account.end (res);
            }
            catch (Error error)
            {
                critical ("Error deleting account: %s", error.message);
                assert_not_reached ();
            }
            main_loop.quit ();
        });
        return false;
    });

    main_loop.run ();
}

void applicationplugin_create ()
{
    var manager = new Ag.Manager ();
    var account = manager.create_account ("MyProvider");
    var application = manager.get_application ("Gallery");
    assert (application != null);

    var plugin = new TestApplicationPlugin (application, account);

    assert (plugin.account == account);
    assert (plugin.get_account () == account);
    assert (plugin.application == application);
    assert (plugin.get_application () == application);

    /* Make sure that the virtual method is called */
    plugin.widget_to_build = new Gtk.Label ("Hello world!");
    assert (plugin.build_widget () == plugin.widget_to_build);
}
