/**
 * GMyth Library
 *
 * @file gmyth/gmyth_vlc.c
 *
 * @brief <p> GMythVLC library provides functions that
 * interact with a VLC server running telnet interface.
 *
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia.
 * @author Artur Duque de Souza <artur.souza@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>

#include "gmyth_vlc.h"
#include "gmyth_debug.h"
#include "gmyth_socket.h"

/*
 * static functions 
 */
static int      _socket_send(int socket, gchar * msg);
static int      _socket_recv(int socket, gchar * buf);

/** Primitive function to send a message through the socket
 *
 * @param socket 
 * @param msg the message itself
 * @return 0 if success
 */
static int
_socket_send(int socket, gchar * msg)
{
    size_t          size = strlen(msg) + 2; // (\n + \0)
    gchar          *final = (gchar *) g_malloc0(sizeof(gchar) * size);

    g_snprintf(final, size, "%s\n", msg);

    if (send(socket, final, strlen(final), 0) == -1)
        perror("GMyth_VLC: send error");

    g_free(final);
    return 0;
}


/** Primitive function to receive a message through the socket
 *
 * @param socket 
 * @param buf Buffer to put the message
 * @return 0 if success
 */
static int
_socket_recv(int socket, gchar * buf)
{
    int             numbytes = 0;

    if ((numbytes = recv(socket, buf, BUFFER - 1, 0)) == -1) {
        perror("GMyth_VLC: recv error");
        return -1;
    }

    buf[numbytes - 1] = '\0';
    return numbytes;
}


/** Function that adds options to the output media
 *
 * @param vlc structure with options for vlc
 * @param output the number of the output media
 * @param kind the type of option we'll change
 * @param the params for the option
 * @return 0 if success
 */
int
gmyth_vlc_setup_output(GMythVlc * vlc, int output, gchar * kind,
                       gchar * opts)
{
    int             ret;

    size_t          size = strlen(opts) + 25;
    gchar          *msg = g_malloc0(sizeof(gchar) * size);

    g_snprintf(msg, size, "setup output%d %s %s", output, kind, opts);

    ret = _socket_send(vlc->sockfd, msg);

    g_free(msg);
    return ret;
}


/** Function to clean the playlist
 *
 * @param vlc structure with options for vlc
 * @param output the number of the output media
 * @param file the file we want to insert in the playlist
 * @return 0 if success
 */
int
gmyth_vlc_clean_playlist(GMythVlc * vlc)
{
    return _socket_send(vlc->sockfd, "del all");
}


/** Function to control the playlist
 *
 * @param vlc structure with options for vlc
 * @param output the number of the output media
 * @param command play, stop or pause(just for vod)
 * @return 0 if success
 */
int
gmyth_vlc_control_input(GMythVlc * vlc, int output, gchar * command)
{
    size_t          size = 25;
    gchar          *msg = g_malloc0(sizeof(gchar) * size);

    g_snprintf(msg, size, "control output%d %s", output, command);

    int             ret = _socket_send(vlc->sockfd, msg);

    g_free(msg);
    return ret;
}



/** Function to insert an item in the playlist
 *
 * @param vlc structure with options for vlc
 * @param output the number of the output media
 * @param file the file we want to insert in the playlist
 * @return 0 if success
 */
int
gmyth_vlc_create_input(GMythVlc * vlc, int output, gchar * file)
{
    return gmyth_vlc_setup_output(vlc, output, "input", file);
}


/** Function to create a channel in vlc
 *
 * @param vlc structure with options for vlc
 * @param type the type of channel (broadcast, vod...)
 * @param port
 * @return 0 if success
 */
int
gmyth_vlc_create_channel(GMythVlc * vlc, gchar * type, int port,
                         int vcodec)
{
    int             ret;
    size_t          size = 40;
    gchar          *msg = (gchar *) g_malloc0(sizeof(gchar) * size);

    g_snprintf(msg, size, "new output%d %s enabled loop", vlc->n_outputs,
               type);

    ret = _socket_send(vlc->sockfd, msg);

    if (ret > -1) {
        gmyth_vlc_setup_output(vlc, vlc->n_outputs, "option",
                               "sout-keep=1");

        g_free(msg);
        size = 256;
        msg = (gchar *) g_malloc0(sizeof(gchar) * size);

        if (vcodec == MPEG1)
            // Best transcode option for N800 (MP1V)
            g_snprintf(msg, size, "#transcode{vcodec=mp1v,vb=384,"
                       "fps=25.0,scale=1,acodec=mpga,"
                       "ab=64,channels=1}:duplicate{dst="
                       "std{access=http,mux=mpeg1,dst=" ":%d}}", port);
        else
            // Best transcode option for N800 (THEORA)
            g_snprintf(msg, size, "#transcode{vcodec=theo,vb=384,"
                       "fps=25.0,scale=1,acodec=vorb,"
                       "ab=64,channels=1}:duplicate{dst="
                       "std{access=http,mux=ogg,dst=" ":%d}}", port);

        ret = gmyth_vlc_setup_output(vlc, vlc->n_outputs, "output", msg);

        vlc->n_outputs++;
    }

    g_free(msg);

    return ret;
}


/** Function to connect to vlc on the backend
 *
 * @param vlc structure with options for vlc
 * @param backend_info infos about the backend
 * @param passwd the password for telnet interface
 * @return 0 if success
 */
int
gmyth_vlc_connect(GMythVlc * vlc, GMythBackendInfo * backend_info,
                  gchar * passwd, int port)
{
    int             numbytes;

    if ((vlc->he = gethostbyname(backend_info->hostname)) == NULL) {
        herror("GMyth_VLC: gethostbyname error");
        return -1;
    }

    if ((vlc->sockfd = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
        perror("GMyth_VLC: socket error");
        return -1;
    }
    // Socket properties
    vlc->their_addr.sin_family = AF_INET;
    vlc->their_addr.sin_port = htons(port);
    vlc->their_addr.sin_addr = *((struct in_addr *) vlc->he->h_addr);
    memset(&(vlc->their_addr.sin_zero), '\0', 8);

    if (connect(vlc->sockfd, (struct sockaddr *) &(vlc->their_addr),
                sizeof(struct sockaddr)) == -1) {
        perror("GMyth_VLC: connect error. Check VLC's telnet interface");
        return -1;
    }
    // Receive the Password's Prompt
    numbytes = _socket_recv(vlc->sockfd, vlc->buf);

    // Send the Password. We don't have to
    // care about passwords being sent in plain text
    // because telnet protocol does it.
    _socket_send(vlc->sockfd, passwd);

    // Receive the Welcome msg
    numbytes = _socket_recv(vlc->sockfd, vlc->buf);
    if (numbytes > -1)
        if (strncmp(vlc->buf, "\r\nWrong password.", 17) == 0) {
            perror("Gmyth_VLC: passwd error. Check your passwd");
            return -2;
        }


    return 0;
}


/** Function to disconnect from vlc
 *
 * @param vlc structure with options for vlc
 * @param backend_info infos about the backend
 * @return 0 if success
 */
int
gmyth_vlc_disconnect(GMythVlc * vlc)
{

    int             ret;

    ret = gmyth_vlc_clean_playlist(vlc);

    if (ret > -1) {
        vlc->n_outputs = 0;
        vlc->n_inputs = 0;
    }

    return close(vlc->sockfd);
}
