/**
 * GMyth Library
 *
 * @file gmyth/gmyth_backend_info.c
 * 
 * @brief <p> This component represents all the MythTV backend server
 * 						configuration information.
 *
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia.
 * @author Hallyson Melo <hallyson.melo@indt.org.br>
 * @author Rosfran Borges <rosfran.borges@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "gmyth_backendinfo.h"
#include "gmyth_uri.h"
#include "gmyth_debug.h"

static void     gmyth_backend_info_class_init(GMythBackendInfoClass *
                                              klass);
static void     gmyth_backend_info_init(GMythBackendInfo * object);

static void     gmyth_backend_info_dispose(GObject * object);
static void     gmyth_backend_info_finalize(GObject * object);

G_DEFINE_TYPE(GMythBackendInfo, gmyth_backend_info, G_TYPE_OBJECT);
static void     gmyth_backend_info_class_init(GMythBackendInfoClass *
                                                  klass)
{
    GObjectClass   *gobject_class;

    gobject_class = (GObjectClass *) klass;

    gobject_class->dispose = gmyth_backend_info_dispose;
    gobject_class->finalize = gmyth_backend_info_finalize;
}

static void
gmyth_backend_info_init(GMythBackendInfo * backend_info)
{
    backend_info->hostname = NULL;
    backend_info->port = -1;
    backend_info->username = NULL;
    backend_info->password = NULL;
    backend_info->db_name = NULL;
    backend_info->db_port = 0;
    backend_info->status_port = -1;
}

static void
gmyth_backend_info_dispose(GObject * object)
{
    GMythBackendInfo *backend_info = GMYTH_BACKEND_INFO(object);

    g_free(backend_info->hostname);
    g_free(backend_info->username);
    g_free(backend_info->password);
    g_free(backend_info->db_name);

    if (backend_info->sock)
        g_object_unref (backend_info->sock);

    backend_info->hostname = NULL;
    backend_info->port = -1;
    backend_info->username = NULL;
    backend_info->password = NULL;
    backend_info->db_name = NULL;
    backend_info->db_port = 0;
    backend_info->status_port = -1;
    backend_info->sock = NULL;

    G_OBJECT_CLASS(gmyth_backend_info_parent_class)->dispose(object);
}

static void
gmyth_backend_info_finalize(GObject * object)
{
    g_signal_handlers_destroy(object);

    G_OBJECT_CLASS(gmyth_backend_info_parent_class)->finalize(object);
}

/** 
 * Creates a new instance of GMythBackendInfo.
 * 
 * @return a new instance of GMythBackendInfo.
 */
GMythBackendInfo *
gmyth_backend_info_new()
{
    GMythBackendInfo *backend_info =
        GMYTH_BACKEND_INFO(g_object_new(GMYTH_BACKEND_INFO_TYPE, NULL));

    return backend_info;
}

/** 
 * Creates a new instance of GMythBackendInfo, based on a given set of 
 * configuration parameters.
 * 
 * @param hostname The hostname to the MythTV backend server.
 * @param username The user name to the MythTV backend MySQL server.
 * @param password The password to the user of the MythTV backend MySQL server.
 * @param db_name The database name of the MythTV backend, stored on the MySQL server.
 * @param port The port number of the MythTV backend server (commonly is 6543).
 * 
 * @return a new instance of GMythBackendInfo.
 */
GMythBackendInfo *
gmyth_backend_info_new_full(const gchar * hostname, const gchar * username,
                            const gchar * password, const gchar * db_name,
                            gint port)
{
    GMythBackendInfo *backend_info =
        GMYTH_BACKEND_INFO(g_object_new(GMYTH_BACKEND_INFO_TYPE, NULL));

    gmyth_backend_info_set_hostname(backend_info, hostname);
    gmyth_backend_info_set_username(backend_info, username);
    gmyth_backend_info_set_password(backend_info, password);
    gmyth_backend_info_set_db_name(backend_info, db_name);
    gmyth_backend_info_set_port(backend_info, port);

    return backend_info;
}

/** 
 * Creates a new instance of GMythBackendInfo, based on the 
 * MythTV's backend server URI string.
 * 
 * @param uri_str The URI string pointing to the MythTV backend server.
 * 
 * @return a new instance of GMythBackendInfo.
 */
GMythBackendInfo *
gmyth_backend_info_new_with_uri(const gchar * uri_str)
{
    GMythBackendInfo *backend_info;
    GMythURI       *uri;
    gchar         **path_parts;
    gchar          *db;

    backend_info = GMYTH_BACKEND_INFO(g_object_new(GMYTH_BACKEND_INFO_TYPE, NULL));
    uri = gmyth_uri_new_with_value (uri_str);
    path_parts = g_strsplit(gmyth_uri_get_path(uri), "&", -1);
    gmyth_backend_info_set_hostname(backend_info, gmyth_uri_get_host(uri));
    gmyth_backend_info_set_username(backend_info, gmyth_uri_get_user(uri));
    gmyth_backend_info_set_password(backend_info,
                                    gmyth_uri_get_password(uri));

    /*
     * gets the path info to database name, from the URI, and removes the
     * trash chars 
     */
    if ((path_parts != NULL) && (strlen (path_parts[0]) > 0))
    {
        db =  path_parts[0]+2;
    }
    else
    {
        db = gmyth_uri_get_path(uri);
    }

    gmyth_backend_info_set_db_name(backend_info, db);

    gmyth_backend_info_set_port(backend_info, gmyth_uri_get_port(uri));

    g_object_unref(uri);
    g_strfreev(path_parts);

    return backend_info;
}

void
gmyth_backend_info_set_hostname(GMythBackendInfo * backend_info,
                                const gchar * hostname)
{
    g_return_if_fail(backend_info != NULL);

    if (NULL == hostname || strlen(hostname) <= 0) {
        gmyth_debug("Error trying to set a hostname equals to NULL.");
    } else {
        backend_info->hostname = g_strdup(hostname);
    }
}

void
gmyth_backend_info_set_username(GMythBackendInfo * backend_info,
                                const gchar * username)
{
    g_return_if_fail(backend_info != NULL);

    backend_info->username = g_strdup(username);
}

void
gmyth_backend_info_set_password(GMythBackendInfo * backend_info,
                                const gchar * password)
{
    g_return_if_fail(backend_info != NULL);

    backend_info->password = g_strdup(password);
}

void
gmyth_backend_info_set_db_name(GMythBackendInfo * backend_info,
                               const gchar * db_name)
{
    g_return_if_fail(backend_info != NULL);

    backend_info->db_name = g_strdup(db_name);
}

void
gmyth_backend_info_set_db_port(GMythBackendInfo * backend_info, gint db_port)
{
    g_return_if_fail(backend_info != NULL);

    if (db_port <= 0) {
        gmyth_debug("Error trying to set a port less than 0.");
    } else {
        backend_info->db_port = db_port;
    }
}

void
gmyth_backend_info_set_port(GMythBackendInfo * backend_info, gint port)
{
    g_return_if_fail(backend_info != NULL);

    if (port <= 0) {
        gmyth_debug("Error trying to set a port less than 0.");
    } else {
        backend_info->port = port;
    }
}

void
gmyth_backend_info_set_status_port(GMythBackendInfo * backend_info,
                                   gint port)
{
    g_return_if_fail(backend_info != NULL);

    if (port <= 0) {
        gmyth_debug
            ("Error trying to set the status port to less than zero.");
    } else {
        backend_info->status_port = port;
    }
}

const gchar    *
gmyth_backend_info_get_hostname(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, NULL);

    return backend_info->hostname;
}

const gchar    *
gmyth_backend_info_get_username(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, NULL);

    return backend_info->username;
}

const gchar    *
gmyth_backend_info_get_password(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, NULL);

    return backend_info->password;
}

const gchar    *
gmyth_backend_info_get_db_name(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, NULL);

    return backend_info->db_name;
}

gint
gmyth_backend_info_get_idb_port(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, -1);

    return backend_info->db_port;
}


gint
gmyth_backend_info_get_port(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, -1);

    return backend_info->port;
}

/** 
 * Creates a new instance of GMythURI, based on the GMythBackendInfo instance to the 
 * MythTV's backend server.
 * 
 * @param backend_info The GMythBackendInfo instance.
 * 
 * @return an instance of GMythURI, created from a GMythBackendInfo.
 */
GMythURI       *
gmyth_backend_info_get_uri(GMythBackendInfo * backend_info)
{
    GMythURI       *uri = NULL;
    gchar          *uri_str = NULL;
    gchar          *user_info = NULL;
    gchar          *db_data = NULL;

    if ((backend_info->username != NULL
         && strlen(backend_info->username) > 0))
        user_info =
            g_strdup_printf("%s:%s@", backend_info->username,
                            backend_info->password);

    if (backend_info->db_name != NULL && strlen(backend_info->db_name) > 0) {
        if ((g_strrstr(backend_info->db_name, "_") != NULL))
            db_data = g_strdup(backend_info->db_name);
        else
            db_data = g_strdup_printf("?%s&", backend_info->db_name);
    }
    // else if ( ( ( g_strrstr( backend_info->path, "livetv" ) != NULL )
    // || 
    // ( g_strrstr( backend_info->path, "/?" ) != NULL ) )

    uri_str = g_strdup_printf("myth://%s%s:%d/%s", user_info != NULL
                              && strlen(user_info) > 0 ? user_info : "",
                              backend_info->hostname, backend_info->port,
                              db_data != NULL
                              && strlen(db_data) > 0 ? db_data : "");
    uri = gmyth_uri_new_with_value(uri_str);

    if (user_info != NULL)
        g_free(user_info);

    if (db_data != NULL)
        g_free(db_data);

    g_free(uri_str);

    return uri;
}

gboolean
gmyth_backend_info_is_local_file(GMythBackendInfo * backend_info)
{
    g_return_val_if_fail(backend_info != NULL, FALSE);

    return
        gmyth_uri_is_local_file(gmyth_backend_info_get_uri(backend_info));
}

GMythSocket*
gmyth_backend_info_get_connected_socket (GMythBackendInfo *backend_info)
{
    if (backend_info->sock == NULL) {
        gboolean res;
        backend_info->sock = gmyth_socket_new ();
        res =  gmyth_socket_connect_to_backend (backend_info->sock,
                                                backend_info->hostname,
                                                backend_info->port,
                                                TRUE);
        if (res == FALSE) {
            g_object_unref (backend_info->sock);
            backend_info->sock = NULL;
            return NULL;
        }
    }

    return g_object_ref (backend_info->sock);
}
