/* This file is part of the KDE Project
   Copyright (c) 2006 Lukas Tinkl <ltinkl@suse.cz>
   Copyright (c) 2008 Lubos Lunak <l.lunak@suse.cz>
   Copyright (c) 2009 Ivo Anjo <knuckles@gmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "freespacenotifier.h"

#include <sys/vfs.h>
#include <unistd.h>

#include <QtCore/QDir>
#include <QtCore/QFile>
#include <QtGui/QLabel>
#include <QtGui/QSpinBox>

#include <QtDBus/QtDBus>

#include <KDebug>
#include <KLocale>
#include <KRun>
#include <KConfigDialog>

#include "settings.h"
#include "ui_freespacenotifier_prefs_base.h"

FreeSpaceNotifier::FreeSpaceNotifier( QObject* parent )
    : QObject( parent )
    , lastAvailTimer( NULL )
    , notification( NULL )
    , lastAvail( -1 )
{
    // If we are running, notifications are enabled
    FreeSpaceNotifierSettings::setEnableNotification( true );
    
    connect( &timer, SIGNAL( timeout() ), SLOT( checkFreeDiskSpace() ) );
    timer.start( 1000 * 60 /* 1 minute */ );
}

FreeSpaceNotifier::~FreeSpaceNotifier()
{
    // The notification is automatically destroyed when it goes away, so we only need to do this if
    // it is still being shown
    if ( notification ) notification->deref();
}

void FreeSpaceNotifier::checkFreeDiskSpace()
{
    if ( notification || !FreeSpaceNotifierSettings::enableNotification() )
        return;
    struct statfs sfs;
    if ( statfs( QFile::encodeName( QDir::homePath() ), &sfs ) == 0 )
    {
        long avail = ( getuid() ? sfs.f_bavail : sfs.f_bfree );

        if (avail < 0 || sfs.f_blocks <= 0)
            return; // we better not say anything about it
        
        long limit = FreeSpaceNotifierSettings::minimumSpace(); // MiB
        int availpct = int( 100 * avail / sfs.f_blocks );
        avail = ((long long)avail) * sfs.f_bsize / ( 1024 * 1024 ); // to MiB
        bool warn = false;
        if( avail < limit ) // avail disk space dropped under a limit
        {
            if( lastAvail < 0 ) // always warn the first time
            {
                lastAvail = avail;
                warn = true;
            }
            else if( avail > lastAvail ) // the user freed some space
                lastAvail = avail;       // so warn if it goes low again
            else if( avail < lastAvail * 0.5 ) // available dropped to a half of previous one, warn again
            {
                warn = true;
                lastAvail = avail;
            }
            // do not change lastAvail otherwise, to handle free space slowly going down
        }
        if ( warn )
        {
            notification = new KNotification( "freespacenotif", 0, KNotification::Persistent );

            notification->setText( i18n( "You are running low on disk space on your home partition (currently %2%, %1 MiB free).\nWould you like to run a file manager to free some disk space?", avail, availpct ) );
            notification->setActions( QStringList() << i18n( "Open File Manager" ) << i18n( "Do Nothing" ) << i18n( "Configure Warning" ) );
            //notification->setPixmap( ... ); // TODO: Maybe add a picture here?

            connect( notification, SIGNAL( action1Activated() ), SLOT( openFileManager() ) );
            connect( notification, SIGNAL( action2Activated() ), SLOT( cleanupNotification() ) );
            connect( notification, SIGNAL( action3Activated() ), SLOT( showConfiguration() ) );
            connect( notification, SIGNAL( closed() ),           SLOT( cleanupNotification() ) );

            notification->setComponentData( KComponentData( "freespacenotifier" ) );
            notification->sendEvent();
        }
    }
}

void FreeSpaceNotifier::openFileManager()
{
    cleanupNotification();
    new KRun( KUrl( QDir::homePath() ), 0 );
}

void FreeSpaceNotifier::showConfiguration()
{
    cleanupNotification();

    if ( KConfigDialog::showDialog( "settings" ) )  {
        return;
    }

    KConfigDialog *dialog = new KConfigDialog( 0, "settings", FreeSpaceNotifierSettings::self() );
    QWidget *generalSettingsDlg = new QWidget();

    Ui::freespacenotifier_prefs_base preferences;
    preferences.setupUi( generalSettingsDlg );

    dialog->addPage( generalSettingsDlg, i18n( "General" ), "system-run" );
    connect( dialog, SIGNAL( finished() ), this, SLOT( configDialogClosed() ) );
    dialog->setAttribute( Qt::WA_DeleteOnClose );
    dialog->show();
}

void FreeSpaceNotifier::cleanupNotification()
{
    notification = NULL;

    // warn again if constantly below limit for too long
    if( lastAvailTimer == NULL )
    {
        lastAvailTimer = new QTimer( this );
        connect( lastAvailTimer, SIGNAL( timeout() ), SLOT( resetLastAvailable() ) );
    }
    lastAvailTimer->start( 1000 * 60 * 60 /* 1 hour*/ );
}

void FreeSpaceNotifier::resetLastAvailable()
{
    lastAvail = -1;
    lastAvailTimer->deleteLater();
    lastAvailTimer = NULL;
}

void FreeSpaceNotifier::configDialogClosed()
{
    if ( !FreeSpaceNotifierSettings::enableNotification() )
        disableFSNotifier();
}

/* The idea here is to disable ourselves by telling kded to stop autostarting us, and
 * to kill the current running instance.
 */
void FreeSpaceNotifier::disableFSNotifier()
{
    QDBusInterface iface( "org.kde.kded", "/kded", "org.kde.kded" );
    if ( dbusError( iface ) ) return;

    // Disable current module autoload
    iface.call( "setModuleAutoloading", "freespacenotifier", false );
    if ( dbusError( iface ) ) return;

    // Unload current module
    iface.call( "unloadModule", "freespacenotifier" );
    if ( dbusError( iface ) ) return;
}

bool FreeSpaceNotifier::dbusError( QDBusInterface &iface )
{
    QDBusError err = iface.lastError();
    if ( err.isValid() )
    {
        kError() << "Failed to perform operation on kded [" << err.name() << "]:" << err.message();
        return true;
    }
    return false;
}
