# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.utils import signal

import pgm
from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.graph.text import Text

class ContectButton(Group):
    
    def __init__(self):
        Group.__init__(self)
        
        self._image = Image()
        self._image.bg_color = (0, 0, 0, 0)
        self._image.layout = pgm.IMAGE_SCALED
        self._image.visible = True
        self.add(self._image)

    def set_from_file(self, img_path, size=None):
        if img_path:
            self._image.set_from_file(img_path, size)
            
    def image__get(self):
        return self._image

class ButtonPanel(Group):

    index_changed = signal.Signal('index_changed', int)
    
    #in percent
    icon_margin = 0.15
    right_margin = 0.50
    unselected_opacity = 80
    
    _current_index = -1

    def __init__(self):
        Group.__init__(self)
        self.active = True

    def _get_index_from_drawable(self, drawable):
        i = 0
        for button in self.children:
            if drawable == button.image:
                return i
            i +=1
        
        return -1

    def add_button(self, image_path, size=None):
        b = ContectButton()
        if image_path is not None:
            b.set_from_file(image_path, size)
            b.image.connect('clicked', self._button_clicked_callback)
        Group.add(self, b)
        b.visible = True
        
    def _button_clicked_callback(self, drawable, x, y, z, button, time):
        if not self.active:
            return False
        index = self._get_index_from_drawable(drawable)
        self.index_changed.emit(index)
        return True
        
    def compute_layout(self):
        if not self.parent:
            return
        p = 1
        margin = self.icon_margin * self.parent.height
        right_margin = self.right_margin * self.parent.height
        index = 0
        for i in reversed(self.children):
            i.size = (self.parent.height - margin*2, self.parent.height - margin * 2)
            x = self.parent.width - i.width * p - margin * (p-1) - right_margin
            i.x = x
            i.y += margin
            p += 1
            if self._current_index > -1 and index == self._current_index:
                i.opacity = 255
            else:
                i.opacity = self.unselected_opacity
            index += 1
        
    
    def current_index__set(self, value):
        if len(self.children) == 0:
            return
            
        if self._current_index > -1:
            previous_widget = self.children[self._current_index]
            previous_widget.opacity = self.unselected_opacity
            
        if value > -1:
            current_widget = self.children[value]
            current_widget.opacity = 255
            
        self._current_index = value
        
    def current_index__get(self):
        return self._current_index
        


class ContextBar(Group):

    def __init__(self):
        Group.__init__(self)
        
        self._button_panel = ButtonPanel()
        self._button_panel.visible = True
        self.add(self._button_panel)
        #FIXME : find a better way to have this widget on top
        self._button_panel.z = 0.05
    
        self.background = Image()
        self.background.bg_color = (0, 0, 0, 0)
        self.background.layout = pgm.IMAGE_FILLED
        self.background.interp = pgm.IMAGE_NEAREST
        self.background.visible = True
        self.add(self.background)

        self.icon = Image()
        self.icon.bg_color = (0, 0, 0, 0)
        self.icon.layout = pgm.IMAGE_SCALED
        self.icon.size = (self.width*0.08, self.height*0.8)
        self.icon.position = (0.0, (self.height-self.icon.height)/2.0, 0.0)
        self.icon.visible = True
        self.add(self.icon)
        
        self._label = Text()
        self._label.bg_color = (0, 0, 0, 0)
        self._label.font_family = "Bitstream DejaVu"
        self._label.weight = pgm.TEXT_WEIGHT_BOLD
        self._label.font_height = 0.7
        self._label.position = (self.icon.width, self.height*0.17, 0.0)
        self._label.size = (self.width-self._label.x, self.height-self._label.y)
        self._label.visible = True
        self.add(self._label)
    
    def compute_layout(self):
        self._button_panel.compute_layout()
        
    def label__set(self, label):
        self._label.label = label

    def label__get(self):
        return self._label
    
    def button_panel__get(self):
        return self._button_panel

if __name__ == "__main__":
    def on_key_press(viewport, event, widget):
        if event.type == pgm.KEY_PRESS:
            # quit on q or ESC
            if event.keyval in (pgm.keysyms.q, pgm.keysyms.Escape):
                pgm.main_quit()

    def on_delete(viewport, event):
        pgm.main_quit()
        
    # OpenGL viewport creation
    factory = pgm.ViewportFactory('opengl')
    gl = factory.create()
    gl.title = 'Context bar widget'

    # Canvas and image drawable creation
    canvas = pgm.Canvas()
    
    # Bind the canvas to the OpenGL viewport
    gl.set_canvas(canvas)
    gl.show()
    
    context = ContextBar()
    context.background.set_from_file("theme/context_bar_bg.png")
    context.label = "Music"
    context.icon.set_from_file("../poblenou_frontend/tango_theme/music.png")
    
    context.canvas = canvas
    context.position = (0.0, 0.0, 0.0)
    context.width = canvas.width
    context.height = 0.2
    context.visible = True

    # Let's start a mainloop
    gl.connect('key-press-event', on_key_press, context)
    gl.connect('delete-event', on_delete)
    pgm.main()
