# Copyright 2009-2010 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>
#          Vincenzo Di Somma <vincenzo.di.somma@canonical.com>

"""Tests for the RecordDict object on which the Contacts API is built."""

import doctest
import os

from collections import MutableSequence, MutableMapping

import desktopcouch
import desktopcouch.application.tests as test_environment

from desktopcouch.application.tests import TestCase
from desktopcouch.application.server import DesktopDatabase
from desktopcouch.records import (
    Record, RecordDict, MergeableList, record_factory, IllegalKeyException,
    validate, NoRecordTypeSpecified, is_mergeable_list)


class TestRecordDict(TestCase):
    """Test the record dictionary functionality."""

    def test_abc(self):
        """Test RecordDict is a mutable mapping."""
        record_dict = RecordDict({})
        self.assertTrue(isinstance(record_dict, MutableMapping))


class TestMergeableList(TestCase):
    """Test the mergeable list functionality."""

    def test_is_mergeable_list(self):
        """Test the is_mergeable_list checker."""
        self.assertTrue(is_mergeable_list({'_order': []}))
        self.assertTrue(is_mergeable_list({
            '_order': [
                'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3',
                'e47455fb-da05-481e-a2c7-88f14d5cc163'],
            'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 'foo',
            'e47455fb-da05-481e-a2c7-88f14d5cc163': 'baz'}))
        self.assertFalse(is_mergeable_list(None))
        self.assertFalse(is_mergeable_list({}))
        self.assertFalse(is_mergeable_list({
            'foo': 'bar',
            'baz': 'qux'}))

    def test_abc(self):
        """Test MergeableList is a mutable sequence."""
        mergeable_list = MergeableList(['foo', 'bar'])
        self.assertTrue(isinstance(mergeable_list, MutableSequence))

    def test_insert(self):
        """Test the insert method."""
        mergeable_list = MergeableList(['foo', 'bar'])
        mergeable_list.insert(0, 'baz')
        self.assertEquals(
            ['baz', 'foo', 'bar'], [item for item in mergeable_list])

        mergeable_list.insert(-1, 'qux')
        self.assertEquals(
            ['baz', 'foo', 'qux', 'bar'],
            [item for item in mergeable_list])
        mergeable_list.insert(2, 'fnord')
        self.assertEquals(
            ['baz', 'foo', 'fnord', 'qux', 'bar'],
            [item for item in mergeable_list])

    def test_index(self):
        """Test the subset of list behavior we'll be supporting"""
        mergeable_list = MergeableList(['foo', 'bar'])
        self.assertEqual("foo", mergeable_list[0])
        self.assertEqual("bar", mergeable_list[1])
        self.assertEqual("bar", mergeable_list[-1])

    def test_eq(self):
        """Test comparison of regular lists to mergeable lists."""
        the_list = ['foo', 'bar', 'baz']
        mergeable_list = MergeableList(the_list)
        self.assertEqual(mergeable_list, the_list)
        self.assertEqual(mergeable_list, mergeable_list)

    def test_eq2(self):
        """Test comparison of regular lists to mergeable lists."""
        the_list = [{'foo': 'bar'}, {'baz': 'qux'}, {'fnord': 'fierljeppen'}]
        mergeable_list = MergeableList(the_list)
        self.assertEqual(mergeable_list, the_list)
        self.assertEqual(mergeable_list, mergeable_list)

    def test_ne(self):
        """Test comparison of regular lists to mergeable lists."""
        the_list = ['foo', 'bar', 'baz']
        mergeable_list = MergeableList(the_list)
        self.assertNotEqual([], mergeable_list)
        self.assertNotEqual([1, 2, 3], mergeable_list)
        self.assertNotEqual('peanut butter & chocolate', mergeable_list)

    def test_del(self):
        """Test deletion of uuid keys."""
        mergeable_list = MergeableList(['foo', 'bar', 'baz'])
        del mergeable_list[1]
        self.assertEqual(2, len(mergeable_list))
        self.assertEqual("foo", mergeable_list[0])
        self.assertEqual("baz", mergeable_list[1])

    def test_append(self):
        """Test append."""
        mergeable_list = MergeableList(['foo', 'bar'])
        mergeable_list.append('baz')
        self.assertEqual(3, len(mergeable_list))
        self.assertEqual('baz', mergeable_list[-1])

    def test_remove(self):
        """Test remove a normal value"""
        mergeable_list = MergeableList(['foo', 'bar'])
        value = 'baz'
        mergeable_list.append(value)
        mergeable_list.remove(value)
        self.assertFalse(value in mergeable_list)

    def test_remove_missing(self):
        """Test missing data rmeoval"""
        mergeable_list = MergeableList(['foo', 'bar'])
        self.assertRaises(ValueError, mergeable_list.remove, 'qux')

    def test_pop(self):
        """Test the pop method when working with a correct index."""
        value = [1, 2, 3, 4, 5]
        mergeable_list = MergeableList(value)
        # test with negative index
        popped_value = mergeable_list.pop(-2)
        self.assertEqual(value[-2], popped_value)
        # test with positive index
        popped_value = mergeable_list.pop(1)
        self.assertEqual(value[1], popped_value)

    def test_pop_wrong_index(self):
        """Test pop when index is out or range."""
        value = [1, 2, 3, 4, 5]
        mergeable_list = MergeableList(value)
        self.assertRaises(IndexError, mergeable_list.pop,
            len(value) * 2)

    def test_pop_no_index(self):
        """Test pop without an index argument."""
        value = [1, 2, 3, 4, 5]
        mergeable_list = MergeableList(value)
        popped_value = mergeable_list.pop()
        self.assertEqual(value[-1], popped_value)


class TestRecords(TestCase):
    """Test the record functionality"""

    # pylint: disable=C0103
    def setUp(self):
        """Test setup."""
        super(TestRecords, self).setUp()
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                    "field11s": "value11s",
                    "field12s": "value12s"},
            "subfield_uuid": {
                '_order': [
                    'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3',
                    'e47455fb-da05-481e-a2c7-88f14d5cc163'],
                "e47455fb-da05-481e-a2c7-88f14d5cc163": {
                    "field11": "value11",
                    "field12": "value12"},
                "d6d2c23b-279c-45c8-afb2-ec84ee7c81c3": {
                    "field21": "value21",
                    "field22": "value22"}},
            "application_annotations": {
                "my_awesome_app": {"foo": "bar", "some_setting": "set_to"}},
            "record_type": "http://fnord.org/smorgasbord",
        }
        self.record = Record(self.dict)
    # pylint: enable=C0103

    def test_revision(self):
        """Test document always has a revision field and that the revision
        changes when the document is updated"""
        self.assertEquals(self.record.record_revision, None)

        def set_rev(rec):
            """Set revision."""
            rec.record_revision = "1"
        self.assertRaises(AttributeError, set_rev, self.record)

        ctx = test_environment.test_context
        db = DesktopDatabase('testing', create=True, ctx=ctx)
        record_id = db.put_record(self.record)

        db.get_record(record_id)
        self.assertNotEquals(self.record.record_revision, None)

        first = self.record.record_revision

        record_id = db.put_record(self.record)
        db.get_record(record_id)
        second = self.record.record_revision

        self.assertTrue(first < second)

        db.delete_record(record_id)

    def test_delitem(self):
        """Test removing a field from the record"""
        def delete_id(record):
            """Delete record id."""
            del record["_id"]
        self.assertRaises(KeyError, delete_id, self.record)

        del self.record["a"]

    def test_iter(self):
        """Tests it is possible to iterate over the record fields"""
        self.assertEquals(sorted(list(iter(self.record))),
            ['a', 'b', 'subfield', 'subfield_uuid'])

    def test_setitem_internal(self):
        """Test it is not possible to set a private field on a record"""
        def set_id(record):
            """Set document id."""
            record["_id"] = "new!"
        self.assertRaises(IllegalKeyException, set_id, self.record)

    def test_no_record_type(self):
        """Test that creating a record with no record type fails"""
        self.assertRaises(NoRecordTypeSpecified, Record, {})

    def test_get_item(self):
        """Does a RecordDict basically wrap a dict properly?"""
        self.assertEqual(self.dict["a"], self.record["a"])
        self.assertRaises(KeyError,
                          self.record.__getitem__, "application_annotations")

    def test_get(self):
        """Does a RecordDict get() work?"""
        self.assertEqual(self.dict["a"], self.record.get("a"))
        self.assertEqual(None, self.record.get("application_annotations"))

    def test_keys(self):
        """Test getting the keys."""
        self.assertEqual(
            ['a', 'b', 'subfield', 'subfield_uuid'],
            sorted(self.record.keys()))
        self.assertIn("a", self.record)
        self.assertNotIn('f', self.record)
        self.assertNotIn("_id", self.record)  # is internal.  play dumb.

    def test_application_annotations(self):
        """Test getting application specific data."""
        my_app_data = self.record.application_annotations["my_awesome_app"]
        self.assertEqual(
            'bar',
            my_app_data['foo'])
        my_app_data["foo"] = "baz"
        self.assertEqual(
            RecordDict(
                {'my_awesome_app': {'foo': 'baz', "some_setting": "set_to"}}),
            self.record.application_annotations)
        self.assertEqual(
            'baz',
            my_app_data['foo'])

    def test_loads_dict_subdict(self):
        """Are subdicts supported?"""
        self.assertEqual(2, len(self.record["subfield"]))
        subfield_single = self.record["subfield"]
        self.assertEqual(
            subfield_single["field11s"],
            self.dict["subfield"]["field11s"])

    def test_loads_dict_multi_subdict(self):
        """Are subdicts with multiple entries supported?"""
        self.assertEqual(2, len(self.record["subfield_uuid"]))

    def test_list(self):
        """Test assigning lists to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = [1, 2, 3, 4]
        self.assert_(isinstance(rec['key'], list))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_mergeable_list(self):
        """Test assigning lists to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = MergeableList([1, 2, 3, 4])
        self.assert_(isinstance(rec['key'], MergeableList))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_dictionary_access_to_mergeable_list(self):
        """Test that appropriate errors are raised."""
        # pylint: disable=W0212
        keys = self.record["subfield_uuid"]._data.keys()
        # pylint: enable=W0212
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__getitem__, keys[0])
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__setitem__, keys[0], 'stuff')

    def test_validate(self):
        """Test that validation of mixed keys raises appropriate
        error."""
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3})
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'baz':
             {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3},
             'qux': 5})

    def test_uuid_like_keys(self):
        """Test that appropriate errors are raised."""
        self.assertRaises(
            IllegalKeyException,
            self.record["subfield"].__setitem__,
            'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3', 'stuff')

    def test_record_type(self):
        """Test the record_type property."""
        self.assertEqual('http://fnord.org/smorgasbord',
          self.record.record_type)

    def test_run_doctests(self):
        """Run all doc tests from here to set the proper context (ctx)"""
        ctx = test_environment.test_context
        globs = {"db": DesktopDatabase('testing', create=True, ctx=ctx)}

        records_tests_path = os.path.dirname(
            desktopcouch.__file__) + '/records/doc/records.txt'

        app_tests_path = os.path.dirname(
            desktopcouch.__file__) + '/records/doc/an_example_application.txt'
        try:
            results = doctest.testfile(records_tests_path,
                                       globs=globs,
                                       module_relative=False)
            results = doctest.testfile(app_tests_path, module_relative=False)
        except IOError:
            results = doctest.testfile(
                '../desktopcouch/records/doc/records.txt',
                globs=globs,
                module_relative=False)
            results = doctest.testfile(
                '../desktopcouch/records/doc/an_example_application.txt',
                module_relative=False)
        self.assertEqual(0, results.failed)

    def test_record_id(self):
        """Test all passible way to assign a record id"""
        data = {"_id": "recordid"}
        record = Record(data, record_type="url")
        self.assertEqual(data["_id"], record.record_id)
        data = {}
        record_id = "recordid"
        record = Record(data, record_type="url", record_id=record_id)
        self.assertEqual(record_id, record.record_id)
        data = {"_id": "differentid"}
        self.assertRaises(ValueError,
            Record, data, record_id=record_id, record_type="url")

    def test_record_type_version(self):
        """Test record type version support"""
        data = {"_id": "recordid"}
        record1 = Record(data, record_type="url")
        self.assertIs(None, record1._data.get(  # pylint: disable=W0212
            '_record_type_version'))
        self.assertIs(None, record1.record_type_version)
        record2 = Record(data, record_type="url", record_type_version='1.0')
        self.assertEqual(record2._data[  # pylint: disable=W0212
            '_record_type_version'], '1.0')
        self.assertEqual(record2.record_type_version, '1.0')


class TestRecordFactory(TestCase):
    """Test Record/Mergeable List factories."""

    # pylint: disable=C0103
    def setUp(self):
        """Test setup."""
        super(TestRecordFactory, self).setUp()
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                "field11s": "value11s",
                "field12s": "value12s"},
            "subfield_uuid":
                [
                {"field11": "value11",
                 "field12": "value12"},
                {"field21": "value21",
                 "field22": "value22"}],
            "record_type": "http://fnord.org/smorgasbord",
        }
    # pylint: enable=C0103

    def test_build(self):
        """Test RecordDict/MergeableList factory method."""
        record = record_factory(self.dict)
        self.assertEquals('A', record['a'])
        self.assertEquals(
            'value12s', record['subfield']['field12s'])
