# Copyright 2010 Canonical Ltd.
#
# desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
"""Desktopcouch application plugins."""

import logging
import os

DESKTOPCOUCH_PLUGIN_PATHS = [os.path.join(os.path.dirname(__file__))]


def load_plugins(couchdb_port, blocking_semaphores, gobject):
    """Load the desktopcouch application plug-ins.

    The blocking_semaphores set is OPTIONALLY mutated by any plugin to signal
    that the service is not ready until a plugin has finished its asynchronous
    operations.  Plugins may add a distinguishing object to the set, and it
    must remove what it adds when it is finished.

    couchdb -- the integer of the port number of the running couchdb
    blocking_semaphores -- the set() of semaphores, which we will mutate
    gobject -- the mainloop module. always 'gobject' except when testing.
    """
    plugin_names = set()
    for path in DESKTOPCOUCH_PLUGIN_PATHS:
        try:
            for _file in os.listdir(path):
                if _file == '__init__':
                    continue
                elif _file.endswith('.py') and _file != '__init__.py':
                    plugin_names.add(os.path.join(
                            'desktopcouch', 'application', 'plugins', _file))
        except OSError:
            continue

    for name in plugin_names:
        modpath = name.replace(os.path.sep, '.')[:-3]
        try:
            plugin = __import__(modpath, None, None, [''])
            plugin.plugin_init(couchdb_port, blocking_semaphores, gobject)
        except (ImportError, AttributeError):
            logging.warning('Failed to load plug-in: %s', modpath)
