# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Manuel de la Pena <manuel.delapena@canonical.com>
"""Keyring  implementation on windows."""

import base64

from itertools import count

from desktopcouch.application.util import make_random_string


class Keyring(object):
    """Implementation to be used on windows.

    Provides the implementation of the keyring operations on windows
    using the crypto lib and the registry key to store the info.
    """

    def __init__(self, make_random_string_fn=make_random_string,
                 registry=None, crypto=None):
        super(Keyring, self).__init__()
        # ignore pylint, we might not be on windows
        # pylint: disable=F0401
        if registry is None:
            import _winreg
            registry = _winreg

        if crypto is None:
            import win32crypt
            crypto = win32crypt
        # pylint: enable=F0401

        self.make_random_string = make_random_string_fn
        self.registry = registry
        self.crypto = crypto

    def _registry_value_exists(self, value):
        """Return if the required key exists in the system."""
        # ignore the fact that we do not specify the expcetion, we do
        # this so that the tests can run on linux.
        # pylint: disable=W0702
        # try to open it, if not, return false
        try:
            access_rights = self.registry.KEY_ALL_ACCESS
            canonical = self.registry.OpenKey(self.registry.HKEY_CURRENT_USER,
                'Canonical', 0, access_rights)
            keyrings = self.registry.OpenKey(canonical,
                'Keyrings', 0, access_rights)
            default = self.registry.OpenKey(keyrings,
                'Default', 0, access_rights)
            desktopcouch = self.registry.OpenKey(default,
                'Desktopcouch', 0, access_rights)
            # enum until we get a exception
            for index in count():
                try:
                    info = self.registry.EnumValue(
                        desktopcouch, index)
                    if info[0] == value:
                        return True
                except:
                    return False
        except:
            return False
        # pylint: enable=W0702

    def _open_registry_key(self):
        """Open the required registry key"""
        # we need to open the key from the keyring location in the
        # registry, the easiest way is to open step by step each key
        # using CreateKey since it will create the key if required
        canonical_key = self.registry.CreateKey(
            self.registry.HKEY_CURRENT_USER,
            'Canonical')
        keyring_key = self.registry.CreateKey(
            canonical_key, 'Keyrings')
        default_key = self.registry.CreateKey(
            keyring_key, 'Default')
        return self.registry.CreateKey(
            default_key, 'Desktopcouch')

    def get_user_name_password(self):
        """Return the user name and passwd used to access desktopcouch."""
        admin_username = None
        admin_password = None
        data_exists = self._registry_value_exists('basic')
        key = self._open_registry_key()
        if data_exists:
            # read and return it
            secret = self.registry.QueryValueEx(key, 'basic')
            secret = base64.b64decode(secret[0])
            secret = self.crypto.CryptUnprotectData(secret,
                None, None, None, 0)
            admin_username, admin_password = secret[1].split(':', 1)
        else:
            admin_username = self.make_random_string(10)
            admin_password = self.make_random_string(10)
            # encrypt the info before we store it
            secret = ':'.join([admin_username, admin_password])
            secret = self.crypto.CryptProtectData(
                secret, u'basic', None, None, None, 0)
            secret = base64.b64encode(secret)
            # store the secured data
            self.registry.SetValueEx(key, 'basic', 0, self.registry.REG_SZ,
                secret)
        return (admin_username, admin_password)

    def get_oauth_data(self):
        """Return the oauth data used to connect with couchdb."""
        consumer_key = self.make_random_string(10)
        consumer_secret = self.make_random_string(10)
        token = self.make_random_string(10)
        token_secret = self.make_random_string(10)
        # Save the new OAuth creds so that 3rd-party apps can
        # authenticate by accessing the keyring first.  This is
        # one-way.  We don't read from keyring.
        key = self._open_registry_key()
        secret = ':'.join([consumer_key, consumer_secret, token, token_secret])
        secret = self.crypto.CryptProtectData(secret, u'oauth',
            None, None, None, 0)
        secret = base64.b64encode(secret)
        self.registry.SetValueEx(key, 'oauth', 0, self.registry.REG_SZ,
            secret)
        return (consumer_key, consumer_secret, token, token_secret)


class TestKeyring(object):
    """Keyring that will store the secrets on memory for testing reasons."""

    def __init__(self):
        """Create a new instance of the class."""
        self.items = {}

    def get_user_name_password(self):
        """Get username and password from memory."""
        # try to get the data from the items, on keyerror create them since
        # the data is missing
        admin_username = None
        admin_password = None
        try:
            admin_username, admin_password = self.items['basic'].split(':', 1)
        except KeyError:
            admin_username = make_random_string(10)
            admin_password = make_random_string(10)
            self.items['basic'] = ':'.join([admin_username, admin_password])
        return (admin_username, admin_password)

    def get_oauth_data(self):
        """Get oauth data."""
        consumer_key = make_random_string(10)
        consumer_secret = make_random_string(10)
        token = make_random_string(10)
        token_secret = make_random_string(10)
        return (consumer_key, consumer_secret, token, token_secret)
