"""Test the desktopcouch service."""

import logging
import signal

from mocker import MockerTestCase, ANY

from desktopcouch.application.service import (
    DesktopcouchService)

logging.basicConfig(level=logging.ERROR)


def set_up_logging(_):
    """ Set logging to be silence."""
    logging.basicConfig(level=logging.ERROR)


class TestService(MockerTestCase):
    """Test the class that manages the service."""

    def setUp(self):
        super(TestService, self).setUp()
        # set up the dependencies used by the service object
        self._pid_result = 32
        self._port_result = 32323
        self._mainloop = self.mocker.mock()
        self._pid_finder = self.mocker.mock()
        self._port_finder = self.mocker.replace(
            'desktopcouch.application.platform.find_port', passthrough=False)
        self._ctx = self.mocker.mock()
        self._stop_couchdb = self.mocker.mock()
        self._fork = self.mocker.mock()
        self._nice = self.mocker.mock()
        self._kill = self.mocker.mock()
        self._sleep = self.mocker.mock()
        self._replication = self.mocker.mock()
        self._advertiser = self.mocker.mock()
        self._resources = self.mocker.mock()
        self._service = DesktopcouchService(self._mainloop,
            pid_finder=self._pid_finder,
            port_finder=self._port_finder,
            ctx=self._ctx,
            stop_couchdb=self._stop_couchdb,
            replication_actions=self._replication,
            advertiser_factory=self._advertiser,
            set_logging=set_up_logging,
            fork=self._fork,
            nice=self._nice,
            kill=self._kill,
            sleep=self._sleep)

    def test_start_new_desktopcouch_no_extensions(self):
        """Test that desktopcouch is started.

        We test that when the pid cannot be found we ensure
        that the desktopcouch instance is started and that the
        start as the dbus service,
        """
        self._pid_finder(start_if_not_running=False, ctx=self._ctx)
        self.mocker.result(None)
        self._pid_finder(start_if_not_running=True, ctx=self._ctx)
        self.mocker.result(self._pid_result)
        self._port_finder(pid=self._pid_result, ctx=self._ctx)
        self.mocker.result(self._port_result)
        self._fork()
        self.mocker.result(234)
        self._fork()
        self.mocker.result(234)
        # XXX: call this?
        self._mainloop.stop             # pylint: disable=W0104
        self.mocker.result(ANY)
        self._advertiser(ANY, self._ctx)
        self._mainloop.run()
        self._stop_couchdb(ctx=self._ctx)
        self._kill(234, signal.SIGTERM)
        self._sleep(1)
        self._kill(234, signal.SIGKILL)
        self.mocker.replay()
        self._service.start()

    def test_start_new_desktopcouch_extensions(self):
        """Test that desktopcouch is started.

        We test that when the pid cannot be found we ensure
        that the desktopcouch instance is started and that the
        start as the dbus service,
        """
        self._pid_finder(start_if_not_running=False, ctx=self._ctx)
        self.mocker.result(None)
        self._pid_finder(start_if_not_running=True, ctx=self._ctx)
        self.mocker.result(self._pid_result)
        self._port_finder(pid=self._pid_result, ctx=self._ctx)
        self.mocker.result(self._port_result)
        self._fork()
        self.mocker.result(234)
        self._fork()
        self.mocker.result(234)
        # XXX: call this?
        self._mainloop.stop             # pylint: disable=W0104
        self.mocker.result(ANY)
        self._advertiser(ANY, self._ctx)
        self._mainloop.run()
        self._stop_couchdb(ctx=self._ctx)
        self._kill(234, signal.SIGTERM)
        self._sleep(1)
        self._kill(234, signal.SIGKILL)
        self.mocker.replay()
        self._service.start()

    def test_start_desktopcouch_replication(self):
        """ Test that the repliciation works.

        We test that when desktopcouch is already running we
        will not start it again and we will perform the replication.
        """
        self._pid_finder(start_if_not_running=False, ctx=self._ctx)
        self.mocker.result(self._pid_result)
        self._port_finder(pid=self._pid_result, ctx=self._ctx)
        self.mocker.result(self._port_result)
        self._fork()
        self.mocker.result(0)
        self._nice(10)
        self._replication.set_up(ANY)
        self._mainloop.run()
        self.mocker.replay()
        self._service.start()

    def test_start_migrate_data(self):
        """Test that desktopcouch is started.

        We test that when the pid cannot be found we ensure
        that the desktopcouch instance is started and that the
        start as the dbus service,
        """

        server_class = self.mocker.replace("desktopcouch.application." +
                "server.DesktopServer", passthrough=False)
        database_class = self.mocker.replace("desktopcouch.application." +
                "server.DesktopDatabase", passthrough=False)

        self._pid_finder(start_if_not_running=False, ctx=self._ctx)
        self.mocker.result(None)
        self._pid_finder(start_if_not_running=True, ctx=self._ctx)
        self.mocker.result(self._pid_result)
        self._port_finder(pid=self._pid_result, ctx=self._ctx)
        self.mocker.result(self._port_result)
        self._fork()
        self.mocker.result(567)
        self._fork()
        self.mocker.result(0)
        self._sleep(ANY)
        self._ctx.db_dir  # searching for files    # pylint: disable=W0104
        self.mocker.result("/tmp/migration-data/does/not/exist")

        self._port_finder(ctx=self._ctx)     # pylint: disable=W0104
        self.mocker.result(self._port_result)

        server_class('http://localhost:' + str(self._port_result),
                     ctx=self._ctx, oauth_tokens=None)
        self.mocker.result(["fakedb1"])

        d = database_class(database='fakedb1', ctx=self._ctx)
        d.add_view('migrate_deleted', design_doc='dc_migration', map_js=ANY)
        view = self.mocker.mock()
        d.execute_view              # pylint: disable=W0104
        self.mocker.result(view)
        view(design_doc='dc_migration',
             view_name='migrate_deleted')  # pylint: disable=W0104
        record = self.mocker.mock()
        self.mocker.result([record])
        record.id              # pylint: disable=W0104
        self.mocker.result("rid")
        d.delete_record("rid")    # pylint: disable=W0104

        self.mocker.replay()
        self._service.start()
