# Copyright (C) 2008-2011  Canonical, Ltd.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, version 3 of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Gtk ListStore model for backing the TreeViews of cruft."""

from __future__ import absolute_import, unicode_literals

__metaclass__ = type
__all__ = [
    'Store',
    'optimize',
    'unused',
    ]


import gobject

from gi.repository import Gtk


# XXX 2010-03-04 barry: Use a munepy enum.
class ListStoreColumns:
    # The cruft name as known by the dbus service.
    name = 0
    # The short name of the cruft.
    short_name = 1
    # The text displayed in the cell.
    text = 2
    # Should the cruft be shown?
    show = 3
    # Is the cruft active locally (i.e. toggle button is set)?
    active = 4
    # Is the cruft being ignored in the dbus service?
    server_ignored = 5
    # Is the cruft's description expanded?
    expanded = 6
    # Is this cruft package cruft or some other kind of cruft?
    is_package_cruft = 7


class Store:
    """The higher level wrapper around the Gtk.ListStore."""

    def __init__(self, janitord):
        """Create the Store.

        :param janitord: The `dbus.Interface` for talking to the Computer
            Janitor dbus service.
        """
        self.janitord = janitord
        # This ListStore is the backing data structure for the TreeViews in
        # the main window.  It holds information about the individual pieces
        # of cruft.  See ListStoreColumns for details.
        #
        # XXX 2010-03-04 barry: pygtk does not like 'unicode'.
        self.store = Gtk.ListStore(str, str, str, bool, bool, bool, bool, bool)

    def find_cruft(self):
        """Find cruft and populate the backing store.

        This actually just asks the backend dbus service to find all the cruft
        asynchronously.  When the dbus service completes the task, it sends a
        `find_finished` signal with the new list of cruft.  Upon receipt of
        that signal, the ListStore is updated.
        """
        self.janitord.connect_to_signal('find_finished', self.on_find_finished)
        self.janitord.find_async()

    def on_find_finished(self, all_cruft_names):
        """dbus signal handler.

        This actually updates the store with the new cruft results.
        """
        ignored_cruft = set(self.janitord.ignored())
        all_cruft_names = set(all_cruft_names);
        pkg_cruft_names = set(
            cruft_name for cruft_name in all_cruft_names
            if self.janitord.get_details(cruft_name)[0].lower()
            == 'packagecruft')
        self.store.clear()
        # Cruft always starts out in the active, not-expanded state.  Package
        # cruft goes in the 'unused' column while non-package cruft goes in
        # the 'optimize' column.
        for cruft_name in all_cruft_names:
            cruft_shortname = self.janitord.get_shortname(cruft_name)
            self.store.append((
                cruft_name, cruft_shortname,
                # What is being displayed in the cell.
                gobject.markup_escape_text(cruft_shortname),
                # Show the cruft be shown?
                cruft_name not in ignored_cruft,
                # Is the cruft active locally?
                cruft_name not in ignored_cruft,
                # Is the cruft ignored in the dbus service?
                cruft_name in ignored_cruft,
                # Is the cruft's description expanded?
                False,
                # Is this package cruft?
                cruft_name in pkg_cruft_names,
                ))

    # Forwards to the underlying store, for convenience.  Really, these should
    # use generic delegates, or clients should just use self.store.store.

    def get_value(self, *args, **kws):
        return self.store.get_value(*args, **kws)

    def set_value(self, *args, **kws):
        return self.store.set_value(*args, **kws)

    def filter_new(self, *args, **kws):
        return self.store.filter_new(*args, **kws)

    def foreach(self, *args, **kws):
        return self.store.foreach(*args, **kws)

    def reorder(self, *args, **kws):
        return self.store.reorder(*args, **kws)

    def get_iter_first(self, *args, **kws):
        return self.store.get_iter_first(*args, **kws)

    def iter_next(self, *args, **kws):
        return self.store.iter_next(*args, **kws)

    def clear(self, *args, **kws):
        return self.store.clear(*args, **kws)


# Filter functions for use with TreeView column display.

def unused(store, iter, data):
    """True if the cruft is not being ignored and is package cruft.

    :param store: The ListStore instance.
    :param iter: The TreeIter instance.
    """
    show = store.get_value(iter, ListStoreColumns.show)
    is_package_cruft = store.get_value(iter, ListStoreColumns.is_package_cruft)
    return show and is_package_cruft


def optimize(store, iter, data):
    """True if the cruft is not package cruft.

    :param store: The ListStore instance.
    :param iter: The TreeIter instance.
    """
    show = store.get_value(iter, ListStoreColumns.show)
    is_package_cruft = store.get_value(iter, ListStoreColumns.is_package_cruft)
    return show and not is_package_cruft
