#!/usr/bin/python

"""Show paths used by bzr itself and push/pull locations for current branch
Written by Alexander Belchenko, 2006
"""

from bzrlib.commands import register_command
from command import BzrToolsCommand
from bzrlib.option import Option


FORMAT = '%18s: %s'


def _bzr_system_info(to_file):
    import os
    import site
    import sys

    import bzrlib
    from bzrlib.config import config_dir
    from bzrlib import osutils
    from bzrlib.plugin import DEFAULT_PLUGIN_PATH
    from bzrlib import plugins
    from bzrlib import trace

    print >>to_file, FORMAT % ('Python interpreter', sys.executable)
    print >>to_file, FORMAT % ('Python library',
                               os.path.dirname(site.__file__))

    print >>to_file, FORMAT % ('bzr executable', osutils.realpath(sys.argv[0]))
    print >>to_file, FORMAT % ('bzrlib', osutils.realpath(bzrlib.__path__[0]))

    print >>to_file, FORMAT % ('bzr config dir', osutils.realpath(config_dir()))

    dirs = os.environ.get('BZR_PLUGIN_PATH', DEFAULT_PLUGIN_PATH).split(os.pathsep)
    dirs.insert(0, os.path.dirname(plugins.__file__))
    if len(dirs) == 1:
        print >>to_file, FORMAT % ('bzr plugins dir', osutils.realpath(dirs[0]))
    else:
        print >>to_file, FORMAT % ('bzr plugins dirs', osutils.realpath(dirs[0]))
        for dir_ in dirs[1:]:
            print >>to_file, FORMAT % ('', osutils.realpath(dir_))

    print >>to_file, FORMAT % ('.bzr.log', osutils.realpath(trace._bzr_log_file.name))
#/def _bzr_system_info


class cmd_show_paths(BzrToolsCommand):
    """Show paths used by bzr itself and for current branch.

    The standard "bzr info" command now includes this functionality.

    If you run this command from branch you'll see
    saved path locations for current branch:

        * branch root - root directory of current branch
        * pull from   - default location for pull command
        * push to     - default location for push command
        * bound to    - for checkouts: location of repository branch
        * submit to   - default reference location for bundle generation
    """

    takes_options = [Option('system',
                            help='Show full bzr system information.'),
                    ]
    takes_args = ['branch?']

    hidden = True

    _see_also = ['info']

    def run(self, system=False, branch=u'.'):
        import sys

        from bzrlib.branch import Branch
        from bzrlib.errors import NotBranchError, NoWorkingTree
        from bzrlib import urlutils
        from bzrlib.workingtree import WorkingTree

        def local_path(path):
            if path.startswith("file://"):
                return urlutils.local_path_from_url(path)
            else:
                return urlutils.unescape(path)

        to_file = getattr(self, 'outf', sys.stdout)

        if system:
            print >>to_file, "*** Main bzr paths info ***"
            _bzr_system_info(to_file)

        try:
            branch = Branch.open_containing(branch)[0]

            if system:
                print >>to_file
            print >>to_file, "*** Current branch paths info ***"

            branch_root = branch.bzrdir.root_transport.base
            print >>to_file, FORMAT % ('branch root', local_path(branch_root))

            pull_loc = branch.get_parent() or 'None'
            print >>to_file, FORMAT % ('branch pull from', local_path(pull_loc))

            push_loc = branch.get_push_location() or 'None'
            print >>to_file, FORMAT % ('branch push to', local_path(push_loc))

            bound_loc = branch.get_bound_location() or 'None'
            print >>to_file, FORMAT % ('bound to branch', local_path(bound_loc))

            if hasattr(branch, 'get_submit_branch'):
                submit_loc = branch.get_submit_branch() or 'None'
                print >>to_file, FORMAT % ('submit to branch', local_path(submit_loc))

        except NotBranchError:
            if not system:
                raise
#/class cmd_show_paths


register_command(cmd_show_paths)
