#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Provides AptWorker which processes transactions."""
# Copyright (C) 2008-2009 Sebastian Heinlein <devel@glatzor.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

__author__  = "Sebastian Heinlein <devel@glatzor.de>"

import fcntl
import logging
import os
import struct
import sys
import time
import traceback

import apt
import apt.cache
import apt.debfile
import apt_pkg
from gettext import gettext as _
import gobject
from softwareproperties.AptAuth import AptAuth

from enums import *
from errors import *
from progress import DaemonOpenProgress, \
                     DaemonInstallProgress, \
                     DaemonFetchProgress, \
                     DaemonDpkgInstallProgress, \
                     DaemonDpkgRecoverProgress

log = logging.getLogger("AptDaemon.Worker")


class LockFailedError(Exception):
    """The locking of file failed."""
    def __init__(self, flock, process=None):
        """Return a new LockFailedError instance.

        Keyword arguments:
        flock -- the path of the file lock
        process -- the process which holds the lock or None
        """
        self.flock = flock
        self.process = process


class AptWorker(gobject.GObject):

    """Worker which processes transactions from the queue."""

    __gsignals__ = {"transaction-done":(gobject.SIGNAL_RUN_FIRST,
                                        gobject.TYPE_NONE,
                                        (gobject.TYPE_STRING,))}

    def __init__(self):
        """Initialize a new AptWorker instance."""
        gobject.GObject.__init__(self)
        self.trans = None
        self.last_action_timestamp = time.time()
        self._cache = None
        self._lock_fd = -1

    def run(self, transaction):
        """Process the given transaction in the background.

        Keyword argument:
        transaction -- core.Transcation instance to run
        """
        log.info("Processing transaction %s", transaction.tid)
        if self.trans:
            raise Exception("There is already a running transacion")
        self.trans = transaction
        gobject.idle_add(self._process_transaction)

    def _emit_transaction_done(self, tid):
        """Emit the transaction-done signal.

        Keyword argument:
        tid -- the id of the finished transaction
        """
        log.debug("Emitting transaction-done: %s", tid)
        self.emit("transaction-done", tid)

    def _process_transaction(self):
        """Run the worker"""
        self.last_action_timestamp = time.time()
        self.trans.status = STATUS_RUNNING
        self.trans.progress = 0
        try:
            # Prepare the package cache
            self._lock_cache()
            if self.trans.role == ROLE_FIX_INCOMPLETE_INSTALL:
                self.fix_incomplete_install()
            elif not is_dpkg_journal_clean():
                raise TransactionFailed(ERROR_INCOMPLETE_INSTALL)
            self._open_cache()
            # Process transaction which can handle a broken dep cache
            if self.trans.role == ROLE_FIX_BROKEN_DEPENDS:
                self.fix_broken_depends()
            elif self.trans.role == ROLE_UPDATE_CACHE:
                self.update_cache()
            elif self.trans.role == ROLE_ADD_VENDOR_KEY_FILE:
                self.add_vendor_key_from_file(**self.trans.kwargs)
            elif self.trans.role == ROLE_REMOVE_VENDOR_KEY:
                self.remove_vendor_key(**self.trans.kwargs)
            # Process the transactions which require a consistent cache
            elif self._cache.broken_count:
                broken = [pkg.name for pkg in self._cache if pkg.is_now_broken]
                raise TransactionFailed(ERROR_CACHE_BROKEN, " ".join(broken))
            elif self.trans.role == ROLE_INSTALL_PACKAGES:
                self.install_packages(self.trans.packages[0])
            elif self.trans.role == ROLE_INSTALL_FILE:
                self.install_file(**self.trans.kwargs)
            elif self.trans.role == ROLE_REMOVE_PACKAGES:
                self.remove_packages(self.trans.packages[2])
            elif self.trans.role == ROLE_UPGRADE_SYSTEM:
                self.upgrade_system(**self.trans.kwargs)
            elif self.trans.role == ROLE_UPGRADE_PACKAGES:
                self.upgrade_packages(self.trans.packages[4])
            elif self.trans.role == ROLE_COMMIT_PACKAGES:
                self.commit_packages(*self.trans.packages)
        except TransactionCancelled:
            self.trans.exit = EXIT_CANCELLED
        except TransactionFailed, excep:
            self.trans.error = excep
            self.trans.exit = EXIT_FAILED
        except (KeyboardInterrupt, SystemExit):
            self.trans.exit = EXIT_CANCELLED
        except Exception, excep:
            self.trans.error = TransactionFailed(ERROR_UNKNOWN,
                                                 traceback.format_exc())
            self.trans.exit = EXIT_FAILED
        else:
            self.trans.exit = EXIT_SUCCESS
        finally:
            self.trans.progress = 100
            self.last_action_timestamp = time.time()
            tid = self.trans.tid[:]
            self.trans = None
            self._emit_transaction_done(tid)
            self._unlock_cache()
            log.info("Finished transaction %s", tid)
        return False

    def commit_packages(self, install, reinstall, remove, purge, upgrade):
        """Perform a complex package operation.

        Keyword arguments:
        install - list of package names to install
        reinstall - list of package names to reinstall
        remove - list of package names to remove
        purge - list of package names to purge including configuration files
        upgrade - list of package names to upgrade
        """
        log.info("Committing packages: %s, %s, %s, %s, %s",
                 install, reinstall, remove, purge, upgrade)
        #FIXME python-apt 0.8 introduced a with statement
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_installation(install, resolver)
        self._mark_packages_for_installation(reinstall, resolver,
                                             reinstall=True)
        self._mark_packages_for_removal(remove, resolver)
        self._mark_packages_for_removal(purge, resolver, purge=True)
        self._mark_packages_for_upgrade(upgrade, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _resolve_depends(self, resolver):
        """Resolve the dependencies using the given ProblemResolver."""
        resolver.install_protect()
        try:
            resolver.resolve()
        except SystemError:
            broken = [pkg.name for pkg in self._cache if pkg.is_inst_broken]
            raise TransactionFailed(ERROR_DEP_RESOLUTION_FAILED,
                                    " ".join(broken))

    def install_packages(self, package_names):
        """Install packages.

        Keyword argument:
        package_names -- list of package name which should be installed
        """
        log.debug("Installing packages: %s", package_names)
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_installation(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _check_unauthenticated(self):
        """Check if any of the cache changes get installed from an
        unauthenticated repository"""
        if self.trans.allow_unauthenticated:
            return
        unauthenticated = []
        for pkg in self._cache:
            if (pkg.markedInstall or
                pkg.markedDowngrade or
                pkg.markedUpgrade or
                pkg.markedReinstall):
                trusted = False
                for origin in pkg.candidate.origins:
                    trusted |= origin.trusted
                if not trusted:
                    unauthenticated.append(pkg.name)
        if unauthenticated:
            raise TransactionFailed(ERROR_PACKAGE_UNAUTHENTICATED,
                                    " ".join(sorted(unauthenticated)))

    def _mark_packages_for_installation(self, package_names, resolver,
                                        reinstall=False):
        """Mark packages for installation."""
        for pkg_name in package_names:
            if self._cache.has_key(pkg_name):
                pkg = self._cache[pkg_name]
            else:
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            if reinstall:
                if not pkg.isInstalled:
                    raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                            "Package %s isn't installed" % \
                                            pkg.name)
            else:
                #FIXME: Turn this into a non-critical message
                if pkg.isInstalled:
                    raise TransactionFailed(ERROR_PACKAGE_ALREADY_INSTALLED,
                                            "Package %s is already installed" %\
                                            pkg_name)
            pkg.markInstall(False, True, True)
            resolver.clear(pkg)
            resolver.protect(pkg)
 
    def add_vendor_key_from_file(self, path):
        """Add the signing key from the given file to the trusted vendors.

        Keyword argument:
        path -- absolute path to the key file
        """
        log.info("Adding vendor key from file: %s", path)
        try:
            #FIXME: use gobject.spawn_async or reactor.spawn
            #FIXME: use --dry-run before?
            auth = AptAuth()
            auth.add(os.path.expanduser(path))
        except Exception, error:
            raise TransactionFailed(ERROR_KEY_NOT_INSTALLED,
                                    "Key file %s couldn't be installed: %s" % \
                                    (path, error))

    def remove_vendor_key(self, fingerprint):
        """Remove repository key.

        Keyword argument:
        fingerprint -- fingerprint of the key to remove
        """
        log.info("Removing vendor key: %s", fingerprint)
        try:
            #FIXME: use gobject.spawn_async or reactor.spawn
            #FIXME: use --dry-run before?
            auth = AptAuth()
            auth.rm(fingerprint)
        except Exception, error:
            raise TransactionFailed(ERROR_KEY_NOT_REMOVED,
                                    "Key with fingerprint %s couldn't be "
                                    "removed: %s" % (fingerprint, error))

    def install_file(self, path):
        """Install local package file.

        Keyword argument:
        path -- absolute path to the package file
        """
        log.info("Installing local package file: %s", path)
        # Check if the dpkg can be installed at all
        deb = apt.debfile.DebPackage(path, self._cache)
        if not deb.check():
            raise TransactionFailed(ERROR_DEP_RESOLUTION_FAILED,
                                    deb._failure_string)
        # Check for required changes and apply them before
        (install, remove, unauth) = deb.required_changes
        if len(install) > 0 or len(remove) > 0:
            dpkg_range = (64, 99)
            self._commit_changes(fetch_range=(5, 33),
                                 install_range=(34, 63))
            self._lock_cache()
        # Install the dpkg file
        if deb.install(DaemonDpkgInstallProgress(self.trans,
                                                 begin=64, end=95)):
            raise TransactionFailed(ERROR_UNKNOWN, deb._failure_string)

    def remove_packages(self, package_names):
        """Remove packages.

        Keyword argument:
        package_names -- list of package name which should be installed
        """
        log.info("Removing packages: '%s'", package_names)
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_removal(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes(fetch_range=(10, 10), install_range=(10, 90))
        #FIXME: should we use a persistant cache? make a check?
        #self._open_cache(prange=(90,99))
        #for p in pkgs:
        #    if self._cache.has_key(p) and self._cache[p].isInstalled:
        #        self.ErrorCode(ERROR_UNKNOWN, "%s is still installed" % p)
        #        self.Finished(EXIT_FAILED)
        #        return

    def _mark_packages_for_removal(self, package_names, resolver, purge=False):
        """Mark packages for installation."""
        for pkg_name in package_names:
            if not self._cache.has_key(pkg_name):
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            pkg = self._cache[pkg_name]
            if not pkg.isInstalled:
                raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                        "Package %s isn't installed" % pkg_name)
            if pkg._pkg.Essential == True:
                raise TransactionFailed(ERROR_NOT_REMOVE_ESSENTIAL_PACKAGE,
                                        "Package %s cannot be removed." % \
                                        pkg_name)
            pkg.markDelete(False, purge=purge)
            resolver.clear(pkg)
            resolver.protect(pkg)
            resolver.remove(pkg)

    def _check_obsoleted_dependencies(self):
        """Mark obsoleted dependencies of to be removed packages for removal."""
        if not self.trans.remove_obsoleted_depends:
            return
        installed_deps = set()
        ac = self._cache.actiongroup()
        for pkg in self._cache:
            if pkg.markedDelete:
                installed_deps = self._installed_dependencies(pkg.name,
                                                              installed_deps)
        for dep_name in installed_deps:
            if dep_name in self._cache:
                pkg = self._cache[dep_name]
                if pkg.isInstalled and pkg.isAutoRemovable:
                    pkg.markDelete(False)
        ac.release()

    def _installed_dependencies(self, pkg_name, all_deps=None):
        """Recursively return all installed dependencies of a given package."""
        #FIXME: Should be part of python-apt, since it makes use of non-public
        #       API. Perhaps by adding a recursive argument to
        #       apt.package.Version.get_dependencies()
        if not all_deps:
            all_deps = set()
        if not pkg_name in self._cache:
            return all_deps
        cur = self._cache[pkg_name]._pkg.CurrentVer
        if not cur:
            return all_deps
        for t in ("PreDepends", "Depends", "Recommends"):
            try:
                for dep in cur.DependsList[t]:
                    dep_name = dep[0].TargetPkg.Name
                    if not dep_name in all_deps:
                        all_deps.add(dep_name)
                        all_deps |= self._installed_dependencies(dep_name,
                                                                 all_deps)
            except KeyError:
                pass
        return all_deps

    def upgrade_packages(self, package_names):
        """Upgrade packages.

        Keyword argument:
        package_names -- list of package name which should be upgraded
        """
        log.info("Upgrading packages: %s", package_names)
        ac = self._cache.actiongroup()
        resolver = apt.cache.ProblemResolver(self._cache)
        self._mark_packages_for_upgrade(package_names, resolver)
        self._resolve_depends(resolver)
        ac.release()
        self._commit_changes()

    def _mark_packages_for_upgrade(self, package_names, resolver):
        """Mark packages for upgrade."""
        for pkg_name in package_names:
            if not self._cache.has_key(pkg_name):
                raise TransactionFailed(ERROR_NO_PACKAGE,
                                        "Package %s isn't available" % pkg_name)
            pkg = self._cache[pkg_name]
            if not pkg.isInstalled:
                raise TransactionFailed(ERROR_PACKAGE_NOT_INSTALLED,
                                        "Package %s isn't installed" % pkg_name)
            auto = not self._cache._depcache.IsAutoInstalled(pkg._pkg)
            pkg.markInstall(False, True, auto)
            resolver.clear(pkg)
            resolver.protect(pkg)

    def update_cache(self):
        """Update the cache."""
        log.info("Updating cache")
        progress = DaemonFetchProgress(self.trans, begin=10, end=95)
        try:
            self._cache.update(progress, raiseOnError=True)
        except apt.cache.FetchFailedException:
            failed_downloads = ""
            for uri, status in progress.items.iteritems():
                #FIXME: are there other files we want to ignore?
                if status == apt.progress.FetchProgress.dlFailed and \
                   not uri.endswith("Packages.diff/Index"):
                    failed_downloads += uri + "\n"
            if failed_downloads:
                raise TransactionFailed(ERROR_REPO_DOWNLOAD_FAILED,
                                        failed_downloads)
        except apt.cache.FetchCancelledException:
            raise TransactionCancelled()
        except SystemError, excep:
            raise TransactionFailed(ERROR_REPO_DOWNLOAD_FAILED,
                                    excep.message)

    def upgrade_system(self, safe_mode=True):
        """Upgrade the system.

        Keyword argument:
        safe_mode -- if additional software should be installed or removed to
                     satisfy the dependencies the an updates
        """
        log.info("Upgrade system with safe mode: %s" % safe_mode)
        # Check for available updates
        updates = filter(lambda p: p.isUpgradable,
                         self._cache)
        #FIXME: What to do if already uptotdate? Add error code?
        self._cache.upgrade(distUpgrade=not safe_mode)
        # Check for blocked updates
        outstanding = []
        changes = self._cache.getChanges()
        for pkg in updates:
            if not pkg in changes or not pkg.markedUpgrade:
                outstanding.append(pkg)
        #FIXME: Add error state if system could not be fully updated
        self._commit_changes()

    def fix_incomplete_install(self):
        """Run dpkg --configure -a to recover from a failed installation."""
        log.info("Fixing incomplete installs")
        self.trans.status = STATUS_CLEANING_UP
        self._unlock_cache()
        progress = DaemonDpkgRecoverProgress(self.trans)
        progress.startUpdate()
        progress.run()
        progress.finishUpdate()
        if progress._child_exit != 0:
            raise TransactionFailed(ERROR_PACKAGE_MANAGER_FAILED,
                                    progress.output)

    def fix_broken_depends(self):
        """Try to fix broken dependencies."""
        log.info("Fixing broken depends")
        try:
            self._cache._depcache.FixBroken()
        except SystemError:
            broken = [pkg.name for pkg in self._cache if pkg.is_inst_broken]
            raise TransactionFailed(ERROR_DEP_RESOLUTION_FAILED,
                                    " ".join(broken))
        self._commit_changes()

    def _open_cache(self, begin=0, end=5, quiet=False):
        """Open the APT cache.

        Keyword arguments:
        start -- the begin of the progress range
        end -- the end of the the progress range
        quiet -- if True do no report any progress
        """
        self.trans.status = STATUS_LOADING_CACHE
        try:
            progress = DaemonOpenProgress(self.trans, begin=begin, end=end,
                                          quiet=quiet)
            if not isinstance(self._cache, apt.cache.Cache):
                self._cache = apt.cache.Cache(progress)
            else:
                self._cache.open(progress)
        except Exception, excep:
            raise TransactionFailed(ERROR_NO_CACHE, excep.message)

    def _lock_cache(self):
        """Lock the APT cache."""
        try:
            self._lock_fd = lock_pkg_system()
        except LockFailedError, error:
            logging.error("Failed to lock the cache")
            self.trans.paused = True
            self.trans.status = STATUS_WAITING_LOCK
            if error.process:
                #TRANSLATORS: %s is the name of a package manager
                msg = "Waiting for %s to exit" % error.process
                self.trans.status_details = msg
            lock_watch = gobject.timeout_add_seconds(3, self._watch_lock)
            while self.trans.paused and not self.trans.cancelled:
                gobject.main_context_default().iteration()
            gobject.source_remove(lock_watch)
            if self.trans.cancelled:
                raise TransactionCancelled()

    def _watch_lock(self):
        """Unpause the transaction if the lock can be obtained."""
        try:
            lock_pkg_system()
        except LockFailedError:
            return True
        self.trans.paused = False
        return False

    def _unlock_cache(self):
        """Unlock the APT cache."""
        if self._lock_fd != -1:
            os.close(self._lock_fd)
            self._lock_fd = -1

    def _commit_changes(self, fetch_range=(5, 50), install_range=(50, 90)):
        """Commit previously marked changes to the cache.

        Keyword arguments:
        fetch_range -- tuple containing the start and end point of the
                       download progress
        install_range -- tuple containing the start and end point of the
                         install progress
        """
        self._check_obsoleted_dependencies()
        self._check_unauthenticated()
        if self.trans.cancelled:
            raise TransactionCancelled()
        self.trans.cancellable = False
        fetch_progress = DaemonFetchProgress(self.trans,
                                             begin=fetch_range[0],
                                             end=fetch_range[1])
        inst_progress = DaemonInstallProgress(self.trans,
                                              begin=install_range[0],
                                              end=install_range[1])
        self._unlock_cache()
        try:
            self._cache.commit(fetch_progress, inst_progress)
        except apt.cache.FetchFailedException, excep:
            raise TransactionFailed(ERROR_PACKAGE_DOWNLOAD_FAILED, 
                                    str(excep))
        except apt.cache.FetchCancelledException:
            raise TransactionCancelled()
        except SystemError, excep:
            # Run dpkg --configure -a to recover from a failed transaction
            self.trans.status = STATUS_CLEANING_UP
            self._unlock_cache()
            progress = DaemonDpkgRecoverProgress(self.trans, begin=90, end=95)
            progress.startUpdate()
            progress.run()
            progress.finishUpdate()
            output = inst_progress.output + progress.output
            raise TransactionFailed(ERROR_PACKAGE_MANAGER_FAILED,
                                    "%s: %s" % (excep, output))


class DummyWorker(AptWorker):

    """Allows to test the daemon without making any changes to the system."""

    def run(self, transaction):
        """Process the given transaction in the background.

        Keyword argument:
        transaction -- core.Transcation instance to run
        """
        log.info("Processing transaction %s", transaction.tid)
        if self.trans:
            raise Exception("There is already a running transacion")
        self.trans = transaction
        self.last_action_timestamp = time.time()
        self.trans.status = STATUS_RUNNING
        self.trans.progress = 0
        self.trans.cancellable = True
        gobject.timeout_add(200, self._process_transaction, transaction)

    def _process_transaction(self, trans):
        """Run the worker"""
        if trans.cancelled:
            trans.exit = EXIT_CANCELLED
        elif trans.progress == 100:
            trans.exit = EXIT_SUCCESS
        else:
            if trans.role == ROLE_INSTALL_PACKAGES:
                if trans.progress == 1:
                    trans.status = STATUS_RESOLVING_DEP
                elif trans.progress == 5:
                    trans.status = STATUS_DOWNLOADING
                elif trans.progress == 50:
                    trans.status = STATUS_COMMITTING
                    trans.status_details = "Heyas!"
                elif trans.progress == 55:
                    trans.paused = True
                    trans.status = STATUS_WAITING_CONFIG_FILE_PROMPT
                    trans.config_file_conflict = "/etc/fstab", "/etc/mtab"
                    while trans.paused:
                        gobject.main_context_default().iteration()
                    trans.config_file_conflict_resolution = None
                    trans.config_file_conflict = None
                    trans.status = STATUS_COMMITTING
                elif trans.progress == 60:
                    trans.required_medium = ("Debian Lenny 5.0 CD 1",
                                             "USB CD-ROM")
                    trans.paused = True
                    trans.status = STATUS_WAITING_MEDIUM
                    while trans.paused:
                        gobject.main_context_default().iteration()
                    trans.status = STATUS_DOWNLOADING
                elif trans.progress == 70:
                    trans.status_details = "Servus!"
                elif trans.progress == 90:
                    trans.status_deatils = ""
                    trans.status = STATUS_CLEANING_UP
            elif trans.role == ROLE_REMOVE_PACKAGES:
                if trans.progress == 1:
                    trans.status = STATUS_RESOLVING_DEP
                elif trans.progress == 5:
                    trans.status = STATUS_COMMITTING
                    trans.status_details = "Heyas!"
                elif trans.progress == 50:
                    trans.status_details = "Hola!"
                elif trans.progress == 70:
                    trans.status_details = "Servus!"
                elif trans.progress == 90:
                    trans.status_deatils = ""
                    trans.status = STATUS_CLEANING_UP
            self.trans.progress += 1
            return True
        self.status = STATUS_FINISHED
        self.last_action_timestamp = time.time()
        tid = self.trans.tid[:]
        self.trans = None
        self._emit_transaction_done(tid)
        log.info("Finished transaction %s", tid)
        return False


def lock_pkg_system():
    """Lock the package system and provide information if this cannot be done.

    This is a reemplemenataion of apt_pkg.PkgSystemLock(), since we want to
    handle an incomplete dpkg run separately.
    """
    def get_lock_fd(lock_path):
        """Return the file descriptor of the lock file or raise
        LockFailedError if the lock cannot be obtained.
        """
        fd_lock = apt_pkg.GetLock(lock_path)
        if fd_lock < 0:
            process = None
            try:
                # Get the pid of the locking application
                fd_lock_read = open(lock_path, "r")
                flk = struct.pack('hhQQi', fcntl.F_WRLCK, os.SEEK_SET, 0, 0, 0)
                flk_ret = fcntl.fcntl(fd_lock_read, fcntl.F_GETLK, flk)
                pid = struct.unpack("hhQQi", flk_ret)[4]
                # Get the command of the pid
                fd_status = open("/proc/%s/status" % pid, "r")
                try:
                    for key, value in (line.split(":") for line in \
                                       fd_status.readlines()):
                        if key == "Name":
                            process = value.strip()
                            break
                finally:
                    fd_status.close()
            except:
                pass
            finally:
                fd_lock_read.close()
            raise LockFailedError(lock_path, process)
        else:
            return fd_lock

    # Try the lock in /var/cache/apt/archive/lock first
    # this is because apt-get install will hold it all the time
    # while the dpkg lock is briefly given up before dpkg is
    # forked off. this can cause a race (LP: #437709)
    lock_archive = os.path.join(apt_pkg.Config.FindDir("Dir::Cache::Archives"),
                                "lock")
    lock_fd_archive = get_lock_fd(lock_archive)
    os.close(lock_fd_archive)
    # Then the status lock
    status_file = apt_pkg.Config.FindFile("Dir::State::status")
    lock_sys = os.path.join(os.path.dirname(status_file), "lock")
    return get_lock_fd(lock_sys)

def is_dpkg_journal_clean():
    """Return False if there are traces of incomplete dpkg status updates."""
    status_file = apt_pkg.Config.FindFile("Dir::State::status")
    status_updates = os.path.join(os.path.dirname(status_file), "updates/")
    for dentry in os.listdir(status_updates):
        if dentry.isdigit():
            return False
    return True

# vim:ts=4:sw=4:et
