/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2003 Joop Stakenborg <pa4tu@amsat.org>
 *
 * This program is free oftware; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* callbacks_otherdialogs.c
 *
 * This is where the callbacks for all the different dialogs reside. 
 * For callbacks related to the widgets in the main window, see the files
 * called callbacks_<widgetname>.[c|h].
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>
#include <gtk/gtk.h>

#include "callbacks_otherdialogs.h"
#include "interface.h"
#include "support.h"
#include "types.h"
#include "log.h"
#include "utils.h"
#include "gtkutils.h"

#define SEARCHHISTORY 10

extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern preferencestype preferences;
extern statetype state;
extern GtkWidget *b4dialog;
extern GList *searchhistory;
extern GList *logwindowlist;
gchar *globtemp;
gint globtype = 1;

/* qsort/bsearch callback functions */
typedef int (*compar_callback_fn) (const void *, const void *);

/* SETUP DIALOG */
void
on_setupbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_show (mainwindow);
  gdk_window_move_resize (mainwindow->window, preferences.x, preferences.y,
			  preferences.width, preferences.height);
}

/* setup dialog is deleted */
gboolean
on_setupdialog_delete_event (GtkWidget * widget, GdkEvent * event,
			     gpointer user_data)
{
  gtk_widget_show (mainwindow);
  gdk_window_move_resize (mainwindow->window, preferences.x, preferences.y,
			  preferences.width, preferences.height);
  return FALSE;
}

/* ABOUT DIALOG */
void
on_aboutbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* SAVE DIALOG */
void
on_saveokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_exit (0);
}

/* cancel clicked, nothing saved */
void
on_savecancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* save dialog is deleted */
gboolean
on_savedialog_delete_event (GtkWidget * widget, GdkEvent * event,
			    gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


/* SEARCH DIALOG */
void
on_searchokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gchar *searchstr, *upsearchstr, *callsign, *labeltext;
  GtkWidget *searchdialog, *searchresultdialog, *searchcomboentry,
    *resultlabel, *searchcheckbutton;
  gint i, row;
  gboolean result = FALSE, searchopen;
  GList *node;
  logtype *logwindow;

  searchdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  searchcomboentry = lookup_widget (searchdialog, "searchcomboentry");
  searchcheckbutton = lookup_widget (searchdialog, "searchcheckbutton");

  searchopen = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					     (searchcheckbutton));

  searchstr = gtk_editable_get_chars (GTK_EDITABLE (searchcomboentry), 0, -1);

  /* add to the searchlist if it is a new one */
  node = g_list_find_custom (searchhistory, g_strdup (searchstr),
			     (GCompareFunc) strncmp);
  if (!node)
    searchhistory = g_list_prepend (searchhistory, g_strdup (searchstr));
  if (g_list_length (searchhistory) > SEARCHHISTORY)
    searchhistory = g_list_remove (searchhistory,
				   g_list_last (searchhistory)->data);

  upsearchstr = g_strdup (searchstr);
  g_strup (upsearchstr);

  /* search all logs and select corresponding QSO */
  for (i = state.searchlog; i < g_list_length (logwindowlist); i++)
    {
      logwindow = g_list_nth_data (logwindowlist, i);
      for (row = state.searchqso; row < logwindow->qsos; row++)
	{
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
			      CALL, &callsign);
	  if (strstr (callsign, upsearchstr))
	    {
	      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook), i);
	      gtk_clist_moveto (GTK_CLIST (logwindow->clist), row, 0, 0.5, 0);
	      gtk_clist_select_row (GTK_CLIST (logwindow->clist), row, 0);
	      result = TRUE;
	    }
	  if (result)
	    break;
	}
      /* reset search */
      if (row == logwindow->qsos
	  && state.searchlog < g_list_length (logwindowlist))
	{
	  state.searchqso = 0;
	  state.searchlog++;
	  if (state.searchlog == g_list_length (logwindowlist))
	    state.searchlog = 0;
	}
      else
	state.searchqso = row + 1;
      if (result)
	break;
    }

  /* not found, show the result */
  if (!result)
    {
      state.searchqso = 0;
      searchresultdialog = create_searchresultdialog ();
      resultlabel = lookup_widget (searchresultdialog, "resultlabel");
      labeltext = g_strdup_printf (_("\'%s\' was not found"), searchstr);
      gtk_label_set_text (GTK_LABEL (resultlabel), labeltext);
      g_free (labeltext);
      gtk_widget_show (searchresultdialog);
    }

  g_free (searchstr);
  g_free (upsearchstr);

  if (!searchopen)
    {
      gtk_widget_set_sensitive (mainwindow, 1);
      gtk_widget_destroy (searchdialog);
    }
}


/* search cancel */
void
on_searchcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* search changed, reset saved state */
void
on_searchcomboentry_changed (GtkEditable * editable, gpointer user_data)
{
  state.searchqso = 0;
  state.searchlog = 0;
}

/* toggle 'keep this dialog open' */
void
on_searchcheckbutton_toggled (GtkToggleButton * togglebutton,
			      gpointer user_data)
{
  GtkWidget *searchdialog, *searchcheckbutton;
  gboolean searchopen;

  searchdialog = gtk_widget_get_toplevel (GTK_WIDGET (togglebutton));
  searchcheckbutton = lookup_widget (searchdialog, "searchcheckbutton");
  searchopen =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (searchcheckbutton));
  if (searchopen)
    gtk_widget_set_sensitive (mainwindow, 1);
  else
    gtk_widget_set_sensitive (mainwindow, 0);
}

gboolean
on_searchdialog_delete_event (GtkWidget * widget, GdkEvent * event,
			      gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* SEARCHRESULT DIALOG */
void
on_searchresultokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}


/* NEW LOG DIALOG */
void
on_newlogokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *newlogentry, *newlogdialog, *menubar, *log_editor_menu;
  gchar *entry, *filename, *xlogbackupfile;
  logtype *logwindow = NULL;
  gint i;
  LOGDB *lp;
  gboolean logopen = FALSE, logexist = FALSE;
  GString *logs;

  newlogdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  newlogentry = lookup_widget (newlogdialog, "newlogentry");
  entry = gtk_editable_get_chars (GTK_EDITABLE (newlogentry), 0, -1);

  if (strlen (entry) > 0)
    {
      for (i = 0; i < g_list_length (logwindowlist); i++)
	{			/* see if the log is already open */
	  logwindow = g_list_nth_data (logwindowlist, i);
	  if (g_strcasecmp (entry, logwindow->logname) == 0)
	    {
	      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook), i);
	      set_qsoframe (logwindow);
	      logopen = TRUE;
	      break;
	    }
	}
      if (!logopen)
	{			/* check if the log exists and open it */
	  filename = g_strconcat (entry, ".xlog", 0);
	  logs = getlogs (preferences.savedir, 0);
	  if (((logs->len) > 0) && (strstr (logs->str, filename)))
	    {
	      logexist = TRUE;
	      filename =
		g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, entry,
			     ".xlog", NULL);
	      lp = log_file_open (filename, TYPE_FLOG);
	      if (lp)
		{
		  logwindow = openlog (lp, entry, state.logwindows++);
		  log_file_qso_foreach (lp, fillin_clist, logwindow);
		  log_file_close (lp);
		  logwindowlist = g_list_append (logwindowlist, logwindow);
		  gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook),
					 state.logwindows - 1);
		  set_qsoframe (logwindow);
		  xlogbackupfile = g_strconcat (filename, ".backup", NULL);
		  savelog (logwindow, xlogbackupfile, TYPE_FLOG);
		  g_free (xlogbackupfile);
		}
	      g_free (filename);
	      g_string_free (logs, TRUE);
	    }
	}
      if ((!logopen) && (!logexist))
	{			/* create a new log */
	  filename =
	    g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S, entry,
			 ".xlog", NULL);
	  lp =
	    log_file_create (filename, TYPE_FLOG, qso_field_nr, qso_fields,
			     qso_widths);
	  if (lp)
	    {
	      logwindow = openlog (lp, entry, state.logwindows++);
	      log_file_close (lp);
	      logwindowlist = g_list_append (logwindowlist, logwindow);
	      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook),
				     state.logwindows - 1);
	      set_qsoframe (logwindow);
	    }
	  g_free (filename);
	}
      g_free (entry);
    }

  gtk_widget_destroy (newlogdialog);
  gtk_widget_set_sensitive (mainwindow, 1);

  /* fire up the logeditor */
  if ((!logopen) && (!logexist))
    {
      gtk_widget_set_sensitive (mainwindow, 0);
      menubar = lookup_widget (mainwindow, "menubar");
      log_editor_menu = lookup_widget (menubar, "log_editor");
      gtk_signal_emit_by_name (GTK_OBJECT (log_editor_menu), "activate");
    }
}


/* no new log */
void
on_newlogcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* new log dialog is deleted */
gboolean
on_newlogdialog_delete_event (GtkWidget * widget, GdkEvent * event,
			      gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* OPEN DIALOG */
void
on_fileselection_ok_button_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *fileselection, *filemenu, *menu, *item;
  gchar *logn, *sellog, *xlogbackupfile, *temp;
  gpointer filen;
  gboolean logexist = FALSE;
  gint i, filetype;
  logtype *logwindow;
  LOGDB *lp;

  fileselection = gtk_widget_get_toplevel (GTK_WIDGET (button));

  /* skip all if nothing filled in */
  if (strlen
      (gtk_editable_get_chars
       (GTK_EDITABLE (GTK_FILE_SELECTION (fileselection)->selection_entry), 0,
	-1)) > 0)
    {
      filemenu = lookup_widget (fileselection, "filemenu");
      menu = GTK_OPTION_MENU (filemenu)->menu;
      item = gtk_menu_get_active (GTK_MENU (menu));
      /* skip old log format (0) */
      filetype = g_list_index (GTK_MENU_SHELL (menu)->children, item) + 1;

      /* don't free sellog */
      sellog =
	gtk_file_selection_get_filename (GTK_FILE_SELECTION (fileselection));
      filen = basename (sellog);
      logn = logname ((gchar *) filen);

      /* if log exist jump to the page in the notebook */
      for (i = 0; i < g_list_length (logwindowlist); i++)
	{
	  logwindow = g_list_nth_data (logwindowlist, i);
	  if (g_strcasecmp (logn, logwindow->logname) == 0)
	    {
	      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook), i);
	      set_qsoframe (logwindow);
	      logexist = TRUE;
	      break;
	    }
	}
      if (!logexist)
	{
	  lp = log_file_open (sellog, filetype);
	  if (lp)
	    {
	      logwindow = openlog (lp, logn, state.logwindows++);
	      log_file_qso_foreach (lp, fillin_clist, logwindow);
	      log_file_close (lp);
	      if ((lp->type) == TYPE_FLOG)
		{
		  xlogbackupfile = g_strconcat (sellog, ".backup", NULL);
		  savelog (logwindow, xlogbackupfile, TYPE_FLOG);
		  g_free (xlogbackupfile);
		}
	      logwindowlist = g_list_append (logwindowlist, logwindow);
	      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook),
				     state.logwindows - 1);
	      set_qsoframe (logwindow);
	    }
	}
      g_free (logn);
    }
  else
    {
      temp = g_strdup_printf (_("No filename entered, open cancelled"));
      update_statusbar (temp);
      g_free (temp);
    }
  gtk_widget_destroy (fileselection);
  gtk_widget_set_sensitive (mainwindow, 1);
}


/* file selection cancelled */
void
on_fileselection_cancel_button_clicked (GtkButton * button,
					gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* file selection dialog is deleted */
gboolean
on_fileselection_delete_event (GtkWidget * widget, GdkEvent * event,
			       gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


/* CLOSE DIALOG */
gboolean
on_closedialog_delete_event (GtkWidget * widget, GdkEvent * event,
			     gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


void
on_closeokbutton_clicked (GtkButton * button, gpointer user_data)
{
  logtype *logwindow;
  gint page;

  page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mainnotebook));
  logwindow = g_list_nth_data (logwindowlist, page);
  logwindowlist = g_list_remove (logwindowlist, logwindow);
  g_free (logwindow->logname);
  g_free (logwindow);
  state.logwindows--;
  gtk_notebook_remove_page (GTK_NOTEBOOK (mainnotebook), page);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}


void
on_closecancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* LOG EDITOR */
gboolean
on_logeditorwindow_delete_event (GtkWidget * widget, GdkEvent * event,
				 gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* update logfields and return number of columns */
static gint
update_logfields (gint column, gint logfields[], gchar * showhide)
{
  gint i, j;

  for (i = 0; i < QSO_FIELDS; i++)
    {
      if (logfields[i] == column)
	break;
    }
  /*
   * If the column is not present, add it to the end of the array
   * if 'Show' is selected
   */
  if (i == QSO_FIELDS)
    {
      if (g_strcasecmp (showhide, _("Show")) == 0)
	{
	  for (j = 0; j < QSO_FIELDS; j++)
	    if (logfields[j] == 0)
	      {
		logfields[j] = column;
		break;
	      }
	}
    }

  /*
   * If the column is present, remove from the array
   * if 'Hide' is selected
   */
  else
    {
      if (g_strcasecmp (showhide, _("Hide")) == 0)
	{
	  logfields[i] = 0;
	  for (j = 0; j < QSO_FIELDS; j++)
	    {
	      if (logfields[j] == 0)
		{
		  if (logfields[j + 1] == 0)
		    break;
		  /* Shift '0' to the right */
		  logfields[j] = logfields[j + 1];
		  logfields[j + 1] = 0;
		}
	    }
	}
    }

  for (i = 0; i < QSO_FIELDS; i++)
    {
      if (logfields[i] == 0)
	break;
    }
  return (i);
}

static gint
compare_fields (void *f1, void *f2)
{
  gint *p = f1, *q = f2;
  if (*p - *q == *p)
    return 0;
  else
    return (*p - *q);
}

void
on_logeditorokbutton_clicked (GtkButton * button, gpointer user_data)
{
  logtype *logw;
  GtkWidget *logeditor, *lecomboentry3, *lecomboentry9_10, *lecomboentry11,
    *lecomboentry12, *lecomboentry13, *lecomboentry14, *lecomboentry15,
    *lecomboentry16, *leentry15, *leentry16, *lecomboentry17;
  GtkWidget *unknownlabel1, *unknownlabel2, *columnlabel;
  gchar *temp;
  gint i, columns, fields[QSO_FIELDS];

  logw = g_list_nth_data (logwindowlist,
			  gtk_notebook_get_current_page (GTK_NOTEBOOK
							 (mainnotebook)));

  /* fill the field array, unused columns with '0' */
  for (i = 0; i < logw->columns; i++)
    fields[i] = logw->logfields[i];
  for (i = logw->columns; i < QSO_FIELDS; i++)
    fields[i] = 0;

  logeditor = gtk_widget_get_toplevel (GTK_WIDGET (button));

  lecomboentry3 = lookup_widget (logeditor, "lecomboentry3");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry3), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), GMTEND, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), GMTEND, TRUE);
  columns = update_logfields (GMTEND, fields, temp);

  lecomboentry9_10 = lookup_widget (logeditor, "lecomboentry9_10");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry9_10), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    {
      gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QSLOUT,
				       FALSE);
      gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QSLIN, FALSE);
    }
  else
    {
      gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QSLOUT, TRUE);
      gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QSLIN, TRUE);
    }
  columns = update_logfields (QSLOUT, fields, temp);
  columns = update_logfields (QSLIN, fields, temp);

  lecomboentry11 = lookup_widget (logeditor, "lecomboentry11");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry11), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), POWER, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), POWER, TRUE);
  columns = update_logfields (POWER, fields, temp);

  lecomboentry12 = lookup_widget (logeditor, "lecomboentry12");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry12), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), NAME, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), NAME, TRUE);
  columns = update_logfields (NAME, fields, temp);

  lecomboentry13 = lookup_widget (logeditor, "lecomboentry13");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry13), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QTH, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), QTH, TRUE);
  columns = update_logfields (QTH, fields, temp);

  lecomboentry14 = lookup_widget (logeditor, "lecomboentry14");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry14), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), LOCATOR, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), LOCATOR, TRUE);
  columns = update_logfields (LOCATOR, fields, temp);

  lecomboentry15 = lookup_widget (logeditor, "lecomboentry15");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry15), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), U1, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), U1, TRUE);
  columns = update_logfields (U1, fields, temp);

  lecomboentry16 = lookup_widget (logeditor, "lecomboentry16");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry16), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), U2, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), U2, TRUE);
  columns = update_logfields (U2, fields, temp);

  lecomboentry17 = lookup_widget (logeditor, "lecomboentry17");
  temp = gtk_editable_get_chars (GTK_EDITABLE (lecomboentry17), 0, -1);
  if (g_strcasecmp (temp, _("Hide")) == 0)
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), REMARKS, FALSE);
  else
    gtk_clist_set_column_visibility (GTK_CLIST (logw->clist), REMARKS, TRUE);
  columns = update_logfields (REMARKS, fields, temp);

  /* see if the array has changed */
  for (i = 0; i < QSO_FIELDS; i++)
    if (fields[i] != logw->logfields[i])
      break;

  /* copy and re-order the array */
  qsort (fields, QSO_FIELDS, sizeof (gint), (compar_callback_fn) compare_fields);
  if (i < QSO_FIELDS)
    {
      for (i = 0; i < QSO_FIELDS; i++)
	logw->logfields[i] = fields[i];
      logw->columns = columns;
    }

  /* check names of free fields and set column labels and labels on the QSO frame */
  leentry15 = lookup_widget (logeditor, "leentry15");
  leentry16 = lookup_widget (logeditor, "leentry16");
  temp = gtk_editable_get_chars (GTK_EDITABLE (leentry15), 0, -1);
  if (g_strcasecmp (temp, preferences.freefield1) != 0)
    {
      if (strlen (temp) == 0)
	temp = g_strdup ("UNKNOWN");
      preferences.freefield1 = g_strdup (temp);
      unknownlabel1 = lookup_widget (mainwindow, "unknownlabel1");
      gtk_label_set_text (GTK_LABEL (unknownlabel1), temp);
      columnlabel = gtk_label_new (temp);
      gtk_clist_set_column_widget (GTK_CLIST (logw->clist), U1, columnlabel);
    }
  temp = gtk_editable_get_chars (GTK_EDITABLE (leentry16), 0, -1);
  if (g_strcasecmp (temp, preferences.freefield2) != 0)
    {
      if (strlen (temp) == 0)
	temp = g_strdup ("UNKNOWN");
      preferences.freefield2 = g_strdup (temp);
      unknownlabel2 = lookup_widget (mainwindow, "unknownlabel2");
      gtk_label_set_text (GTK_LABEL (unknownlabel2), temp);
      columnlabel = gtk_label_new (temp);
      gtk_clist_set_column_widget (GTK_CLIST (logw->clist), U2, columnlabel);
    }

  g_free (temp);
  gtk_widget_destroy (logeditor);
  set_qsoframe (logw);
  gtk_widget_set_sensitive (mainwindow, 1);
}


void
on_logeditorcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* SAVE AS DIALOG */
void
on_saveasok_button_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *saveasdialog, *saveasmenu, *menu, *item, *savewarningdialog,
    *savewarninglabel;
  gchar *temp, *rev;
  gint index;
  logtype *logwindow;
  gboolean warning = FALSE;

  saveasdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  saveasmenu = lookup_widget (saveasdialog, "saveasmenu");
  menu = GTK_OPTION_MENU (saveasmenu)->menu;
  item = gtk_menu_get_active (GTK_MENU (menu));
  index = g_list_index (GTK_MENU_SHELL (menu)->children, item);

  if (index == 0)
    globtype = 1;
  else if (index == 1)
    globtype = 3;
  else
    globtype = 4;

  globtemp =
    g_strdup (gtk_file_selection_get_filename
	      (GTK_FILE_SELECTION (saveasdialog)));
  rev = g_strdup (globtemp);
  g_strreverse (rev);
  if (!g_strncasecmp (rev, "/", 1))
    {
      temp = g_strdup_printf (_("No filename entered, log not saved"));
      update_statusbar (temp);
    }
  else
    {
      if (!fileexist (globtemp))
	{
	  logwindow = g_list_nth_data (logwindowlist,
				       gtk_notebook_get_current_page
				       (GTK_NOTEBOOK (mainnotebook)));
	  savelog (logwindow, globtemp, globtype);
	  temp = g_strdup_printf (_("Log saved to %s"), globtemp);
	  update_statusbar (temp);
	}
      else
	{
	  savewarningdialog = create_savewarningdialog ();
	  savewarninglabel =
	    lookup_widget (savewarningdialog, "savewarninglabel");
	  temp =
	    g_strdup_printf (_("%s already exist, do want to overwrite it?"),
			     globtemp);
	  gtk_label_set_text (GTK_LABEL (savewarninglabel), temp);
	  gtk_widget_show (savewarningdialog);
	  warning = TRUE;
	}
    }
  g_free (temp);
  g_free (rev);

  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  if (!warning)
    gtk_widget_set_sensitive (mainwindow, 1);
}

/* delete save as dialog */
gboolean
on_saveasdialog_delete_event (GtkWidget * widget, GdkEvent * event,
			      gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


/* cancel save as */
void
on_saveas_cancel_button_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}


/* DUPE CHECK DIALOG */
void
on_dupecheckokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *dupecheckdialog, *dupecheckthisradiobutton, *dupelistdialog,
    *duperesultclist, *dupecheckstatusbar, *dupecheckprogressbar,
    *dupecheckbuttonmode, *dupecheckbuttonband, *duperesultdialog;
  GList *dupelist = NULL;
  logtype *logwindow;
  gint i, page, savedpage, pages, row, step, duperow = 0;
  gchar *logn, *prevlogn, *nr, *prevnr, *date, *gmt, *callsign, *band,
    *mode, *temp, *str, *dupestr;
  gchar **dupe = NULL;
  gfloat new_val;
  GtkAdjustment *adj;
  gboolean dupefound = FALSE;

  dupe = g_new0 (gchar *, 7);
  for (i = 0; i < 7; i++)
    dupe[i] = g_new0 (gchar, 100);

  dupecheckdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  dupecheckthisradiobutton = lookup_widget (dupecheckdialog,
					    "dupecheckthisradiobutton");
  dupecheckbuttonmode =
    lookup_widget (dupecheckdialog, "dupecheckbuttonmode");
  dupecheckbuttonband =
    lookup_widget (dupecheckdialog, "dupecheckbuttonband");
  dupecheckstatusbar = lookup_widget (dupecheckdialog, "dupecheckstatusbar");
  dupecheckprogressbar =
    lookup_widget (dupecheckdialog, "dupecheckprogressbar");
  gtk_statusbar_pop (GTK_STATUSBAR (dupecheckstatusbar), 1);
  gtk_statusbar_push (GTK_STATUSBAR (dupecheckstatusbar), 1,
		      _("Searching ..."));
  gtkut_widget_wait_for_draw (GTK_WIDGET (dupecheckstatusbar)->parent);

  /* this or all logs ? */
  if (gtk_toggle_button_get_active
      (GTK_TOGGLE_BUTTON (dupecheckthisradiobutton)))
    state.dupecheck = 0;
  else
    state.dupecheck = 1;
  state.notdupecheckmode =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dupecheckbuttonmode));
  state.notdupecheckband =
    gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dupecheckbuttonband));

  /* set autosort and sort item for the result dialog */
  dupelistdialog = create_dupelistdialog ();
  duperesultclist = lookup_widget (dupelistdialog, "duperesultclist");
  gtk_clist_set_sort_column (GTK_CLIST (duperesultclist), 4);
  gtk_clist_set_auto_sort (GTK_CLIST (duperesultclist), TRUE);
  for (i = 0; i < 7; i++)
    gtk_clist_set_column_auto_resize (GTK_CLIST (duperesultclist), i, TRUE);

  if (state.dupecheck == 0)
    {
      page = gtk_notebook_get_current_page (GTK_NOTEBOOK (mainnotebook));
      pages = page + 1;
    }
  else
    {
      page = 0;
      pages = g_list_length (logwindowlist);
    }

  /* traverse 2 times down the log(s) */
  savedpage = page;
  while (page < pages)
    {
      logwindow = g_list_nth_data (logwindowlist, page);
      /* compose a list with items to scan for */
      for (row = 0; row < logwindow->qsos; row++)
	{
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, CALL, &temp);
	  if (!state.notdupecheckband)
	    {
	      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, BAND,
				  &band);
	      str = findband (g_strdup (band));
	      temp = g_strconcat (temp, str, NULL);
	      g_free (str);
	    }
	  if (!state.notdupecheckmode)
	    {
	      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, MODE,
				  &mode);
	      temp = g_strconcat (temp, mode, NULL);
	    }
	  dupelist = g_list_append (dupelist, temp);
	}
      page++;
    }

  page = savedpage;
  while (page < pages)
    {
      logwindow = g_list_nth_data (logwindowlist, page);
      for (row = 0; row < logwindow->qsos; row++)
	{
	  if (row % 10 == 0)	/* advance progress bar */
	    {
	      new_val =
		gtk_progress_get_value (GTK_PROGRESS (dupecheckprogressbar)) +
		1;
	      adj = GTK_PROGRESS (dupecheckprogressbar)->adjustment;
	      if (new_val > adj->upper)
		new_val = adj->lower;
	      gtk_progress_set_value (GTK_PROGRESS (dupecheckprogressbar),
				      new_val);
	      while (gtk_events_pending ())
		gtk_main_iteration ();
	    }
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, CALL, &temp);
	  if (!state.notdupecheckband)
	    {
	      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, BAND,
				  &band);
	      str = findband (g_strdup (band));
	      temp = g_strconcat (temp, str, NULL);
	      g_free (str);
	    }
	  if (!state.notdupecheckmode)
	    {
	      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, MODE,
				  &mode);
	      temp = g_strconcat (temp, mode, NULL);
	    }
	  dupestr = g_strdup (temp);
	  for (step = 0; step < logwindow->qsos; step++)
	    {
	      if (g_strcasecmp (dupestr, g_list_nth_data (dupelist, step)) ==
		  0)
		{		/* dupe found ? */
		  gtk_clist_get_text (GTK_CLIST (logwindow->clist), row, NR,
				      &nr);
		  if ((logwindow->qsos) - step != atoi (nr))
		    {
		      dupe[0] = g_strdup (logwindow->logname);
		      dupe[1] = g_strdup (nr);
		      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
					  DATE, &date);
		      dupe[2] = g_strdup (date);
		      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
					  GMT, &gmt);
		      dupe[3] = g_strdup (gmt);
		      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
					  CALL, &callsign);
		      dupe[4] = g_strdup (callsign);
		      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
					  BAND, &band);
		      dupe[5] = g_strdup (band);
		      gtk_clist_get_text (GTK_CLIST (logwindow->clist), row,
					  MODE, &mode);
		      dupe[6] = g_strdup (mode);
		      gtk_clist_append (GTK_CLIST (duperesultclist), dupe);
		      dupefound = TRUE;
		    }
		}
	      duperow++;
	    }
	}
      page++;
    }

  g_list_free (dupelist);
  g_free (temp);
  for (i = 0; i < 7; i++)
    g_free (dupe[i]);
  g_free (dupe);

  /* show result */
  gtk_widget_destroy (dupecheckdialog);
  if (dupefound)
    {				/* remove double entries (this happens with more than 2 dupes) */
      for (row = 0; row < GTK_CLIST (duperesultclist)->rows; row++)
	{
	  gtk_clist_get_text (GTK_CLIST (duperesultclist), row, 0, &logn);
	  gtk_clist_get_text (GTK_CLIST (duperesultclist), row, 1, &nr);
	  if (row > 0)
	    {
	      gtk_clist_get_text (GTK_CLIST (duperesultclist), row - 1, 0,
				  &prevlogn);
	      gtk_clist_get_text (GTK_CLIST (duperesultclist), row - 1, 1,
				  &prevnr);
	      if ((g_strcasecmp (prevnr, nr) == 0)
		  && (g_strcasecmp (prevlogn, logn) == 0))
		gtk_clist_remove (GTK_CLIST (duperesultclist), row - 1);
	    }
	}
      gtk_widget_show (dupelistdialog);
      gtk_widget_set_sensitive (mainwindow, 1);
    }
  else
    {
      duperesultdialog = create_duperesultdialog ();
      gtk_widget_destroy (dupelistdialog);
      gtk_widget_show (duperesultdialog);
    }
}

/* delete the dupecheck dialog */
gboolean
on_dupecheckdialog_delete_event (GtkWidget * widget, GdkEvent * event,
				 gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}

/* cancel dupe check */
void
on_dupecheckcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}


/* DUPE LIST DIALOG */
void
on_dupelistokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* entry is selected in the duperesult list, select corresponding entry in
   the main log */
void
on_duperesultclist_select_row (GtkCList * clist, gint row, gint column,
			       GdkEvent * event, gpointer user_data)
{
  gchar *nr, *lognr, *logn;
  gint i, logrow = -1;
  logtype *logwindow = NULL;
  gboolean qsofound = FALSE;

  gtk_clist_get_text (GTK_CLIST (clist), row, 1, &nr);
  gtk_clist_get_text (GTK_CLIST (clist), row, 0, &logn);

  for (i = 0; i < g_list_length (logwindowlist); i++)
    {
      logwindow = g_list_nth_data (logwindowlist, i);
      for (logrow = 0; logrow < logwindow->qsos; logrow++)
	{
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist), logrow, 0,
			      &lognr);
	  if ((g_strcasecmp (nr, lognr) == 0)
	      && (g_strcasecmp (logn, logwindow->logname) == 0))
	    {
	      qsofound = TRUE;
	      break;
	    }
	}
      if (qsofound)
	break;
    }
  if (qsofound)
    {
      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook), i);
      gtk_clist_moveto (GTK_CLIST (logwindow->clist), logrow, 0, 0.5, 0);
      gtk_clist_select_row (GTK_CLIST (logwindow->clist), logrow, 0);
    }
}


/* DUPE RESULT DIALOG */
void
on_duperesultdialogokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* delete the duperesult dialog */
gboolean
on_duperesultdialog_delete_event (GtkWidget * widget, GdkEvent * event,
				  gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


/* SAVE AS WARNING DIALOG */
gboolean
on_savewarningdialog_delete_event (GtkWidget * widget, GdkEvent * event,
				   gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


void
on_savewarningokbutton_clicked (GtkButton * button, gpointer user_data)
{
  gchar *temp;
  logtype *logwindow;

  logwindow = g_list_nth_data (logwindowlist,
			       gtk_notebook_get_current_page (GTK_NOTEBOOK
							      (mainnotebook)));
  savelog (logwindow, globtemp, globtype);
  temp = g_strdup_printf (_("Log saved to %s"), globtemp);
  update_statusbar (temp);
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}


void
on_savewarningcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}


/* DEFAULTS DIALOG */

gboolean
on_defaultsdialog_delete_event (GtkWidget * widget, GdkEvent * event,
				gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


void
on_defaultsokbutton_clicked (GtkButton * button, gpointer user_data)
{
  GtkWidget *defaultsmhzentry, *defaultsmodeentry, *defaultstxrstentry,
    *defaultsrxrstentry, *defaultspowerentry, *defaultsdialog,
    *defaultsfreefield1entry, *defaultsfreefield2entry, *defaultsremarksentry;
  gchar *temp;

  defaultsdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  defaultsmhzentry = lookup_widget (defaultsdialog, "defaultsmhzentry");
  defaultsmodeentry = lookup_widget (defaultsdialog, "defaultsmodeentry");
  defaultstxrstentry = lookup_widget (defaultsdialog, "defaultstxrstentry");
  defaultsrxrstentry = lookup_widget (defaultsdialog, "defaultsrxrstentry");
  defaultspowerentry = lookup_widget (defaultsdialog, "defaultspowerentry");
  defaultsfreefield1entry =
    lookup_widget (defaultsdialog, "defaultsfreefield1entry");
  defaultsfreefield2entry =
    lookup_widget (defaultsdialog, "defaultsfreefield2entry");
  defaultsremarksentry =
    lookup_widget (defaultsdialog, "defaultsremarksentry");

  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultsmhzentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultmhz = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultmhz, temp))
    preferences.defaultmhz = g_strdup (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultsmodeentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultmode = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultmode, temp))
    preferences.defaultmode = g_strdup (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultstxrstentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaulttxrst = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaulttxrst, temp))
    preferences.defaulttxrst = g_strdup (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultsrxrstentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultrxrst = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultrxrst, temp))
    preferences.defaultrxrst = g_strdup (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultspowerentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultpower = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultpower, temp))
    preferences.defaultpower = g_strdup (temp);
  temp =
    gtk_editable_get_chars (GTK_EDITABLE (defaultsfreefield1entry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultfreefield1 = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultfreefield1, temp))
    preferences.defaultfreefield1 = g_strdup (temp);
  temp =
    gtk_editable_get_chars (GTK_EDITABLE (defaultsfreefield2entry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultfreefield2 = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultfreefield2, temp))
    preferences.defaultfreefield2 = g_strdup (temp);
  temp = gtk_editable_get_chars (GTK_EDITABLE (defaultsremarksentry), 0, -1);
  if (strlen (temp) == 0)
    preferences.defaultremarks = g_strdup ("?");
  else if (g_strcasecmp (preferences.defaultremarks, temp))
    preferences.defaultremarks = g_strdup (temp);

  g_free (temp);
  gtk_widget_destroy (defaultsdialog);
  gtk_widget_set_sensitive (mainwindow, 1);
}


void
on_defaultscancelbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* WORKED B4 DIALOG */

void
on_b4clist_select_row (GtkCList * clist, gint row, gint column,
		       GdkEvent * event, gpointer user_data)
{
  gchar *nr, *lognr, *logn;
  gint i, logrow = -1;
  logtype *logwindow = NULL;
  gboolean qsofound = FALSE;

  gtk_clist_get_text (GTK_CLIST (clist), row, 1, &nr);
  gtk_clist_get_text (GTK_CLIST (clist), row, 0, &logn);

  for (i = 0; i < g_list_length (logwindowlist); i++)
    {
      logwindow = g_list_nth_data (logwindowlist, i);
      for (logrow = 0; logrow < logwindow->qsos; logrow++)
	{
	  gtk_clist_get_text (GTK_CLIST (logwindow->clist), logrow, 0,
			      &lognr);
	  if ((g_strcasecmp (nr, lognr) == 0)
	      && (g_strcasecmp (logn, logwindow->logname) == 0))
	    {
	      qsofound = TRUE;
	      break;
	    }
	}
      if (qsofound)
	break;
    }
  if (qsofound)
    {
      gtk_notebook_set_page (GTK_NOTEBOOK (mainnotebook), i);
      gtk_clist_moveto (GTK_CLIST (logwindow->clist), logrow, 0, 0.5, 0);
      gtk_clist_select_row (GTK_CLIST (logwindow->clist), logrow, 0);
    }
}

void
on_b4okbutton_clicked (GtkButton * button, gpointer user_data)
{
  gtk_widget_hide (b4dialog);
}

/* DIALOGS DIALOG */

gboolean
on_dialogsdialog_delete_event(GtkWidget * widget, GdkEvent * event,
                                        gpointer user_data)
{
  gtk_widget_set_sensitive (mainwindow, 1);
  return FALSE;
}


void
on_dialogsokbutton_clicked(GtkButton * button, gpointer user_data)
{
  GtkWidget *dialogsdialog, *checkbutton, *b4clist;
  gchar *widgetname, **states;
  gboolean check;
  gint i;

  states = g_new0 (gchar *, 20);
  dialogsdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
  b4clist = lookup_widget(b4dialog, "b4clist");
  for (i = 0; i < 19; i++)
  { /* look up name of the  button and get it's togglestate */
    widgetname = g_strdup_printf("b4%d", i);
    checkbutton = lookup_widget(dialogsdialog, widgetname);
    check = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (checkbutton));
    if (GTK_WIDGET_VISIBLE (b4dialog)) 
      gtk_clist_set_column_visibility (GTK_CLIST (b4clist), i, check);
    states[i] = g_strdup(check ? "1" : "0");
    g_free(widgetname);
  }
  states[19] = NULL; /* terminator for g_strjoinv */
  preferences.b4columns = g_strjoinv (",", states);

  for (i = 0; i < 19; i++) 
    g_free(states[i]);
  g_free(states);

  gtk_widget_destroy (dialogsdialog);
  if (GTK_WIDGET_VISIBLE (b4dialog)) 
    gtk_widget_set_sensitive (b4dialog, 1);
  gtk_widget_set_sensitive (mainwindow, 1);
}


void
on_dialogscancelbutton_clicked(GtkButton * button, gpointer user_data)
{
  gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
  gtk_widget_set_sensitive (mainwindow, 1);
}

/* end of file */
