<?php
/*
 * Gallery - a web based photo album viewer and editor
 * Copyright (C) 2000-2007 Bharat Mediratta
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street - Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
 * Exif Module
 *
 * This module provides support for adding exifs to items
 *
 * @package Exif
 * @author Bharat Mediratta <bharat@menalto.com>
 * @author Georg Rehfeld <rehfeld@georg-rehfeld.de>
 * @author Jozsef R.Nagy <jozsef.rnagy@site.hu>
 * @version $Revision: 16034 $
 */
class ExifModule extends GalleryModule {

    function ExifModule() {
	global $gallery;

	$this->setId('exif');
	$this->setName($gallery->i18n('EXIF/IPTC'));
	$this->setDescription($gallery->i18n('Extract EXIF/IPTC data from JPEG photos'));
	$this->setVersion('1.1.0'); /* Update upgrade() function below too */
	$this->setGroup('data', $gallery->i18n('Extra Data'));
	$this->setCallbacks('getSiteAdminViews');
	$this->setRequiredCoreApi(array(7, 10));
	$this->setRequiredModuleApi(array(3, 0));
    }

    /**
     * @see GalleryModule::upgrade
     */
    function upgrade($currentVersion) {
	global $gallery;

	if (!isset($currentVersion)) {
	    /* Initial install */
	    GalleryCoreApi::requireOnce('modules/exif/classes/ExifHelper.class');
	    $ret = ExifHelper::setDefaultProperties(EXIF_SUMMARY);
	    if ($ret) {
		return $ret;
	    }

	    $ret = ExifHelper::setDefaultProperties(EXIF_DETAILED);
	    if ($ret) {
		return $ret;
	    }

	    /* Import IPTC keywords into item keywords, see comment below for reason. */
	    $ret = $this->setParameter('addOption', IPTC_ITEM_KEYWORDS);
	    if ($ret) {
		return $ret;
	    }
	} else {
	    switch ($currentVersion) {
	    case '0.8.2':
	    case '0.8.3':
		$ret = $this->setParameter('addOption', '0');
		if ($ret) {
		    return $ret;
		}

	    case '0.8.4':
	    case '0.8.5':
	    case '0.8.6':
	    case '0.8.7':
	    case '0.8.8':
	    case '0.8.9':
	    case '0.9.1':
		/*
		 * New IPTC capabilities in 0.9.2, we add reasonable IPTC default properties to
		 * summary and detailed view. We also enable extraction of IPTC keywords into item
		 * keywords: item keywords are nowhere else set automatically and users adding
		 * IPTC keywords to their images (which must be done actively, as opposed to EXIF
		 * data generated by the camera) most probably will want to have them extracted.
		 */
		GalleryCoreApi::requireOnce('modules/exif/classes/ExifHelper.class');
		$ret = ExifHelper::addDefaultIptcProperties(EXIF_SUMMARY);
		if ($ret) {
		    return $ret;
		}

		$ret = ExifHelper::addDefaultIptcProperties(EXIF_DETAILED);
		if ($ret) {
		    return $ret;
		}

		list ($ret, $addOption) = $this->getParameter('addOption');
		if ($ret) {
		    return $ret;
		}
		$ret = $this->setParameter('addOption', $addOption | IPTC_ITEM_KEYWORDS);
		if ($ret) {
		    return $ret;
		}

	    case '0.9.2':
	    case '0.9.3':
	    case '0.9.4':
	    case '0.9.5':
	    case '0.9.6':
	    case '0.9.7':
	    case '0.9.8':
	    case '1.0.0':
	    case '1.0.1':
		/* Added EXIF GPS tags */
	    case '1.0.2':
		/* Upgrade to Exifixer 1.5 and add support for image/x-dcraw */
	    case '1.0.3':
	    case '1.0.4':
	    case '1.0.5':
	    case '1.0.6':
		/* Rotate images based on exifdata */
	    case '1.0.7':
	    case '1.0.8':
	    case '1.0.9':

	    case 'end of upgrade path':
		/*
		 * Leave this bogus case at the end of the legitimate case statements so that we
		 * always properly terminate our upgrade path with a break.
		 */
		break;

	    default:
		return GalleryCoreApi::error(ERROR_BAD_PLUGIN, __FILE__, __LINE__,
					    sprintf('Unknown module version %s', $currentVersion));
	    }
	}

	return null;
    }

    /**
     * @see GalleryModule::performFactoryRegistrations
     */
    function performFactoryRegistrations() {
	$ret = GalleryCoreApi::registerFactoryImplementation(
	    'ExifInterface_1_0', 'ExifExtractor', 'Exif',
	    'modules/exif/classes/ExifExtractor.class', 'exif', null);
	if ($ret) {
	    return $ret;
	}

	$ret = GalleryCoreApi::registerFactoryImplementation(
	    'GalleryToolkit', 'ExifToolkit', 'Exif',
	    'modules/exif/classes/ExifToolkit.class', 'exif', null);
	if ($ret) {
	    return $ret;
	}

	$ret = GalleryCoreApi::registerFactoryImplementation('ItemAddOption',
	    'ExifDescriptionOption', 'ExifDescriptionOption',
	    'modules/exif/ExifDescriptionOption.inc', 'exif', null);
	if ($ret) {
	    return $ret;
	}

	return null;
    }

    /**
     * @see GalleryModule::activate
     */
    function activate($postActivationEvent=true) {
	global $gallery;
	$ret = GalleryCoreApi::registerToolkitProperty(
		'Exif',
		array('image/jpeg', 'image/pjpeg',
		      'image/jpeg-cmyk', 'image/pjpeg-cmyk', 'image/x-dcraw'),
		'originationTimestamp',
		'int',
		$gallery->i18n('Get the origination timestamp'));
	if ($ret) {
	    return array($ret, null);
	}

	list ($ret, $redirect) = parent::activate($postActivationEvent);
	if ($ret) {
	    return array($ret, null);
	}

	return array(null, $redirect);
    }

    /**
     * @see GalleryModule::getSiteAdminViews
     */
    function getSiteAdminViews() {
	global $gallery;

	return array(null,
		     array(array('name' => $this->translate('EXIF/IPTC'),
				 'view' => 'exif.AdminExif')));
    }
}
?>
