module SSL;

export {
	const SSLv2  = 0x0002;
	const SSLv3  = 0x0300;
	const TLSv10 = 0x0301;
	const TLSv11 = 0x0302;
	const TLSv12 = 0x0303;
	const TLSv13 = 0x0304;

	const DTLSv10 = 0xFEFF;
	# DTLSv11 does not exist
	const DTLSv12 = 0xFEFD;

	## Mapping between the constants and string values for SSL/TLS versions.
	const version_strings: table[count] of string = {
		[SSLv2] = "SSLv2",
		[SSLv3] = "SSLv3",
		[TLSv10] = "TLSv10",
		[TLSv11] = "TLSv11",
		[TLSv12] = "TLSv12",
		[TLSv13] = "TLSv13",
		[DTLSv10] = "DTLSv10",
		[DTLSv12] = "DTLSv12"
	} &default=function(i: count):string
		{
		if ( i/0xFF == 0x7F ) # TLS 1.3 draft
		  return fmt("TLSv13-draft%d", i % 0x7F );

		return fmt("unknown-%d", i);
		};

	# TLS content types:
	const CHANGE_CIPHER_SPEC = 20;
	const ALERT = 21;
	const HANDSHAKE = 22;
	const APPLICATION_DATA = 23;
	const HEARTBEAT = 24;
	const V2_ERROR = 300;
	const V2_CLIENT_HELLO = 301;
	const V2_CLIENT_MASTER_KEY = 302;
	const V2_SERVER_HELLO = 304;

	# TLS Handshake types:
	const HELLO_REQUEST       = 0;
	const CLIENT_HELLO        = 1;
	const SERVER_HELLO        = 2;
	const HELLO_VERIFY_REQUEST = 3; # RFC 6347
	const SESSION_TICKET      = 4; # RFC 5077
	const HELLO_RETRY_REQUEST = 6; # draft-ietf-tls-tls13-16
	const ENCRYPTED_EXTENSIONS = 8; # draft-ietf-tls-tls13-16
	const CERTIFICATE         = 11;
	const SERVER_KEY_EXCHANGE = 12;
	const CERTIFICATE_REQUEST = 13;
	const SERVER_HELLO_DONE   = 14;
	const CERTIFICATE_VERIFY  = 15;
	const CLIENT_KEY_EXCHANGE = 16;
	const FINISHED            = 20;
	const CERTIFICATE_URL     = 21; # RFC 3546
	const CERTIFICATE_STATUS  = 22; # RFC 3546
	const SUPPLEMENTAL_DATA   = 23; # RFC 4680
	const KEY_UPDATE          = 24; # draft-ietf-tls-tls13-16

	## Mapping between numeric codes and human readable strings for alert
	## levels.
	const alert_levels: table[count] of string = {
		[1] = "warning",
		[2] = "fatal",
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable strings for hash
	## algorithms.
	const hash_algorithms: table[count] of string = {
		[0] = "none",
		[1] = "md5",
		[2] = "sha1",
		[3] = "sha224",
		[4] = "sha256",
		[5] = "sha384",
		[6] = "sha512",
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable strings for signature
	## algorithms.
	const signature_algorithms: table[count] of string = {
		[0] = "anonymous",
		[1] = "rsa",
		[2] = "dsa",
		[3] = "ecdsa",
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable strings for alert
	## descriptions.
	const alert_descriptions: table[count] of string = {
		[0] = "close_notify",
		[10] = "unexpected_message",
		[20] = "bad_record_mac",
		[21] = "decryption_failed",
		[22] = "record_overflow",
		[30] = "decompression_failure",
		[40] = "handshake_failure",
		[41] = "no_certificate",
		[42] = "bad_certificate",
		[43] = "unsupported_certificate",
		[44] = "certificate_revoked",
		[45] = "certificate_expired",
		[46] = "certificate_unknown",
		[47] = "illegal_parameter",
		[48] = "unknown_ca",
		[49] = "access_denied",
		[50] = "decode_error",
		[51] = "decrypt_error",
		[60] = "export_restriction",
		[70] = "protocol_version",
		[71] = "insufficient_security",
		[80] = "internal_error",
		[86] = "inappropriate_fallback",
		[90] = "user_canceled",
		[100] = "no_renegotiation",
		[110] = "unsupported_extension",
		[111] = "certificate_unobtainable",
		[112] = "unrecognized_name",
		[113] = "bad_certificate_status_response",
		[114] = "bad_certificate_hash_value",
		[115] = "unknown_psk_identity",
		[120] = "no_application_protocol",
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable strings for SSL/TLS
	## extensions.
	# More information can be found here:
	# http://www.iana.org/assignments/tls-extensiontype-values/tls-extensiontype-values.xml
	const extensions: table[count] of string = {
		[0] = "server_name",
		[1] = "max_fragment_length",
		[2] = "client_certificate_url",
		[3] = "trusted_ca_keys",
		[4] = "truncated_hmac",
		[5] = "status_request",
		[6] = "user_mapping",
		[7] = "client_authz",
		[8] = "server_authz",
		[9] = "cert_type",
		[10] = "supported_groups", # old name: elliptic_curves - draft-ietf-tls-negotiated-ff-dhe
		[11] = "ec_point_formats",
		[12] = "srp",
		[13] = "signature_algorithms",
		[14] = "use_srtp",
		[15] = "heartbeat",
		[16] = "application_layer_protocol_negotiation",
		[17] = "status_request_v2",
		[18] = "signed_certificate_timestamp",
		[19] = "client_certificate_type",
		[20] = "server_certificate_type",
		[21] = "padding",
		[22] = "encrypt_then_mac",
		[23] = "extended_master_secret",
		[24] = "token_binding", # temporary till 2017-03-06 - draft-ietf-tokbind-negotiation
		[25] = "cached_info",
		[35] = "SessionTicket TLS",
		[40] = "key_share", # new for TLS 1.3; was used for extended_random before. State as of TLS 1.3 draft 16
		[41] = "pre_shared_key", # new for 1.3, state of draft-16
		[42] = "early_data", # new for 1.3, state of draft-16
		[43] = "supported_versions", # new for 1.3, state of draft-16
		[44] = "cookie", # new for 1.3, state of draft-16
		[45] = "psk_key_exchange_modes", # new for 1.3, state of draft-18
		[46] = "TicketEarlyDataInfo", # new for 1.3, state of draft-16
		[47] = "certificate_authorities", # new for 1.3, state of draft-18
		[48] = "oid_filters", # new for 1.3, state of draft-18
		[13172] = "next_protocol_negotiation",
		[13175] = "origin_bound_certificates",
		[13180] = "encrypted_client_certificates",
		[30031] = "channel_id",
		[30032] = "channel_id_new",
		[35655] = "padding",
		[65281] = "renegotiation_info"
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable string for SSL/TLS elliptic curves.
	# See http://www.iana.org/assignments/tls-parameters/tls-parameters.xhtml#tls-parameters-8
	const ec_curves: table[count] of string = {
		[1] = "sect163k1", # 1-23 are TLS 1.3 obsoleted
		[2] = "sect163r1",
		[3] = "sect163r2",
		[4] = "sect193r1",
		[5] = "sect193r2",
		[6] = "sect233k1",
		[7] = "sect233r1",
		[8] = "sect239k1",
		[9] = "sect283k1",
		[10] = "sect283r1",
		[11] = "sect409k1",
		[12] = "sect409r1",
		[13] = "sect571k1",
		[14] = "sect571r1",
		[15] = "secp160k1",
		[16] = "secp160r1",
		[17] = "secp160r2",
		[18] = "secp192k1",
		[19] = "secp192r1",
		[20] = "secp224k1",
		[21] = "secp224r1",
		[22] = "secp256k1",
		[23] = "secp256r1", # TLS 1.3 valid
		[24] = "secp384r1", # TLS 1.3 valid
		[25] = "secp521r1", # TLS 1.3 valid
		[26] = "brainpoolP256r1", # 26-28 are TLS 1.3 obsoleted
		[27] = "brainpoolP384r1",
		[28] = "brainpoolP512r1",
		# Temporary till 2017-01-09 - draft-ietf-tls-rfc4492bis
		[29] = "x25519", # TLS 1.3 valid
		[30] = "x448", # TLS 1.3 valid
		# draft-ietf-tls-negotiated-ff-dhe-10
		[256] = "ffdhe2048", # 256-260 are TLS 1.3 valid
		[257] = "ffdhe3072",
		[258] = "ffdhe4096",
		[259] = "ffdhe6144",
		[260] = "ffdhe8192",
		[0xFF01] = "arbitrary_explicit_prime_curves",
		[0xFF02] = "arbitrary_explicit_char2_curves"
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	## Mapping between numeric codes and human readable string for SSL/TLS EC point formats.
	# See http://www.iana.org/assignments/tls-parameters/tls-parameters.xhtml#tls-parameters-9
	const ec_point_formats: table[count] of string = {
		[0] = "uncompressed",
		[1] = "ansiX962_compressed_prime",
		[2] = "ansiX962_compressed_char2"
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

	# SSLv2
	const SSLv20_CK_RC4_128_WITH_MD5 = 0x010080;
	const SSLv20_CK_RC4_128_EXPORT40_WITH_MD5 = 0x020080;
	const SSLv20_CK_RC2_128_CBC_WITH_MD5 = 0x030080;
	const SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5 = 0x040080;
	const SSLv20_CK_IDEA_128_CBC_WITH_MD5 = 0x050080;
	const SSLv20_CK_DES_64_CBC_WITH_MD5 = 0x060040;
	const SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5 = 0x0700C0;

	# TLS
	const TLS_NULL_WITH_NULL_NULL = 0x0000;
	const TLS_RSA_WITH_NULL_MD5 = 0x0001;
	const TLS_RSA_WITH_NULL_SHA = 0x0002;
	const TLS_RSA_EXPORT_WITH_RC4_40_MD5 = 0x0003;
	const TLS_RSA_WITH_RC4_128_MD5 = 0x0004;
	const TLS_RSA_WITH_RC4_128_SHA = 0x0005;
	const TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5 = 0x0006;
	const TLS_RSA_WITH_IDEA_CBC_SHA = 0x0007;
	const TLS_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0008;
	const TLS_RSA_WITH_DES_CBC_SHA = 0x0009;
	const TLS_RSA_WITH_3DES_EDE_CBC_SHA = 0x000A;
	const TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x000B;
	const TLS_DH_DSS_WITH_DES_CBC_SHA = 0x000C;
	const TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA = 0x000D;
	const TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x000E;
	const TLS_DH_RSA_WITH_DES_CBC_SHA = 0x000F;
	const TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA = 0x0010;
	const TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA = 0x0011;
	const TLS_DHE_DSS_WITH_DES_CBC_SHA = 0x0012;
	const TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA = 0x0013;
	const TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA = 0x0014;
	const TLS_DHE_RSA_WITH_DES_CBC_SHA = 0x0015;
	const TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA = 0x0016;
	const TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5 = 0x0017;
	const TLS_DH_ANON_WITH_RC4_128_MD5 = 0x0018;
	const TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA = 0x0019;
	const TLS_DH_ANON_WITH_DES_CBC_SHA = 0x001A;
	const TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA = 0x001B;
	const SSL_FORTEZZA_KEA_WITH_NULL_SHA = 0x001C;
	const SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA = 0x001D;
	const TLS_KRB5_WITH_DES_CBC_SHA = 0x001E;
	const TLS_KRB5_WITH_3DES_EDE_CBC_SHA = 0x001F;
	const TLS_KRB5_WITH_RC4_128_SHA = 0x0020;
	const TLS_KRB5_WITH_IDEA_CBC_SHA = 0x0021;
	const TLS_KRB5_WITH_DES_CBC_MD5 = 0x0022;
	const TLS_KRB5_WITH_3DES_EDE_CBC_MD5 = 0x0023;
	const TLS_KRB5_WITH_RC4_128_MD5 = 0x0024;
	const TLS_KRB5_WITH_IDEA_CBC_MD5 = 0x0025;
	const TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA = 0x0026;
	const TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA = 0x0027;
	const TLS_KRB5_EXPORT_WITH_RC4_40_SHA = 0x0028;
	const TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5 = 0x0029;
	const TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5 = 0x002A;
	const TLS_KRB5_EXPORT_WITH_RC4_40_MD5 = 0x002B;
	const TLS_RSA_WITH_AES_128_CBC_SHA = 0x002F;
	const TLS_DH_DSS_WITH_AES_128_CBC_SHA = 0x0030;
	const TLS_DH_RSA_WITH_AES_128_CBC_SHA = 0x0031;
	const TLS_DHE_DSS_WITH_AES_128_CBC_SHA = 0x0032;
	const TLS_DHE_RSA_WITH_AES_128_CBC_SHA = 0x0033;
	const TLS_DH_ANON_WITH_AES_128_CBC_SHA = 0x0034;
	const TLS_RSA_WITH_AES_256_CBC_SHA = 0x0035;
	const TLS_DH_DSS_WITH_AES_256_CBC_SHA = 0x0036;
	const TLS_DH_RSA_WITH_AES_256_CBC_SHA = 0x0037;
	const TLS_DHE_DSS_WITH_AES_256_CBC_SHA = 0x0038;
	const TLS_DHE_RSA_WITH_AES_256_CBC_SHA = 0x0039;
	const TLS_DH_ANON_WITH_AES_256_CBC_SHA = 0x003A;
	const TLS_RSA_WITH_NULL_SHA256 = 0x003B;
	const TLS_RSA_WITH_AES_128_CBC_SHA256 = 0x003C;
	const TLS_RSA_WITH_AES_256_CBC_SHA256 = 0x003D;
	const TLS_DH_DSS_WITH_AES_128_CBC_SHA256 = 0x003E;
	const TLS_DH_RSA_WITH_AES_128_CBC_SHA256 = 0x003F;
	const TLS_DHE_DSS_WITH_AES_128_CBC_SHA256 = 0x0040;
	const TLS_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0041;
	const TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0042;
	const TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0043;
	const TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA = 0x0044;
	const TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA = 0x0045;
	const TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA = 0x0046;
	const TLS_RSA_EXPORT1024_WITH_RC4_56_MD5 = 0x0060;
	const TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5 = 0x0061;
	const TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA = 0x0062;
	const TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA = 0x0063;
	const TLS_RSA_EXPORT1024_WITH_RC4_56_SHA = 0x0064;
	const TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA = 0x0065;
	const TLS_DHE_DSS_WITH_RC4_128_SHA = 0x0066;
	const TLS_DHE_RSA_WITH_AES_128_CBC_SHA256 = 0x0067;
	const TLS_DH_DSS_WITH_AES_256_CBC_SHA256 = 0x0068;
	const TLS_DH_RSA_WITH_AES_256_CBC_SHA256 = 0x0069;
	const TLS_DHE_DSS_WITH_AES_256_CBC_SHA256 = 0x006A;
	const TLS_DHE_RSA_WITH_AES_256_CBC_SHA256 = 0x006B;
	const TLS_DH_ANON_WITH_AES_128_CBC_SHA256 = 0x006C;
	const TLS_DH_ANON_WITH_AES_256_CBC_SHA256 = 0x006D;
	# draft-ietf-tls-openpgp-keys-06
	const TLS_DHE_DSS_WITH_3DES_EDE_CBC_RMD = 0x0072;
	const TLS_DHE_DSS_WITH_AES_128_CBC_RMD = 0x0073;
	const TLS_DHE_DSS_WITH_AES_256_CBC_RMD = 0x0074;
	const TLS_DHE_RSA_WITH_3DES_EDE_CBC_RMD = 0x0077;
	const TLS_DHE_RSA_WITH_AES_128_CBC_RMD = 0x0078;
	const TLS_DHE_RSA_WITH_AES_256_CBC_RMD = 0x0079;
	const TLS_RSA_WITH_3DES_EDE_CBC_RMD = 0x007C;
	const TLS_RSA_WITH_AES_128_CBC_RMD = 0x007D;
	const TLS_RSA_WITH_AES_256_CBC_RMD = 0x007E;
	# draft-chudov-cryptopro-cptls-04
	const TLS_GOSTR341094_WITH_28147_CNT_IMIT = 0x0080;
	const TLS_GOSTR341001_WITH_28147_CNT_IMIT = 0x0081;
	const TLS_GOSTR341094_WITH_NULL_GOSTR3411 = 0x0082;
	const TLS_GOSTR341001_WITH_NULL_GOSTR3411 = 0x0083;
	const TLS_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0084;
	const TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0085;
	const TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0086;
	const TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA = 0x0087;
	const TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA = 0x0088;
	const TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA = 0x0089;
	const TLS_PSK_WITH_RC4_128_SHA = 0x008A;
	const TLS_PSK_WITH_3DES_EDE_CBC_SHA = 0x008B;
	const TLS_PSK_WITH_AES_128_CBC_SHA = 0x008C;
	const TLS_PSK_WITH_AES_256_CBC_SHA = 0x008D;
	const TLS_DHE_PSK_WITH_RC4_128_SHA = 0x008E;
	const TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA = 0x008F;
	const TLS_DHE_PSK_WITH_AES_128_CBC_SHA = 0x0090;
	const TLS_DHE_PSK_WITH_AES_256_CBC_SHA = 0x0091;
	const TLS_RSA_PSK_WITH_RC4_128_SHA = 0x0092;
	const TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA = 0x0093;
	const TLS_RSA_PSK_WITH_AES_128_CBC_SHA = 0x0094;
	const TLS_RSA_PSK_WITH_AES_256_CBC_SHA = 0x0095;
	const TLS_RSA_WITH_SEED_CBC_SHA = 0x0096;
	const TLS_DH_DSS_WITH_SEED_CBC_SHA = 0x0097;
	const TLS_DH_RSA_WITH_SEED_CBC_SHA = 0x0098;
	const TLS_DHE_DSS_WITH_SEED_CBC_SHA = 0x0099;
	const TLS_DHE_RSA_WITH_SEED_CBC_SHA = 0x009A;
	const TLS_DH_ANON_WITH_SEED_CBC_SHA = 0x009B;
	const TLS_RSA_WITH_AES_128_GCM_SHA256 = 0x009C;
	const TLS_RSA_WITH_AES_256_GCM_SHA384 = 0x009D;
	const TLS_DHE_RSA_WITH_AES_128_GCM_SHA256 = 0x009E;
	const TLS_DHE_RSA_WITH_AES_256_GCM_SHA384 = 0x009F;
	const TLS_DH_RSA_WITH_AES_128_GCM_SHA256 = 0x00A0;
	const TLS_DH_RSA_WITH_AES_256_GCM_SHA384 = 0x00A1;
	const TLS_DHE_DSS_WITH_AES_128_GCM_SHA256 = 0x00A2;
	const TLS_DHE_DSS_WITH_AES_256_GCM_SHA384 = 0x00A3;
	const TLS_DH_DSS_WITH_AES_128_GCM_SHA256 = 0x00A4;
	const TLS_DH_DSS_WITH_AES_256_GCM_SHA384 = 0x00A5;
	const TLS_DH_ANON_WITH_AES_128_GCM_SHA256 = 0x00A6;
	const TLS_DH_ANON_WITH_AES_256_GCM_SHA384 = 0x00A7;
	const TLS_PSK_WITH_AES_128_GCM_SHA256 = 0x00A8;
	const TLS_PSK_WITH_AES_256_GCM_SHA384 = 0x00A9;
	const TLS_DHE_PSK_WITH_AES_128_GCM_SHA256 = 0x00AA;
	const TLS_DHE_PSK_WITH_AES_256_GCM_SHA384 = 0x00AB;
	const TLS_RSA_PSK_WITH_AES_128_GCM_SHA256 = 0x00AC;
	const TLS_RSA_PSK_WITH_AES_256_GCM_SHA384 = 0x00AD;
	const TLS_PSK_WITH_AES_128_CBC_SHA256 = 0x00AE;
	const TLS_PSK_WITH_AES_256_CBC_SHA384 = 0x00AF;
	const TLS_PSK_WITH_NULL_SHA256 = 0x00B0;
	const TLS_PSK_WITH_NULL_SHA384 = 0x00B1;
	const TLS_DHE_PSK_WITH_AES_128_CBC_SHA256 = 0x00B2;
	const TLS_DHE_PSK_WITH_AES_256_CBC_SHA384 = 0x00B3;
	const TLS_DHE_PSK_WITH_NULL_SHA256 = 0x00B4;
	const TLS_DHE_PSK_WITH_NULL_SHA384 = 0x00B5;
	const TLS_RSA_PSK_WITH_AES_128_CBC_SHA256 = 0x00B6;
	const TLS_RSA_PSK_WITH_AES_256_CBC_SHA384 = 0x00B7;
	const TLS_RSA_PSK_WITH_NULL_SHA256 = 0x00B8;
	const TLS_RSA_PSK_WITH_NULL_SHA384 = 0x00B9;
	const TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BA;
	const TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BB;
	const TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BC;
	const TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BD;
	const TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BE;
	const TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256 = 0x00BF;
	const TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C0;
	const TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C1;
	const TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C2;
	const TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C3;
	const TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C4;
	const TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256 = 0x00C5;
	# draft-ietf-tls-tls13-16
	const TLS_AES_128_GCM_SHA256 = 0x1301;
	const TLS_AES_256_GCM_SHA384 = 0x1302;
	const TLS_CHACHA20_POLY1305_SHA256 = 0x1303;
	const TLS_AES_128_CCM_SHA256 = 0x1304;
	const TLS_AES_128_CCM_8_SHA256 = 0x1305;
	# draft-bmoeller-tls-downgrade-scsv-01
	const TLS_FALLBACK_SCSV = 0x5600;
	# RFC 4492
	const TLS_ECDH_ECDSA_WITH_NULL_SHA = 0xC001;
	const TLS_ECDH_ECDSA_WITH_RC4_128_SHA = 0xC002;
	const TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC003;
	const TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA = 0xC004;
	const TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA = 0xC005;
	const TLS_ECDHE_ECDSA_WITH_NULL_SHA = 0xC006;
	const TLS_ECDHE_ECDSA_WITH_RC4_128_SHA = 0xC007;
	const TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA = 0xC008;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA = 0xC009;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA = 0xC00A;
	const TLS_ECDH_RSA_WITH_NULL_SHA = 0xC00B;
	const TLS_ECDH_RSA_WITH_RC4_128_SHA = 0xC00C;
	const TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA = 0xC00D;
	const TLS_ECDH_RSA_WITH_AES_128_CBC_SHA = 0xC00E;
	const TLS_ECDH_RSA_WITH_AES_256_CBC_SHA = 0xC00F;
	const TLS_ECDHE_RSA_WITH_NULL_SHA = 0xC010;
	const TLS_ECDHE_RSA_WITH_RC4_128_SHA = 0xC011;
	const TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA = 0xC012;
	const TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA = 0xC013;
	const TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA = 0xC014;
	const TLS_ECDH_ANON_WITH_NULL_SHA = 0xC015;
	const TLS_ECDH_ANON_WITH_RC4_128_SHA = 0xC016;
	const TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA = 0xC017;
	const TLS_ECDH_ANON_WITH_AES_128_CBC_SHA = 0xC018;
	const TLS_ECDH_ANON_WITH_AES_256_CBC_SHA = 0xC019;
	const TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA = 0xC01A;
	const TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA = 0xC01B;
	const TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA = 0xC01C;
	const TLS_SRP_SHA_WITH_AES_128_CBC_SHA = 0xC01D;
	const TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA = 0xC01E;
	const TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA = 0xC01F;
	const TLS_SRP_SHA_WITH_AES_256_CBC_SHA = 0xC020;
	const TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA = 0xC021;
	const TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA = 0xC022;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC023;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC024;
	const TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256 = 0xC025;
	const TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384 = 0xC026;
	const TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256 = 0xC027;
	const TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384 = 0xC028;
	const TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256 = 0xC029;
	const TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384 = 0xC02A;
	const TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256 = 0xC02B;
	const TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384 = 0xC02C;
	const TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256 = 0xC02D;
	const TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384 = 0xC02E;
	const TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256 = 0xC02F;
	const TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384 = 0xC030;
	const TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256 = 0xC031;
	const TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384 = 0xC032;
	const TLS_ECDHE_PSK_WITH_RC4_128_SHA = 0xC033;
	const TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA = 0xC034;
	const TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA = 0xC035;
	const TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA = 0xC036;
	const TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256 = 0xC037;
	const TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384 = 0xC038;
	const TLS_ECDHE_PSK_WITH_NULL_SHA = 0xC039;
	const TLS_ECDHE_PSK_WITH_NULL_SHA256 = 0xC03A;
	const TLS_ECDHE_PSK_WITH_NULL_SHA384 = 0xC03B;
	# RFC 6209
	const TLS_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC03C;
	const TLS_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC03D;
	const TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256 = 0xC03E;
	const TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384 = 0xC03F;
	const TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC040;
	const TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC041;
	const TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256 = 0xC042;
	const TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384 = 0xC043;
	const TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC044;
	const TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC045;
	const TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256 = 0xC046;
	const TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384 = 0xC047;
	const TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256 = 0xC048;
	const TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384 = 0xC049;
	const TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256 = 0xC04A;
	const TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384 = 0xC04B;
	const TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC04C;
	const TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC04D;
	const TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256 = 0xC04E;
	const TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384 = 0xC04F;
	const TLS_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC050;
	const TLS_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC051;
	const TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC052;
	const TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC053;
	const TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC054;
	const TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC055;
	const TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256 = 0xC056;
	const TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384 = 0xC057;
	const TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256 = 0xC058;
	const TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384 = 0xC059;
	const TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256 = 0xC05A;
	const TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384 = 0xC05B;
	const TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256 = 0xC05C;
	const TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384 = 0xC05D;
	const TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256 = 0xC05E;
	const TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384 = 0xC05F;
	const TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC060;
	const TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC061;
	const TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256 = 0xC062;
	const TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384 = 0xC063;
	const TLS_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC064;
	const TLS_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC065;
	const TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC066;
	const TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC067;
	const TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC068;
	const TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC069;
	const TLS_PSK_WITH_ARIA_128_GCM_SHA256 = 0xC06A;
	const TLS_PSK_WITH_ARIA_256_GCM_SHA384 = 0xC06B;
	const TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256 = 0xC06C;
	const TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384 = 0xC06D;
	const TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256 = 0xC06E;
	const TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384 = 0xC06F;
	const TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256 = 0xC070;
	const TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384 = 0xC071;
	# RFC 6367
	const TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC072;
	const TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC073;
	const TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC074;
	const TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC075;
	const TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC076;
	const TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC077;
	const TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256 = 0xC078;
	const TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384 = 0xC079;
	const TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC07A;
	const TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC07B;
	const TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC07C;
	const TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC07D;
	const TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC07E;
	const TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC07F;
	const TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256 = 0xC080;
	const TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384 = 0xC081;
	const TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256 = 0xC082;
	const TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384 = 0xC083;
	const TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256 = 0xC084;
	const TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384 = 0xC085;
	const TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC086;
	const TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC087;
	const TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC088;
	const TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC089;
	const TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC08A;
	const TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC08B;
	const TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256 = 0xC08C;
	const TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384 = 0xC08D;
	const TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256 = 0xC08E;
	const TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384 = 0xC08F;
	const TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256 = 0xC090;
	const TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384 = 0xC091;
	const TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256 = 0xC092;
	const TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384 = 0xC093;
	const TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC094;
	const TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC095;
	const TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC096;
	const TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC097;
	const TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC098;
	const TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC099;
	const TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256 = 0xC09A;
	const TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384 = 0xC09B;
	# RFC 6655
	const TLS_RSA_WITH_AES_128_CCM = 0xC09C;
	const TLS_RSA_WITH_AES_256_CCM = 0xC09D;
	const TLS_DHE_RSA_WITH_AES_128_CCM = 0xC09E;
	const TLS_DHE_RSA_WITH_AES_256_CCM = 0xC09F;
	const TLS_RSA_WITH_AES_128_CCM_8 = 0xC0A0;
	const TLS_RSA_WITH_AES_256_CCM_8 = 0xC0A1;
	const TLS_DHE_RSA_WITH_AES_128_CCM_8 = 0xC0A2;
	const TLS_DHE_RSA_WITH_AES_256_CCM_8 = 0xC0A3;
	const TLS_PSK_WITH_AES_128_CCM = 0xC0A4;
	const TLS_PSK_WITH_AES_256_CCM = 0xC0A5;
	const TLS_DHE_PSK_WITH_AES_128_CCM = 0xC0A6;
	const TLS_DHE_PSK_WITH_AES_256_CCM = 0xC0A7;
	const TLS_PSK_WITH_AES_128_CCM_8 = 0xC0A8;
	const TLS_PSK_WITH_AES_256_CCM_8 = 0xC0A9;
	const TLS_PSK_DHE_WITH_AES_128_CCM_8 = 0xC0AA;
	const TLS_PSK_DHE_WITH_AES_256_CCM_8 = 0xC0AB;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CCM = 0xC0AC;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CCM = 0xC0AD;
	const TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8 = 0xC0AE;
	const TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8 = 0xC0AF;
	# draft-agl-tls-chacha20poly1305-02
	const TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD = 0xCC13;
	const TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256_OLD = 0xCC14;
	const TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD = 0xCC15;
	# RFC 7905
	const TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = 0xCCA8;
	const TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256 = 0xCCA9;
	const TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256 = 0xCCAA;
	const TLS_PSK_WITH_CHACHA20_POLY1305_SHA256 = 0xCCAB;
	const TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = 0xCCAC;
	const TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256 = 0xCCAD;
	const TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256 = 0xCCAE;
	# draft-ietf-tls-ecdhe-psk-aead-05
	const TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256 = 0xD001;
	const TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384 = 0xD002;
	const TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256 = 0xD003;
	const TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256 = 0xD004;

	const SSL_RSA_FIPS_WITH_DES_CBC_SHA = 0xFEFE;
	const SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA = 0xFEFF;
	const SSL_RSA_FIPS_WITH_DES_CBC_SHA_2 = 0xFFE1;
	const SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2 = 0xFFE0;
	const SSL_RSA_WITH_RC2_CBC_MD5 = 0xFF80;
	const SSL_RSA_WITH_IDEA_CBC_MD5 = 0xFF81;
	const SSL_RSA_WITH_DES_CBC_MD5 = 0xFF82;
	const SSL_RSA_WITH_3DES_EDE_CBC_MD5 = 0xFF83;
	const TLS_EMPTY_RENEGOTIATION_INFO_SCSV = 0x00FF;

	## This is a table of all known cipher specs.  It can be used for
	## detecting unknown ciphers and for converting the cipher spec
	## constants into a human readable format.
	const cipher_desc: table[count] of string = {
		[SSLv20_CK_RC4_128_EXPORT40_WITH_MD5] =
			"SSLv20_CK_RC4_128_EXPORT40_WITH_MD5",
		[SSLv20_CK_RC4_128_WITH_MD5] = "SSLv20_CK_RC4_128_WITH_MD5",
		[SSLv20_CK_RC2_128_CBC_WITH_MD5] = "SSLv20_CK_RC2_128_CBC_WITH_MD5",
		[SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5] =
			"SSLv20_CK_RC2_128_CBC_EXPORT40_WITH_MD5",
		[SSLv20_CK_IDEA_128_CBC_WITH_MD5] = "SSLv20_CK_IDEA_128_CBC_WITH_MD5",
		[SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5] =
			"SSLv20_CK_DES_192_EDE3_CBC_WITH_MD5",
		[SSLv20_CK_DES_64_CBC_WITH_MD5] = "SSLv20_CK_DES_64_CBC_WITH_MD5",

		[TLS_NULL_WITH_NULL_NULL] = "TLS_NULL_WITH_NULL_NULL",
		[TLS_RSA_WITH_NULL_MD5] = "TLS_RSA_WITH_NULL_MD5",
		[TLS_RSA_WITH_NULL_SHA] = "TLS_RSA_WITH_NULL_SHA",
		[TLS_RSA_EXPORT_WITH_RC4_40_MD5] = "TLS_RSA_EXPORT_WITH_RC4_40_MD5",
		[TLS_RSA_WITH_RC4_128_MD5] = "TLS_RSA_WITH_RC4_128_MD5",
		[TLS_RSA_WITH_RC4_128_SHA] = "TLS_RSA_WITH_RC4_128_SHA",
		[TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5] = "TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5",
		[TLS_RSA_WITH_IDEA_CBC_SHA] = "TLS_RSA_WITH_IDEA_CBC_SHA",
		[TLS_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_RSA_WITH_DES_CBC_SHA] = "TLS_RSA_WITH_DES_CBC_SHA",
		[TLS_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_DSS_WITH_DES_CBC_SHA] = "TLS_DH_DSS_WITH_DES_CBC_SHA",
		[TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_RSA_WITH_DES_CBC_SHA] = "TLS_DH_RSA_WITH_DES_CBC_SHA",
		[TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DHE_DSS_WITH_DES_CBC_SHA] = "TLS_DHE_DSS_WITH_DES_CBC_SHA",
		[TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DHE_RSA_WITH_DES_CBC_SHA] = "TLS_DHE_RSA_WITH_DES_CBC_SHA",
		[TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5] = "TLS_DH_ANON_EXPORT_WITH_RC4_40_MD5",
		[TLS_DH_ANON_WITH_RC4_128_MD5] = "TLS_DH_ANON_WITH_RC4_128_MD5",
		[TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA] = "TLS_DH_ANON_EXPORT_WITH_DES40_CBC_SHA",
		[TLS_DH_ANON_WITH_DES_CBC_SHA] = "TLS_DH_ANON_WITH_DES_CBC_SHA",
		[TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA] = "TLS_DH_ANON_WITH_3DES_EDE_CBC_SHA",
		[SSL_FORTEZZA_KEA_WITH_NULL_SHA] = "SSL_FORTEZZA_KEA_WITH_NULL_SHA",
		[SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA] = "SSL_FORTEZZA_KEA_WITH_FORTEZZA_CBC_SHA",
		[TLS_KRB5_WITH_DES_CBC_SHA] = "TLS_KRB5_WITH_DES_CBC_SHA",
		[TLS_KRB5_WITH_3DES_EDE_CBC_SHA] = "TLS_KRB5_WITH_3DES_EDE_CBC_SHA",
		[TLS_KRB5_WITH_RC4_128_SHA] = "TLS_KRB5_WITH_RC4_128_SHA",
		[TLS_KRB5_WITH_IDEA_CBC_SHA] = "TLS_KRB5_WITH_IDEA_CBC_SHA",
		[TLS_KRB5_WITH_DES_CBC_MD5] = "TLS_KRB5_WITH_DES_CBC_MD5",
		[TLS_KRB5_WITH_3DES_EDE_CBC_MD5] = "TLS_KRB5_WITH_3DES_EDE_CBC_MD5",
		[TLS_KRB5_WITH_RC4_128_MD5] = "TLS_KRB5_WITH_RC4_128_MD5",
		[TLS_KRB5_WITH_IDEA_CBC_MD5] = "TLS_KRB5_WITH_IDEA_CBC_MD5",
		[TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA] = "TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA",
		[TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA] = "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA",
		[TLS_KRB5_EXPORT_WITH_RC4_40_SHA] = "TLS_KRB5_EXPORT_WITH_RC4_40_SHA",
		[TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5] = "TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5",
		[TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5] = "TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5",
		[TLS_KRB5_EXPORT_WITH_RC4_40_MD5] = "TLS_KRB5_EXPORT_WITH_RC4_40_MD5",
		[TLS_RSA_WITH_AES_128_CBC_SHA] = "TLS_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DH_DSS_WITH_AES_128_CBC_SHA] = "TLS_DH_DSS_WITH_AES_128_CBC_SHA",
		[TLS_DH_RSA_WITH_AES_128_CBC_SHA] = "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DHE_DSS_WITH_AES_128_CBC_SHA] = "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
		[TLS_DHE_RSA_WITH_AES_128_CBC_SHA] = "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
		[TLS_DH_ANON_WITH_AES_128_CBC_SHA] = "TLS_DH_ANON_WITH_AES_128_CBC_SHA",
		[TLS_RSA_WITH_AES_256_CBC_SHA] = "TLS_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DH_DSS_WITH_AES_256_CBC_SHA] = "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
		[TLS_DH_RSA_WITH_AES_256_CBC_SHA] = "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DHE_DSS_WITH_AES_256_CBC_SHA] = "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
		[TLS_DHE_RSA_WITH_AES_256_CBC_SHA] = "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
		[TLS_DH_ANON_WITH_AES_256_CBC_SHA] = "TLS_DH_ANON_WITH_AES_256_CBC_SHA",
		[TLS_RSA_WITH_NULL_SHA256] = "TLS_RSA_WITH_NULL_SHA256",
		[TLS_RSA_WITH_AES_128_CBC_SHA256] = "TLS_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_WITH_AES_256_CBC_SHA256] = "TLS_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DH_DSS_WITH_AES_128_CBC_SHA256] = "TLS_DH_DSS_WITH_AES_128_CBC_SHA256",
		[TLS_DH_RSA_WITH_AES_128_CBC_SHA256] = "TLS_DH_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_DHE_DSS_WITH_AES_128_CBC_SHA256] = "TLS_DHE_DSS_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA] = "TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA",
		[TLS_RSA_EXPORT1024_WITH_RC4_56_MD5] = "TLS_RSA_EXPORT1024_WITH_RC4_56_MD5",
		[TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5] = "TLS_RSA_EXPORT1024_WITH_RC2_CBC_56_MD5",
		[TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA] = "TLS_RSA_EXPORT1024_WITH_DES_CBC_SHA",
		[TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA] = "TLS_DHE_DSS_EXPORT1024_WITH_DES_CBC_SHA",
		[TLS_RSA_EXPORT1024_WITH_RC4_56_SHA] = "TLS_RSA_EXPORT1024_WITH_RC4_56_SHA",
		[TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA] = "TLS_DHE_DSS_EXPORT1024_WITH_RC4_56_SHA",
		[TLS_DHE_DSS_WITH_RC4_128_SHA] = "TLS_DHE_DSS_WITH_RC4_128_SHA",
		[TLS_DHE_RSA_WITH_AES_128_CBC_SHA256] = "TLS_DHE_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_DH_DSS_WITH_AES_256_CBC_SHA256] = "TLS_DH_DSS_WITH_AES_256_CBC_SHA256",
		[TLS_DH_RSA_WITH_AES_256_CBC_SHA256] = "TLS_DH_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DHE_DSS_WITH_AES_256_CBC_SHA256] = "TLS_DHE_DSS_WITH_AES_256_CBC_SHA256",
		[TLS_DHE_RSA_WITH_AES_256_CBC_SHA256] = "TLS_DHE_RSA_WITH_AES_256_CBC_SHA256",
		[TLS_DH_ANON_WITH_AES_128_CBC_SHA256] = "TLS_DH_ANON_WITH_AES_128_CBC_SHA256",
		[TLS_DH_ANON_WITH_AES_256_CBC_SHA256] = "TLS_DH_ANON_WITH_AES_256_CBC_SHA256",
		[TLS_DHE_DSS_WITH_3DES_EDE_CBC_RMD] = "TLS_DHE_DSS_WITH_3DES_EDE_CBC_RMD",
		[TLS_DHE_DSS_WITH_AES_128_CBC_RMD] = "TLS_DHE_DSS_WITH_AES_128_CBC_RMD",
		[TLS_DHE_DSS_WITH_AES_256_CBC_RMD] = "TLS_DHE_DSS_WITH_AES_256_CBC_RMD",
		[TLS_DHE_RSA_WITH_3DES_EDE_CBC_RMD] = "TLS_DHE_RSA_WITH_3DES_EDE_CBC_RMD",
		[TLS_DHE_RSA_WITH_AES_128_CBC_RMD] = "TLS_DHE_RSA_WITH_AES_128_CBC_RMD",
		[TLS_DHE_RSA_WITH_AES_256_CBC_RMD] = "TLS_DHE_RSA_WITH_AES_256_CBC_RMD",
		[TLS_RSA_WITH_3DES_EDE_CBC_RMD] = "TLS_RSA_WITH_3DES_EDE_CBC_RMD",
		[TLS_RSA_WITH_AES_128_CBC_RMD] = "TLS_RSA_WITH_AES_128_CBC_RMD",
		[TLS_RSA_WITH_AES_256_CBC_RMD] = "TLS_RSA_WITH_AES_256_CBC_RMD",
		[TLS_GOSTR341094_WITH_28147_CNT_IMIT] = "TLS_GOSTR341094_WITH_28147_CNT_IMIT",
		[TLS_GOSTR341001_WITH_28147_CNT_IMIT] = "TLS_GOSTR341001_WITH_28147_CNT_IMIT",
		[TLS_GOSTR341094_WITH_NULL_GOSTR3411] = "TLS_GOSTR341094_WITH_NULL_GOSTR3411",
		[TLS_GOSTR341001_WITH_NULL_GOSTR3411] = "TLS_GOSTR341001_WITH_NULL_GOSTR3411",
		[TLS_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA] = "TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA",
		[TLS_PSK_WITH_RC4_128_SHA] = "TLS_PSK_WITH_RC4_128_SHA",
		[TLS_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_PSK_WITH_AES_128_CBC_SHA] = "TLS_PSK_WITH_AES_128_CBC_SHA",
		[TLS_PSK_WITH_AES_256_CBC_SHA] = "TLS_PSK_WITH_AES_256_CBC_SHA",
		[TLS_DHE_PSK_WITH_RC4_128_SHA] = "TLS_DHE_PSK_WITH_RC4_128_SHA",
		[TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_DHE_PSK_WITH_AES_128_CBC_SHA] = "TLS_DHE_PSK_WITH_AES_128_CBC_SHA",
		[TLS_DHE_PSK_WITH_AES_256_CBC_SHA] = "TLS_DHE_PSK_WITH_AES_256_CBC_SHA",
		[TLS_RSA_PSK_WITH_RC4_128_SHA] = "TLS_RSA_PSK_WITH_RC4_128_SHA",
		[TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_RSA_PSK_WITH_AES_128_CBC_SHA] = "TLS_RSA_PSK_WITH_AES_128_CBC_SHA",
		[TLS_RSA_PSK_WITH_AES_256_CBC_SHA] = "TLS_RSA_PSK_WITH_AES_256_CBC_SHA",
		[TLS_RSA_WITH_SEED_CBC_SHA] = "TLS_RSA_WITH_SEED_CBC_SHA",
		[TLS_DH_DSS_WITH_SEED_CBC_SHA] = "TLS_DH_DSS_WITH_SEED_CBC_SHA",
		[TLS_DH_RSA_WITH_SEED_CBC_SHA] = "TLS_DH_RSA_WITH_SEED_CBC_SHA",
		[TLS_DHE_DSS_WITH_SEED_CBC_SHA] = "TLS_DHE_DSS_WITH_SEED_CBC_SHA",
		[TLS_DHE_RSA_WITH_SEED_CBC_SHA] = "TLS_DHE_RSA_WITH_SEED_CBC_SHA",
		[TLS_DH_ANON_WITH_SEED_CBC_SHA] = "TLS_DH_ANON_WITH_SEED_CBC_SHA",
		[TLS_RSA_WITH_AES_128_GCM_SHA256] = "TLS_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_RSA_WITH_AES_256_GCM_SHA384] = "TLS_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_RSA_WITH_AES_128_GCM_SHA256] = "TLS_DHE_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_RSA_WITH_AES_256_GCM_SHA384] = "TLS_DHE_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DH_RSA_WITH_AES_128_GCM_SHA256] = "TLS_DH_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_DH_RSA_WITH_AES_256_GCM_SHA384] = "TLS_DH_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_DSS_WITH_AES_128_GCM_SHA256] = "TLS_DHE_DSS_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_DSS_WITH_AES_256_GCM_SHA384] = "TLS_DHE_DSS_WITH_AES_256_GCM_SHA384",
		[TLS_DH_DSS_WITH_AES_128_GCM_SHA256] = "TLS_DH_DSS_WITH_AES_128_GCM_SHA256",
		[TLS_DH_DSS_WITH_AES_256_GCM_SHA384] = "TLS_DH_DSS_WITH_AES_256_GCM_SHA384",
		[TLS_DH_ANON_WITH_AES_128_GCM_SHA256] = "TLS_DH_ANON_WITH_AES_128_GCM_SHA256",
		[TLS_DH_ANON_WITH_AES_256_GCM_SHA384] = "TLS_DH_ANON_WITH_AES_256_GCM_SHA384",
		[TLS_PSK_WITH_AES_128_GCM_SHA256] = "TLS_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_PSK_WITH_AES_256_GCM_SHA384] = "TLS_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_DHE_PSK_WITH_AES_128_GCM_SHA256] = "TLS_DHE_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_DHE_PSK_WITH_AES_256_GCM_SHA384] = "TLS_DHE_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_RSA_PSK_WITH_AES_128_GCM_SHA256] = "TLS_RSA_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_RSA_PSK_WITH_AES_256_GCM_SHA384] = "TLS_RSA_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_PSK_WITH_AES_128_CBC_SHA256] = "TLS_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_PSK_WITH_AES_256_CBC_SHA384] = "TLS_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_PSK_WITH_NULL_SHA256] = "TLS_PSK_WITH_NULL_SHA256",
		[TLS_PSK_WITH_NULL_SHA384] = "TLS_PSK_WITH_NULL_SHA384",
		[TLS_DHE_PSK_WITH_AES_128_CBC_SHA256] = "TLS_DHE_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_DHE_PSK_WITH_AES_256_CBC_SHA384] = "TLS_DHE_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_DHE_PSK_WITH_NULL_SHA256] = "TLS_DHE_PSK_WITH_NULL_SHA256",
		[TLS_DHE_PSK_WITH_NULL_SHA384] = "TLS_DHE_PSK_WITH_NULL_SHA384",
		[TLS_RSA_PSK_WITH_AES_128_CBC_SHA256] = "TLS_RSA_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_RSA_PSK_WITH_AES_256_CBC_SHA384] = "TLS_RSA_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_RSA_PSK_WITH_NULL_SHA256] = "TLS_RSA_PSK_WITH_NULL_SHA256",
		[TLS_RSA_PSK_WITH_NULL_SHA384] = "TLS_RSA_PSK_WITH_NULL_SHA384",
		[TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DH_ANON_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256] = "TLS_DH_ANON_WITH_CAMELLIA_256_CBC_SHA256",
		[TLS_AES_128_GCM_SHA256] = "TLS_AES_128_GCM_SHA256",
		[TLS_AES_256_GCM_SHA384] = "TLS_AES_256_GCM_SHA384",
		[TLS_CHACHA20_POLY1305_SHA256] = "TLS_CHACHA20_POLY1305_SHA256",
		[TLS_AES_128_CCM_SHA256] = "TLS_AES_128_CCM_SHA256",
		[TLS_AES_128_CCM_8_SHA256] = "TLS_AES_128_CCM_8_SHA256",
		[TLS_FALLBACK_SCSV] = "TLS_FALLBACK_SCSV",
		[TLS_ECDH_ECDSA_WITH_NULL_SHA] = "TLS_ECDH_ECDSA_WITH_NULL_SHA",
		[TLS_ECDH_ECDSA_WITH_RC4_128_SHA] = "TLS_ECDH_ECDSA_WITH_RC4_128_SHA",
		[TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA] = "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_NULL_SHA] = "TLS_ECDHE_ECDSA_WITH_NULL_SHA",
		[TLS_ECDHE_ECDSA_WITH_RC4_128_SHA] = "TLS_ECDHE_ECDSA_WITH_RC4_128_SHA",
		[TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDH_RSA_WITH_NULL_SHA] = "TLS_ECDH_RSA_WITH_NULL_SHA",
		[TLS_ECDH_RSA_WITH_RC4_128_SHA] = "TLS_ECDH_RSA_WITH_RC4_128_SHA",
		[TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_RSA_WITH_AES_128_CBC_SHA] = "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_RSA_WITH_AES_256_CBC_SHA] = "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_NULL_SHA] = "TLS_ECDHE_RSA_WITH_NULL_SHA",
		[TLS_ECDHE_RSA_WITH_RC4_128_SHA] = "TLS_ECDHE_RSA_WITH_RC4_128_SHA",
		[TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
		[TLS_ECDH_ANON_WITH_NULL_SHA] = "TLS_ECDH_ANON_WITH_NULL_SHA",
		[TLS_ECDH_ANON_WITH_RC4_128_SHA] = "TLS_ECDH_ANON_WITH_RC4_128_SHA",
		[TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDH_ANON_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDH_ANON_WITH_AES_128_CBC_SHA] = "TLS_ECDH_ANON_WITH_AES_128_CBC_SHA",
		[TLS_ECDH_ANON_WITH_AES_256_CBC_SHA] = "TLS_ECDH_ANON_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA",
		[TLS_SRP_SHA_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA",
		[TLS_SRP_SHA_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA",
		[TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA] = "TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256] = "TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256",
		[TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384] = "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256] = "TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256",
		[TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384] = "TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384",
		[TLS_ECDHE_PSK_WITH_RC4_128_SHA] = "TLS_ECDHE_PSK_WITH_RC4_128_SHA",
		[TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA] = "TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA] = "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA] = "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA",
		[TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256] = "TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256",
		[TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384] = "TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384",
		[TLS_ECDHE_PSK_WITH_NULL_SHA] = "TLS_ECDHE_PSK_WITH_NULL_SHA",
		[TLS_ECDHE_PSK_WITH_NULL_SHA256] = "TLS_ECDHE_PSK_WITH_NULL_SHA256",
		[TLS_ECDHE_PSK_WITH_NULL_SHA384] = "TLS_ECDHE_PSK_WITH_NULL_SHA384",
		[TLS_RSA_WITH_ARIA_128_CBC_SHA256] = "TLS_RSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_RSA_WITH_ARIA_256_CBC_SHA384] = "TLS_RSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256] = "TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256",
		[TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384] = "TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384",
		[TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256] = "TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384] = "TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256] = "TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256",
		[TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384] = "TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384",
		[TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256] = "TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384] = "TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256] = "TLS_DH_ANON_WITH_ARIA_128_CBC_SHA256",
		[TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384] = "TLS_DH_ANON_WITH_ARIA_256_CBC_SHA384",
		[TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256] = "TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384] = "TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256] = "TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384] = "TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256] = "TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384] = "TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256] = "TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256",
		[TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384] = "TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384",
		[TLS_RSA_WITH_ARIA_128_GCM_SHA256] = "TLS_RSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_RSA_WITH_ARIA_256_GCM_SHA384] = "TLS_RSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256] = "TLS_DHE_RSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384] = "TLS_DHE_RSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256] = "TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384] = "TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256] = "TLS_DHE_DSS_WITH_ARIA_128_GCM_SHA256",
		[TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384] = "TLS_DHE_DSS_WITH_ARIA_256_GCM_SHA384",
		[TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256] = "TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256",
		[TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384] = "TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384",
		[TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256] = "TLS_DH_ANON_WITH_ARIA_128_GCM_SHA256",
		[TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384] = "TLS_DH_ANON_WITH_ARIA_256_GCM_SHA384",
		[TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256] = "TLS_ECDHE_ECDSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384] = "TLS_ECDHE_ECDSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256] = "TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384] = "TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256] = "TLS_ECDHE_RSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384] = "TLS_ECDHE_RSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256] = "TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256",
		[TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384] = "TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384",
		[TLS_PSK_WITH_ARIA_128_CBC_SHA256] = "TLS_PSK_WITH_ARIA_128_CBC_SHA256",
		[TLS_PSK_WITH_ARIA_256_CBC_SHA384] = "TLS_PSK_WITH_ARIA_256_CBC_SHA384",
		[TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256] = "TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256",
		[TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384] = "TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384",
		[TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256] = "TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256",
		[TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384] = "TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384",
		[TLS_PSK_WITH_ARIA_128_GCM_SHA256] = "TLS_PSK_WITH_ARIA_128_GCM_SHA256",
		[TLS_PSK_WITH_ARIA_256_GCM_SHA384] = "TLS_PSK_WITH_ARIA_256_GCM_SHA384",
		[TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256] = "TLS_DHE_PSK_WITH_ARIA_128_GCM_SHA256",
		[TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384] = "TLS_DHE_PSK_WITH_ARIA_256_GCM_SHA384",
		[TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256] = "TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256",
		[TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384] = "TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384",
		[TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256] = "TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256",
		[TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384] = "TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384",
		[TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DHE_RSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DHE_RSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DHE_DSS_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DHE_DSS_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DH_ANON_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DH_ANON_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_ECDHE_RSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_ECDHE_RSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_DHE_PSK_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_DHE_PSK_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256] = "TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256",
		[TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384] = "TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384",
		[TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256] = "TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256",
		[TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384] = "TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384",
		[TLS_RSA_WITH_AES_128_CCM] = "TLS_RSA_WITH_AES_128_CCM",
		[TLS_RSA_WITH_AES_256_CCM] = "TLS_RSA_WITH_AES_256_CCM",
		[TLS_DHE_RSA_WITH_AES_128_CCM] = "TLS_DHE_RSA_WITH_AES_128_CCM",
		[TLS_DHE_RSA_WITH_AES_256_CCM] = "TLS_DHE_RSA_WITH_AES_256_CCM",
		[TLS_RSA_WITH_AES_128_CCM_8] = "TLS_RSA_WITH_AES_128_CCM_8",
		[TLS_RSA_WITH_AES_256_CCM_8] = "TLS_RSA_WITH_AES_256_CCM_8",
		[TLS_DHE_RSA_WITH_AES_128_CCM_8] = "TLS_DHE_RSA_WITH_AES_128_CCM_8",
		[TLS_DHE_RSA_WITH_AES_256_CCM_8] = "TLS_DHE_RSA_WITH_AES_256_CCM_8",
		[TLS_PSK_WITH_AES_128_CCM] = "TLS_PSK_WITH_AES_128_CCM",
		[TLS_PSK_WITH_AES_256_CCM] = "TLS_PSK_WITH_AES_256_CCM",
		[TLS_DHE_PSK_WITH_AES_128_CCM] = "TLS_DHE_PSK_WITH_AES_128_CCM",
		[TLS_DHE_PSK_WITH_AES_256_CCM] = "TLS_DHE_PSK_WITH_AES_256_CCM",
		[TLS_PSK_WITH_AES_128_CCM_8] = "TLS_PSK_WITH_AES_128_CCM_8",
		[TLS_PSK_WITH_AES_256_CCM_8] = "TLS_PSK_WITH_AES_256_CCM_8",
		[TLS_PSK_DHE_WITH_AES_128_CCM_8] = "TLS_PSK_DHE_WITH_AES_128_CCM_8",
		[TLS_PSK_DHE_WITH_AES_256_CCM_8] = "TLS_PSK_DHE_WITH_AES_256_CCM_8",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CCM] = "TLS_ECDHE_ECDSA_WITH_AES_128_CCM",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CCM] = "TLS_ECDHE_ECDSA_WITH_AES_256_CCM",
		[TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8] = "TLS_ECDHE_ECDSA_WITH_AES_128_CCM_8",
		[TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8] = "TLS_ECDHE_ECDSA_WITH_AES_256_CCM_8",
		[TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD] = "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD",
		[TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256_OLD] = "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256_OLD",
		[TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD] = "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256_OLD",
		[TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256] = "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256] = "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256] = "TLS_DHE_RSA_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_PSK_WITH_CHACHA20_POLY1305_SHA256] = "TLS_PSK_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256] = "TLS_ECDHE_PSK_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256] = "TLS_DHE_PSK_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256] = "TLS_RSA_PSK_WITH_CHACHA20_POLY1305_SHA256",
		[TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256] = "TLS_ECDHE_PSK_WITH_AES_128_GCM_SHA256",
		[TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384] = "TLS_ECDHE_PSK_WITH_AES_256_GCM_SHA384",
		[TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256] = "TLS_ECDHE_PSK_WITH_AES_128_CCM_8_SHA256",
		[TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256] = "TLS_ECDHE_PSK_WITH_AES_128_CCM_SHA256",
		[SSL_RSA_FIPS_WITH_DES_CBC_SHA] = "SSL_RSA_FIPS_WITH_DES_CBC_SHA",
		[SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA] = "SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA",
		[SSL_RSA_FIPS_WITH_DES_CBC_SHA_2] = "SSL_RSA_FIPS_WITH_DES_CBC_SHA_2",
		[SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2] = "SSL_RSA_FIPS_WITH_3DES_EDE_CBC_SHA_2",
		[SSL_RSA_WITH_RC2_CBC_MD5] = "SSL_RSA_WITH_RC2_CBC_MD5",
		[SSL_RSA_WITH_IDEA_CBC_MD5] = "SSL_RSA_WITH_IDEA_CBC_MD5",
		[SSL_RSA_WITH_DES_CBC_MD5] = "SSL_RSA_WITH_DES_CBC_MD5",
		[SSL_RSA_WITH_3DES_EDE_CBC_MD5] = "SSL_RSA_WITH_3DES_EDE_CBC_MD5",
		[TLS_EMPTY_RENEGOTIATION_INFO_SCSV] = "TLS_EMPTY_RENEGOTIATION_INFO_SCSV",
	} &default=function(i: count):string { return fmt("unknown-%d", i); };

}
