# -*- coding: utf-8 -*-
#
# ccwatcher (http://ccwatcher.sourceforge.net/)
# Copyright (C) 2009-2013 Xaver Wurzenberger <xaverxn at users.sourceforge.net>
#
# This program is free software; you can redistribute and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY, without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.

"""ccwatcher_gui/__init__.py
"""

from PyQt4 import QtGui, QtCore
import shutil
import time
import os
import subprocess
from ccwatcher_py.ccw_messages import str_thisisccw_pt1, str_thisisccw_pt2, list_plot_mode,  list_plot_mode_qwt, list_multi_mode, list_verbosity_str

__version__ = "1.3.1"


class BestPointsDialog(QtGui.QDialog):
    def __init__(self, filenames, parent = None):
        super(BestPointsDialog, self).__init__(parent)
        
        self.setWindowTitle("Lowest scan energies")
        self.textwid = QtGui.QTextEdit()
        self.textwid.setReadOnly(True)
        self.buttonBox = QtGui.QDialogButtonBox(QtGui.QDialogButtonBox.Cancel)

        self.layout = QtGui.QVBoxLayout()
        self.layout.addWidget(self.textwid)
        self.layout.addWidget(self.buttonBox)
        self.setLayout(self.layout)
        
        self.connect(self.buttonBox, QtCore.SIGNAL("rejected()"), QtCore.SLOT("reject()"))



class MultiDialog(QtGui.QDialog):
    def __init__(self, filenames, parent = None):
        super(MultiDialog, self).__init__(parent)

        self.multiComboBox = QtGui.QComboBox()
        self.multiComboBox.insertItems(1, list_multi_mode)
        self.multiComboBox.setCurrentIndex(1) #we could set it to the settingsfile entry, but that would mean calling readsettings again...
        self.multiComboBoxLabel = QtGui.QLabel("&Multi-parse mode:")
        self.multiComboBoxLabel.setBuddy(self.multiComboBox)
        self.buttonBox = QtGui.QDialogButtonBox(QtGui.QDialogButtonBox.Save|QtGui.QDialogButtonBox.Cancel)

        self.fileOrderListWidget = QtGui.QListWidget()
        self.fileOrderListWidget.addItems(filenames)
#        self.fileOrderListWidget.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.fileOrderListWidget.setCurrentRow(len(filenames)-1) #select the last entry by default
        self.fileOrderListLabel = QtGui.QLabel("&Text-parse which file?")
        self.fileOrderListLabel.setBuddy(self.fileOrderListWidget)

        self.multiTabLayout = QtGui.QGridLayout()
        self.multiTabLayout.addWidget(self.multiComboBoxLabel, 0, 0)
        self.multiTabLayout.addWidget(self.multiComboBox, 0, 1)
        self.multiTabLayout.addWidget(self.fileOrderListLabel, 1, 0)
        self.multiTabLayout.addWidget(self.fileOrderListWidget, 2, 0, 1, 2)
        self.multiTabLayout.addWidget(self.buttonBox, 3, 1)
        self.setLayout(self.multiTabLayout)

        self.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), QtCore.SLOT("accept()"))#QtGui.QDialog.accept(self))
        self.connect(self.buttonBox, QtCore.SIGNAL("rejected()"), QtCore.SLOT("reject()"))

        self.setWindowTitle("Choose multifile parsing mode and main file...")


class SettingsDialog(QtGui.QDialog):
    """Here, the current settings from the ini file are displayed (NOT the current ones and NOT the hardcoded ones!). Clicking the reset button restores also these and NOT the hardcoded ones. Calling setup.py is the way to restore the latter..."""
    def __init__(self, parent = None):
        super(SettingsDialog, self).__init__(parent)
        
        import ccwatcher_py.ccw_settingshandler
        self.ccw_settingshandler = ccwatcher_py.ccw_settingshandler #needed for writing it back at the end of 'accept()'

        self.settingsfile = self.ccw_settingshandler.getSettingsFileLocation()
        self.dict_settings = self.ccw_settingshandler.readSettings(self.settingsfile)
        self.olddict = self.dict_settings.copy() #needed for reset() function

        self.multiComboBox = QtGui.QComboBox()
        self.multiComboBox.insertItems(1, list_multi_mode)
        self.multiComboBoxLabel = QtGui.QLabel("&Multi-parse mode:")
        self.multiComboBoxLabel.setBuddy(self.multiComboBox)
        self.multiComboBox.setCurrentIndex(self.dict_settings['num_multifile_opt'])
        
        self.verbosityComboBox = QtGui.QComboBox()
        self.verbosityComboBox.insertItems(1, list_verbosity_str) #repr is necessary because it needs a QStringList otherwise
        self.verbosityComboBox.setCurrentIndex(self.dict_settings['num_verbosity'])
        self.verbosityComboBox.setEditable(False)
        self.verbosityComboBoxLabel = QtGui.QLabel("&Verbosity:")
        self.verbosityComboBoxLabel.setBuddy(self.verbosityComboBox)
        
        self.fileageSpinBox = QtGui.QSpinBox()
        self.fileageSpinBox.setRange(1, 600)
        self.fileageSpinBox.setValue(self.dict_settings['num_max_file_age_minutes'])
        self.fileageSpinBox.setSuffix(" Minutes")
        self.fileageLabel = QtGui.QLabel("Ma&x. file age:")
        self.fileageLabel.setBuddy(self.fileageSpinBox)
        
        self.useguiCheckBox = QtGui.QCheckBox("Always &use GUI")
        self.useguiCheckBox.setChecked(self.dict_settings['use_gui'])

        self.convergenceBox = QtGui.QCheckBox("&No 0%-Convergence-Messages")
        self.convergenceBox.setChecked(self.dict_settings['no_zero_convergence'])       

        self.guessModeBox = QtGui.QCheckBox("&Do not guess logfile: start GUI empty")
        self.guessModeBox.setChecked(self.dict_settings['no_guess_mode'])   

        self.avopathEdit = QtGui.QLineEdit(self.dict_settings['str_avogadro_path'])
        self.avopathLabel = QtGui.QLabel("&Avogadro path:")
        self.avopathLabel.setBuddy(self.avopathEdit)
        
        self.gnuplotpathEdit = QtGui.QLineEdit(self.dict_settings['str_gnuplot_path'])
        self.gnuplotpathLabel = QtGui.QLabel("&Gnuplot path:")
        self.gnuplotpathLabel.setBuddy(self.gnuplotpathEdit)
    
        self.optionsTabLayout = QtGui.QGridLayout()
        self.optionsTabLayout.addWidget(self.verbosityComboBoxLabel, 0, 0)
        self.optionsTabLayout.addWidget(self.verbosityComboBox, 0, 1)
        self.optionsTabLayout.addWidget(self.multiComboBoxLabel, 1, 0)
        self.optionsTabLayout.addWidget(self.multiComboBox, 1, 1)        
        self.optionsTabLayout.addWidget(self.fileageLabel, 2, 0)
        self.optionsTabLayout.addWidget(self.fileageSpinBox, 2, 1)
        self.optionsTabLayout.addWidget(self.avopathLabel, 3, 0)
        self.optionsTabLayout.addWidget(self.avopathEdit, 3, 1)
        self.optionsTabLayout.addWidget(self.gnuplotpathLabel, 4, 0)
        self.optionsTabLayout.addWidget(self.gnuplotpathEdit, 4, 1)
        self.optionsTabLayout.addWidget(self.useguiCheckBox, 5, 0)
        self.optionsTabLayout.addWidget(self.convergenceBox, 5, 1, QtCore.Qt.AlignRight)
        self.optionsTabLayout.addWidget(self.guessModeBox, 6, 0, 1, 2)
        
        self.optionsTab = QtGui.QWidget()
        self.optionsTab.setLayout(self.optionsTabLayout)


        self.keywordsLabel = QtGui.QLabel("Add keyword:")

        self.keywordsListWidget_gaussian = QtGui.QListWidget()
        self.keywordsListWidget_gaussian.addItems(self.dict_settings['list_keywords_gaussian'])
        self.keywordsListWidget_gaussian.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_gaussian = QtGui.QLineEdit()
        self.keywordaddButton_gaussian = QtGui.QPushButton("&Add")
        self.keywordaddButton_gaussian = QtGui.QPushButton("&Add")
        self.mymenu_gaussian = QtGui.QMenu()
        self.removeAction_gaussian = QtGui.QAction("Remove", self)
        self.mymenu_gaussian.addAction(self.removeAction_gaussian)
        self.keywordsTabLayout_gaussian = QtGui.QGridLayout()
        self.keywordsTabLayout_gaussian.addWidget(self.keywordsListWidget_gaussian, 0, 0, 1, 3)
        self.keywordsTabLayout_gaussian.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_gaussian.addWidget(self.keywordsLineEdit_gaussian, 1, 1)
        self.keywordsTabLayout_gaussian.addWidget(self.keywordaddButton_gaussian, 1, 2)
        self.keywordsTab_gaussian = QtGui.QWidget()
        self.keywordsTab_gaussian.setLayout(self.keywordsTabLayout_gaussian)
        
        self.keywordsListWidget_orca = QtGui.QListWidget()
        self.keywordsListWidget_orca.addItems(self.dict_settings['list_keywords_orca'])
        self.keywordsListWidget_orca.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_orca = QtGui.QLineEdit()
        self.keywordaddButton_orca = QtGui.QPushButton("&Add")
        self.mymenu_orca = QtGui.QMenu()
        self.removeAction_orca = QtGui.QAction("Remove", self)
        self.mymenu_orca.addAction(self.removeAction_orca)
        self.keywordsTabLayout_orca = QtGui.QGridLayout()
        self.keywordsTabLayout_orca.addWidget(self.keywordsListWidget_orca, 0, 0, 1, 3)
        self.keywordsTabLayout_orca.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_orca.addWidget(self.keywordsLineEdit_orca, 1, 1)
        self.keywordsTabLayout_orca.addWidget(self.keywordaddButton_orca, 1, 2)
        self.keywordsTab_orca = QtGui.QWidget()
        self.keywordsTab_orca.setLayout(self.keywordsTabLayout_orca)

        self.keywordsListWidget_jaguar = QtGui.QListWidget()
        self.keywordsListWidget_jaguar.addItems(self.dict_settings['list_keywords_jaguar'])
        self.keywordsListWidget_jaguar.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_jaguar = QtGui.QLineEdit()
        self.keywordaddButton_jaguar = QtGui.QPushButton("&Add")
        self.mymenu_jaguar = QtGui.QMenu()
        self.removeAction_jaguar = QtGui.QAction("Remove", self)
        self.mymenu_jaguar.addAction(self.removeAction_jaguar)
        self.keywordsTabLayout_jaguar = QtGui.QGridLayout()
        self.keywordsTabLayout_jaguar.addWidget(self.keywordsListWidget_jaguar, 0, 0, 1, 3)
        self.keywordsTabLayout_jaguar.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_jaguar.addWidget(self.keywordsLineEdit_jaguar, 1, 1)
        self.keywordsTabLayout_jaguar.addWidget(self.keywordaddButton_jaguar, 1, 2)
        self.keywordsTab_jaguar = QtGui.QWidget()
        self.keywordsTab_jaguar.setLayout(self.keywordsTabLayout_jaguar)

        self.keywordsListWidget_gamess_us = QtGui.QListWidget()
        self.keywordsListWidget_gamess_us.addItems(self.dict_settings['list_keywords_gamess_us'])
        self.keywordsListWidget_gamess_us.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_gamess_us = QtGui.QLineEdit()
        self.keywordaddButton_gamess_us = QtGui.QPushButton("&Add")
        self.mymenu_gamess_us = QtGui.QMenu()
        self.removeAction_gamess_us = QtGui.QAction("Remove", self)
        self.mymenu_gamess_us.addAction(self.removeAction_gamess_us)
        self.keywordsTabLayout_gamess_us = QtGui.QGridLayout()
        self.keywordsTabLayout_gamess_us.addWidget(self.keywordsListWidget_gamess_us, 0, 0, 1, 3)
        self.keywordsTabLayout_gamess_us.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_gamess_us.addWidget(self.keywordsLineEdit_gamess_us, 1, 1)
        self.keywordsTabLayout_gamess_us.addWidget(self.keywordaddButton_gamess_us, 1, 2)
        self.keywordsTab_gamess_us = QtGui.QWidget()
        self.keywordsTab_gamess_us.setLayout(self.keywordsTabLayout_gamess_us)

        self.keywordsListWidget_nwchem = QtGui.QListWidget()
        self.keywordsListWidget_nwchem.addItems(self.dict_settings['list_keywords_nwchem'])
        self.keywordsListWidget_nwchem.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_nwchem = QtGui.QLineEdit()
        self.keywordaddButton_nwchem = QtGui.QPushButton("&Add")
        self.mymenu_nwchem = QtGui.QMenu()
        self.removeAction_nwchem = QtGui.QAction("Remove", self)
        self.mymenu_nwchem.addAction(self.removeAction_nwchem)
        self.keywordsTabLayout_nwchem = QtGui.QGridLayout()
        self.keywordsTabLayout_nwchem.addWidget(self.keywordsListWidget_nwchem, 0, 0, 1, 3)
        self.keywordsTabLayout_nwchem.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_nwchem.addWidget(self.keywordsLineEdit_nwchem, 1, 1)
        self.keywordsTabLayout_nwchem.addWidget(self.keywordaddButton_nwchem, 1, 2)
        self.keywordsTab_nwchem = QtGui.QWidget()
        self.keywordsTab_nwchem.setLayout(self.keywordsTabLayout_nwchem)

        self.keywordsListWidget_adf2012 = QtGui.QListWidget()
        self.keywordsListWidget_adf2012.addItems(self.dict_settings['list_keywords_adf2012'])
        self.keywordsListWidget_adf2012.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_adf2012 = QtGui.QLineEdit()
        self.keywordaddButton_adf2012 = QtGui.QPushButton("&Add")
        self.mymenu_adf2012 = QtGui.QMenu()
        self.removeAction_adf2012 = QtGui.QAction("Remove", self)
        self.mymenu_adf2012.addAction(self.removeAction_adf2012)
        self.keywordsTabLayout_adf2012 = QtGui.QGridLayout()
        self.keywordsTabLayout_adf2012.addWidget(self.keywordsListWidget_adf2012, 0, 0, 1, 3)
        self.keywordsTabLayout_adf2012.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_adf2012.addWidget(self.keywordsLineEdit_adf2012, 1, 1)
        self.keywordsTabLayout_adf2012.addWidget(self.keywordaddButton_adf2012, 1, 2)
        self.keywordsTab_adf2012 = QtGui.QWidget()
        self.keywordsTab_adf2012.setLayout(self.keywordsTabLayout_adf2012)

        self.keywordsListWidget_turbomole = QtGui.QListWidget()
        self.keywordsListWidget_turbomole.addItems(self.dict_settings['list_keywords_turbomole'])
        self.keywordsListWidget_turbomole.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_turbomole = QtGui.QLineEdit()
        self.keywordaddButton_turbomole = QtGui.QPushButton("&Add")
        self.mymenu_turbomole = QtGui.QMenu()
        self.removeAction_turbomole = QtGui.QAction("Remove", self)
        self.mymenu_turbomole.addAction(self.removeAction_turbomole)
        self.keywordsTabLayout_turbomole = QtGui.QGridLayout()
        self.keywordsTabLayout_turbomole.addWidget(self.keywordsListWidget_turbomole, 0, 0, 1, 3)
        self.keywordsTabLayout_turbomole.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_turbomole.addWidget(self.keywordsLineEdit_turbomole, 1, 1)
        self.keywordsTabLayout_turbomole.addWidget(self.keywordaddButton_turbomole, 1, 2)
        self.keywordsTab_turbomole = QtGui.QWidget()
        self.keywordsTab_turbomole.setLayout(self.keywordsTabLayout_turbomole)

        self.keywordsListWidget_molpro = QtGui.QListWidget()
        self.keywordsListWidget_molpro.addItems(self.dict_settings['list_keywords_molpro'])
        self.keywordsListWidget_molpro.setContextMenuPolicy(QtCore.Qt.CustomContextMenu)
        self.keywordsLineEdit_molpro = QtGui.QLineEdit()
        self.keywordaddButton_molpro = QtGui.QPushButton("&Add")
        self.mymenu_molpro = QtGui.QMenu()
        self.removeAction_molpro = QtGui.QAction("Remove", self)
        self.mymenu_molpro.addAction(self.removeAction_molpro)
        self.keywordsTabLayout_molpro = QtGui.QGridLayout()
        self.keywordsTabLayout_molpro.addWidget(self.keywordsListWidget_molpro, 0, 0, 1, 3)
        self.keywordsTabLayout_molpro.addWidget(self.keywordsLabel, 1, 0)
        self.keywordsTabLayout_molpro.addWidget(self.keywordsLineEdit_molpro, 1, 1)
        self.keywordsTabLayout_molpro.addWidget(self.keywordaddButton_molpro, 1, 2)
        self.keywordsTab_molpro = QtGui.QWidget()
        self.keywordsTab_molpro.setLayout(self.keywordsTabLayout_molpro)


        self.tabs = QtGui.QTabWidget()
        self.tabs.addTab(self.optionsTab, "&Options")
        programsdict = {"_gaussian":"Gaussian", "_adf2012":"ADF2012", "_turbomole":"Turbomole", "_gamess_us":"Gamess(US)", "_molpro":"Molpro", "_orca":"ORCA", "_nwchem":"NWChem", "_jaguar":"Jaguar"}
        for program in programsdict.iterkeys():
            self.tabs.addTab(eval(str('self.keywordsTab'+program)),  "&"+programsdict.get(program))
#        for program in ("_gaussian", "_adf2012", "_turbomole", "_gamess_us", "_molpro", "_orca", "_nwchem", "_jaguar"):
#            self.tabs.addTab(eval(str('self.keywordsTab'+program)),  "&"+program[1:])
        
        self.buttonBox = QtGui.QDialogButtonBox(QtGui.QDialogButtonBox.Save|QtGui.QDialogButtonBox.Cancel)
        self.resetButton = QtGui.QPushButton("&Reset")
        
        self.buttonslayout = QtGui.QHBoxLayout()
        self.buttonslayout.addWidget(self.resetButton)
        self.buttonslayout.addWidget(self.buttonBox)
        
        self.overallLayout = QtGui.QVBoxLayout()
        self.overallLayout.addWidget(self.tabs)
        self.overallLayout.addLayout(self.buttonslayout)
        self.setLayout(self.overallLayout)
        
        self.connect(self.keywordaddButton_gaussian, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_gaussian)
        self.connect(self.keywordaddButton_orca, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_orca)
        self.connect(self.keywordaddButton_jaguar, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_jaguar)
        self.connect(self.keywordaddButton_gamess_us, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_gamess_us)
        self.connect(self.keywordaddButton_nwchem, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_nwchem)
        self.connect(self.keywordaddButton_turbomole, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_turbomole)
        self.connect(self.keywordaddButton_adf2012, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_adf2012)
        self.connect(self.keywordaddButton_molpro, QtCore.SIGNAL("clicked()"), self.updateKeywordsListWidget_molpro)
        self.connect(self.buttonBox, QtCore.SIGNAL("accepted()"), self.accept)
        self.connect(self.buttonBox, QtCore.SIGNAL("rejected()"), QtCore.SLOT("reject()"))
        self.connect(self.removeAction_gaussian,QtCore.SIGNAL('triggered()'), self.removeKeyword_gaussian)
        self.connect(self.removeAction_orca,QtCore.SIGNAL('triggered()'), self.removeKeyword_orca)
        self.connect(self.removeAction_jaguar,QtCore.SIGNAL('triggered()'), self.removeKeyword_jaguar)
        self.connect(self.removeAction_gamess_us,QtCore.SIGNAL('triggered()'), self.removeKeyword_gamess_us)
        self.connect(self.removeAction_turbomole,QtCore.SIGNAL('triggered()'), self.removeKeyword_turbomole)
        self.connect(self.removeAction_adf2012,QtCore.SIGNAL('triggered()'), self.removeKeyword_adf2012)
        self.connect(self.removeAction_molpro,QtCore.SIGNAL('triggered()'), self.removeKeyword_molpro)
        self.connect(self.removeAction_nwchem,QtCore.SIGNAL('triggered()'), self.removeKeyword_nwchem)
        self.connect(self.keywordsListWidget_gaussian, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_gaussian)
        self.connect(self.keywordsListWidget_orca, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_orca)
        self.connect(self.keywordsListWidget_jaguar, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_jaguar)
        self.connect(self.keywordsListWidget_gamess_us, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_gamess_us)
        self.connect(self.keywordsListWidget_turbomole, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_turbomole)
        self.connect(self.keywordsListWidget_adf2012, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_adf2012)
        self.connect(self.keywordsListWidget_molpro, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_molpro)
        self.connect(self.keywordsListWidget_nwchem, QtCore.SIGNAL("customContextMenuRequested(QPoint)"), self.raiseRemoveMenu_nwchem)
        self.connect(self.resetButton, QtCore.SIGNAL("clicked()"), self.reset)
        
        self.setWindowTitle("Manage default settings")
        

    
    def updateKeywordsListWidget_gaussian(self):
        str_item = str(self.keywordsLineEdit_gaussian.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_gaussian.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_gaussian.addItem(str_item)
                self.keywordsLineEdit_gaussian.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")

    def updateKeywordsListWidget_orca(self):
        str_item = str(self.keywordsLineEdit_orca.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_orca.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_orca.addItem(str_item)
                self.keywordsLineEdit_orca.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")


    def updateKeywordsListWidget_jaguar(self):
        str_item = str(self.keywordsLineEdit_jaguar.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_jaguar.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_jaguar.addItem(str_item)
                self.keywordsLineEdit_jaguar.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")
            
            
    def updateKeywordsListWidget_gamess_us(self):
        str_item = str(self.keywordsLineEdit_gamess_us.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_gamess_us.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_gamess_us.addItem(str_item)
                self.keywordsLineEdit_gamess_us.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")

    def updateKeywordsListWidget_molpro(self):
        str_item = str(self.keywordsLineEdit_molpro.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_molpro.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_molpro.addItem(str_item)
                self.keywordsLineEdit_molpro.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")


    def updateKeywordsListWidget_adf2012(self):
        str_item = str(self.keywordsLineEdit_adf2012.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_adf2012.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_adf2012.addItem(str_item)
                self.keywordsLineEdit_adf2012.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")
 

    def updateKeywordsListWidget_nwchem(self):
        str_item = str(self.keywordsLineEdit_nwchem.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_nwchem.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_nwchem.addItem(str_item)
                self.keywordsLineEdit_nwchem.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")


    def updateKeywordsListWidget_turbomole(self):
        str_item = str(self.keywordsLineEdit_turbomole.text()).strip() #QString
        if str_item.find(",") == -1:
            if self.keywordsListWidget_turbomole.findItems(str_item, QtCore.Qt.MatchCaseSensitive):
                QtGui.QMessageBox.warning(self, "Duplicate","Already in the list!")
            elif len(str_item) < 2:
                QtGui.QMessageBox.warning(self, "Too short", "Enter two characters at least")
            else:
                self.keywordsListWidget_turbomole.addItem(str_item)
                self.keywordsLineEdit_turbomole.clear()
        else:
            QtGui.QMessageBox.warning(self, "Illegal character","Keywords must not contain commas. Please file a bug report if that creates problems for you.")

    
    def raiseRemoveMenu_gaussian(self, qpoint_screen):
#        print self.keywordsListWidget.itemAt(qpoint_screen).text()
        self.removeAction_gaussian.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_gaussian.mapToGlobal(qpoint_screen)
        self.mymenu_gaussian.exec_(qpoint_widget)
    

    def raiseRemoveMenu_nwchem(self, qpoint_screen):
        self.removeAction_nwchem.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_nwchem.mapToGlobal(qpoint_screen)
        self.mymenu_nwchem.exec_(qpoint_widget)        


    def raiseRemoveMenu_orca(self, qpoint_screen):
        self.removeAction_orca.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_orca.mapToGlobal(qpoint_screen)
        self.mymenu_orca.exec_(qpoint_widget)

    def raiseRemoveMenu_jaguar(self, qpoint_screen):
        self.removeAction_jaguar.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_jaguar.mapToGlobal(qpoint_screen)
        self.mymenu_jaguar.exec_(qpoint_widget)

    def raiseRemoveMenu_gamess_us(self, qpoint_screen):
        self.removeAction_gamess_us.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_gamess_us.mapToGlobal(qpoint_screen)
        self.mymenu_gamess_us.exec_(qpoint_widget)

    def raiseRemoveMenu_molpro(self, qpoint_screen):
        self.removeAction_molpro.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_molpro.mapToGlobal(qpoint_screen)
        self.mymenu_molpro.exec_(qpoint_widget)
        
    def raiseRemoveMenu_turbomole(self, qpoint_screen):
        self.removeAction_turbomole.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_turbomole.mapToGlobal(qpoint_screen)
        self.mymenu_turbomole.exec_(qpoint_widget)
        
    def raiseRemoveMenu_adf2012(self, qpoint_screen):
        self.removeAction_adf2012.setData(QtCore.QVariant(qpoint_screen))
        qpoint_widget = self.keywordsListWidget_adf2012.mapToGlobal(qpoint_screen)
        self.mymenu_adf2012.exec_(qpoint_widget)

    def removeKeyword_gaussian(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_gaussian.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_gaussian.row(item)
            self.keywordsListWidget_gaussian.takeItem(num_row)

    def removeKeyword_orca(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_orca.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_orca.row(item)
            self.keywordsListWidget_orca.takeItem(num_row)

    def removeKeyword_jaguar(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_jaguar.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_jaguar.row(item)
            self.keywordsListWidget_jaguar.takeItem(num_row)

    def removeKeyword_gamess_us(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_gamess_us.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_gamess_us.row(item)
            self.keywordsListWidget_gamess_us.takeItem(num_row)

    def removeKeyword_adf2012(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_adf2012.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_adf2012.row(item)
            self.keywordsListWidget_adf2012.takeItem(num_row)


    def removeKeyword_turbomole(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_turbomole.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_turbomole.row(item)
            self.keywordsListWidget_turbomole.takeItem(num_row)


    def removeKeyword_nwchem(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_nwchem.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_nwchem.row(item)
            self.keywordsListWidget_nwchem.takeItem(num_row)


    def removeKeyword_molpro(self):
        action = self.sender()
        if action.data():
            item = self.keywordsListWidget_molpro.itemAt(action.data().toPoint())
            print "Removed '", item.text(), "' from keyword list"
            num_row = self.keywordsListWidget_molpro.row(item)
            self.keywordsListWidget_molpro.takeItem(num_row)


    def reset(self):
        self.fileageSpinBox.setValue(self.olddict['num_max_file_age_minutes'])
        self.useguiCheckBox.setChecked(self.olddict['use_gui'])
        self.convergenceBox.setChecked(self.olddict['no_zero_convergence'])
        self.guessModeBox.setChecked(self.olddict['no_guess_mode'])
        self.avopathEdit.setText(self.olddict['str_avogadro_path'])
        self.keywordsListWidget.clear()
        self.keywordsListWidget.addItems(self.dict_settings['list_keywords_gaussian'])
        self.keywordsListWidget_orca.clear()
        self.keywordsListWidget_orca.addItems(self.dict_settings['list_keywords_orca'])
        self.keywordsListWidget_jaguar.clear()
        self.keywordsListWidget_jaguar.addItems(self.dict_settings['list_keywords_jaguar'])
        self.keywordsListWidget_gamess_us.clear()
        self.keywordsListWidget_gamess_us.addItems(self.dict_settings['list_keywords_gamess_us'])
        self.keywordsListWidget_molpro.clear()
        self.keywordsListWidget_molpro.addItems(self.dict_settings['list_keywords_molpro'])
        self.keywordsListWidget_adf2012.clear()
        self.keywordsListWidget_adf2012.addItems(self.dict_settings['list_keywords_adf2012'])        
        self.keywordsListWidget_turbomole.clear()
        self.keywordsListWidget_turbomole.addItems(self.dict_settings['list_keywords_turbomole'])        
        self.keywordsListWidget_nwchem.clear()
        self.keywordsListWidget_nwchem.addItems(self.dict_settings['list_keywords_nwchem'])        


    def accept(self):
        
        keywordslist = []
        for x in range(self.keywordsListWidget_gaussian.count()):
            keystring = str(self.keywordsListWidget_gaussian.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_gaussian'] = keywordslist

        keywordslist = []
        for x in range(self.keywordsListWidget_jaguar.count()):
            keystring = str(self.keywordsListWidget_jaguar.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_jaguar'] = keywordslist

        keywordslist = []
        for x in range(self.keywordsListWidget_orca.count()):
            keystring = str(self.keywordsListWidget_orca.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_orca'] = keywordslist
        
        keywordslist = []
        for x in range(self.keywordsListWidget_gamess_us.count()):
            keystring = str(self.keywordsListWidget_gamess_us.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_gamess_us'] = keywordslist

        keywordslist = []
        for x in range(self.keywordsListWidget_adf2012.count()):
            keystring = str(self.keywordsListWidget_adf2012.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_adf2012'] = keywordslist
 
        keywordslist = []
        for x in range(self.keywordsListWidget_turbomole.count()):
            keystring = str(self.keywordsListWidget_turbomole.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_turbomole'] = keywordslist
         
        keywordslist = []
        for x in range(self.keywordsListWidget_nwchem.count()):
            keystring = str(self.keywordsListWidget_nwchem.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_nwchem'] = keywordslist
        
        keywordslist = []
        for x in range(self.keywordsListWidget_molpro.count()):
            keystring = str(self.keywordsListWidget_molpro.item(x).text())
            keywordslist.append(keystring)
        self.dict_settings['list_keywords_molpro'] = keywordslist
        

        self.dict_settings['str_avogadro_path'] = str(self.avopathEdit.text())
        self.dict_settings['use_gui'] = self.useguiCheckBox.isChecked()
        self.dict_settings['no_zero_convergence'] = self.convergenceBox.isChecked()
        self.dict_settings['no_guess_mode'] = self.guessModeBox.isChecked()
        self.dict_settings['num_max_file_age_minutes'] = self.fileageSpinBox.value()
#        self.dict_settings['num_perc_plot_ysurplus'] = float(self.plotsurplusSpinBox.value()) / 100
#        self.dict_settings['num_plot_mode'] = self.plotComboBox.currentIndex()
        self.dict_settings['num_verbosity'] = self.verbosityComboBox.currentIndex()
        self.dict_settings['num_multifile_opt'] = self.multiComboBox.currentIndex()
        
        result = self.ccw_settingshandler.writeSettings(self.dict_settings, self.settingsfile)
        QtGui.QMessageBox.information(self, "Settings changed", "<br >"+result)

        QtGui.QDialog.accept(self)



class ParserWidget(QtGui.QTabWidget):
    def __init__(self, parser, parent = None):
        super(ParserWidget, self).__init__(parent)
        
        self.savePlotImageAction = QtGui.QAction('&Save Image...', self)
        self.saveParsedTextAction = QtGui.QAction('&Save Text...', self)
        self.exportPlotDataAction = QtGui.QAction('&Export Data...', self)

        self.instance_parser = parser
        self.textField = QtGui.QTextEdit()
        self.textField.setReadOnly(True)
        self.textField.setToolTip('<p>This is the text output of the parser.</p> <b>Right-click</b> to save to a file.')
        self.textField.setContextMenuPolicy(QtCore.Qt.ActionsContextMenu)
        self.textField.addAction(self.saveParsedTextAction)
        self.textField.zoomOut()

        if self.instance_parser.dict_settings['str_plot_system'] == 'gnuplot':
            self.plotImageLabel = QtGui.QLabel(self)
            self.plotImageLabel.setPixmap(QtGui.QPixmap(os.path.join(os.path.dirname(__file__),'images', 'no_plot.png')))
    #        self.plotImageLabel.setPixmap(QtGui.QPixmap(':/trolltech/styles/commonstyle/images/standardbutton-cancel-128.png'))    fallback?
            self.plotImageLabel.setAlignment(QtCore.Qt.AlignVCenter|QtCore.Qt.AlignHCenter)
            self.plotImageLabel.setToolTip('<p>This is a plot of the SCF energies vs. steps needed.</p> <b>Right-click</b> to save to a file.')
            self.plotImageLabel.setContextMenuPolicy(QtCore.Qt.ActionsContextMenu)
            self.plotImageLabel.addAction(self.savePlotImageAction)
            self.plotImageLabel.addAction(self.exportPlotDataAction)
        else:
            self.plotImageLabel = self.instance_parser.instance_plotter #rename plotImageLabel sometime?

        
        self.quitButton = QtGui.QPushButton("&Quit")
        self.quitButton.setIcon(QtGui.QIcon(":/trolltech/styles/commonstyle/images/standardbutton-close-16.png"))
        self.showPointsButton = QtGui.QPushButton("Show best points")
        self.startAvogadroButton = QtGui.QPushButton("Start &Avogadro")
        self.startAvogadroButton.setIcon(QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', 'avogadro.png')))
        self.loadLogfileButton = QtGui.QPushButton("&Load logfile...")
        self.loadLogfileButton.setIcon(QtGui.QIcon(":/trolltech/styles/commonstyle/images/filecontents-16.png"))
        self.openDirButton = QtGui.QPushButton("&Open Directory")
        self.openDirButton.setIcon(QtGui.QIcon(":/trolltech/styles/commonstyle/images/diropen-16.png"))
        
        self.comboBoxLabel = QtGui.QLabel("Plot mode:")
        self.comboBox = QtGui.QComboBox()
        self.comboBox.insertItems(1, list_plot_mode)
        self.comboBox.setCurrentIndex(self.instance_parser.dict_settings['num_plot_mode'])
        self.comboBoxLabel.setBuddy(self.comboBox)
        
        self.showLogLabel = QtGui.QLabel("Display Logfile")
        self.showLogBox = QtGui.QCheckBox()
        self.showLogLabel.setBuddy(self.showLogBox)
        showLogHbox = QtGui.QHBoxLayout()
        showLogHbox.addWidget(self.showLogBox)
        showLogHbox.addWidget(self.showLogLabel)
        
        self.refreshButton = QtGui.QPushButton()
        self.refreshButton.setIcon(self.style().standardIcon(QtGui.QStyle.SP_BrowserReload))
        #action.setData(QtCore.QVariant(self.instance_parser.tuple_files))
        showLogHbox.addStretch(1)
        showLogHbox.addWidget(self.refreshButton)
        
        self.comboBoxQwt = QtGui.QComboBox()
        self.comboBoxQwt.insertItems(1, list_plot_mode_qwt)
        if self.instance_parser.dict_settings['num_plot_mode'] != 0:
            qwtindex = self.instance_parser.dict_settings['num_plot_mode']-1
        else:
            qwtindex = 0
        self.comboBoxQwt.setCurrentIndex(qwtindex)


        comboHbox = QtGui.QHBoxLayout()
        comboHbox.addWidget(self.comboBoxLabel)
        if self.instance_parser.dict_settings['str_plot_system'] == 'gnuplot':
            comboHbox.addWidget(self.comboBox)
        else:
            comboHbox.addWidget(self.comboBoxQwt)
            
        lowerVbox = QtGui.QVBoxLayout()
        lowerVbox.addWidget(self.loadLogfileButton)
        lowerVbox.addWidget(self.openDirButton)
        lowerVbox.addLayout(comboHbox)
        lowerVbox.addLayout(showLogHbox)
        lowerVbox.addWidget(self.showPointsButton)
        lowerVbox.addWidget(self.startAvogadroButton)
        lowerVbox.addWidget(self.quitButton)

        upperHbox = QtGui.QHBoxLayout()
        upperHbox.addWidget(self.textField)
        lowerHbox = QtGui.QHBoxLayout()
        lowerHbox.addWidget(self.plotImageLabel)
        lowerHbox.addLayout(lowerVbox)
        
        lowerHbox.setStretch(0, 100) #mind that a stretch factor is a priority and not a size number
        lowerHbox.insertStretch(1, 10)
        lowerHbox.addStretch(20)
        lowerHbox.insertSpacing(1, 20)
        lowerHbox.addSpacing(20)

        vbox = QtGui.QVBoxLayout()
        vbox.addLayout(upperHbox)
        vbox.addLayout(lowerHbox)

        self.mainpage = QtGui.QWidget()
        self.mainpage.setLayout(vbox) #remove the command 4 lines above when activating
        self.addTab(self.mainpage, QtGui.QIcon(":/trolltech/styles/commonstyle/images/filecontents-16.png"), "Monitor")


class logViewer(QtGui.QWidget):
    def __init__(self, parent = None):
        super(logViewer, self).__init__(parent)
        
        self.lv_textedit = QtGui.QTextEdit()
        self.lv_textedit.setReadOnly(True)
        
        lv_findlabel = QtGui.QLabel('&Find')
        self.lv_findfield = QtGui.QLineEdit()
        lv_findlabel.setBuddy(self.lv_findfield)
        lv_upbutton = QtGui.QPushButton()
        lv_upbutton.setIcon(self.style().standardIcon(QtGui.QStyle.SP_ArrowUp))
        lv_downbutton = QtGui.QPushButton()
        lv_downbutton.setIcon(self.style().standardIcon(QtGui.QStyle.SP_ArrowDown))
        
        self.connect(lv_upbutton, QtCore.SIGNAL("clicked()"), self.searchUp)
        self.connect(lv_downbutton, QtCore.SIGNAL("clicked()"), self.searchDown)
        
        lv_findlayout = QtGui.QHBoxLayout()
        lv_findlayout.addWidget(lv_findlabel)
        lv_findlayout.addWidget(self.lv_findfield)
        lv_findlayout.addWidget(lv_upbutton)
        lv_findlayout.addWidget(lv_downbutton)
        
        lv_layout = QtGui.QVBoxLayout()
        lv_layout.addWidget(self.lv_textedit)
        lv_layout.addLayout(lv_findlayout)
        self.setLayout(lv_layout)
    
    def searchUp(self):
        if self.lv_findfield.text():
            result= self.lv_textedit.find(self.lv_findfield.text(), QtGui.QTextDocument.FindCaseSensitively|QtGui.QTextDocument.FindBackward)
            if result:
                self.lv_findfield.setStyleSheet("QLineEdit{background: white;}") #white background
            else:
                self.lv_findfield.setStyleSheet("QLineEdit{background: rgb(255, 102, 102);}") #red
        
    def searchDown(self):
        if self.lv_findfield.text():
            result= self.lv_textedit.find(self.lv_findfield.text(), QtGui.QTextDocument.FindCaseSensitively)
            if result:
                self.lv_findfield.setStyleSheet("QLineEdit{background: white;}") #white background
            else:
                self.lv_findfield.setStyleSheet("QLineEdit{background: rgb(255, 102, 102);}") #red
    


class CCW_mainwin(QtGui.QMainWindow):

    def __init__(self, parser, parent = None):
        super(CCW_mainwin, self).__init__(parent)
        
        self.list_logtabs = []
        self.setAttribute(QtCore.Qt.WA_DeleteOnClose, True)
        
        self.instance_parser = parser
        self.parserwid = ParserWidget(parser)
        self.setCentralWidget(self.parserwid)
        
        settings = QtCore.QSettings()
        self.restoreGeometry(settings.value("Geometry").toByteArray())
        self.recentfiles = settings.value("RecentFiles").toStringList() #yields list of QStrings; reverse because upcoming 'updateRecentFiles' pre-pends files and this would revert the youngest-to-oldest order otherwise
        while len(self.recentfiles):
            self.instance_parser.updateRecentFiles(str(self.recentfiles.takeLast())) #reversed order because updateRecent pre-pends -> right order again

        self.statuslabel = QtGui.QLabel("")
        self.statusBar().addWidget(self.statuslabel)
        self.statuslabel.setText('Ready')
        self.ctimer = QtCore.QTimer()
        self.setWindowTitle('ccwatcher GUI')
        self.setWindowIcon(QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', 'ccwatcher.svg'))) #at least until I set 128 for aboutbox and 32 for here again
        self.trayicon_on = QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', 'ccwatcher_trayicon_active.png')) # KDE 4.5.2 doesn't handle svg trayicons right it seems...
        self.trayicon_off = QtGui.QIcon(os.path.join(os.path.dirname(__file__),'images', 'ccwatcher_trayicon_inactive.png'))
        self.ccwtrayicon = QtGui.QSystemTrayIcon(self.trayicon_on, self)
        self.ccwtrayicon.setVisible(True)
        self.ccwtrayicon.setToolTip("CCWatcher is parsing")
        
        
        self.restoreWindowAction = QtGui.QAction('&Restore Window', self)
        saveParsedTextAction = QtGui.QAction('&Save Parsed Text...', self)
        stopParsingAction = QtGui.QAction('S&top Parser', self)
        zoomOutAction = QtGui.QAction('S&maller text', self)
        zoomInAction = QtGui.QAction('B&igger text', self)
        savePlotImageAction = QtGui.QAction('&Save Image...', self)
        exportPlotDataAction = QtGui.QAction('&Export Data...', self)
        aboutDialogAction = QtGui.QAction('&About', self)
        aboutQtDialogAction = QtGui.QAction('About &Qt', self)
        self.loadMultiAction = QtGui.QAction('Load &files...', self)
        self.openDirAction = QtGui.QAction('Open &Directory...', self)
        self.quitAction = QtGui.QAction('&Quit', self)
        
        self.traymenu = QtGui.QMenu()
        self.traymenu.addAction(self.restoreWindowAction)
        self.traymenu.addAction(self.quitAction)
        self.ccwtrayicon.setContextMenu(self.traymenu)
        
        menubar = self.menuBar()
        self.fileMenu = menubar.addMenu('&File') #remember that this menu is overwritten by updateFileMenuSlot
        self.fileMenu.addAction(self.loadMultiAction)
        self.fileMenu.addAction(self.openDirAction)
        self.fileMenu.addAction(self.quitAction)
        
        parserMenu = menubar.addMenu('&Parser')
        parserMenu.addAction(stopParsingAction)
        parserMenu.addAction(saveParsedTextAction)
        parserMenu.addAction(zoomInAction)
        parserMenu.addAction(zoomOutAction)

        plotMenu = menubar.addMenu('Plo&t')
        plotMenu.addAction(savePlotImageAction)
        plotMenu.addAction(exportPlotDataAction)
        
        settingsAction = QtGui.QAction('&Settings...', self)
        menubar.addAction(settingsAction)
        
        helpMenu = menubar.addMenu('&Help')
        helpMenu.addAction(aboutDialogAction)
        helpMenu.addAction(aboutQtDialogAction)
        

        self.connect(self.parserwid.loadLogfileButton, QtCore.SIGNAL("clicked()"), self.getLoadLogNames)
        self.connect(self.parserwid.openDirButton, QtCore.SIGNAL("clicked()"), self.getDirName)
        self.connect(self.parserwid.showPointsButton, QtCore.SIGNAL("clicked()"), self.showPointsSlot)
        self.connect(self.parserwid.startAvogadroButton, QtCore.SIGNAL("clicked()"), self.avogadroSlot)
        self.connect(self.parserwid.quitButton, QtCore.SIGNAL("clicked()"), self.close)
        self.connect(self.parserwid.comboBox, QtCore.SIGNAL("activated(int)"), self.changePlotModeSlot)
        self.connect(self.parserwid.comboBoxQwt, QtCore.SIGNAL("activated(int)"), self.changePlotModeSlot)
        self.connect(self.parserwid.savePlotImageAction, QtCore.SIGNAL('triggered()'), self.savePlotImageSlot)
        self.connect(self.parserwid.saveParsedTextAction, QtCore.SIGNAL('triggered()'), self.saveParsedTextSlot)
        self.connect(self.parserwid.exportPlotDataAction, QtCore.SIGNAL('triggered()'), self.savePlotDataSlot)
        self.connect(self.parserwid.showLogBox, QtCore.SIGNAL('stateChanged(int)'), self.adjustLogDisplay)
        self.connect(self.parserwid.refreshButton, QtCore.SIGNAL("clicked()"), self.getLoadLogNames)
        
        self.connect(self, QtCore.SIGNAL("stopTimer()"), self.stop_timer)
        self.connect(self, QtCore.SIGNAL("logStatusMessage"), self.msgappend)
        self.connect(self, QtCore.SIGNAL("logNewText"), self.logappend)
        self.connect(self, QtCore.SIGNAL("logImpText"), self.impappend)
        self.connect(self, QtCore.SIGNAL("statusbar_finished()"), self.change_statusbar_finished)
        self.connect(self, QtCore.SIGNAL('lastWindowClosed()'), self.close)
        
        self.connect(self.restoreWindowAction, QtCore.SIGNAL('triggered()'), self.showNormal)
        self.connect(self.quitAction, QtCore.SIGNAL('triggered()'), QtCore.SLOT('close()'))
        self.connect(stopParsingAction, QtCore.SIGNAL('triggered()'), self.stop_timer)
        self.connect(zoomOutAction, QtCore.SIGNAL('triggered()'), self.zoomParserOut)
        self.connect(zoomInAction, QtCore.SIGNAL('triggered()'), self.zoomParserIn)      
        self.connect(self.loadMultiAction, QtCore.SIGNAL('triggered()'), self.getLoadLogNames)
        self.connect(self.openDirAction, QtCore.SIGNAL('triggered()'), self.getDirName)
        self.connect(saveParsedTextAction, QtCore.SIGNAL('triggered()'), self.saveParsedTextSlot)
        self.connect(savePlotImageAction, QtCore.SIGNAL('triggered()'), self.savePlotImageSlot)
        self.connect(exportPlotDataAction, QtCore.SIGNAL('triggered()'), self.savePlotDataSlot)
        self.connect(aboutDialogAction, QtCore.SIGNAL('triggered()'), self.slotAbout)
        self.connect(aboutQtDialogAction, QtCore.SIGNAL('triggered()'), self.slotAboutQt)

        self.connect(self.fileMenu, QtCore.SIGNAL('aboutToShow()'), self.updateFileMenuSlot)
        self.connect(settingsAction, QtCore.SIGNAL('triggered()'), self.settingsdlgSlot)

        
        self.show()
        self.instance_parser.log.debug("Qt GUI successfully loaded.")
        
        if len(self.instance_parser.tuple_files[0]): #or better tuple[index]?
            self.loadLogfile()
            


    def start_timer(self):
        """
        Start the constant parser cycle timer
        """
        
        self.msgappend("***Starting parser***")
        self.connect(self.ctimer, QtCore.SIGNAL("timeout()"), self.startParserSlot)
        self.statuslabel.setText('Reading logfile... (Last logfile change: %s)' % time.ctime(self.instance_parser.get_logfile_time()[0]))
        self.ccwtrayicon.setIcon(self.trayicon_on)
        self.ccwtrayicon.setToolTip("CCWatcher is parsing")
        self.ctimer.start(3000) #start a continuous 3 second timer
        self.startParserSlot() #...but start the first parsing right away

    def stop_timer(self):
        """
        Stops the constant parser cycle timer
        """
        self.ctimer.stop()
        self.msgappend("***Stopping parser***")
        self.emit(QtCore.SIGNAL("statusbar_finished()"))
        self.disconnect(self.ctimer, QtCore.SIGNAL("timeout()"), self.startParserSlot)
        self.ccwtrayicon.setIcon(self.trayicon_off)
        self.ccwtrayicon.setToolTip("CCWatcher is inactive")


    def zoomParserIn(self):
        """
        Zooms in on the text in the parser text window (i.e., bigger font size)
        """
        self.parserwid.textField.zoomIn()        


    def zoomParserOut(self):
        """
        Zooms in on the text in the parser text window (i.e., bigger font size)
        """
        self.parserwid.textField.zoomOut()   


    def logappend(self,string):
        """
        Appends normal text to QTextEdit
        """
        
        self.parserwid.textField.moveCursor(QtGui.QTextCursor.End)
        self.parserwid.textField.setFontItalic(False)
        self.parserwid.textField.insertPlainText(string)
        self.parserwid.textField.ensureCursorVisible()


    def impappend(self,string):
        """
        Appends red text to QTextEdit
        """
        
        self.parserwid.textField.moveCursor(QtGui.QTextCursor.End)
        self.parserwid.textField.setFontItalic(False)
        self.parserwid.textField.insertHtml('<br><font color="red">'+string+'</font><br>')
        self.parserwid.textField.ensureCursorVisible()

    def msgappend(self,string):
        """
        Appends messages to QTextEdit which are centered and italic
        """
        self.parserwid.textField.append("\n")
        self.parserwid.textField.setFontItalic(True)
        self.parserwid.textField.append(string)
        self.parserwid.textField.setAlignment(QtCore.Qt.AlignHCenter)
        self.parserwid.textField.append("\n")
        self.parserwid.textField.setFontItalic(False)
        self.parserwid.textField.setAlignment(QtCore.Qt.AlignLeft)

    def change_statusbar_finished(self):
        self.statuslabel.setText('Finished reading logfile (reason: see above). Last logfile change: %s' % time.ctime(self.instance_parser.get_logfile_time()[0]) )


    def adjustLogDisplay(self,  state):
        """Accepts the checkbox signals and removes the log display tabs - or adds them and starts a reparse."""
        
        if state == 0:
          self.instance_parser.log.debug("The DisplayLog-Checkbox is unchecked now.")
          self.instance_parser.forwardLogText = False
          while len(self.list_logtabs):
            self.parserwid.removeTab(self.list_logtabs.pop())
          
        elif state == 2:
            self.instance_parser.log.debug("The DisplayLog-Checkbox is checked now.")
            self.instance_parser.forwardLogText = True
            if self.instance_parser.tuple_files:
                self.instance_parser.setInputFiles(self.instance_parser.tuple_files)
                self.loadLogfile()
            
        else:
            self.instance_parser.log.error("Warning: The GUI checkbox is in an illdefined state! Check situation and inform developer.")


    def startParserSlot(self):
        output = self.instance_parser.parse()
        lstolst_output = output[0]
        lstolst_text = output[1]
        
        for lst_line in lstolst_output:
            if  lst_line[0] == 0:
                self.emit(QtCore.SIGNAL("logNewText"),lst_line[1])
            elif lst_line[0] == 1:
                self.emit(QtCore.SIGNAL("logImpText"),lst_line[1])
            elif lst_line[0] == 2:
                self.emit(QtCore.SIGNAL("logStatusMessage"),lst_line[1].strip())
                self.instance_parser.log.info(lst_line[1].strip()) #this echoes the string on the CLI, additionally
#            elif lst_line[0] == 3:
#                self.emit(QtCore.SIGNAL(lst_line[1]))
            else:
                self.emit(QtCore.SIGNAL("logStatusMessage"),"FIXME: Illegal output code received! Text is: "+lst_line[1].strip())
                self.instance_parser.log.info(lst_line[1].strip())

        if self.instance_parser.forwardLogText == True:
            viewer = self.parserwid.widget(self.list_logtabs[-1])
            for lst_line in lstolst_text:
                viewer.lv_textedit.append(lst_line)

        #time.sleep(0.1)
        self.statuslabel.setText('Reading logfile... (Last logfile change: %s)' % time.ctime(self.instance_parser.get_logfile_time()[0]))
        if os.path.exists(self.instance_parser.str_plotfilename):
            self.parserwid.plotImageLabel.setPixmap(QtGui.QPixmap(self.instance_parser.str_plotfilename))

        if hasattr(self, 'progress'):
            self.progress.setValue(100)
            self.progress.hide()
            #self.progress.reject()
            
        if len(self.instance_parser.tuple_files) > self.instance_parser.int_current_file_index + 1: # old: and self.instance_parser.file_terminated == True:#old:and self.instance_parser.bool_no_more_lines == True:
            self.loadLogfile()
        elif self.instance_parser.file_terminated == True or self.instance_parser.dict_settings.get('single_run', False) == True:
            self.emit(QtCore.SIGNAL("stopTimer()"))



    def getLoadLogNames(self):
        action = self.sender()
        
        if action == self.parserwid.refreshButton: #if the refreshButton was pressed
            qlist_names = [] #fake
            self.stop_timer()
            self.instance_parser.setInputFiles(self.instance_parser.tuple_files)
            self.loadLogfile()
            
            
        elif isinstance(action, QtGui.QAction) and len(action.data().toString()) > 0: # isinstance check evaluates to True also for simple File-Load and button clicks!
            qlist_names = QtCore.QStringList(action.data().toString())  #type afterwards: QStringList of QStrings

        else:
            if self.instance_parser.filehandle:#tuple_files[self.instance_parser.int_current_file_index]:
                dirname = os.path.dirname(self.instance_parser.tuple_files[self.instance_parser.int_current_file_index])
            else:
                dirname = ""
            qlist_names = QtGui.QFileDialog.getOpenFileNames(self,"Open log file ...",dirname,
                                                                                                        "Log files (*.out *.log* gradient);;Zipped files (*.zip *.gz *.bz *.bz2 *.tar.bz2);;All files (*.*)") #type: QString

        if len(qlist_names) == 1:
            tuple_names = (str(qlist_names[0]),)
            self.stop_timer()
            self.instance_parser.setInputFiles(tuple_names)
            self.loadLogfile()

        elif len(qlist_names) > 1:
            multiSettingsDialog = MultiDialog(qlist_names, self)
            if multiSettingsDialog.exec_():
                self.instance_parser.dict_settings['num_multifile_opt'] = multiSettingsDialog.multiComboBox.currentIndex()
                filenumber = multiSettingsDialog.fileOrderListWidget.currentRow() #implement fetching the whole file order for more than two files
                parsefilename = qlist_names.takeAt(filenumber)
                qlist_names.append(parsefilename) #put the one to actually parse at the end
                self.stop_timer()
                self.instance_parser.setInputFiles(tuple(map(str, qlist_names)))
                self.parserwid.textField.clear() #this is necessary for multi_opt=3 cases to clear screen BEFORE multiparsing (for non-3-multiparsing it happens in loadLogFile).
                self.loadLogfile()
        
        else: # no filename given -> nothing to do
            pass



    def getDirName(self):
        
        if self.instance_parser.filehandle:#tuple_files[self.instance_parser.int_current_file_index]:
            dirname = os.path.dirname(self.instance_parser.tuple_files[self.instance_parser.int_current_file_index])
        else:
            dirname = ""
            
        qstr_dirname = QtGui.QFileDialog.getExistingDirectory(self, "Open Directory", dirname, QtGui.QFileDialog.ShowDirsOnly | QtGui.QFileDialog.DontResolveSymlinks)
        #qstr_dirname is already ABSOLUTE path
        self.stop_timer()
        self.instance_parser.setInputFiles((str(qstr_dirname)), )
        self.parserwid.textField.clear() #this is necessary for multi_opt=3 cases to clear screen BEFORE multiparsing (for non-3-multiparsing it happens in loadLogFile).
        self.loadLogfile()


    def loadLogfile(self): #stop_timer has happened before (needs to be BEFORE setInputFiles); here we call openNextFile and start_timer

        #Step1 : close old tabs
        if self.instance_parser.dict_settings['num_multifile_opt'] > 0 and self.instance_parser.int_current_file_index > -1: #keep all logatbs open during multifile loads, but not between them
            pass
        else:
            while len(self.list_logtabs):
                self.parserwid.removeTab(self.list_logtabs.pop())
                
        #Step2 : clear plot and text so there's no confusion when there's problems loading the new file
        if not self.instance_parser.dict_settings['num_multifile_opt'] == 3: #to remove old output BEFORE a multi-3 load, there is an additional clear() in getLoadLogNames
            self.parserwid.textField.clear() #also called at a non-3 multifile load procedure in loadMultiFile so it's cleared before a new multi-0 load        
        if self.instance_parser.dict_settings['str_plot_system'] == 'gnuplot':
            self.parserwid.plotImageLabel.setPixmap(QtGui.QPixmap(os.path.join(os.path.dirname(__file__),'images', 'no_plot.png'))) #or already earlier?
        else: #case qwt: clearing happens in parser
            pass 
        
        #Step3 : Show a progress bar dialog and make parser open the file       
        self.progress = QtGui.QProgressDialog("Please wait", "Cancel", 0, 100, self)
        self.progress.setWindowModality(QtCore.Qt.WindowModal)
        #self.progress.setAutoReset(True)
        #self.progress.setAutoClose(True)
        self.progress.setMinimum(0)
        self.progress.setMaximum(100)
#        progress.resize(800,220)
        self.progress.setWindowTitle("Loading...")
        self.progress.setValue(0)
        self.progress.show()
        
        self.int_successreturnvalue = self.instance_parser.openNextFile()

        #Step4 : When there's no error from the parser, adjust title, open tabs and start parsing
        if self.int_successreturnvalue: #check for return value of 1 because ccwatcher can reject to open files
            self.instance_parser.log.debug("GUI: File exists and seems valid. (Re-)Starting timer...")
            if self.instance_parser.forwardLogText == True:
               newViewer = logViewer()
               tabIndex = self.parserwid.addTab(newViewer, os.path.basename(self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]))
               self.list_logtabs.append(tabIndex)
            self.setWindowTitle("ccwatcher GUI - %s" % self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]) 
            self.progress.setValue(50)
            self.start_timer()

        elif self.int_successreturnvalue == 0:
            QtGui.QMessageBox.warning(self,"Error", "An error occurred while reading the file. See stdout for further details.")



    def saveParsedTextSlot(self):
        txtFilename = QtGui.QFileDialog.getSaveFileName(self,"Save parser output to ...","","Text file (*.txt)")
        if txtFilename != "":
            if str(txtFilename).find(".") == -1:
                txtFilename = txtFilename+".txt"
            txtFile = open(txtFilename, 'w+')
            txtFile.write(str(self.parserwid.textField.toPlainText()))
            txtFile.close()



    def savePlotDataSlot(self):
        dataFilename = QtGui.QFileDialog.getSaveFileName(self,"Save plot data to ...","","Text file (*.txt)")
        if dataFilename != "":
            if str(dataFilename).find(".") == -1:
                dataFilename = dataFilename+".txt"
            txtFile = open(dataFilename, 'w+')
            data = self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]+":\n"
            data += 'Steps\t\tOverall steps\tE/Hartree\t\tE\kJmol-1\tE/Hartree (normalized)\tE/kJmol-1 (normalized)\n'
            data += 118*'-'+'\n'
            data += '\n'
            for line in self.instance_parser.list_scf:
                for element in line:
                    data += str(element)+'\t\t'
                data += '\n'
            filenumber = 0
            for file in self.instance_parser.preservedSCF:
                data += '\n'
                data += "Reference "+str(filenumber)+": "+self.instance_parser.tuple_files[filenumber]+':\n'
                data += 'Steps\t\tOverall steps\tE/Hartree\t\tE\kJmol-1\tE/Hartree (normalized)\tE/kJmol-1 (normalized)\n'
                data += 118*'-'+'\n'
                data += '\n'
                for scfvalues in file:
                    for value in scfvalues:
                        data += str(value)+'\t\t'
                    data += '\n'
                data += '\n'
                filenumber += 1
            data += '\n\n'
            txtFile.write(data)
            txtFile.close()



    def savePlotImageSlot(self):
        imgFilename = QtGui.QFileDialog.getSaveFileName(self,"Save plot image to ...","","Portable network graphics (*.png)")
        if imgFilename != "":
            if str(imgFilename).find(".") == -1:
                imgFilename = imgFilename+".png"
            shutil.copyfile(self.instance_parser.str_plotfilename,imgFilename)



    def slotAbout(self):
        self.aboutBox = QtGui.QMessageBox.about(self,"About CCWatcher",str_thisisccw_pt1+"\n"+str_thisisccw_pt2)



    def slotAboutQt(self):
        self.qtAboutBox = QtGui.QMessageBox.aboutQt(self)



    def changePlotModeSlot(self,num_plot_mode_new):
        self.instance_parser.log.info("GUI: New plot mode:"+str(num_plot_mode_new))
        if self.instance_parser.dict_settings['str_plot_system'] != 'gnuplot':
            num_plot_mode_new = num_plot_mode_new+1 #this maps the qwt-modes to the correct 'normal' modes: Auto(0)->Customized(1), FPZH-(1)>FPZH(2),...
        self.instance_parser.set_plot_mode(num_plot_mode_new)
        time.sleep(0.1)
        if os.path.exists(self.instance_parser.str_plotfilename):
            if self.instance_parser.dict_settings['str_plot_system'] == 'gnuplot':
                self.parserwid.plotImageLabel.setPixmap(QtGui.QPixmap(self.instance_parser.str_plotfilename))



    def updateFileMenuSlot(self):
        self.fileMenu.clear()
        self.fileMenu.addAction(self.loadMultiAction)
        self.fileMenu.addAction(self.openDirAction)
        self.fileMenu.addSeparator()
        for i, name in enumerate(self.instance_parser.list_recent_filenames):
            #if name != self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]: #so list is always 5 entries long
                action = QtGui.QAction("&%d %s" % (i+1, name), self)
                action.setData(QtCore.QVariant(name))
                self.connect(action,  QtCore.SIGNAL("triggered()"), self.getLoadLogNames)
                self.fileMenu.addAction(action)
        self.fileMenu.addSeparator()
        self.fileMenu.addAction(self.quitAction)


    
    def settingsdlgSlot(self):
        """Manipulation of core settings. Should be applied immediately because otherwise we need to think about the write-back process."""
        settingsDialog = SettingsDialog(self)
        if settingsDialog.exec_():
            self.stop_timer()
            self.instance_parser.dict_settings['num_max_file_age_minutes'] = settingsDialog.dict_settings['num_max_file_age_minutes']
            self.instance_parser.dict_settings['no_zero_convergence'] = settingsDialog.dict_settings['no_zero_convergence']
            self.instance_parser.dict_settings['no_guess_mode'] = settingsDialog.dict_settings['no_guess_mode']
            self.instance_parser.dict_settings['num_verbosity'] = settingsDialog.dict_settings['num_verbosity']
            self.instance_parser.dict_settings['str_avogadro_path'] = settingsDialog.dict_settings['str_avogadro_path']
            self.instance_parser.dict_settings['list_keywords_gaussian'] = settingsDialog.dict_settings['list_keywords_gaussian']
            self.instance_parser.dict_settings['list_keywords_nwchem'] = settingsDialog.dict_settings['list_keywords_nwchem']
            self.instance_parser.dict_settings['list_keywords_orca'] = settingsDialog.dict_settings['list_keywords_orca']
            self.instance_parser.dict_settings['list_keywords_jaguar'] = settingsDialog.dict_settings['list_keywords_jaguar']
            self.instance_parser.dict_settings['list_keywords_gamess_us'] = settingsDialog.dict_settings['list_keywords_gamess_us']
            self.instance_parser.dict_settings['list_keywords_molpro'] = settingsDialog.dict_settings['list_keywords_molpro']  
            self.instance_parser.dict_settings['list_keywords_turbomole'] = settingsDialog.dict_settings['list_keywords_turbomole']    
            self.instance_parser.dict_settings['list_keywords_adf2012'] = settingsDialog.dict_settings['list_keywords_adf2012']    

            #think about which settings to update here (=> take effect immediately). ATM, use_gui does not make sense. Also think about which e.g. widgets need to be updated for continuity (here num_plot_mode combobox)

            if self.instance_parser.dict_settings['str_plot_system'] == 'gnuplot':
                self.parserwid.plotImageLabel.setPixmap(QtGui.QPixmap(os.path.join(os.path.dirname(__file__),'images', 'no_plot.png'))) #just to be safe?
            if self.instance_parser.tuple_files[0]:
                self.instance_parser.setInputFiles(self.instance_parser.tuple_files)
                self.loadLogfile()


    def showPointsSlot(self):
        if self.instance_parser.filetype == 'ADF' and self.instance_parser.num_adf_subversion >= 2012 or self.instance_parser.filetype in ('Gaussian', 'ORCA'):
            str_points = ""
            for i in self.instance_parser.lst_bestpoints:
                if i[-1] < 0:
                    str_points += str(i[0])+"\n"
            #QtGui.QMessageBox.information(self, "Lowest scan energies", str_points)
            bestPointsDialog = BestPointsDialog(self)
            bestPointsDialog.textwid.append(str_points)
            bestPointsDialog.exec_()
        else:
            QtGui.QMessageBox.warning(self, "Unsupported", "This filetype ("+self.instance_parser.filetype+") is currently not supported")


    def avogadroSlot(self):
        try:
            if os.path.exists(os.path.join(self.instance_parser.dict_settings['str_avogadro_path'], 'avogadro')):    
                avo = subprocess.Popen([os.path.join(self.instance_parser.dict_settings['str_avogadro_path'], 'avogadro'),self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]])
            elif os.path.exists(self.instance_parser.dict_settings['str_avogadro_path']):
                avo = subprocess.Popen([self.instance_parser.dict_settings['str_avogadro_path'], self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]])
            else:
                avo = subprocess.Popen(["avogadro", self.instance_parser.tuple_files[self.instance_parser.int_current_file_index]])
        except OSError:
            self.instance_parser.log.error("GUI: The avogadro path from the ini-File, if any, is incorrect. It's also not in your PATH!")


    def closeEvent(self, event):
        settings = QtCore.QSettings()
        settings.setValue("RecentFiles", QtCore.QVariant(self.instance_parser.list_recent_filenames))
        settings.setValue("Geometry", QtCore.QVariant(self.saveGeometry()))
        #cleanup is handled by __init__ (in a finally clause)
