#!/usr/bin/python
# -*- coding: utf-8 -*-
"""
   = = = = = = = = = = = = = = = = = = = = = =
   =   opsi linux bootimage - master script  =
   = = = = = = = = = = = = = = = = = = = = = =
   
   This module is part of the desktop management solution opsi
   (open pc server integration) http://www.opsi.org
   
   Copyright (C) 2006 - 2015 uib GmbH
   
   http://www.uib.de/
   
   All rights reserved.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   
   @copyright:	uib GmbH <info@uib.de>
   @author: Jan Schneider <j.schneider@uib.de>
   @author: Erol Ueluekmen <e.ueluekmen@uib.de>
   @author: Mathias Radtke <m.radtke@uib.de>
   @license: GNU General Public License version 2
"""

__version__ = '4.0.6.2'


# Constants
LOG_FILE     = u'/tmp/log'
DEPOT_MOUNT  = u'/mnt/opsi'
OPSI_TMP_DIR = u'/var/tmp/opsi'
PATCHA_IN    = OPSI_TMP_DIR + u'/patcha.in'
SYSCONF_INI  = OPSI_TMP_DIR + u'/sysconf.ini'

# Imports
import os, time, sys, re, gettext, getopt, codecs, tftpy
from signal import *
import requests



# OPSI imports
from OPSI.Logger import *
# Get Logger instance
logger = Logger()
logger.setConsoleLevel(LOG_NONE)
logger.setLogFile(LOG_FILE)
logger.setFileLevel(LOG_DEBUG)
logger.setLogFormat(u'[%l] [%D] %M (%F|%N)')

from OPSI.System import *
from OPSI.UI import *
from OPSI.Object import *
from OPSI.Types import *
from OPSI.Backend.Backend import *
from OPSI.Backend.JSONRPC import JSONRPCBackend
from OPSI.Util import *
from OPSI.Util.WindowsDrivers import *
from OPSI.Util.Message import *
from OPSI.Util.Repository import *
from OPSI.Util.File import *
import os
import urllib

try:
	t = gettext.translation('opsi-linux-bootimage', '/usr/share/locale')
	_ = t.ugettext
except Exception, e:
	logger.error(u"Locale not found: %s" % e)
	def _(string):
		return string

global rebootWanted
global shutdownWanted
global adminBackend
global nextserver
global termRequested

DHCLIENT_LEASES_FILE = '/var/lib/dhcp/dhclient.leases'

ui = None
fqdn = u''
clientId = u''
hostId = u''
productId = u''
backend = None
productOnClient = None
adminBackend = None
nextserver = u''
hardwareAddress = u''
pckey = u''
actionScript = u''
SCRIPT_PATH = u''
product = {}
bootimageVersion = u''
termRequested = False
rebootWanted = False
shutdownWanted = False
depot = None

class ScriptMessageObserver(MessageObserver):
	def messageChanged(self, subject, message):
		ui.getMessageBox().addText(u"%s\n" % message)



class BootimageUIHook(SystemHook, RepositoryHook):
	def __init__(self, ui):
		self._ui = ui
		self._copyProgressBox = None
		self._copyDualProgressBox = None
		self._shredProgressBox = None
		self._saveImageProgressBox = None
		self._restoreImageProgressBox = None
		
	def pre_getDirectorySize(self, path):
		return path
	
	def post_getDirectorySize(self, path, result):
		return result
	
	def error_getDirectorySize(self, path, result, exception):
		pass
	
	
	def pre_getSize(self, path):
		return path
	
	def post_getSize(self, path, result):
		return result
	
	def error_getSize(self, path, result, exception):
		pass
	
	
	def pre_countFiles(self, path):
		return path
	
	def post_countFiles(self, path, result):
		return result
	
	def error_countFiles(self, path, result, exception):
		pass
	
	
	def pre_getCountAndSize(self, path):
		return path
	
	def post_getCountAndSize(self, path, result):
		return result
	
	def error_getCountAndSize(self, path, result, exception):
		pass
	
	
	def pre_copy(self, src, dst, progressSubject):
		title = _(u'Copy from %s to %s') % (src, dst)
		if not progressSubject:
			progressSubject = ProgressSubject(id = 'copy_progress', title = title)
		self._copyProgressBox = self._ui.createCopyProgressBox(width = 90, height = 30, total = 100, title = title, text = u'')
		progressSubject.attachObserver(self._copyProgressBox)
		return (src, dst, progressSubject)
	
	def post_copy(self, src, dst, progressSubject):
		if self._copyProgressBox and progressSubject:
			progressSubject.detachObserver(self._copyProgressBox)
			self._copyProgressBox.hide()
		return None
	
	def error_copy(self, src, dst, progressSubject, exception):
		if self._copyProgressBox and progressSubject:
			progressSubject.detachObserver(self._copyProgressBox)
			self._copyProgressBox.hide()
	
	
	def pre_Repository_copy(self, source, destination, overallProgressSubject, currentProgressSubject):
		title = _(u'Copy from %s to %s') % (source, destination)
		if not overallProgressSubject:
			overallProgressSubject = ProgressSubject(id = 'copy_overall', title = title)
		if not currentProgressSubject:
			currentProgressSubject = ProgressSubject(id = 'copy_current', title = title)
		
		self._copyDualProgressBox = self._ui.createCopyDualProgressBox(width = 90, height = 30, total = 100, title = title, text = u'')
		self._copyDualProgressBox.setOverallProgressSubject(overallProgressSubject)
		self._copyDualProgressBox.setCurrentProgressSubject(currentProgressSubject)
		
		return (source, destination, overallProgressSubject, currentProgressSubject)
	
	def post_Repository_copy(self, source, destination, overallProgressSubject, currentProgressSubject):
		if self._copyDualProgressBox:
			if overallProgressSubject:
				overallProgressSubject.detachObserver(self._copyDualProgressBox)
			if currentProgressSubject:
				currentProgressSubject.detachObserver(self._copyDualProgressBox)
			self._copyDualProgressBox.hide()
		return None
	
	def error_Repository_copy(self, source, destination, overallProgressSubject, currentProgressSubject, exception):
		if self._copyDualProgressBox:
			if overallProgressSubject:
				overallProgressSubject.detachObserver(self._copyDualProgressBox)
			if currentProgressSubject:
				currentProgressSubject.detachObserver(self._copyDualProgressBox)
			self._copyDualProgressBox.hide()
	
	def pre_reboot(self, wait):
		global rebootWanted
		self._ui.getMessageBox().addText(_(u"Reboot triggered\n"))
		if not rebootWanted and not shutdownWanted:
			wait = 60
		else:
			wait = 0
		rebootWanted = True
		return wait
	
	def post_reboot(self, wait):
		return None
	
	def error_reboot(self, wait, exception):
		pass
	
	
	def pre_halt(self, wait):
		global shutdownWanted
		self._ui.getMessageBox().addText(_(u"Shutdown triggered\n"))
		if not rebootWanted and not shutdownWanted:
			wait = 60
		else:
			wait = 0
		shutdownWanted = True
		return wait
	
	def post_halt(self, wait):
		return None
	
	def error_halt(self, wait, exception):
		pass
	
	
	def pre_Harddisk_deletePartitionTable(self, harddisk):
		self._ui.getMessageBox().addText(_(u"Deleting partition table on %s\n") % harddisk.device)
		return None
	
	def post_Harddisk_deletePartitionTable(self, harddisk):
		return None
	
	def error_Harddisk_deletePartitionTable(self, harddisk, exception):
		pass
	
	
	def pre_Harddisk_writePartitionTable(self, harddisk):
		return None
	
	def post_Harddisk_writePartitionTable(self, harddisk):
		return None
	
	def error_Harddisk_writePartitionTable(self, harddisk, exception):
		pass
	
	
	def pre_Harddisk_readPartitionTable(self, harddisk):
		return None
	
	def post_Harddisk_readPartitionTable(self, harddisk):
		return None
	
	def error_Harddisk_readPartitionTable(self, harddisk, exception):
		pass
	
	
	def pre_Harddisk_setPartitionBootable(self, harddisk, partition, bootable):
		return (partition, bootable)
	
	def post_Harddisk_setPartitionBootable(self, harddisk, partition, bootable):
		return None
	
	def error_Harddisk_setPartitionBootable(self, harddisk, partition, bootable, exception):
		pass
	
	
	def pre_Harddisk_setPartitionId(self, harddisk, partition, id):
		return (partition, id)
	
	def post_Harddisk_setPartitionId(self, harddisk, partition, id):
		return None
	
	def error_Harddisk_setPartitionId(self, harddisk, partition, id, exception):
		pass
	
	
	def pre_Harddisk_readMasterBootRecord(self, harddisk):
		return None
	
	def post_Harddisk_readMasterBootRecord(self, harddisk, result):
		return result
	
	def error_Harddisk_readMasterBootRecord(self, harddisk, exception):
		pass
	
	
	def pre_Harddisk_writeMasterBootRecord(self, harddisk, system):
		return system
	
	def post_Harddisk_writeMasterBootRecord(self, harddisk, system):
		return None
	
	def error_Harddisk_writeMasterBootRecord(self, harddisk, system, exception):
		pass
	
	
	def pre_Harddisk_readPartitionBootRecord(self, harddisk, partition):
		return partition
	
	def post_Harddisk_readPartitionBootRecord(self, harddisk, partition, result):
		return result
	
	def error_Harddisk_readPartitionBootRecord(self, harddisk, partition, exception):
		pass
	
	
	def pre_Harddisk_writePartitionBootRecord(self, harddisk, partition, fsType):
		self._ui.getMessageBox().addText(_(u"Writing partition boot record on %s%s\n") % (harddisk.device, partition))
		return (partition, fsType)
	
	def post_Harddisk_writePartitionBootRecord(self, harddisk, partition, fsType):
		return None
	
	def error_Harddisk_writePartitionBootRecord(self, harddisk, partition, fsType, exception):
		pass
	
	
	def pre_Harddisk_setNTFSPartitionStartSector(self, harddisk, partition, sector):
		return (partition, sector)
	
	def post_Harddisk_setNTFSPartitionStartSector(self, harddisk, partition, sector):
		return None
	
	def error_Harddisk_setNTFSPartitionStartSector(self, harddisk, partition, sector, exception):
		pass
	
	
	def pre_Harddisk_createPartition(self, harddisk, start, end, fs, type, boot, lba):
		self._ui.getMessageBox().addText(_(u"Creating partition (start: %s, end: %s, fs: %s, type: %s, boot: %s, lba: %s) on disk %s\n") \
							% (start, end, fs, type, boot, lba, harddisk.device))
		return (start, end, fs, type, boot, lba)
	
	def post_Harddisk_createPartition(self, harddisk, start, end, fs, type, boot, lba):
		return None
	
	def error_Harddisk_createPartition(self, harddisk, start, end, fs, type, boot, lba, exception):
		pass
	
	
	def pre_Harddisk_deletePartition(self, harddisk, partition):
		self._ui.getMessageBox().addText(_(u"Deleting partition %s on disk %s\n") % (partition, harddisk.device))
		return partition
	
	def post_Harddisk_deletePartition(self, harddisk, partition):
		return None
	
	def error_Harddisk_deletePartition(self, harddisk, partition, e):
		pass
	
	
	def pre_Harddisk_mountPartition(self, harddisk, partition, mountpoint, **options):
		self._ui.getMessageBox().addText(_(u"Mounting partition %s of disk %s\n") % (partition, harddisk.device))
		return (partition, mountpoint, options)
	
	def post_Harddisk_mountPartition(self, harddisk, partition, mountpoint, **options):
		return None
	
	def error_Harddisk_mountPartition(self, harddisk, partition, mountpoint, exception, **options):
		pass
	
	
	def pre_Harddisk_umountPartition(self, harddisk, partition):
		self._ui.getMessageBox().addText(_(u"Umounting partition %s of disk %s\n") % (partition, harddisk.device))
		return partition
	
	def post_Harddisk_umountPartition(self, harddisk, partition):
		return None
	
	def error_Harddisk_umountPartition(self, harddisk, partition, exception):
		pass
	
	
	def pre_Harddisk_createFilesystem(self, harddisk, partition, fs):
		self._ui.getMessageBox().addText(_(u"Creating filesystem %s on partition %s of disk %s\n") % (fs, partition, harddisk.device))
		return (partition, fs)
	
	def post_Harddisk_createFilesystem(self, harddisk, partition, fs):
		return None
	
	def error_Harddisk_createFilesystem(self, harddisk, partition, fs, exception):
		pass
	
	
	def pre_Harddisk_resizeFilesystem(self, harddisk, partition, size, fs):
		size = forceInt(size)
		if (size <= 0):
			self._ui.getMessageBox().addText(_(u"Resizing filesystem on partition %s of disk %s to %0.2f MB\n") % (partition, harddisk.device, (float(size)/(1024*1024))) )
		else:
			self._ui.getMessageBox().addText(_(u"Resizing filesystem on partition %s of disk %s to partition size\n") % (partition, harddisk.device))
		return (partition, size, fs)
	
	def post_Harddisk_resizeFilesystem(self, harddisk, partition, size, fs):
		return None
	
	def error_Harddisk_resizeFilesystem(self, harddisk, partition, size, fs, exception):
		pass
	
	
	def pre_Harddisk_shred(self, harddisk, partition, iterations, progressSubject):
		partition = forceInt(partition)
		title = _(u'Overwriting data on harddisk %s (%d iterations)') % (harddisk.device, iterations)
		if partition:
			title = _(u'Overwriting data on partition %s (%d iterations)') % (harddisk.getPartition(partition)['device'], iterations)
		if not progressSubject:
			progressSubject = ProgressSubject(id = 'shred_progress', title = title)
		self._shredProgressBox = self._ui.createProgressBox(width = 60, height = 20, total = 100, title = title, text = u'')
		progressSubject.attachObserver(self._shredProgressBox)
		return (partition, iterations, progressSubject)
	
	def post_Harddisk_shred(self, harddisk, partition, iterations, progressSubject):
		if self._shredProgressBox and progressSubject:
			progressSubject.detachObserver(self._shredProgressBox)
			self._shredProgressBox.hide()
		return None
	
	def error_Harddisk_shred(self, harddisk, partition, iterations, progressSubject, exception):
		if self._shredProgressBox and progressSubject:
			progressSubject.detachObserver(self._shredProgressBox)
			self._shredProgressBox.hide()
	
	
	def pre_Harddisk_fill(self, harddisk, partition, infile, progressSubject):
		return (partition, infile, progressSubject)
	
	def post_Harddisk_fill(self, harddisk, partition, infile, progressSubject):
		return None
	
	def error_Harddisk_fill(self, harddisk, partition, infile, progressSubject, exception):
		pass
	
	
	def pre_Harddisk_saveImage(self, harddisk, partition, imageFile, progressSubject):
		partition = forceInt(partition)
		title = _(u'Creating image of partition %s') % harddisk.getPartition(partition)['device']
		if not progressSubject:
			progressSubject = ProgressSubject(id = 'saveImage_progress', title = title)
		self._saveImageProgressBox = self._ui.createProgressBox(width = 60, height = 20, total = 100, title = title, text = u'')
		progressSubject.attachObserver(self._saveImageProgressBox)
		return (partition, imageFile, progressSubject)
	
	def post_Harddisk_saveImage(self, harddisk, partition, imageFile, progressSubject):
		if self._saveImageProgressBox and progressSubject:
			progressSubject.detachObserver(self._saveImageProgressBox)
			self._saveImageProgressBox.hide()
		return None
	
	def error_Harddisk_saveImage(self, harddisk, partition, imageFile, progressSubject, exception):
		if self._saveImageProgressBox and progressSubject:
			progressSubject.detachObserver(self._saveImageProgressBox)
			self._saveImageProgressBox.hide()
	
	
	def pre_Harddisk_restoreImage(self, harddisk, partition, imageFile, progressSubject):
		title = _(u'Restoring partition %s from image') % harddisk.getPartition(partition)['device']
		if not progressSubject:
			progressSubject = ProgressSubject(id = 'restoreImage_progress', title = title)
		self._restoreImageProgressBox = self._ui.createProgressBox(width = 60, height = 20, total = 100, title = title, text = u'')
		progressSubject.attachObserver(self._restoreImageProgressBox)
		return (partition, imageFile, progressSubject)
	
	def post_Harddisk_restoreImage(self, harddisk, partition, imageFile, progressSubject):
		if self._restoreImageProgressBox and progressSubject:
			progressSubject.detachObserver(self._restoreImageProgressBox)
			self._restoreImageProgressBox.hide()
		return None
	
	def error_Harddisk_restoreImage(self, harddisk, partition, imageFile, progressSubject, exception):
		if self._restoreImageProgressBox and progressSubject:
			progressSubject.detachObserver(self._restoreImageProgressBox)
			self._restoreImageProgressBox.hide()
	
	
	def pre_auditHardware(self, config, hostId, progressSubject):
		self._ui.getMessageBox().addText(_(u"Running hardware audit\n"))
		return (config, hostId, progressSubject)
	
	def post_auditHardware(self, config, hostId, result):
		return result
	
	def error_auditHardware(self, config, hostId, progressSubject, exception):
		pass
		
	

def getAdminBackend():
	global adminBackend
	global nextserver
	while not adminBackend:
		logger.info(u"Creating admin backend")
		(adminUser, adminPass) = (u'', u'')
		values = ui.getValues( [ { 'name': _('Server:'),	'value': nextserver },
					 { 'name': _('Username:'),	'value': adminUser },
					 { 'name': _('Password:'),	'value': adminPass, 'password': True } ],
					title = _(u'Admin login') )
		if not values:
			raise CanceledException(_(u'Canceled'))
		
		nextserver = values[0]['value']
		adminUser = values[1]['value']
		adminPass = values[2]['value']
		
		if not (nextserver and adminUser and adminPass):
			continue
		
		logger.addConfidentialString(adminPass)
		ui.addConfidentialString(adminPass)
		
		logger.info(u"Creating JSONRPCBackend instance")
		try:
			adminBackend = JSONRPCBackend(	address	    = nextserver,
							application = u'opsi linux bootimage %s' % bootimageVersion,
							username    = adminUser,
							password    = adminPass )
			if not adminBackend.isLegacyOpsi():
				adminBackend.setDeflate(True)
		except Exception, e:
			ui.showError(text = _(u'Login failed: %s') % e)
			continue
	
	return adminBackend

def getOnetimeBackend():
	onetimeBackend = None
	global nextserver
	global hardwareAddress
	while not onetimeBackend:
		logger.info(u"Creating onetime backend")
		onetimePassword = u''
		values = ui.getValues( [ { 'name': _('Server:'),	'value': nextserver },
					 { 'name': _('Username:'),	'value': hardwareAddress },
					 { 'name': _('Password:'),	'value': onetimePassword } ],
					title = _(u'Onetime password login') )
		if not values:
			raise CanceledException(_(u'Canceled'))
		
		nextserver = values[0]['value']
		hardwareAddress = forceHardwareAddress(values[1]['value'])
		onetimePassword = values[2]['value']
		
		if not (nextserver and hardwareAddress and onetimePassword):
			continue
		
		logger.info(u"Creating JSONRPCBackend instance")
		try:
			onetimeBackend = JSONRPCBackend(address	    = nextserver,
							application = u'opsi linux bootimage %s' % bootimageVersion,
							username    = hardwareAddress,
							password    = onetimePassword )
			if not onetimeBackend.isLegacyOpsi():
				onetimeBackend.setDeflate(True)
		except Exception, e:
			ui.showError(text = _(u'Login failed: %s') % e)
			continue
	
	return onetimeBackend
	
def signalHandler(signo, stackFrame):
	logger.info(u"Received signal %s" % signo)
	if (signo == SIGTERM or signo == SIGINT):
		logger.notice(u"Need to terminate")
		global termRequested
		termRequested = True

try:
	logLevelOpt = False
	try:
		(opts, args) = getopt.getopt(sys.argv[1:], "l:p:")
		for (opt, arg) in opts:
			if (opt == "-l"):
				logLevelOpt = True
				logger.setFileLevel(arg)
			if (opt == "-p"):
				productId = arg
		
	except getopt.GetoptError:
		pass
	
	signal(SIGHUP,  signalHandler)
	signal(SIGTERM, signalHandler)
	signal(SIGINT,  signalHandler)
	
	cmdline = getKernelParams()
	
	if not logLevelOpt and not cmdline.get('loglevel') is None:
		logger.setFileLevel(forceInt(cmdline['loglevel']))
	
	logger.info(u"Kernel commandline: %s" % cmdline)
	
	execute(u'echo 0 > /proc/sys/kernel/printk')
	try:
		execute(u'setterm -msg off -blank 0 -powersave off < /dev/tty6 > /dev/tty6')
	except Exception, e:
		logger.warning(e)
		
	ui = UIFactory()
	ui.drawRootText(1, 1,u'opsi linux bootimage')
	uihook = BootimageUIHook(ui)
	addSystemHook(uihook)
	
	if os.path.exists(u'/version'):
		f = codecs.open(u'/version', 'r', 'utf-8')
		bootimageVersion = f.readline().strip()
		f.close()
	
	hostname = cmdline.get('hn', u'')
	domain = cmdline.get('dn', u'')
	nextserver = cmdline.get('service', u'')
	pckey = cmdline.get('pckey', u'')
	if pckey:
		logger.addConfidentialString(pckey)
		ui.addConfidentialString(pckey)
	pcpatchPassword = cmdline.get('pcpatch_passwd', u'')
	if pcpatchPassword:
		logger.addConfidentialString(pcpatchPassword)
		ui.addConfidentialString(pcpatchPassword)
	
	execute(u'dmesg')
	execute(u'lsmod')
	execute(u'cat /etc/environment')
	execute(u'env 2>&1')
	execute(u'locale 2>&1')
	
	# Free disks
	execute(u'swapoff -a')
	
	usedNetworkDevice = u''
	ipaddress = u''
	netmask = u''
	gateway = u''
	nameserver = u''
	ethernetDevices = getEthernetDevices()
	depotId = None
	
	linkUpDevices = []


	for device in ethernetDevices:
		try:
			execute(u"%s %s up" % (which('ifconfig'), device))
		except:
			pass
		try:
			execute(u'mii-tool %s' % device)
		except:
			pass
		try:
			for line in execute(u'ethtool %s' % device):
				match = re.search('link detected:\s*(\S+)\s*$', line, re.IGNORECASE)
				if match and (match.group(1).lower() == 'yes'):
					logger.notice(u"Detected link on device '%s'" % device)
					linkUpDevices.append(device)
		except:
			pass
	
	leasesFiles = {}
	for device in ethernetDevices:
		if os.path.exists('/var/lib/dhcp/dhclient.%s.leases' % device):
			logger.notice(u"Found dhclient.%s.leases" % device)
			leasesFiles.update({'%s'%device:'/var/lib/dhcp/dhclient.%s.leases'%device})
		else:
			logger.notice(u"No file dhclient.%s.leases found" % device)

	opsiEthernetDevices = []
	interface = None
	address = None
	unused = None
	interfaceAddress = {}
	
	noDhclientLeases = {}
	if not leasesFiles:
		for device in ethernetDevices:
			noDhclientLeases.update({'%s'%device:DHCLIENT_LEASES_FILE})
		leasesFiles = noDhclientLeases

	for lease in leasesFiles.itervalues():
		with open(lease) as f:
			for line in f:
				fine = line.strip()
				if line.startswith('interface'):
					unused,interface = line.split(None, 1)
					interface = interface[:-1]
				elif line.startswith('option dhcp-server-identifier'):
					unused,unused,address = line.split(None, 2)
					address = address[:-1]
				if interface and address:
					interfaceAddress[interface.replace('"', '')] = address
					interface = address = None

	for device, address in interfaceAddress.items():
		try:
			response = requests.get('https://%s:4447' % address, verify = False)
			if response.status_code == 200:
				opsiEthernetDevices.append(device)
		except:
			pass

	if opsiEthernetDevices:
		logger.notice(u"Only using network devices with opsi server link detected: %s" % ','.join(opsiEthernetDevices))
		del ethernetDevices[:]
		ethernetDevices = opsiEthernetDevices
		logger.notice(u"copied following elements: %s" % ','.join(ethernetDevices))
	
	if cmdline.has_key("nodhcp"):
		logger.notice(u"DHCP disabled by kernel cmdline")
	else:
		ui.drawRootText(2, 3, _(u'Waiting for network devices to come up...'))
		logger.notice(u"Waiting for network devices to get configured by dhcp")
		done = False
		for j in range(2):
			for i in range(15):
				configuredFound = False
				done = True
				for device in ethernetDevices:
					if not getNetworkDeviceConfig(device).get('ipAddress'):
						logger.info(u"Waiting for device: %s" % device)
						done = False
					else:
						configuredFound = True
				if done:
					logger.notice(u"All network devices configured")
					ui.drawRootText(2, 4, _(u'All network devices up'))
					break
				time.sleep(2)
			if done:
				break

			if configuredFound:
				logger.warning(u"Not all network devices configured, but we have configured interfaces, we will try to work")
				break

			logger.warning(u"Timed out while waiting for network devices to get configured")
			ui.drawRootText(2, 4, _(u'Timed out waiting for network devices'))
			try:
				execute(u'ifdown -a --exclude=lo')
			except:
				pass
			try:
				execute(u'ifup -a --exclude=lo')
			except:
				pass
		for device, lease in leasesFiles.items():
			logger.notice(u"leases: %s:%s"% (device, lease))

		dhcpResult = {}
		for device, lease in leasesFiles.items():
			logger.notice(u"Getting dhcp result for device: %s" % device)
			dhcpResult[device] = getDHCPResult(device, lease)
			logger.info(u"DHCP result for device %s:\n%s" % (device, objectToBeautifiedText(dhcpResult[device])))
		
		for (device, config) in dhcpResult.items():
			if config.get('ip'):
				# Interface has been configured by dhcp
				if not usedNetworkDevice:
					# No interface selected so far
					usedNetworkDevice = device
				elif (config.get('nextserver') and not dhcpResult[usedNetworkDevice].get('nextserver')):
					logger.info(u"Found tftpserver on interface '%s', prefering this interface" % device)
					usedNetworkDevice = device
		if usedNetworkDevice:
			ipaddress  = dhcpResult[usedNetworkDevice].get('ip', u'')
			netmask    = dhcpResult[usedNetworkDevice].get('netmask', u'')
			gateway    = dhcpResult[usedNetworkDevice].get('gateways', u'').split(u',')[0].split(u' ')[0]
			nameserver = dhcpResult[usedNetworkDevice].get('nameservers', u'').split(u',')[0].split(u' ')[0]
			if not hostname:
				hostname = dhcpResult[usedNetworkDevice].get('hostname', u'')
			if not domain:
				domain = dhcpResult[usedNetworkDevice].get('domain', u'')
			if not nextserver:
				nextserver = dhcpResult[usedNetworkDevice].get('nextserver', u'')
	
	if not usedNetworkDevice:
		logger.warning(u"No ethernet device configured by dhcp")
		if ethernetDevices:
			usedNetworkDevice = ethernetDevices[0]
	
	if usedNetworkDevice:
		hardwareAddress = getNetworkDeviceConfig(usedNetworkDevice).get('hardwareAddress', u'')
	
	
	logger.info(u'Current network config:')
	logger.info(u'   device          : %s' % usedNetworkDevice)
	logger.info(u'   hardwareAddress : %s' % hardwareAddress)
	logger.info(u'   ipaddress       : %s' % ipaddress)
	logger.info(u'   netmask         : %s' % netmask)
	logger.info(u'   gateway         : %s' % gateway)
	logger.info(u'   nameserver      : %s' % nameserver)
	logger.info(u'   hostname        : %s' % hostname)
	logger.info(u'   domain          : %s' % domain)
	logger.info(u'   nextserver      : %s' % nextserver)
	
	insertExec = False
	if cmdline.get('pre-execute') or cmdline.get('pre-script'):
		preExecuteScript = cmdline.get('pre-execute', '')
		if not preExecuteScript:
			preExecuteScript = cmdline.get('pre-script', '')
			if preExecuteScript and preExecuteScript.lower().endswith(".py"):
				insertExec = True
		targetScript = "/tmp/pre-execute-script"
		logger.info("PreExecute-Script: '%s' was given." % preExecuteScript)
		# Possible Values starts with: tftp:// or http:// or https://

		if preExecuteScript.startswith("tftp://"):
			#get Script per tftp
			# tftp://172.16.166/linux/test.py
			tftpserver = ''
			tftpPort = 69
			requestedFile = ''
			#tftpy.setLogLevel(logging.DEBUG)
			
			urlparts = preExecuteScript[7:].split("/")
			if len(urlparts) <= 2:
				logger.error(u"preExecuteScript: given url is not correct, please check your settings.")
			else:
				tftpserver = urlparts[0]
				if ":" in tftpserver:
					tftpPort = int(tftpserver.split(":")[1])
					tftpserver = tftpserver.split(":")[0]
				#### requestedFile and outputfile must be str.
				#### tftpyClient.download does not support unicode
				requestedFile = str("/".join(urlparts[1:]))
				
			if tftpserver and tftpPort and requestedFile:
				try:
					logger.info(u"Connecting to tftpserver '%s:%d'" % (tftpserver, tftpPort))
					tftpclient = tftpy.TftpClient(tftpserver, tftpPort)
					logger.info(u"Trying to load '%s' to '%s'" % (requestedFile, targetScript))
					tftpclient.download(requestedFile, targetScript, timeout=30)
					
				except Exception, e:
					logger.error(e)
			
		elif preExecuteScript.startswith("http://") or preExecuteScript.startswith("https://"):
			#get Script per wget
			try:
				logger.info(u"Trying to download preExecuteScript Script: '%s'" % preExecuteScript)
				execute("wget --no-check-certificate -O %s %s" % (targetScript, preExecuteScript))
			except Exception, e:
				logger.error(e)
		else:
			logger.error(u"Not Supported pre-execute attribute: '%s', supported modes are: tftp://... or http://..." % preExecuteScript)
		
		if os.path.exists(targetScript):
			try:
				messageBox = ui.createMessageBox(title = 'Executing preScript')
				messageBox.show()
				os.chmod(targetScript, 755)
				if insertExec:
					execfile(targetScript)
				else:
					handle = execute(u'%s 2>&1' % targetScript, getHandle=True)
					text = posix.read(handle.fileno(), 128)
					while text:
						messageBox.addText(text)
						text = posix.read(handle.fileno(), 128)
			
					messageBox.addText(u'-----------------------------------\n')
					messageBox.addText(_(u'Done!'))
					time.sleep(3)
					if handle.close():
						raise Exception(_(u'Failed!'))
			except Exception, e:
				logger.error(e)
				
	
	if cmdline.get('script'):
		DEPOT_MOUNT = u''
		actionScript = cmdline.get('script')
		product['name'] = u'Local script'
		productId = u'script'
	else:
		ui.drawRootText(2, 5, _(u'hostname: %s, domain: %s') % (hostname, domain) )
		ui.drawRootText(2, 6, _(u'network device: %s %s') % (usedNetworkDevice, hardwareAddress) )
		ui.drawRootText(2, 7, _(u'ip: %s, netmask: %s, gw: %s, ns: %s') % (ipaddress, netmask, gateway, nameserver) )
		ui.drawRootText(2, 8, _(u'opsi config server: %s') % nextserver )
		
		if not pckey:
			logger.notice(u"pckey not found in kernel cmdline, trying to get boot configuration by tftp")
			try:
				logger.info(u"Connecting to tftpserver '%s:69'" % nextserver)
				tclient = tftpy.TftpClient(nextserver, 69, {})
				remotePxelinuxConfigFile = u'linux/pxelinux.cfg/01-%s' % hardwareAddress.replace(':', '-')
				localPxelinuxConfigFile = u'/tmp/pxelinux.cfg'
				logger.info(u"Trying to download pxelinux configuration file '%s'" % remotePxelinuxConfigFile)
				tclient.download(remotePxelinuxConfigFile, localPxelinuxConfigFile)
				logger.info(u"pxelinux configuration saved to '%s'" % localPxelinuxConfigFile)
				
				logger.debug(u"pxelinux configuration file content:")
				lines = []
				f = open(localPxelinuxConfigFile)
				for line in f.readlines():
					line = line.rstrip()
					lines.append(line)
					index = line.lower().find('pckey=')
					if (index != -1):
						line = line[:index+6] + u'*'*32 + line[index+38:]
					logger.debug(u"> %s" % line)
				f.close()
				
				logger.debug(u"Parsing pxelinux configuration file")
				default = u''
				label = u''
				append = u''
				for line in lines:
					match = re.search('^\s*default\s+(\S.*)\s*$', line)
					if match:
						default = match.group(1)
						logger.debug(u"Found default label: '%s'" % default)
						continue
					match = re.search('^\s*label\s+(\S.*)\s*$', line)
					if match:
						label = match.group(1)
						logger.debug(u"Found label: '%s'" % label)
						continue
					if label and (label == default):
						match = re.search('^\s*append\s+(\S.*)\s*$', line)
						if match:
							append = match.group(1)
							logger.debug(u"Found default append: '%s'" % append)
							break
				if not append:
					raise Exception(u"Default append line not found in pxelinux configuration file")
				
				for option in append.split():
					keyValue = option.split("=", 1)
					if ( len(keyValue) == 2 ):
						if   (keyValue[0].strip().lower() == 'hn'):
							hostname = keyValue[1].strip()
							logger.debug(u"Hostname set to '%s'" % hostname)
						elif (keyValue[0].strip().lower() == 'dn'):
							domain = keyValue[1].strip()
							logger.debug(u"Domain set to '%s'" % domain)
						elif (keyValue[0].strip().lower() == 'pckey'):
							pckey = keyValue[1].strip()
							logger.addConfidentialString(pckey)
							ui.addConfidentialString(pckey)
							logger.debug(u"Opsi host key set")
						elif (keyValue[0].strip().lower() == 'service'):
							nextserver = keyValue[1].strip()
							logger.debug(u"Service set to '%s'" % nextserver)
						elif (keyValue[0].strip().lower() == 'product'):
							if not productId:
								productId = keyValue[1].strip()
								logger.debug(u"Product id set to '%s'" % productId)
			except Exception, e:
				logger.error(u"Failed to get boot configuration from tftp server: %s" % e)
		
		if not productId:
			productId = cmdline.get('product', u'')
		
		if (cmdline.get('ask_conf') == '1') or not (ipaddress and netmask and hostname and domain and nextserver):
			done = False
			currentIpaddress  = ipaddress
			currentNetmask    = netmask
			currentGateway    = gateway
			currentNameserver = nameserver
			
			title =  _(u'Configuration incomplete')
			if (ipaddress and netmask and hostname and domain and nextserver):
				title =  _(u'Edit configuration (ask_conf set)')
			
			if not netmask:
				netmask = u'255.255.255.0'
			
			while not done:
				logger.notice(u"Getting values from user")
				values = ui.getValues( [  { 'name': _(u'Clients ip address:'),        'value': ipaddress },
							  { 'name': _(u'Netmask:'),                   'value': netmask },
							  { 'name': _(u'Clients hostname:'),          'value': hostname },
							  { 'name': _(u'Clients domainname:'),        'value': domain },
							  { 'name': _(u'Gateway:'),                   'value': gateway },
							  { 'name': _(u'Nameserver:'),                'value': nameserver },
							  { 'name': _(u'Config-server (opsiconfd):'), 'value': nextserver } ],
							  title = title )
				if not values:
					raise CanceledException(_(u'Canceled'))
				
				ipaddress = values[0]['value']
				netmask = values[1]['value']
				hostname = values[2]['value']
				domain = values[3]['value']
				gateway = values[4]['value']
				nameserver = values[5]['value']
				nextserver = values[6]['value']
				
				done = True
				
				try:
					ipaddress = forceIpAddress(ipaddress)
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid ip address: %s") % ipaddress)
					continue
				try:
					netmask = forceNetmask(netmask)
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid netmask: %s") % netmask)
					continue
				try:
					hostname = forceHostname(hostname)
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid hostname: %s") % hostname)
					continue
				try:
					domain = forceDomain(domain)
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid domain: %s") % domain)
					continue
				try:
					if gateway:
						gateway = forceIpAddress(gateway)
					else:
						gateway = u''
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid gateway address: %s") % gateway)
					continue
				try:
					if nameserver:
						nameserver = forceIpAddress(nameserver)
					else:
						nameserver = u''
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid nameserver address: %s") % nameserver)
					continue
				try:
					nextserver = forceUnicode(nextserver)
				except Exception, e:
					done = False
					ui.showError(text = _(u"Invalid host: %s") % nextserver)
					continue
				
				if not (ipaddress and netmask and hostname and domain and nextserver):
					ui.showError(text = _(u'Configuration incomplete'))
					done = False
				
			if (currentIpaddress != ipaddress or currentNetmask != netmask):
				ifconfig(device = u'eth0', address = ipaddress, netmask = netmask)
			if (currentGateway != gateway):
				try:
					execute(u'route del default')
				except Exception, e:
					logger.warning(e)
				if gateway:
					execute(u'route add default gw %s' % gateway)
			if (currentNameserver != nameserver):
				f = open(u'/etc/resolv.conf', 'w')
				f.write(u'domain %s\nsearch %s\nnameserver %s\n' % (domain, domain, nameserver))
				f.close()
		
		netdev = getDefaultNetworkInterfaceName()
		if not netdev:
			netdev = 'eth0'
		usedNetworkDevice = getNetworkDeviceConfig(netdev)
		
		hostId = clientId = fqdn = forceHostId(u'%s.%s' % (hostname, domain))
		logger.info(u"Using FQDN '%s'" % fqdn)
		
		ui.drawRootText(2, 5, _(u'hostname: %s, domain: %s, client id: %s') % (hostname, domain, clientId) )
		ui.drawRootText(2, 6, _(u'network device: %s %s') % (netdev, hardwareAddress) )
		ui.drawRootText(2, 7, _(u'ip: %s, netmask: %s, gw: %s, ns: %s') % (ipaddress, netmask, gateway, nameserver) )
		ui.drawRootText(2, 8, _(u'opsi config server: %s') % nextserver )
		
		if not pckey:
			if ui.yesno(\
				text        = _(u'How to get host key?'), 
				title       = _(u'Host key not available'),
				okLabel     = _(u'Admin account'),
				cancelLabel = _(u'Onetime password') ):
				
				adminBackend = getAdminBackend()
				if clientId not in adminBackend.getClientIds_list():
					logger.info(u"Client '%s' does not exist, creating..." % clientId)
					adminBackend.createClient(hostname, domain, notes = u'Automatically created by linux bootimage', hardwareAddress = hardwareAddress)
				pckey = adminBackend.getOpsiHostKey(clientId)
				logger.addConfidentialString(pckey)
				ui.addConfidentialString(pckey)
			else:
				onetimeBackend = getOnetimeBackend()
				pckey = onetimeBackend.getOpsiHostKey(clientId)
				logger.addConfidentialString(pckey)
				ui.addConfidentialString(pckey)
				onetimeBackend.backend_exit()
		
		product = {}
		productProperties = {}
		productPropertyStates = []
		productPropertyValues = {}
		softwareLicenseKey = ''
		networkConfig = {}
		generalConfig = {}
		configStates = {}
		
		trynum = 0
		connectionException = None
		while (trynum <= 3):
			trynum +=1
			logger.info(u"Creating JSONRPCBackend instance (try #%d)" % trynum)
			try:
				backend = JSONRPCBackend( 	address     = nextserver,
								application = u'opsi linux bootimage %s' % bootimageVersion,
								username    = clientId,
								password    = pckey )
				if not backend.isLegacyOpsi():
					backend.setDeflate(True)
				connectionException = None
				break
			except Exception, e:
				connectionException = e
				logger.error(u"Connection to service '%s' failed: %s" % (nextserver, e))
				time.sleep(2)
		
		if connectionException:
			if isinstance(connectionException, OpsiAuthenticationError):
				if not ui.yesno(	text        = _(u"Connection to service '%s' failed: %s") % (nextserver, connectionException),
							title       = _(u'Authentication failure'),
							okLabel     = _(u'Create client'),
							cancelLabel = _(u'Cancel') ):
					raise CanceledException(_(u'Canceled'))
				
				adminBackend = getAdminBackend()
				
				if adminBackend.isOpsi4():
					hosts = adminBackend.host_getObjects(type = 'OpsiClient', id = clientId)
					if not hosts:
						adminBackend.host_createOpsiClient(id = clientId, notes = notes, hardwareAddress = hardwareAddress)
						hosts = adminBackend.host_getObjects(type = 'OpsiClient', id = clientId)
						if not hosts:
							raise Exception(u"Failed to create client '%s'" % clientId)
					pckey = hosts[0].opsiHostKey
				else:
					if clientId not in adminBackend.getClientIds_list():
						logger.info(u"Client '%s' does not exist, creating..." % clientId)
						adminBackend.createClient(hostname, domain, notes = u'Automatically created by linux bootimage', hardwareAddress = hardwareAddress)
					pckey = adminBackend.getOpsiHostKey(clientId)
				logger.addConfidentialString(pckey)
				ui.addConfidentialString(pckey)
				backend = JSONRPCBackend( 	address     = nextserver,
								application = u'opsi linux bootimage %s' % bootimageVersion,
								username    = clientId,
								password    = pckey )
				if not backend.isLegacyOpsi():
					backend.setDeflate(True)
			else:
				raise Exception(u"Connection to service '%s' failed: %s" % (nextserver, connectionException))
		
		
		if backend.isOpsi4():
			clientToDepotservers = backend.configState_getClientToDepotserver(clientIds = clientId)
			if not clientToDepotservers:
				raise Exception(u"Failed to get depotserver for client '%s'" % clientId)
			
			depotId = clientToDepotservers[0]['depotId']
		
		products = {}
		if backend:
			# Product ID
			if not productId or (cmdline.get('ask_conf') == '1'):
				productIds = []
				try:
					if backend.isOpsi4():
						productOnDepots = backend.productOnDepot_getIdents(
									productType = 'NetbootProduct',
									depotId     = depotId,
									returnType  = 'dict')
						
						for p in backend.product_getObjects(
								id   = productIds,
								type = 'NetbootProduct'):
							for i in range(len(productOnDepots)):
								if (productOnDepots[i]['productId'] == p.id):
									if (productOnDepots[i]['productVersion'] == p.productVersion) and \
									   (productOnDepots[i]['packageVersion'] == p.packageVersion):
										products[productOnDepots[i]['productId']] = p
										productIds.append(productOnDepots[i]['productId'])
									break
					else:
						for pid in backend.getInstallableNetBootProductIds_list(clientId):
                            ### show all netbootproducts
							#if backend.getProduct_hash(pid).get("pxeConfigTemplate", u""):
							#	# Sort out products with special pxe template
							#	continue
							productIds.append(pid)
					productIds.sort()
				except Exception, e:
					logger.error(e)
				if not productIds:
					raise Exception(u"No netboot product installable for client '%s'" % clientId)
				
				if not productId:
					logger.info(u"Fetching product actions...")
					if backend.isOpsi4():
						for ident in backend.productOnClient_getIdents(
									productType   = 'NetbootProduct',
									clientId      = clientId,
									actionRequest = ['setup', 'uninstall', 'update', 'always', 'once', 'custom'],
									returnType    = 'dict'):
							if not ident['productId'] in productIds:
								continue
							productId = ident['productId']
							break
					else:
						for actionRequest in backend.getProductActionRequests_listOfHashes(clientId):
							if not (actionRequest.get('actionRequest') == u'setup'):
								continue
							if not actionRequest.get('productId') in productIds:
								continue
							productId = actionRequest['productId']
							break
				
				values = []
				for p in productIds:
					values.append( { "name": p, "selected": (p == productId) } )
				
				productId = ui.getSelection(values, radio=True, title = _(u"Please select netboot-product to install") )
				if not productId:
					raise CanceledException(_(u'Canceled'))
				productId = productId[0]
				
			logger.notice(u"Product-id is '%s'" % productId)
			
			# Product
			if backend.isOpsi4():
				if products:
					product = products[productId].toHash()
				else:
					productOnDepots = backend.productOnDepot_getIdents(
							productId   = productId,
							productType = 'NetbootProduct',
							depotId     = depotId,
							returnType  = 'dict')
					if not productOnDepots:
						raise Exception(u"Product '%s' not found on depot '%s'" % (productId, depotId))
							
					product = backend.product_getObjects(
							id             = productOnDepots[0]['productId'],
							productVersion = productOnDepots[0]['productVersion'],
							packageVersion = productOnDepots[0]['packageVersion'])[0].toHash()
				productOnClient = ProductOnClient(
					productId      = product['id'],
					productVersion = product['productVersion'],
					packageVersion = product['packageVersion'],
					productType    = product['type'],
					clientId       = clientId)
			else:
				logger.info(u"Fetching product...")
				try:
					product = backend.getProduct_hash(productId)
					logger.info(u"Product-hash:\n%s" % objectToBeautifiedText(product))
				except Exception, e:
					logger.warning(u"Product '%s' not found: %s" % (productId, e))
			
			ui.drawRootText(5, 10, _(u"[ Product: %s ]") % product.get('name', productId))
			
			# ProductProperties
			logger.notice(u"Fetching product properties...")
			if backend.isOpsi4():
				backend.backend_setOptions( { 'addProductPropertyStateDefaults': True } )
				for pps in backend.productPropertyState_getObjects(
									productId = productId,
									objectId  = clientId):
					logger.info(u"Found product property: %s" % pps.toHash())
					productPropertyStates.append(pps)
					productPropertyValues[pps.propertyId] = pps.getValues()
					productProperties[pps.propertyId] = u','.join(forceUnicodeList(pps.getValues()))
			else:
				try:
					productProperties = backend.getProductProperties_hash(productId, clientId)
					logger.info(u"Product-properties:\n%s" % objectToBeautifiedText(productProperties))
				except Exception, e:
					logger.warning(u"No product properties found for productId '%s': %s" % (productId, e))
			
			# NetworkConfig
			if not backend.isOpsi4():
				logger.notice(u"Fetching network config...")
				try:
					networkConfig = backend.getNetworkConfig_hash(clientId)
					logger.info(u"Network config:\n%s" % objectToBeautifiedText(networkConfig))
				except Exception, e:
					logger.warning(u"Failed to get network config: %s" % e)
				
			# GeneralConfig
			if backend.isOpsi4():
				logger.notice(u"Fetching config states...")
				configStates = {}
				backend.backend_setOptions( { 'addConfigStateDefaults': True } )
				for cs in backend.configState_getObjects(objectId = clientId):
					logger.info(u"Found config state: %s" % cs.toHash())
					configStates[cs.configId] = cs.getValues()
					generalConfig[cs.configId] = u','.join(forceUnicodeList(cs.getValues()))
			else:
				logger.info(u"Fetching general config...")
				try:
					generalConfig = backend.getGeneralConfig_hash(clientId)
					logger.info(u"General config:\n%s" % objectToBeautifiedText(generalConfig))
				except Exception, e:
					logger.warning(u"Failed to get general config: %s" % e)
			
			# LicenseKey
			if product.get('licenseRequired', False):
				logger.notice(u"Requesting product license...")
				if backend.isOpsi4():
					if configStates.get('license-management.use', [ False ])[0]:
						logger.notice(u"Using license management to get licensekey")
						softwareLicenseKey = backend.licenseOnClient_getOrCreateObject(
							clientId  = clientId,
							productId = productId).getLicenseKey()
					else:
						logger.notice(u"Using product property to get licensekey")
						softwareLicenseKey = productPropertyValues.get('productkey', [u''])[0]
				else:
					if hasattr(backend, u'getAndAssignSoftwareLicenseKey'):
						# opsi >= 3.4
						softwareLicenseKey = backend.getAndAssignSoftwareLicenseKey(clientId, "", productId)
					else:
						softwareLicenseKey = backend.getLicenseKey(productId, clientId)
				logger.info("Product license key: %s" % softwareLicenseKey)
				
			# PcpatchPassword
			if not pcpatchPassword:
				logger.notice(u"Fetching pcpatch password...")
				encryptedPassword = None
				if backend.isOpsi4():
					encryptedPassword = backend.user_getCredentials(username = u'pcpatch', hostId = clientId)['password']
				else:
					encryptedPassword = backend.getPcpatchPassword(clientId)
				logger.confidential(u"Encrypted password: %s" % encryptedPassword)
				logger.info(u"Decrypting password...")
				pcpatchPassword = blowfishDecrypt( pckey, encryptedPassword )
				logger.addConfidentialString(pcpatchPassword)
				ui.addConfidentialString(pcpatchPassword)
		
		depotUrl = None
		if backend.isOpsi4():
			depots = backend.host_getObjects(type = 'OpsiDepotserver', id = depotId)
			if not depots:
				raise Exception(u"Failed to get info for depot '%s'" % depotId)
			protocol = configStates.get('clientconfig.depot.protocol', [])
			if protocol and protocol[0] and (protocol[0] == 'webdav'):
				depotUrl = depots[0].depotWebdavUrl
			else:
				depotUrl = depots[0].depotRemoteUrl
			
			# For backward compatibility
			networkConfig = {
				"nextBootServiceURL": configStates.get("clientconfig.service.url", [""])[0],
				"depotUrl":           depotUrl,
				"depotId":            depotId,
				"depotDrive":         configStates.get("clientconfig.depot.drive", [""])[0],
				"winDomain":          configStates.get("clientconfig.windows.domain", [""])[0]
			}
		else:
			depotUrl = networkConfig.get('depotUrl')
		
		mkdir(OPSI_TMP_DIR)
		
		pi = codecs.open(PATCHA_IN, 'w', 'utf-8')
		si = codecs.open(SYSCONF_INI, 'w', 'utf-8')
		print >> si, u"[General]";
		print >> pi, u"ipaddress=%s" % ipaddress
		print >> si, u"ipaddress=%s" % ipaddress
		print >> pi, u"netmask=%s" % netmask
		print >> si, u"netmask=%s" % netmask
		print >> pi, u"defaultgateway=%s" % gateway
		print >> si, u"defaultgateway=%s" % gateway
		print >> pi, u"hostname=%s" % hostname
		print >> si, u"hostname=%s" % hostname
		print >> pi, u"pcname=%s" % hostname
		print >> si, u"pcname=%s" % hostname
		print >> pi, u"bootserver=%s" % nextserver
		print >> si, u"bootserver=%s" % nextserver
		print >> pi, u"depottftp=%s" % nextserver
		print >> si, u"depottftp=%s" % nextserver
		print >> pi, u"dnsdomain=%s" % domain
		print >> si, u"dnsdomain=%s" % domain
		print >> pi, u"productid=%s" % productId
		print >> si, u"productid=%s" % productId
		print >> pi, u"productkey=%s" % softwareLicenseKey
		print >> pi, u"PCKEY=%s" % pckey
		for d in [ product, productProperties, networkConfig, generalConfig ]:
			for (key, value) in d.items():
				if key.lower() in ('changelog', 'productkey'):
					continue
				print >> si, u"%s=%s" % (key.lower(), value)
				print >> pi, u"%s=%s" % (key.lower(), value)
		print >> si, ""
		pi.close()
		si.close()
		
		#ui.drawRootText(5, 11, _(u"[ Mounting depotserver share '%s' to '%s' ]") % (depotUrl, DEPOT_MOUNT))
		ui.drawRootText(5, 11, _(u"[ Opening depotserver repository '%s' ]") % depotUrl)
		
		if depotUrl.lower().startswith(u'smb://') or depotUrl.lower().startswith(u'cifs://'):
			username = configStates.get('clientconfig.depot.user', [depotUrl.split(u'/')[2].split('.')[0] + u'\\pcpatch'])[0]
			depot = getRepository(
					url          = depotUrl,
					username     = username,
					password     = pcpatchPassword,
					mountPoint   = DEPOT_MOUNT,
					mountOptions = {'iocharset': 'utf8'})
			# For backward compatibility
			depotUrl = u'smb://%s/%s' % (depotUrl.split(u'/')[2], depotUrl.split(u'/')[3])
			
		elif depotUrl.startswith(u'webdav'):
			depot = getRepository(
					url      = depotUrl,
					username = clientId,
					password = pckey)
		else:
			raise Exception(u"Unhandled depot url '%s'" % depotUrl)
		
		depot.addHook(uihook)
		
		# SetupScript
		if not product.get('setupScript'):
			raise Exception(_(u"No setup script found for product '%s'!") % productId)
		
		actionScript = u'/tmp/%s' % product.get('setupScript')
		scriptFile =  u"%s/%s" % (productId, product.get('setupScript'))
		SCRIPT_PATH = u'/'.join(scriptFile.split(u'/')[:-1])
		
		logger.notice(u"Copying action script '%s' from depot to '%s'" % (scriptFile, actionScript))
		depot.copy(scriptFile, actionScript)
		
		askBeforeInst = True
		for (key, value) in productProperties.items():
			if key.lower() in ('askbeforeinst', 'ask_before_start'):
				askBeforeInst = forceBool(value)
		
		if askBeforeInst:
			if backend.isOpsi4():
				productOnClient.setActionProgress(u'waiting for start')
				backend.productOnClient_updateObjects(productOnClient)
			if not ui.yesno(text        = _(u"Start installation of '%s'?") % product.get('name', productId),
					okLabel     = _(u"Start"),
					cancelLabel = _(u"Cancel")):
				raise CanceledException(_(u'Canceled'))
	
	
	ui.drawRootText(5, 12, _(u"[ Starting script: %s ]") % actionScript)
	
	scriptMessageObserver = ScriptMessageObserver()
	scriptMessageSubject = MessageSubject(id = u'script', title=product.get('name', productId))
	scriptMessageSubject.attachObserver(scriptMessageObserver)
	
	if not SCRIPT_PATH:
		SCRIPT_PATH = os.path.dirname(actionScript)

	logger.debug(u"The name of our actionScript: {0!r}".format(actionScript))
	if actionScript.lower().endswith('.sh'):
		# Installation status installing introduced in opsi 3.1
		if backend:
			if backend.isOpsi4():
				productOnClient.setActionProgress(u'running')
				productOnClient.setActionResult(u'none')
				productOnClient.setInstallationStatus(u'unknown')
				logger.debug("Before submitting request to backend...")
				backend.productOnClient_updateObjects(productOnClient)
				logger.debug("After submitting request to backend...")

				productOnClient.setActionRequest(u'none')
				productOnClient.actionProgress = None
				#productOnClient.setActionResult(u'successful')
			else:
				if 'installing' in backend.getPossibleProductInstallationStatus_list():
					backend.setProductInstallationStatus(productId, clientId, u'installing')
				backend.unsetProductActionRequest(productId, clientId)
		messageBox = ui.createMessageBox(title = product.get('name', productId))
		messageBox.show()
		handle = execute(u'%s 2>&1' % actionScript, getHandle=True)
		text = posix.read(handle.fileno(), 128)
		while text:
			messageBox.addText(text)
			text = posix.read(handle.fileno(), 128)
			#logger.debug(text)
		messageBox.addText(u'-----------------------------------\n')
		messageBox.addText(_(u'Done!'))
		time.sleep(3)
		if handle.close():
			raise Exception(_(u'Failed!'))
	
	elif actionScript.lower().endswith('.py'):
		# Installation status installing introduced in opsi 3.1
		if backend:
			if backend.isOpsi4():
				productOnClient.emptyValues()
				productOnClient.setActionProgress(u'running')
				productOnClient.setActionResult(u'none')
				productOnClient.setInstallationStatus(u'unknown')
				logger.debug("Before submitting request to backend...")
				backend.productOnClient_updateObjects(productOnClient)
				logger.debug("After submitting request to backend...")

				productOnClient.setActionRequest(u'none')
				productOnClient.actionProgress = None
				#productOnClient.setActionResult(u'successful')
			else:
				if 'installing' in backend.getPossibleProductInstallationStatus_list():
					backend.setProductInstallationStatus(productId, clientId, u'installing')
				backend.unsetProductActionRequest(productId, clientId)
		messageBox = ui.createMessageBox(title = product.get('name', productId))
		messageBox.show()
		execfile(actionScript)
	
	else:
		raise Exception(_(u'Unknown script type!'))
	
	logger.notice(u"Script execution successful")
	if productOnClient:
		logger.info(u"Updating productOnClient: %s" % productOnClient.toHash())
		backend.productOnClient_updateObjects(productOnClient)
	
	# Append geo_override log if exists
	if os.path.isfile(u'/tmp/geo_override.log'):
		execute(u'cat /tmp/geo_override.log')
	if os.path.isfile(SYSCONF_INI):
		execute(u'cat %s' % SYSCONF_INI)
	if os.path.isfile(PATCHA_IN):
		execute(u'cat %s' % PATCHA_IN)
	
except CanceledException, e:
	time.sleep(1)
	logger.logException(e)
	try:
		logger.debug(u"Backend: %s" % backend)
		logger.debug(u"ProductId: %s" % productId)
		logger.debug(u"ClientId: %s" % clientId)
		logger.debug(u"ProductOnClient: %s" % productOnClient)
		if backend and backend.isOpsi4():
			if not productOnClient and productId and clientId:
				productOnClient = ProductOnClient(
						productId   = productId,
						productType = 'NetbootProduct',
						clientId    = clientId)
			if productOnClient:
				productOnClient.setActionResult(u'failed')
				productOnClient.setActionRequest(u'none')
				productOnClient.setActionProgress(u'canceled by user')
				logger.info(u"Updating productOnClient: %s" % productOnClient.toHash())
				backend.productOnClient_updateObjects(productOnClient)
			
		elif backend and productId and clientId:
			backend.unsetProductActionRequest(productId, clientId)
			backend.setProductInstallationStatus(productId, clientId, 'failed')
		
	except Exception, e2:
		logger.logException(e2)
	
except Exception, e:
	time.sleep(1)
	logger.logException(e)
	
	try:
		logger.debug(u"Backend: %s" % backend)
		logger.debug(u"ProductId: %s" % productId)
		logger.debug(u"ClientId: %s" % clientId)
		logger.debug(u"ProductOnClient: %s" % productOnClient)
		if backend and backend.isOpsi4():
			if not productOnClient and productId and clientId:
				productOnClient = ProductOnClient(
						productId   = productId,
						productType = 'NetbootProduct',
						clientId    = clientId)
			if productOnClient:
				productOnClient.setActionResult(u'failed')
				productOnClient.setActionRequest(u'none')
				productOnClient.setActionProgress(u'')
				logger.info(u"Updating productOnClient: %s" % productOnClient.toHash())
				backend.productOnClient_updateObjects(productOnClient)
			
		elif backend and productId and clientId:
			backend.unsetProductActionRequest(productId, clientId)
			backend.setProductInstallationStatus(productId, clientId, 'failed')
		
	except Exception, e2:
		logger.logException(e2)
finally:
	logger.notice(u"Writing log to service")

	try:
		with codecs.open(LOG_FILE, 'r', 'utf-8', 'replace') as log:
			data = log.read()

		logger.setFileLevel(LOG_ERROR)

		if backend:
			try:
				ui.getMessageBox().addText(_(u"Writing log to service\n"))
			except Exception as messageBoxError:
				logger.error(u'Showing error in message box failed: {0}'.format(messageBoxError))

			try:
				if backend.isOpsi4():
					backend.log_write(logType='bootimage', data=data, objectId=clientId, append=False)
				else:
					backend.writeLog(type='bootimage', data=data.replace(u'\ufffd', u'?'), objectId=clientId, append=False)
			except Exception as writingLogError:
				logger.error(u"Failed to write log to service: {0}".format(writingLogError))
	except Exception as fileReadingError:
		logger.logException(fileReadingError)

	try:
		ui.showError(text=forceUnicode(e))
	except NameError:
		# probably no error thrown and therefore no object with name 'e'
		pass

time.sleep(1)
ui.exit()

def resetNetworking():
	# Try to reset network device to make it usable for ms windows
	f = None
	try:
		f = open("/proc/net/dev")
		for line in f.readlines():
			line = line.strip()
			if not line or (line.find(':') == -1):
				continue
			device = line.split(':')[0].strip()
			if device.startswith('eth'):
				module = None
				for line2 in execute("ethtool -i %s" % device):
					line2 = line2.strip()
					if line2 and line2.startswith('driver:'):
						module = line2.split(':', 1)[1].strip()
						break
				if not module:
					continue
				try:
					execute(u'mii-tool -R %s' % device)
				except:
					pass
				try:
					execute(u'ifconfig %s down' % device)
				except:
					pass
				try:
					execute(u'modprobe -r %s' % module)
				except:
					pass
				#execute(u'modprobe %s' % module)
				#execute(u'ifconfig %s up' % device)
	except:
		pass
	if f:
		f.close()

if depot:
	try:
		depot.disconnect()
	except:
		pass

if rebootWanted:
	resetNetworking()
	reboot(0)
	time.sleep(1)
elif shutdownWanted:
	resetNetworking()
	halt(0)
	time.sleep(1)



