/*
* Copyright (C) 2014 Ed Trettevik <eat@nodebrain.org>
*
* NodeBrain is free software; you can modify and/or redistribute it under the
* terms of either the MIT License (Expat) or the following NodeBrain License.
*
* Permission to use and redistribute with or without fee, in source and binary
* forms, with or without modification, is granted free of charge to any person
* obtaining a copy of this software and included documentation, provided that
* the above copyright notice, this permission notice, and the following
* disclaimer are retained with source files and reproduced in documention
* included with source and binary distributions. 
*
* Unless required by applicable law or agreed to in writing, this software is
* distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.
*
*=============================================================================
* Program:  NodeBrain
*
* File:     nbtrack.c 
*
* Title:    Simulation Track Routines
*
* Function:
*
*   This file provides functions that manage simulation track files. 
*
* Synopsis:
*
*   #include "nb.h"
*
*   int   nbTrackPlay(nbCELL context,char *file);
*
* Description
*
*   A simulation track file has the following format.
*
*     yyyy-mm-dd hh:mm:ss nnnnnnnnnn [identity,node] > command
*
*   There are two play modes: 1) clocked, and 2) unclocked.  In unclocked
*   mode the nbTrackPlay function opens the file, reads the first line, schedules
*   the execution of the command with a synapse, and returns.
*
*   The synapse alarm function is nbTrackAlarm, which passes the scheduled command
*   to nbCmd, reads the next command and schedules it.  This continues
*   to end of file. 
*
*   The Clock functions understand unclocked mode, and ignore the system clock.
*   Instead, they step throught the scheduled events and time progresses as rapidly
*   as possible, leaping over periods when the rule engine would normal wait for the
*   clock to align with the next scheduled event.
*
*   In clocked mode the nbTrackPlay function opens the file, reads the first line and
*   schedules it to play right away, and returns.  The nbTrackAlarm function passes 
*   the scheduled command to nbCmd, reads the next command and schedules it. But in
*   clocked mode it schedules the next command based on the interval from the
*   prior command.  In this mode, the Clock function operates on the system 
*   clock, so commands are issued at the recorded rate.
*
*   Unclocked mode is intended for NodeBrain regression testing and validation
*   of rule changes against a well understood event stream.  Clocked mode is
*   intended for simulations in real-time, enabling interaction with other
*   components operating normally in real-time.
*
*=============================================================================
* Change History:
*
*    Date    Name/Change
* ---------- -----------------------------------------------------------------
* 2014-04-16 Ed Trettevik - Introduced in 0.9.04
*=============================================================================
*/
#include <nb/nbi.h> 

struct TYPE *nb_TrackType=NULL;
int nb_tracks=0;

#define NB_TRACK_LINE_SIZE 1024

typedef struct NB_TRACK{
  NB_Object  object;     // object header
  NB_Cell   *context;
  char      *filename;	 // simulation file name
  FILE      *file;
  char      line[NB_TRACK_LINE_SIZE];
  } NB_Track;

#define NB_TRACK_LINE_TIME_SIZE 31 // 19+1+10+1
typedef struct NB_TRACK_LINE{
  char timestamp[19];  // "yyyy-mm-dd hh:mm:ss"
  char pad1;           // " "
  char utc[10];        // nnnnnnnnnn
  char pad2;           // " "
  char cmd[NB_TRACK_LINE_SIZE-NB_TRACK_LINE_TIME_SIZE];
  } NB_TrackLine;

/*
*  Show simulation object
*/
static void nbTrackShow(NB_Track *track){
  outPut("(track:%s)",track->filename);
  }

/*
*  Alarm handler for track object
*/
static void nbTrackAlarm(NB_Track *track){
  int tracktime,offset=0;
  char *cursor=((NB_TrackLine *)&(track->line))->cmd;
  char *delim;
  char *cmd=NULL;
  char ident[1024];
  int len;
  NB_Term *term;
  NB_Identity *identity;

  outMsg(0,'I',"Track %s",track->filename);
  if(nb_Clocked) offset=nb_ClockTime-(atoi(((NB_TrackLine *)&(track->line))->utc)+nb_ClockOffset);
  //if(*cursor!='[') outMsg(0,'E',"Track line command must start with '['-->%s",track->line);
  //else{
    //cursor++;
    delim=strchr(cursor,' ');
    if(!delim) outMsg(0,'E',"Track line identity missing terminating space-->%s",track->line);
    else{
      len=delim-cursor; 
      if(len>=sizeof(ident)) outMsg(0,'E',"Track line identity too long-->%s",track->line);
      else{
        strncpy(ident,cursor,len);
        ident[len]=0;
        if((term=nbTermFind(identityC,ident))==NULL) 
          outMsg(0,'E',"Track line identity not defined-->%s",track->line);
        else{
          identity=(NB_Identity *)term->def;
          cursor=delim+1;
          delim=strchr(cursor,' ');
          if(!delim) outMsg(0,'E',"Track line node missing terminating space-->%s",track->line);
          else{
            len=delim-cursor;
            if(len>=sizeof(ident)) outMsg(0,'E',"Track line node term too long-->%s",track->line);
            else{
              strncpy(ident,cursor,len);
              ident[len]=0;
	      if((term=nbTermFind(rootGloss,ident))==NULL)
                outMsg(0,'E',"Track line node not defined-->%s",track->line); 
              else{
	        cursor=delim+1;
                if(*cursor=='>') cmd=cursor;  // find command
                else outMsg(0,'E',"Track line missing command delimiter '>'-->%s",track->line);
                }
              }
            }
          }
        }
      }
    //}
  if(cmd){
    cmd++;
    while(*cmd==' ') cmd++;
    // get the identity here and call nbCmdSid instead
    //nbCmd(track->context,((NB_TrackLine *)&(track->line))->cmd,NB_CMDOPT_ECHO|NB_CMDOPT_RECORD);
    nbCmdSid((nbCELL)term,cmd,NB_CMDOPT_ECHO|NB_CMDOPT_RECORD,identity);
    }
  if(fgets(track->line,sizeof(track->line),track->file)==NULL){
    outMsg(0,'I',"Track %s ended",track->filename);
    free(track->filename);
    fclose(track->file);
    nbFree(track,sizeof(NB_Track));
    // Decrement the simulation count and shut down when we get to zero
    nb_tracks--;
    if(nb_tracks<=0){
      nb_flag_stop=1;
      nb_opt_prompt=0;
      nbMedullaStop();
      }
    return;
    }
  tracktime=atoi(((NB_TrackLine *)&(track->line))->utc)+nb_ClockOffset;
  if(nb_Clocked) tracktime+=offset;
  nbClockSetTimer(tracktime,(nbCELL)track); // schedule next command
  }

/*
*  Initialization
*/
void nbTrackInit(NB_Stem *stem){
  nb_TrackType=nbObjectType(stem,"track",0,0,nbTrackShow,NULL);
  nb_TrackType->alert=nbTrackAlarm; // not necessary
  nb_TrackType->alarm=nbTrackAlarm;
  }

/*
*  Get the start time of a track
*
*  This function may be called before a context is available, and
*  before output function are initialized.  So we have no context
*  parameter and we output with fprintf(stderr,...).
*/
int nbTrackGetStartTime(const char *filename){
  FILE *file;
  char line[4096];
  NB_TrackLine *trackLine=(NB_TrackLine *)&line;
  int trackTime;
  int i;

  file=fopen(filename,"r");
  if(!file){
    fprintf(stderr,"Track %s open error - %s",filename,strerror(errno));
    return(-1);
    }
  if(fgets(line,sizeof(line),file)==NULL){
    fprintf(stderr,"Track %s is empty",filename);
    fclose(file);
    return(-1);
    }
  fclose(file);
  if(strlen(line)<NB_TRACK_LINE_TIME_SIZE || trackLine->pad1!=' ' || trackLine->pad2!=' '){
    fprintf(stderr,"Track %s has invalid format",filename);
    fprintf(stderr,"Need: yyyy-mm-dd hh:mm:ss nnnnnnnnnn \n");
    fprintf(stderr,"Have: %s\n",line);
    return(-1);
    }
  for(i=0;i<10;i++){
    if(trackLine->utc[i]<0 || trackLine->utc[i]>'9'){
      fprintf(stderr,"Track %s has invalid format",filename);
      fprintf(stderr,"Need: yyyy-mm-dd hh:mm:ss nnnnnnnnnn \n");
      fprintf(stderr,"Have: %s\n",line);
      return(-1);
      }
    } 
  trackTime=atoi(trackLine->utc)+nb_ClockOffset;;
  return(trackTime);
  }

/*
*  Play a simulation file
*
*  This function create a simulation object, schedules the first
*  command, and then returns.  The nbTrackAlarm function executes
*  commands in response to alarms and schedules the next command.
*/
int nbTrackPlay(nbCELL context,char *filename){
  NB_Track *track;
  int tracktime;
  
  track=newObject(nb_TrackType,NULL,sizeof(NB_Track));
  track->context=context;
  track->file=fopen(filename,"r");
  if(!track->file){
    outMsg(0,'T',"Track %s open error - %s",filename,strerror(errno));
    nbFree(track,sizeof(NB_Track));
    return(-1);
    }
  if(fgets(track->line,sizeof(track->line),track->file)==NULL){
    outMsg(0,'T',"Track %s is empty",filename);
    fclose(track->file);
    nbFree(track,sizeof(NB_Track));
    return(0);
    }
  track->filename=strdup(filename);
  if(!nb_Clocked){ // schedule at the specified time
    tracktime=atoi(((NB_TrackLine *)&(track->line))->utc)+nb_ClockOffset;
    nbClockSetTimer(tracktime,(nbCELL)track);
    }
  else nbClockSetTimer(nb_ClockTime,(nbCELL)track);  // schedule right away
  nb_tracks++;
  return(0);
  } 
