/*
* Copyright (C) 2014 Ed Trettevik <eat@nodebrain.org>
*
* NodeBrain is free software; you can modify and/or redistribute it under the
* terms of either the MIT License (Expat) or the following NodeBrain License.
*
* Permission to use and redistribute with or without fee, in source and binary
* forms, with or without modification, is granted free of charge to any person
* obtaining a copy of this software and included documentation, provided that
* the above copyright notice, this permission notice, and the following
* disclaimer are retained with source files and reproduced in documention
* included with source and binary distributions. 
*
* Unless required by applicable law or agreed to in writing, this software is
* distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.
*
*=============================================================================
* Program:  NodeBrain
*
* File:     nbset.c
*
* Title:    Set Binary Tree Routines 
*
* Purpose:
*
*   This file contains functions used to manage sets as binary trees, keeping
*   them reasonably well balanced.  The functions in this file are similar to
*   functions in nbtree.c, but enable members to be removed by node address
*   without first creating a tree path structure.
*
*
* Synopsis:
*
*   void *nbSetFind(void *member,NB_SetNode *node);
*
*   void nbSetInsert(NB_SetNode *root,NB_SetNode *parent,NB_SetNode **nodeP,NB_SetNode *node,void *member);
*
*   void *nbSetRemove(NB_SetNode *root,NB_SetNode *node);
*
*
* Reference:
*
*   Search for "AVL tree" on the web for a general explanation of the algorithm.
*
* Description:
*   
*   This code is an adaptation of a small subset of Ben Pfaff's GNU libavl 2.0.2.  
*
*   See nbtree.c for a general description of the tree algorithms used here. 
*
*   The main difference between the nbSet and nbTree routines is the additional
*   maintenance of a parent node pointer to enable member nodes to be removed 
*   without first building a path structure. In the right pattern of usage, this
*   overhead is more than offset by not building a path structure when it isn't
*   necessary.  In other words, we assume a set will be used in cases where 
*   members are frequently found when a request is made to add them, and/or
*   frequently not found when a request is made to delete them.  In these cases,
*   the tree functions will assume they have to build a path structure using a
*   locate function, but then find they don't need it after all, since no insert
*   or remove is required.
*
*   Set nodes are designed to be separate from the member objects they reference,
*   enabling an object to be in any number of sets.  However, their may be special
*   sets associated with an object, where it is appropriate to include one or more
*   set nodes as elements of the object structure.  In these cases, the ability to
*   remove a member from a set without first searching for it via the set structure
*   is handy.  It is for this reason, that the nbSetInsert and nbSetRemove functions
*   leave it to the caller to allocate and free the set nodes as required.
*
*   Since the nbSetFind() function is so simple, there may be cases where it is more
*   efficient for a calling program to use a search loop provided by, or similar
*   to, the NB_SET_FIND macro.
*
*=============================================================================
* Change History:
*
* Date       Name/Change
* ---------- -----------------------------------------------------------------
* 2013-12-30 Ed Trettevik - first prototype based on nbtree.c
*=============================================================================
*/
#include <nb/nbi.h>

// nbSetFind()    - Binary tree search function
//
//   member   - pointer to member object
//   node     - pointer to root node (may be just a left pointer)
//
// Returns node ptr when found, otherwise NULL.
//
// This function is a called version of the code generated by the NB_SET_FIND
// macro.  Calling code can use the macro directly to generate inline code
// and avoid a call.
// 
/*
void *nbSetFind(void *member,NB_SetNode *node){
  NB_SET_FIND(member,node)
  return(node);
  }
*/

// nbSetInsert()  - AVL tree node insertion function for sets
//
//   root   - pointer to root node (address of left pointer)
//   parent - pointer to parent node
//   nodeP  - pointer to the parent node pointer to point to node
//   node   - pointer to node structure (header to callers node structure)
//
void nbSetInsert(NB_SetNode *root,NB_SetNode *parent,NB_SetNode **nodeP,NB_SetNode *node){
  NB_SetNode *y;      // node to balance
  NB_SetNode *w;      // new root of rebalanced subtree
  NB_SetNode *x;

  //outMsg(0,'T',"nbSetInsert: called");
  //outMsg(0,'T',"nbSetInsert: called with root=%p,parent=%p,pointer=%p,node=%p",root,parent,nodeP,node);

  // Link to new node and initialize 

  if(*nodeP!=NULL) nbExit("call to nbSetInsert where link is not currently NULLL"); // temp
  *nodeP=node;
  node->left=node->right=NULL;
  node->balance=0;
  node->parent=parent;  // set pointer to parent node


  // Adjust balance factors up to first unbalanced node

  //outMsg(0,'T',"nbSetInsert: looping up");
  w=node;
  for(y=node->parent;y!=root && y->balance==0;node=y,y=y->parent){
    if(y==w) nbExit("nbSetInsert encountered loop");
    if(y->left==node) y->balance--;
    else y->balance++;
    }
  if(y==root) return;  // still balanced
  //outMsg(0,'T',"nbSetInsert: y->balance=%d",y->balance);
  if(y->left==node) y->balance--;
  else y->balance++;
  //outMsg(0,'T',"nbSetInsert: new y->balance=%d",y->balance);
  if(y->balance==0) return; // balanced
  //outMsg(0,'T',"nbSetInsert: have to rebalance");

  node=y->parent;  // root of unbalanced portion
  if(node->left==y) nodeP=&node->left;
  else nodeP=&node->right;

  //outMsg(0,'T',"nbSetInsert: rebalancing");
  // Rebalance

  if(y->balance==-2){        // rebalance after left insertion
    x=y->left;
    //outMsg(0,'T',"nbSetInsert: -2 x=%p",x);
    if(x->balance==-1){      // Rotate right at y
      //outMsg(0,'T',"nbSetInsert: rotate right at y");
      w=x;
      if((y->left=x->right)) y->left->parent=y;
      x->right=y;
      y->parent=x;
      x->balance=y->balance=0;
      }
    else{                    // Rotate left at x and then right at y
      //outMsg(0,'T',"nbSetInsert: rotate left at x then right at y");
      w=x->right;
      if((x->right=w->left)) x->right->parent=x;
      w->left=x;
      x->parent=w;
      if((y->left=w->right)) y->left->parent=y;
      w->right=y;
      y->parent=w;
      if(w->balance==-1) x->balance=0, y->balance=+1;
      else if(w->balance==0) x->balance= y->balance=0;
      else /* w->balance==+1 */ x->balance=-1, y->balance=0;
      w->balance=0;
      }
    }
  else if(y->balance==+2){   // rebalance after right insertion
    x=y->right;
   // outMsg(0,'T',"nbSetInsert: +2 x=%p",x);
    if(x->balance==+1){      // Rotate left at y
      //outMsg(0,'T',"nbSetInsert: rotate left at y");
      w=x;
      if((y->right=x->left)) y->right->parent=y;
      x->left=y;
      y->parent=x;
      x->balance=y->balance=0;
      }
    else{                    // Rotate right at x then left at y
      //outMsg(0,'T',"nbSetInsert: rotate right at x then left at y");
      w=x->left;
      if((x->left=w->right)) x->left->parent=x;
      w->right=x;
      x->parent=w;
      if((y->right=w->left)) y->right->parent=y;
      w->left=y;
      y->parent=w;
      if(w->balance==+1) x->balance=0, y->balance=-1;
      else if(w->balance==0) x->balance= y->balance=0;
      else /* w->balance == -1 */ x->balance=+1, y->balance=0;
      w->balance=0;
      }
    }
  else return;         // should never happen
  *nodeP=w;            // replace root node for the previously unbalanced portion
  w->parent=node;
  //outMsg(0,'T',"nbSetInsert: leaving as node=%p,parent=%p",w,w->parent);
  }



// nbSetRemove()  - AVL tree node removal routine
//
//   path      - pointer to path structure returned by nbSetLocate() or nbSetLocateValue()    
//   newNode   - pointer to node structure (header to callers node structure)
// 
// Returns address of deleted (removed) node or NULL if not found
//           

void nbSetRemove(NB_SetNode *root,NB_SetNode *node){ 
  NB_SetNode **nodeP; // pointer to node pointer - used for steping left and right
  NB_SetNode *r,*s,*w;// temp node pointers
  NB_SetNode *y;
  NB_SetNode *x;        
  int top,depth=31;   // path depth
  int j=0;       // temp depth index 
  int pathStep[64];
  NB_SetNode *pathNode[64];

  //outMsg(0,'T',"nbSetRemove: called root=%p node=%p",root,node);
  //outMsg(0,'T',"nbSetRemove: node->parent=%p",node->parent);

  // Construct a reverse path so we can use the path based logic for now
  //   Study this in the future to see if we can revise the logic to be
  //   more efficient not using the path array.

  for(y=node,w=node->parent;w!=root;y=w,w=w->parent){
    if(w==node) nbExit("nbSetRemove encountered loop");
    pathNode[depth]=w;
    if(w->left==y) pathStep[depth]=0;
    else pathStep[depth]=1;
    depth--;
    }
  pathNode[depth]=root;
  pathStep[depth]=0;
  top=depth;
  depth=32;

  // Remove node

  if(pathStep[depth-1]) nodeP=&(pathNode[depth-1])->right;
  else nodeP=&(pathNode[depth-1])->left;
  if((r=node->right)==NULL){
    *nodeP=node->left;  // Case 1: Right link is null - replace with left
    if(node->left) node->left->parent=pathNode[depth-1]; 
    }
  else if(r->left==NULL){      // Case 2: Right->left is null - replace with right
    r->left=node->left;
    if(r->left) r->left->parent=r;
    r->balance=node->balance;
    *nodeP=r;
    r->parent=node->parent;
    pathStep[depth]=1;
    pathNode[depth++]=r;
    }
  else{                        // Case 3: Replace with leftmost of right
    j=depth++;
    for(;;){
      pathStep[depth]=0;    // moving left
      pathNode[depth++]=r;
      s=r->left;
      if(s->left==NULL) break;
      r=s;
      }
    s->left=node->left;
    if(s->left) s->left->parent=s;
    r->left=s->right;
    if(r->left) r->left->parent=r;
    s->right=node->right;
    if(s->right) s->right->parent=s;
    s->balance=node->balance;
    if(pathStep[j-1]) pathNode[j-1]->right=s;
    else pathNode[j-1]->left=s;
    s->parent=pathNode[j-1];
    pathStep[j]=1;
    pathNode[j]=s;
    }

// Update balance factors and rebalance

  while(--depth>top){
    y=pathNode[depth];
    if(pathStep[depth-1]) nodeP=&pathNode[depth-1]->right;
    else nodeP=&pathNode[depth-1]->left;
    if(pathStep[depth]==0){     // Update y's balance factor after left-side deletion
      y->balance++;
      if(y->balance==+1) break;
      else if(y->balance==+2){  // rebalance
        x=y->right;
        if(x->balance==-1){  // left side deletion rebalance case 1
          w=x->left;
          x->left=w->right;
          if(x->left) x->left->parent=x;
          w->right=x;
          x->parent=w;
          y->right=w->left;
          if(y->right) y->right->parent=y;
          w->left=y;
          y->parent=w;
          if(w->balance==+1) x->balance=0,y->balance=-1;
          else if(w->balance==0) x->balance=y->balance=0;
          else /* w->balance == -1 */ x->balance=+1,y->balance=0;
          w->balance = 0;
          *nodeP=w;
          w->parent=pathNode[depth-1];
          }
        else{                   // left side deletion rebalance case 2
          y->right=x->left;
          if(y->right) y->right->parent=y;
          x->left=y;
          y->parent=x;
          *nodeP=x;
          x->parent=pathNode[depth-1];
          if(x->balance==0){
            x->balance=-1;
            y->balance=+1;
            break;
            }
          else x->balance=y->balance=0;
          }
        }
      }
    else{             // Update y's balance factor after right-side deletion
      y->balance--;
      if(y->balance==-1) break;
      else if(y->balance==-2){
        x=y->left;
        if(x->balance==+1){  // Rotate left at x then right at y
          w=x->right;
          x->right=w->left;
          if(x->right) x->right->parent=x;
          w->left=x;
          x->parent=w;
          y->left=w->right;
          if(y->left) y->left->parent=y;
          w->right=y;
          y->parent=w;
          if(w->balance==-1) x->balance=0,y->balance=+1;
          else if(w->balance==0) x->balance=y->balance=0;
          else /* w->balance == +1 */ x->balance=-1,y->balance=0;
          w->balance=0;
          *nodeP=w;
          w->parent=pathNode[depth-1];
          }
        else{
          y->left=x->right;
          if(y->left) y->left->parent=y;
          x->right=y;
          y->parent=x;
          *nodeP=x;
          x->parent=pathNode[depth-1];
          if(x->balance==0){
            x->balance=+1;
            y->balance=-1;
            break;
            }
          else x->balance=y->balance=0;
          }
        }
      }
    }
  //outMsg(0,'T',"nbSetRemove: returning");
  //return((void *)node);
  }

/*
*  Recursive routine to flatten a subtree
*    Updates a pointer to the flattened substree (nodeP)
*    Returns the last node in the flattened subtree
*/
NB_SetNode *nbSetFlatten(NB_SetNode **nodeP,NB_SetNode *node){
  NB_SetNode *lastNode;
  if(node->left!=NULL){
    lastNode=nbSetFlatten(nodeP,node->left);  // flatten left side
    node->left=NULL;
    nodeP=&lastNode->right;
    }
  *nodeP=node;
  if(node->right!=NULL) return(nbSetFlatten(&node->right,node->right));
  return(node);
  }

/*
*  Recursive routine to balance a subtree
*    Returns root node and updates next pointer
*    The next pointer identifies the second half of the list
*    which is null when called for the right half.
*/
NB_SetNode *nbSetBalance(NB_SetNode *node,int n,NB_SetNode **nextP){
  NB_SetNode *leftroot;

  switch(n){
    case 0:
      node=NULL;
      break;
    case 1:
      *nextP=node->right;
      node->right=NULL;
      break;
    case 2:
      *nextP=node->right->right;
      node->right->right=NULL;
      break;
    case 3:
      node->right->left=node;
      node=node->right;
      node->left->right=NULL;
      *nextP=node->right->right;
      node->right->right=NULL;
      break;
    default:
      leftroot=nbSetBalance(node,n/2,&node);
      node->left=leftroot;
      node->right=nbSetBalance(node->right,n-n/2-1,nextP);
    }
  return(node);
  }

