/*
* Copyright (C) 2014 Ed Trettevik <eat@nodebrain.org>
*
* NodeBrain is free software; you can modify and/or redistribute it under the
* terms of either the MIT License (Expat) or the following NodeBrain License.
*
* Permission to use and redistribute with or without fee, in source and binary
* forms, with or without modification, is granted free of charge to any person
* obtaining a copy of this software and included documentation, provided that
* the above copyright notice, this permission notice, and the following
* disclaimer are retained with source files and reproduced in documention
* included with source and binary distributions. 
*
* Unless required by applicable law or agreed to in writing, this software is
* distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.
*
*=============================================================================
*
* Name:   nb_snmptrap.c
*
* Title:  SNMP Trap Listener 
*
* Function:
*
*   This program is a NodeBrain skill module for monitoring SNMP V1 
*   and V2 traps.
*   
* Description:
*
*   This is an experimental trap monitor.  Unless you have a special
*   reason to accept SNMP traps directly, you may prefer to use your
*   snmptrapd utility to accept traps and log them to a file.  In that
*   case, you would use a NodeBrain log file listener to monitor the
*   traps. 
*
*   This module listens for SNMP V1/V2 traps (UDP datagrams) and converts
*   them into NodeBrain ALERT commands.  See the comments for the
*   translate() function for a description of the generated command
*   format.
*
*   To use this module, first declare your nodebrain script.  You can
*   use a "?" to let NodeBrain fill in the file extension used on your
*   platform: so, sl, dylib, dll.
*
*     Syntax:
*
*       declare <term> module <dynamic-load-library>;
*
*     Example:
*
*       declare snmptrap module /usr/local/lib/nb/nb_snmptrap.?;
*
*   You may define one or more snmptrap nodes, but only one can listen
*   to a give port on a given interface.
*
*     Syntax:
*
*       define <term> node snmptrap[(<binding>)][:<options>];
*
*       <binding>  -  Bind to a specific interface and/or port
*                   
*                     "<address>"       - bind to interface
*                                         (default is all interfaces)
*                     <port>            - bind alternate port number
*                                         (default is 162)
*                     "<address>:port"  - bind to interface and port
*
*       <options>  -  Options to control the log output 
*
*                     trace   - display every trap received
*                     dump    - display a hex dump of UDP datagrams
*                     silent  - don't echo generated NodeBrain commands
*     Examples:
*
*       define snmptrap node snmptrap; 
*       define snmptrap node snmptrap("127.0.0.1"); # bind to local host 
*       define snmptrap node snmptrap(50162);  # alternate port
*       define snmptrap node snmptrap("127.0.0.1:50162");  # both
*       define snmptrap node snmptrap:dump;  # specify an option
*       define snmptrap node snmptrap(50162):silent; # specify port and option
*
*   To take actions under specific conditions, you need to define rules to
*   respond to the alerts generated by this module.  
*
*     Syntax:
*
*       <context>. define <term> if(<condition>) <action>
*
*     Example:
*
*       snmptrap. define snmpTrapCommunity cell '1.3.6.1.6.3.18.1.4';
*       snmptrap. define r1 if(snmpTrapCommunity!="public"):$ - myscript $${agent}
*
*   The terms used within the context of the node (snmptrap in our examples) are OID
*   values used as quoted terms.
*
*     '<oid>'     - OID term (e.g. '1.3.6.1.6.3.18.1.4')
*      
*   This sample module is not MIB aware, so you are not able to reference variable names like the
*   the following.
*
*     .iso.org.dod.internet.mgmt.mib-2.system.sysDescr.0
*   
*   Instead, you may define your own variable names to make the rules easier to read.  This
*   is accomplished by defining terms that reference the numberic OID terms.  This is standard
*   NodeBrain syntax---nothing special going on here.
*
*     Syntax:
*
*       <context>. define <term> cell '<oid>';
*
*     Example: (Don't worry about the actual OID here, it isn't intended to be appropriate)
*
*       snmptrap. define process cell '1.3.6.1.2.1.1.1.0';  
*
*     Example Rule:
*
*       snmptrap. define r1 if(process="inetd") trouble=5;
*   
*   An snmptrap node will not start listening until you enable it.
*
*     Syntax:
*
*       enable <term>;
*
*     Example:
*
*       enable snmptrap;
*
*===========================================================================
* Sample Rules:
*
*   There are three files distributed with this module to illustrate how
*   you might use it.  They are in the sample directory nb-0.6.2/sample/.
*
*     snmptrap.nb    - sample rule set (mostly comments)
*     snmptrap.sh    - sample script to set test traps
*     snmptrap.log   - sample log data from a successful test
*
*===========================================================================
* NodeBrain API Functions Used:
*
*   Here's an outline of the API functions used by this module.  Refer to
*   the online NodeBrain User's Guide for a description of the NodeBrain
*   Skill Module API.  However, you can probably figure it out by
*   the example of this module using this outline as a starting point.
*   
*   Inteface to NodeBrain skill modules
*
*     nbSkillSetMethod()  - Used to tell NodeBrain about our methods.  
*
*       serverConstruct()  - construct an snmptrap node
*       serverEnable()     - start listening
*       serverDisable()    - stop listening
*       serverCommand()    - handle a custom command
*       serverDestroy()    - clean up when the node is destroyed
*
*   Interface to NodeBrain objects (primarily used in serverConstruct)
*
*     nbCellCreate()      - Create a new cell
*     nbCellGetType()     - Identify the type of a cell
*     nbCellGetString()   - Get a C string from a string cell
*     nbCellGetReal()     - Get a C double value from a real cell
*     nbCellDrop()        - Release a cell (decrement use count)
*
*   Interface to NodeBrain actions
*
*     nbLogMsg()	    	  - write a message to the log
*     nbLogDump()         - write a buffer dump to the log
*     nbCmd()             - Issue a NodeBrain command
*
*   Interface to NodeBrain listeners
*
*     nbListenerAdd()     - Start listening to file/socket
*
*       serverRead()      - Handler passed to nbListenerAdd()   
*
*     nbListenerRemove()  - Stop listening to file/socket
*
*   Interface to Internet Protocol (IP) provided by NodeBrain
*
*     These functions are not an interface to NodeBrain, but are provided
*     by NodeBrain to symplify some common tasks for socket programming.  If you
*     use this module as a model for another, you may replace these functions
*     with your own direct calls to C functions for socket programming.  In other
*     words, NodeBrain doesn't care if you use these or not. 
*
*     nbIpGetUdpServerSocket()  - Obtain a UDP server socket
*     nbIpGetDatagram()         - Read a UDP packet (datagram)
*     nbIpGetSocketAddrString() - Get address of interface we are listening on
*     nbIpGetAddrStr()          - Convert an IP address from internal form to string
*
*=====================================================================
* Change History:
*
* Date       Name/Change
* ---------- ---------------------------------------------------------
* 2005/03/24 Ed Trettevik - original prototype version 0.6.2
* 2005/03/27 eat 0.6.2  Added comments for reference as a sample module
* 2005/04/09 eat 0.6.2  adapted to api changes
* 2005/04/20 eat 0.6.2  included changes by Roland Hajj and Fadi Naffah
*            1) fixed the decoding of oid values
*            2) included all the types of varbinds in the translateValue() function
*            3) included all the SNMP v1 trap parameters
*            4) included support for SNMP v2 traps  
*            (Thanks to both for this contribution.)
* 2005/04/20 eat 0.6.2  Fixed additional problems in oid decoding for large numbers
*            Numbers over 14 bits need more than 2 bytes
* 2005/04/21 eat 0.6.2  Added support for NULLOBJ data type (0x05).
*            Using NodeBrain's Unknown value identified as ??  
* 2005/04/21 eat 0.6.2  Fix interprestation of V1 enterpriseOID
*            For some reason I was previously treating it like the agent address OID
* 2005/04/21 eat 0.6.2  Common NodeBrain alert format for both V1 and V2
*            Used RFC 3584 section 3.1 to translate V1 notification parameters
*            into V2 notification parameters for insertion into alert command.
*            The new format removes the "oid." prefix on OID terms, and
*            references all notication parameters just like variables bindings
*            using the OID term format.  For example, what was 
*            community="public" is now '1.3.6.1.6.3.18.1.4'="public".
*            Remember, an alias can be defined for any OID.
*
*              define snmpTrapCommunity cell '1.3.6.1.6.3.18.1.4';
* 2005/04/21 eat 0.6.2  Inserted sender's address in V2 traps.
*            If the agent address is included in the trap, it will override
*            the one we insert.  This is because of the way NodeBrain
*            handles ALERT/ASSERT commands.  The last assignment wins.
*            In this case, it is exactly what we want.  This ensures
*            that we get an agent address if the sender doesn't include
*            one.  It is likely to be there already if the trap comes
*            from a proxy.
* 2005/05/14 eat 0.6.3  snmptrapBind() updated for moduleHandle
* 2009/06/26 eat 0.7.6  included support for handler nodes
*            By specifying a 'handler' glossary with terms of the form
*            '<SNMP_TRAP_OID>'="<handler>"  you can send traps to handler
*            nodes.  In the example below, any trap having the specified
*            trap OID will be sent to the myhandler node.
*
*              define myhandler node servant...
*              define snmptrap node snmptrap...
*              handler.'1.3.6.1.4.1.6101.141.0.0'="myhandler";
*
*            The handler node is invoked as follows.
*              myhandler:'1.3.6.1.4.1.6101.141.0.0':<variable bindings>
* 2010/02/25 eat 0.7.9  Cleaned up -Wall warning messages
* 2010/02/25 eat 0.7.9  Cleaned up -Wall warning messages (gcc 4.1.2)
* 2012-10-13 eat 0.8.12 Replaced malloc/free with nbAlloc/nbFree
* 2012-12-27 eat 0.8.13 Checker updates
* 2013-01-13 eat 0.8.13 Checker updates
* 2014-06-24 eat 0.9.02 Fixed length of stop on NULLOBJ variable type 
*=====================================================================
*/
#include "config.h"
#include <nb/nb.h>
#include <ctype.h>

/*
*  The following structure is created by the skill module's "construct"
*  function (serverConstruct) defined in this file.  This is a module specific
*  structure.  NodeBrain is only aware of the address of instances of this
*  structure which it stores in a node's "handle".  The handle is passed to
*  various functions defined in this module.
*/
typedef struct NB_MOD_SNMPTRAP{            /* SNMPTRAP node descriptor */
  unsigned int   socket;           /* server socket for datagrams */
  char interfaceAddr[16];              /* interface address to bind listener */
  unsigned short port;             /* UDP port of listener */
  unsigned char  trace;            /* trace option */
  unsigned char  dump;             /* option to dump packets in trace */
  unsigned char  echo;             /* echo option */
  unsigned int   sourceAddr;       /* source address */
  nbCELL handlerContext;
  nbCELL syntaxContext;
  nbCELL attributeContext;
  } NB_MOD_Snmptrap;

/*================================================================================*/

/*
*  SNMP V1 and V2 Trap Datagram Format
*
*  Note: This format description has been quickly reverse engineered and probably
*        isn't actually correct.  It just needs to be close enough to experiment
*        with.  If you use this module and find bugs or deficiencies, please fix 
*        them and share your fix.  When I get a chance I'll actually read the 
*        appropriate RFC's and improve this code.
*
*  2005/04/21 eat 0.6.2 - The reverse engineering has been enhanced by Roland Hajj
*
*  V1 Example:
*
*    0000: 30 3D 02 01  00 04 06 70  75 62 6C 69  63 A4 30 06    0=.....public.0.
*    0016: 08 2B 06 01  04 01 03 01  01 40 04 82  26 FD 39 02    .+.......@..&.9.
*    0032: 01 03 02 01  00 43 04 57  B5 22 10 30  12 30 10 06    .....C.W.".0.0..
*    0048: 08 2B 06 01  02 01 01 01  00 04 04 66  72 65 64       .+.........fred
*
*    30 - object?     see getObjectLength() for more informaion on all lengths
*    3D - length of object - lengths may be multiple bytes
*    02.01.00 - version 1
*    04.06.70.75.62.6c.69.63 community string ("public" length 06)
*    ------------------
*    A4 - v1 unique portion
*    30 - length of object - see getObjectLength()
*    06.08.2B.06.01.04.01.03.01.01  Agent address OID (length 08)       
*    40.04.C0.2A.E5.63  address (length 04) 
*    02.01.03 - generic trap type
*    02.01.00 - specific trap type
*    43.04.57.B5.22.10 - uptime
*    ------------------
*    30.12 variable binding list (length 12)
*    30.10 variable binding (length 10)
*    06.08.23.06.01.02.01.01.01.00 OID (length 08)
*    04.04.66.72.65.64 Value ("fred" length 04)
*  
*  V2 Example:
*
*    0000 30620201 01040670 75626c69 63a75502 0b.....public.U.
*    0010 047ab0e2 da020100 02010030 47301006 .z.........0G0..
*    0020 082b0601 02010103 00430467 047b5530 .+.......C.g.{U0
*    0030 12060a2b 06010603 01010401 0006042a ...+...........*
*    0040 03040530 1006082b 06010201 01010004 ...0...+........
*    0050 04667265 64300d06 032a0304 06062b04 .fred0...*....+.
*    0060 839f7b0a ........ ........ ........ ..{.            
*
*    30 - object?
*    62 - length
*    02.01.01 - version 2
*    04.06.70.75.62.6c.69.63 community string ("public" length 06)
*    ------------------
*    A7 - v2 unique portion
*    30 - length of object - see getObjectLength()
*    02.04.7a.b0.e2.da ?
*    02.01.00
*    02.01.00
*    ------------------
*    30.47 variable binding list (length 47)
*    this section same as V1
*/

/*
* Get the length of an object and advance the cursor
*
*     0x01-0x7f  - one byte length
*     0x81 - one byte length greater than 127 follows
*     0x82 - two byte length follows     
*     0x83 - three byte length follows
*     ...
*/
static int getObjectLength(unsigned char **bufP,unsigned char *bufend){
  int objlen,bytes;
  unsigned char *buf=*bufP;

  objlen=*buf;
  buf++;
  if(objlen>127){
    bytes=objlen&127;   
    objlen=*buf;
    buf++;
    for(bytes--;bytes>0;bytes--){
      objlen=(objlen<<8)|*buf;
      buf++;
      }
    }
  if(buf+objlen>bufend) return(-1);
  *bufP=buf;
  return(objlen);
  }

/*
*  Translate OID from binary coding to character
*/
static char *translateOid(unsigned char **cursorP,unsigned char *bufend,char **cmdP,char *cmdend){
  unsigned char *cursor=*cursorP;
  char *cmdcur=*cmdP;
  int objlen,n=0;

  objlen=getObjectLength(&cursor,bufend);
  if(objlen<0) return("variable binding OID value length error");
  // first byte of OID has two numbers     
  sprintf(cmdcur,"%u.%u.",*cursor/40,*cursor%40);    
  cmdcur+=strlen(cmdcur);   // 2013-01-14 eat - VID 968-0.8.13-3 FP but replaced strchr(cmdcur,0)   
  objlen--;    
  cursor++;    
  // then we get 7 bits per byte    
  // high order bit of 1 means we have another byte    
  for(;objlen>0;objlen--){    
    n=*cursor&0x7f;    
    while(*cursor&0x80){    
      cursor++;    
      n=(n<<7)|(*cursor&0x7f);    
      objlen--;    
      }    
    cursor++;    
    sprintf(cmdcur,"%u.",n);    
    cmdcur+=strlen(cmdcur); // 2013-01-14 eat - VID 962-0.8.13-3 FP but replaced strchr(cmdcur,0)   
    }    
  if(*(cmdcur-1)=='.') cmdcur--;    
  *cursorP=cursor;
  *cmdP=cmdcur;
  return(NULL);
  }

/*
*  Translate an SNMP value into a NodeBrain value
*
*  SNMP Value Types           NodeBrain Value
*
*    i 02 - INTEGER           number n
*    s 04 - OCTET STRING      string "abcdef"
**   x 04 - HEX STRING        "
**   d 04 - DECIMAL STRING    "
**   b 04 - BITS              "
**     04 - DateAndTime       number n - UTC time
*    n 05 - NULLOBJ           unknown value ? 
*    o 06 - OBJID             string "n.n.n.n..."
*    a 40 - IpAddress         string "xxx.xxx.xxx.xxx" 
**   c 41 - Counter32         number n 
**   u 42 - Unsigned          " 
**   t 43 - TimeTicks         " 
*
*  Note 1: Addresses are formated in a non-standard way to simplify comparison.  Leading
*          zeros are included so every number has three digits.  For example, 192.168.1.23
*          is formatted as "192.168.001.023".  This makes "greater than" and "less than"
*          comparison on the string values work.  However, you should convert these values
*          to standard format when using as parameters to standards tools like ping and
*          and traceroute.
*
*  Note 2: Double quotes are replaced with single quotes in strings to avoid terminating
*          NodeBrain strings.
*
*  Note 3: For some reason SNMP elected to double up on value types like 0x04.  We have a
*          default way of representing values, but accept a syntax parameter to direct 
*          the representation when the default is not appropriate.
*
*          04 Default     - as is if all bytes are printable, else convert bytes to hex string
*          04 DateAndTime - number n - UTC Time
*          
*/
static void translateUnquoteString(char *cursor,int len){
  char *delim;
  delim=memchr(cursor,'"',len);
  while(delim){
    *delim='\'';
    len-=delim-cursor;
    cursor=delim;
    delim=memchr(cursor,'"',len);
    }
  }

char mymsg[256];
static char *translateValue(unsigned char **cursorP,unsigned char *bufend,char **cmdP,char *cmdend,char *syntax){
  unsigned char *cursor=*cursorP,*lookahead;
  char *cmdcur=*cmdP,*msg;
  int objlen,n=0,type;
  char *hexchar="0123456789ABCDEF";
  double d=0;
  time_t utime;
  struct tm tm;

  type=*cursor;  /* get date type */
  cursor++;
  objlen=getObjectLength(&cursor,bufend);
  if(objlen*2>cmdend-cmdcur-1){  /* all types except string need twice as much space after converted */
    if(type!=4 || objlen>cmdend-cmdcur-1) return("object is too long for command buffer");
    }
  switch(type){
    case 0x02: /* integer */
      if(objlen<1 || objlen>4) return("variable binding integer value length error");
      for(;objlen>0;objlen--){
        n=(n<<8)|*cursor;
        cursor++;
        } 
      sprintf(cmdcur,"%d",n);
      cmdcur+=strlen(cmdcur);  // 2013-01-14 eat - VID 4164-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    case 0x04: /* string */
      if(objlen<0) return("variable binding string value length error");
      if(objlen>cmdend-cmdcur-1) return("value is too large for buffer");
      if(strcmp(syntax,"DateAndTime")==0){
        if(objlen!=8 && objlen!=11) return("variable binding DateAndTime value length error");
        tm.tm_year=*cursor<<8;
        cursor++;
        tm.tm_year|=*cursor;
        tm.tm_year-=1900;
        cursor++;
        tm.tm_mon=*cursor-1;
        cursor++;
        tm.tm_mday=*cursor;
        cursor++;
        tm.tm_hour=*cursor;
        cursor++;
        tm.tm_min=*cursor;
        cursor++;
        tm.tm_sec=*cursor;
        cursor++;
        cursor++; // ignore deci-seconds
        objlen-=8;
        if(objlen){ 
          if(*cursor=='+'){
            cursor++;
            tm.tm_hour-=*cursor;
            cursor++;
            tm.tm_min-=*cursor;
            }
          else if(*cursor=='-'){
            cursor++;
            tm.tm_hour+=*cursor;
            cursor++;
            tm.tm_min+=*cursor;
            }
          else return("variable binding DateAndtime value has unrecognized direction from UTC");
          cursor++;
          }
        tm.tm_isdst=0;
        utime=nbClockTimeGm(&tm);
        sprintf(cmdcur,"%d",(int)utime);
        cmdcur=strchr(cmdcur,0);
        }
      else{
        *cmdcur='"';
        cmdcur++;
        for(lookahead=cursor+objlen-1;lookahead>=cursor && isprint(*lookahead);lookahead--);
        if(lookahead<cursor){
          strncpy(cmdcur,(char *)cursor,objlen);
          translateUnquoteString(cmdcur,objlen);
          cmdcur+=objlen;
          }
        else{
          for(lookahead=cursor;lookahead<cursor+objlen;lookahead++){
            *cmdcur=*(hexchar+(*lookahead>>4));
            cmdcur++;
            *cmdcur=*(hexchar+(*lookahead&0x0f));
            cmdcur++;
            }
          }
        *cmdcur='"';
        cmdcur++;
        cursor+=objlen;
        }
      break;
    case 0x05: /* NULLOBJ */
      if(objlen!=0) return("variable binding NULLOBJ length error - expecting zero");
      strcpy(cmdcur,"?");  /* we'll use NodeBrain's Unknown value for NULLOBJ */
      cmdcur++; // 2014-06-24 eat - changed from 2 to 1 byte step because "?" replaced "??"
      break;
    case 0x06: /* OID */
      *cmdcur='"';
      cmdcur++;
      cursor=*cursorP+1; /* go back to the length */
      if((msg=translateOid(&cursor,bufend,&cmdcur,cmdend))!=NULL) return(msg);
      *cmdcur='"';
      cmdcur++;
      break;
    case 0x40: /* address */
      if(objlen!=4) return("expecting 4 byte address");
      *cmdcur='"';
      cmdcur++;
      sprintf(cmdcur,"%3.3u.%3.3u.%3.3u.%3.3u",*cursor,*(cursor+1),*(cursor+2),*(cursor+3));
      cmdcur+=15;
      *cmdcur='"';
      cmdcur++;
      cursor+=4;
      break;
    case 0x41: /* counter32 */
      if(objlen<1 || objlen>4) return("variable binding integer value length error");
      for(;objlen>0;objlen--){
        n=(n<<8)|*cursor;
        cursor++;
        } 
      sprintf(cmdcur,"%d",n);
      cmdcur+=strlen(cmdcur); // 2013-01-14 eat - VID 4154-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    case 0x42: /* unsigned */
      if(objlen<1 || objlen>4) return("variable binding integer value length error");
      for(;objlen>0;objlen--){
        n=(n<<8)|*cursor;
        cursor++;
        } 
      sprintf(cmdcur,"%d",n);
      cmdcur+=strlen(cmdcur); // 2013-01-14 eat - VID 4158-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    case 0x43: /* timeticks */
      if(objlen<0) return("variable binding OID value length error");
      for(;objlen>0;objlen--){
      	n=(n<<8)|*cursor;
        cursor++;
        }
      sprintf(cmdcur,"%d",n);
      cmdcur+=strlen(cmdcur); // 2013-01-14 eat - VID 4153-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    case 0x46: // 64 bit number
      if(objlen<1 || objlen>8) return("variable binding Counter64 value length error");
      for(;objlen>0;objlen--){
        d=d*256+*cursor;
        cursor++;
        } 
      sprintf(cmdcur,"%.10g",d);
      cmdcur+=strlen(cmdcur);  // 2013-01-14 eat - VID 4159-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    default:
      sprintf(mymsg,"unrecognized value type %x len=%d",type,objlen);
      return(mymsg);
    }
  *cursorP=cursor;
  *cmdP=cmdcur;
  return(NULL);
  }

/*
*  Translate an SNMP V1 or V2 Trap to a NodeBrain ALERT command
*
*    alert '<oid>'=<value>,...;
*
*    Here's an example of a full alert with a single <variable-binding>.
*  
*    *** include example here ***
*/
static char *translate(NB_MOD_Snmptrap *snmptrap,unsigned char *buf,int len,char *cmd,int cmdlen,char **handlerNameP){
  char *cmdcur=cmd;
  char *cmdend=cmdcur+cmdlen;
  unsigned char *cursor=buf;
  unsigned char *bufend=buf+len;
  unsigned char *enterpriseOid;
  int objlen,generic;
  char *msg,version,senderAddr[40];
  char trapOID[64],*trapOIDCur;  // value of '1.3.6.1.6.3.1.1.4.1.0' (SNMP_TRAP_OID)
  //char *handler;
  nbCELL cell;
  char *oid,*value,*syntax,*attribute;

  fprintf(stderr,"translate: called\n");
  fflush(stderr);

  *trapOID=0;
  if(*cursor!=0x30) return("packet not recognized");
  cursor++;
  objlen=getObjectLength(&cursor,bufend);
  if(objlen<0) return("trap length error");
  if(*cursor!=0x02 || *(cursor+1)!=0x01) return("expecting 02.01 to start trap");
  cursor+=2;
  version=*cursor;
  cursor++;
  sprintf(cmdcur,"alert '1.3.6.1.6.3.18.1.4'=");  // snmpTrapCommunity OID
  cmdcur+=strlen(cmdcur);  // 2013-01-14 eat - VID 976-0.8.13-3 FP but replaced strchr(cmdcur,0)
  if(*cursor!=0x04) return("expecting type 04 (string) for community string");
  if(NULL!=(msg=translateValue(&cursor,bufend,&cmdcur,cmdend,""))) return(msg);

  fprintf(stderr,"translate: checking version specific stuff\n");
  fflush(stderr);

  switch(version){
    case 0: /* snmpV1 trap */
      if(*cursor!=0xA4) return("expecting 0xA4 for V1 trap");
      cursor++;
      objlen=getObjectLength(&cursor,bufend);
      if(objlen<0) return("buffer length confusion");

      if(*cursor!=0x06) return("expecting 0x06 for enterprise OID");
      strcpy(cmdcur,",'1.3.6.1.6.3.1.1.4.3'=");
      cmdcur+=strlen(cmdcur);  // 2013-01-14 eat - VID 4157-0.8.13-3 FP but replaced strchr(cmdcur,0)
      enterpriseOid=cursor;
      if(NULL!=(msg=translateValue(&cursor,bufend,&cmdcur,cmdend,""))) return(msg);

      if(*cursor!=0x40) return("expecting 0x40 for address");
      strcpy(cmdcur,",'1.3.6.1.6.3.18.1.3'=");
      cmdcur+=strlen(cmdcur);  // 2013-01-14 eat - VID 4165-0.8.13-3 FP but replaced strchr(cmdcur,0)
      if(NULL!=(msg=translateValue(&cursor,bufend,&cmdcur,cmdend,""))) return(msg);

      if(*cursor!=0x02) return("expecting integer for trap generic type");
      cursor++;
      if(*cursor>1) return("generic trap type length error - expecting 1");
      cursor++;
      generic=*cursor;
      cursor++;
      if(*cursor!=0x02) return("expecting integer for trap specific type");             
      strcpy(cmdcur,",'1.3.6.1.6.3.1.1.4.1.0'=");  // snmpTrapOID.0
      cmdcur+=strlen(cmdcur);   // 2013-01-14 eat - VID 4162-0.8.13-3 FP but replaced strchr(cmdcur,0)
      if(generic!=6){
        sprintf(cmdcur,"\"1.3.6.1.6.3.1.1.5.%d\"",(*cursor)+1); // see RFC 3584
        cmdcur+=strlen(cmdcur);   // 2013-01-14 eat - VID 4156-0.8.13-3 FP but replaced strchr(cmdcur,0)
        cursor++;
        objlen=getObjectLength(&cursor,bufend);  // step over trap specific type
        if(objlen<0) return("buffer length confusion");
        }
      else{
        trapOIDCur=cmdcur+1;  //save location of trapOId
        if(NULL!=(msg=translateValue(&enterpriseOid,bufend,&cmdcur,cmdend,""))) return(msg);
        cmdcur--;  // back up over ending quote to extend this enterprise oid
        strcpy(cmdcur,".0.");
        cmdcur+=3;
        if(NULL!=(msg=translateValue(&cursor,bufend,&cmdcur,cmdend,""))) return(msg);
        *trapOID='\'';
        objlen=cmdcur-trapOIDCur;
        strncpy(trapOID+1,trapOIDCur,objlen);
        *(trapOID+objlen+1)='\'';
        *(trapOID+objlen+2)=0;
        *cmdcur='"';
        cmdcur++;
        }
      if(*cursor!=0x43) return("expecting 0x43 for uptime");
      strcpy(cmdcur,",'1.3.6.1.2.1.1.3.0'=");
      cmdcur+=strlen(cmdcur);   // 2013-01-14 eat - VID 966-0.8.13-3 FP but replaced strchr(cmdcur,0)
      if(NULL!=(msg=translateValue(&cursor,bufend,&cmdcur,cmdend,""))) return(msg);
      break;

    case 1: /* snmpV2 trap */

      if(*cursor!=0xA7) return("expecting 0xA7 for trap");
      cursor++;
      objlen=getObjectLength(&cursor,bufend);
      if(objlen<0) return("buffer length confusion");
  	
      if(*cursor!=0x02) return("expecting 0x02 for variable 1");
      cursor++;
      objlen=getObjectLength(&cursor,bufend);
      if(objlen<0) return("variable 1 length error");
      cursor+=objlen;  
  	
      if(*cursor!=0x02 || *(cursor+1)!=0x01 || *(cursor+2)!=0x00 || *(cursor+3)!=0x02 || *(cursor+4)!=0x01 || *(cursor+5)!=0x00) return("V2:expecting 02 01 00 02 01 00 to start trap");
      cursor+=6;

      // Insert the sender's address - don't worry, this will be overridden if sender supplies in variable bindings  
      sprintf(cmdcur,",'1.3.6.1.6.3.18.1.3'=\"%s\"",nbIpGetAddrString(senderAddr,snmptrap->sourceAddr));
      cmdcur+=strlen(cmdcur);   // 2013-01-14 eat - VID 980-0.8.13-3 FP but replaced strchr(cmdcur,0)
      break;
    default: return("unrecognized trap version");
    }

  fprintf(stderr,"translate: checking for handler\n");
  fflush(stderr);

  fprintf(stderr,"translate: processing variable bindings\n");
  fflush(stderr);

  // Variable bindings same for V1 and V2
  if(*cursor!=0x30) return("expecting 0x30 for variable binding list");
  cursor++;
  objlen=getObjectLength(&cursor,bufend);
  if(objlen<0) return("variable binding list length error");
  while(cursor<bufend){
    if(*cursor!=0x30) return("expecting 0x30 for variable binding");
    cursor++;
    objlen=getObjectLength(&cursor,bufend);
    if(objlen<0) return("variable binding length error");
    if(*cursor!=0x06) return("expecting OID on left side of variable binding");
    cursor++;
    *cmdcur=',';
    cmdcur++;
    oid=cmdcur;  // save location of OID
    *cmdcur='\'';
    cmdcur++;
    if((msg=translateOid(&cursor,bufend,&cmdcur,cmdend))!=NULL) return(msg);
    *cmdcur='\'';
    cmdcur++;
    *cmdcur=0;
    // look up syntax for value representation
    if(snmptrap->syntaxContext
        && (cell=nbTermLocateHere(snmptrap->syntaxContext,oid))!=NULL
        && (cell=nbTermGetDefinition(snmptrap->syntaxContext,cell))!=NULL
        && nbCellGetType(snmptrap->syntaxContext,cell)==NB_TYPE_STRING){
      syntax=nbCellGetString(snmptrap->syntaxContext,cell);
      }
    else syntax="";
    // look up attribute name and replace if found
    if(snmptrap->attributeContext
        && (cell=nbTermLocateHere(snmptrap->attributeContext,oid))!=NULL
        && (cell=nbTermGetDefinition(snmptrap->attributeContext,cell))!=NULL
        && nbCellGetType(snmptrap->attributeContext,cell)==NB_TYPE_STRING){
      attribute=nbCellGetString(snmptrap->attributeContext,cell);
      strcpy(oid,attribute);
      cmdcur=strchr(oid,0);
      }
    *cmdcur='=';
    cmdcur++;
    value=cmdcur;
    if((msg=translateValue(&cursor,bufend,&cmdcur,cmdend,syntax))!=NULL) return(msg);
    if(strncmp(oid,"'1.3.6.1.6.3.1.1.4.1.0'",23)==0 && *value=='"'){
      oid=trapOID;
      *oid='\'';
      oid++;
      value++;
      while(value<cmdcur && *value!='"') *oid=*value,oid++,value++;
      *oid='\'';
      oid++;
      *oid=0;
      }
    }
  *cmdcur=0;

  // Check for handler
  if(snmptrap->handlerContext
      && (cell=nbTermLocateHere(snmptrap->handlerContext,trapOID))!=NULL
      && (cell=nbTermGetDefinition(snmptrap->handlerContext,cell))!=NULL
      && nbCellGetType(snmptrap->handlerContext,cell)==NB_TYPE_STRING){
    *handlerNameP=nbCellGetString(snmptrap->handlerContext,cell);
    //handler=nbCellGetString(snmptrap->handlerContext,cell);
    //strcpy(cmd,handler);
    //cmdcur=strchr(cmd,0);
    //*cmdcur=':';
    //cmdcur++; 
    //strcpy(cmdcur,trapOID);
    //cmdcur=strchr(cmdcur,0);
    }

  fprintf(stderr,"translate: returning\n");
  fflush(stderr);

  return(NULL);
  }

/*==================================================================================
*
*  M E T H O D S
*
*  The code above this point is very specific to the goals of this skill module. The
*  code below this point is also specific in some of the details, but the general
*  structure applies to any skill module.  The functions below are "methods" called
*  by NodeBrain.  Their parameters must conform to the NodeBrain Skill Module
*  API.  A module is not required to provide all possible methods, so modules may
*  vary in the set of methods they implement.  For example, serverRead() is an
*  example of a method that would only be used by a module that "listens".
*
*=================================================================================*/

/*
*  Read incoming packets
*/
static void serverRead(nbCELL context,int serverSocket,void *handle){
  NB_MOD_Snmptrap *snmptrap=handle;
  unsigned char buffer[NB_BUFSIZE];
  size_t buflen=NB_BUFSIZE;
  int  len;
  unsigned short rport;
  char daddr[40],raddr[40];
  char cmd[NB_BUFSIZE];
  size_t cmdlen=NB_BUFSIZE;
  char *msg;
  char *handlerName=NULL;

  nbIpGetSocketAddrString(serverSocket,daddr);
  len=nbIpGetDatagram(context,serverSocket,&snmptrap->sourceAddr,&rport,buffer,buflen);
  if(snmptrap->trace) nbLogMsg(context,0,'I',"Datagram %s:%5.5u -> %s len=%d\n",nbIpGetAddrString(raddr,snmptrap->sourceAddr),rport,daddr,len);
  if(snmptrap->dump) nbLogDump(context,buffer,len);
  msg=translate(snmptrap,buffer,len,cmd,cmdlen,&handlerName);
  if(msg!=NULL){
    nbLogMsg(context,0,'E',msg);
    return;
    }
  if(snmptrap->trace && !snmptrap->echo) nbLogMsg(context,0,'I',cmd);
  if(handlerName){
    *(cmd+5)=':'; // convert to node command, stepping over "alert" verb
    nbNodeCmd(context,handlerName,cmd+5);
    }
  else nbCmd(context,cmd,snmptrap->echo);
  }

/*
*  construct() method
*
*    define <term> node <skill>[(<args>)][:<text>]
*
*    <args> - port_number or "interface_address[:port_number]"
*    <text> - flag keywords
*               trace   - display input packets
*               dump    - display dump of SNMP UDP packets
*               silent  - don't echo generated NodeBrain commands 
*
*    define snmptrap node snmptrap;
*    define snmptrap node snmptrap:dump,silent;
*    define snmptrap node snmptrap("127.0.0.1");
*    define snmptrap node snmptrap(50162);
*    define snmptrap node snmptrap("127.0.0.1:50162");
*    define snmptrap node snmptrap("127.0.0.1:50162"):silent;
*/
static void *serverConstruct(nbCELL context,void *skillHandle,nbCELL arglist,char *text){
  NB_MOD_Snmptrap *snmptrap;
  nbCELL cell=NULL;
  nbSET argSet;
  char *cursor=text,*delim,saveDelim;
  double r,d;
  char interfaceAddr[16];
  unsigned int port=162;
  int type,trace=0,dump=0,echo=1;
  int len;
  char *str;

  *interfaceAddr=0;

  argSet=nbListOpen(context,arglist);   
  cell=nbListGetCellValue(context,&argSet); 
  if(cell!=NULL){
    type=nbCellGetType(context,cell);
    if(type==NB_TYPE_STRING){
      str=nbCellGetString(context,cell);
      delim=strchr(str,':');
      if(delim==NULL) len=strlen(str);
      else len=delim-str;
      if(len>15){
        nbLogMsg(context,0,'E',"Inteface IP address may not be greater than 15 characters");
        nbCellDrop(context,cell);
        return(NULL);
        }
      strncpy(interfaceAddr,str,len);
      *(interfaceAddr+len)=0; 
      if(delim!=NULL){
        delim++;
        port=(unsigned int)atoi(delim);
        }
      nbCellDrop(context,cell);
      }
    else if(type==NB_TYPE_REAL){
      r=nbCellGetReal(context,cell);
      nbCellDrop(context,cell);
      port=(unsigned int)r;
      d=port;
      if(d!=r || d==0){
        nbLogMsg(context,0,'E',"Expecting non-zero integer UDP port number");
        return(NULL);
        }
      }
    else{
      nbLogMsg(context,0,'E',"Expecting interface (\"address[:port]\") or (port) as argument list");
      return(NULL);
      }
    cell=nbListGetCellValue(context,&argSet); 
    if(cell!=NULL){
      nbLogMsg(context,0,'E',"Only one argument expected - ignoring additional arguments");
      nbCellDrop(context,cell);
      }
    }
  if(*text!=0){
    cursor=text;
    while(*cursor==' ') cursor++;
    while(*cursor!=';' && *cursor!=0){
      delim=strchr(cursor,' ');
      if(delim==NULL) delim=strchr(cursor,',');
      if(delim==NULL) delim=strchr(cursor,';');
      if(delim==NULL) delim=cursor+strlen(cursor);  // 2013-01-14 eat - VID 969-0.8.13-3 FP but replaced strchr(cursor,0)
      saveDelim=*delim;
      *delim=0;
      if(strcmp(cursor,"trace")==0){trace=1;}
      else if(strcmp(cursor,"dump")==0){trace=1;dump=1;}
      else if(strcmp(cursor,"silent")==0) echo=0; 
      *delim=saveDelim;
      cursor=delim;
      if(*cursor==',') cursor++;
      while(*cursor==' ') cursor++;
      }
    }
  snmptrap=nbAlloc(sizeof(NB_MOD_Snmptrap));
  snmptrap->socket=0;
  strcpy(snmptrap->interfaceAddr,interfaceAddr);
  snmptrap->port=port;
  snmptrap->trace=trace;
  snmptrap->dump=dump;
  snmptrap->echo=echo;
  snmptrap->handlerContext=NULL;
  snmptrap->syntaxContext=NULL;
  snmptrap->attributeContext=NULL;
  nbListenerEnableOnDaemon(context);  // sign up to enable when we daemonize
  return(snmptrap);
  }

/*
*  enable() method
*
*    enable <node>
*/
static int serverEnable(nbCELL context,void *skillHandle,NB_MOD_Snmptrap *snmptrap){
  int fd;
  snmptrap->handlerContext=nbTermLocateHere(context,"handler");
  snmptrap->syntaxContext=nbTermLocateHere(context,"syntax");
  snmptrap->attributeContext=nbTermLocateHere(context,"attribute");
  if((fd=nbIpGetUdpServerSocket(context,snmptrap->interfaceAddr,snmptrap->port))<0){ // 2012-12-27 eat 0.8.13 - CID 751574
    nbLogMsg(context,0,'E',"Unable to listen on port %d\n",snmptrap->port);
    return(1);
    }
  snmptrap->socket=fd;
  nbListenerAdd(context,snmptrap->socket,snmptrap,serverRead);
  nbLogMsg(context,0,'I',"Listening on port %u for SNMP Trap Datagrams",snmptrap->port);
  return(0);
  }

/*
*  disable method
* 
*    disable <node>
*/
static int serverDisable(nbCELL context,void *skillHandle,NB_MOD_Snmptrap *snmptrap){
  nbListenerRemove(context,snmptrap->socket);
#if defined(WIN32)
  closesocket(snmptrap->socket);
#else
  close(snmptrap->socket);
#endif
  snmptrap->socket=0;
  return(0);
  }

/*
*  command() method
*
*    <node>[(<args>)][:<text>]
*/
static int *serverCommand(nbCELL context,void *skillHandle,NB_MOD_Snmptrap *snmptrap,nbCELL arglist,char *text){
  if(snmptrap->trace){
    nbLogMsg(context,0,'T',"nb_snmptrap:serverCommand() text=[%s]\n",text);
    }
  /* insert command parsing code here */
  return(0);
  }


/*
*  destroy() method
*
*    undefine <node>
*/
static int serverDestroy(nbCELL context,void *skillHandle,NB_MOD_Snmptrap *snmptrap){
  nbLogMsg(context,0,'T',"serverDestroy called");
  if(snmptrap->socket!=0) serverDisable(context,skillHandle,snmptrap);
  nbFree(snmptrap,sizeof(NB_MOD_Snmptrap));
  return(0);
  }

#if defined(_WINDOWS)
_declspec (dllexport)
#endif
extern void *snmptrapBind(nbCELL context,void *moduleHandle,nbCELL skill,nbCELL arglist,char *text){
  nbSkillSetMethod(context,skill,NB_NODE_CONSTRUCT,serverConstruct);
  nbSkillSetMethod(context,skill,NB_NODE_DISABLE,serverDisable);
  nbSkillSetMethod(context,skill,NB_NODE_ENABLE,serverEnable);
  nbSkillSetMethod(context,skill,NB_NODE_COMMAND,serverCommand);
  nbSkillSetMethod(context,skill,NB_NODE_DESTROY,serverDestroy);
  return(NULL);
  }

#if defined(_WINDOWS)
_declspec (dllexport)
#endif
extern void *serverBind(nbCELL context,void *moduleHandle,nbCELL skill,nbCELL arglist,char *text){
  nbSkillSetMethod(context,skill,NB_NODE_CONSTRUCT,serverConstruct);
  nbSkillSetMethod(context,skill,NB_NODE_DISABLE,serverDisable);
  nbSkillSetMethod(context,skill,NB_NODE_ENABLE,serverEnable);
  nbSkillSetMethod(context,skill,NB_NODE_COMMAND,serverCommand);
  nbSkillSetMethod(context,skill,NB_NODE_DESTROY,serverDestroy);
  return(NULL);
  }

//=======================================================================================
// Client

typedef struct NB_MOD_CLIENT{      /* SNMP Trap client descriptor */
  unsigned int   socket;           /* client socket for datagrams */
  char           address[16];      /* server address */
  unsigned short port;             /* server port */
  unsigned char  trace;            /* trace option */
  unsigned char  dump;             /* option to dump packets in trace */
  unsigned char  echo;             /* echo option */
  unsigned int   sourceAddr;       /* source address */
  } NB_MOD_Client;

/*
* Translate text '<oid>'="<value>" into variable binding
*
*   Initially we are using a fixed OID to do some debugging on some packets
*   that are causing a problem.
*/
static int translateText2Varbinding(nbCELL context,char **packetCursorP,char **textP){
  char *packetCursor=*packetCursorP,*cursor=*textP,*delim;
  size_t len,lbyte;
  char *varLenP;

  if(*cursor!='\''){
    nbLogMsg(context,0,'E',"Expecting single quote in text at:%s",cursor);
    return(-1);
    }
  cursor++;
  *packetCursor=0x30; packetCursor++; // variable binding
  *packetCursor=0x82; packetCursor++; // two byte length follows
  varLenP=packetCursor;
  packetCursor+=2;
  *packetCursor=0x06; packetCursor++; // OID
  *packetCursor=0x09; packetCursor++; // len=9
  *packetCursor=0x2b; packetCursor++; 
  *packetCursor=0x06; packetCursor++; 
  *packetCursor=0x01; packetCursor++; 
  *packetCursor=0x04; packetCursor++; 
  *packetCursor=0x01; packetCursor++; 
  *packetCursor=0xaf; packetCursor++; 
  *packetCursor=0x55; packetCursor++; 
  *packetCursor=0x81; packetCursor++; 
  *packetCursor=0x0d; packetCursor++; 
  delim=strchr(cursor,'\'');
  if(!delim){ // 2012-12-27 eat 0.8.13 - CID 751623   // 2013-01-13 eat - VID 1093-0.8.13-2
    nbLogMsg(context,0,'E',"Unbalanced single quote at:%s",*textP); // 2012-12-27 eat 0.8.13 - CID 7516
    return(-1);
    }
  cursor=delim+1;
  while(*cursor==' ') cursor++;
  if(*cursor!='='){
    nbLogMsg(context,0,'E',"Expecting '=' at:%s",cursor);
    return(-1);
    }
  cursor++;
  while(*cursor==' ') cursor++;
  if(*cursor!='"'){
    nbLogMsg(context,0,'E',"Expecting double quote at:%s",cursor);
    return(-1);
    }
  cursor++;
  delim=strchr(cursor,'"');
  if(!delim){
    nbLogMsg(context,0,'E',"Unbalanced double quote at:%s",cursor);
    return(-1);
    }
  *packetCursor=0x04; packetCursor++;    // string
  len=delim-cursor;
  lbyte=len;
  if(len>0x7f){
    if(len>0xffff){
      *packetCursor=0x83; packetCursor++;
      *packetCursor=(char )((len>>16)%256); packetCursor++;  // 2013-01-16 eat - VID 4152-0.8.13-4
      lbyte&=0xffff;
      *packetCursor=(char)((len>>8)%256); packetCursor++;
      }
    else if(len>0xff){
      *packetCursor=0x82; packetCursor++;
      *packetCursor=len>8; packetCursor++;
      }
    else *packetCursor=0x81; packetCursor++;
    lbyte&=0xff;
    }
  *packetCursor=lbyte; packetCursor++;
  strncpy(packetCursor,cursor,len);
  packetCursor+=len;
  len=packetCursor-varLenP-2;
  *varLenP=len>>8;
  *(varLenP+1)=len%256;   // 2013-01-16 eat
  *packetCursorP=packetCursor;
  cursor=delim+1;
  *textP=cursor;
  return(0);
  }

/*
* Translate text '<oid>'="<value>",'<oid>'="<value>"... into variable binding
*
*   Initially we are using a fixed OID to do some debugging on some packets
*   that are causing a problem.
*/
static int translateText2VarbindingList(nbCELL context,NB_MOD_Client *client,char **packetCursorP,char *text){
  char *packetCursor=*packetCursorP,*cursor=text;
  int len; 
  char *bindingLenP;

  *packetCursor=0x30; packetCursor++; // variable binding list 
  *packetCursor=0x82; packetCursor++; // 2 byte length follows
  bindingLenP=packetCursor;
  packetCursor+=2;
  while(*cursor!=';' && *cursor!=0){
    if(translateText2Varbinding(context,&packetCursor,&cursor)<0){
      nbLogMsg(context,0,'E',"Syntax error in variable binding list at:%s",cursor);
      return(-1);
      }
    if(*cursor==',') cursor++;
    }
  len=packetCursor-*packetCursorP-4;
  *bindingLenP=len>>8;
  *(bindingLenP+1)=len&0xff;
  len=packetCursor-*packetCursorP;
  *packetCursorP=packetCursor;
  return(len);
  }

/*
* Construct a client object
*/
static void *clientConstruct(nbCELL context,void *skillHandle,nbCELL arglist,char *text){
  NB_MOD_Client *client;
  nbCELL cell=NULL;
  nbSET argSet;
  char *cursor=text,*delim,saveDelim;
  char serverAddr[16];
  unsigned int port=0;
  int type,trace=0,dump=0,echo=1;
  int len;
  char *str;
  int clientSocket=0;

  *serverAddr=0;

  argSet=nbListOpen(context,arglist);
  cell=nbListGetCellValue(context,&argSet);
  if(cell==NULL){
    nbLogMsg(context,0,'E',"Expecting \"address:port\" as first argument");
    nbCellDrop(context,cell);
    return(NULL);
    }
  type=nbCellGetType(context,cell);
  if(type!=NB_TYPE_STRING){
    nbLogMsg(context,0,'E',"Expecting \"address:port\" as first argument");
    nbCellDrop(context,cell);
    return(NULL);
    }
  str=nbCellGetString(context,cell);
  delim=strchr(str,':');
  if(delim==NULL) delim=str+strlen(str);  // 2013-01-13 eat - VID 973-0.8.13-3 FP but replaced strchr(str,0)
  len=delim-str;
  if(len>15){
    nbLogMsg(context,0,'E',"Inteface IP address may not be greater than 15 characters");
    nbCellDrop(context,cell);
    return(NULL);
    }
  strncpy(serverAddr,str,len);
  *(serverAddr+len)=0;
  if(!*delim) port=162; // default port 
  else{
    delim++;
    port=(unsigned int)atoi(delim);
    }
  nbCellDrop(context,cell);
  cell=nbListGetCellValue(context,&argSet);
  if(cell!=NULL){
    nbLogMsg(context,0,'E',"The client skill only accepts one argument.");
    return(NULL);
    }
  while(*cursor==' ') cursor++;
  while(*cursor!=';' && *cursor!=0){
    delim=strchr(cursor,' ');
    if(delim==NULL) delim=strchr(cursor,',');
    if(delim==NULL) delim=strchr(cursor,';');
    if(delim==NULL) delim=cursor+strlen(cursor); // 2013-01-13 eat - VID 982-0.8.13-3 FP but replaced strchr(cursor,0)
    saveDelim=*delim;
    *delim=0;
    if(strcmp(cursor,"trace")==0){trace=1;}
    else if(strcmp(cursor,"dump")==0){trace=1;dump=1;}
    else if(strcmp(cursor,"silent")==0) echo=0;
    *delim=saveDelim;
    cursor=delim;
    if(*cursor==',') cursor++;
    while(*cursor==' ') cursor++;
    }
  if((clientSocket=nbIpGetUdpClientSocket(0,serverAddr,port))<0){
    nbLogMsg(context,0,'E',"Unable to obtain client UDP socket %s:%d",serverAddr,port);
    return(NULL);
    }
  client=nbAlloc(sizeof(NB_MOD_Client));
  client->socket=clientSocket;
  strcpy(client->address,serverAddr);
  client->port=port;
  client->trace=trace;
  client->dump=dump;
  client->echo=echo;
  return(client);
  }

/*  enable() method
*
*    enable <node>
*/
static int clientEnable(nbCELL context,void *skillHandle,NB_MOD_Client *client){
  int fd;
  if(client->socket!=0) return(0);
  if((fd=nbIpGetUdpClientSocket(0,client->address,client->port))<0){  // 2012-12-27 eat 0.8.13 - CID 751573
    nbLogMsg(context,0,'E',"Unable to obtain client UDP socket %s:%d\n",client->address,client->port);
    return(1);
    }
  client->socket=fd;
  return(0);
  }

/*
*  disable method
*
*    disable <node>
*/
static int clientDisable(nbCELL context,void *skillHandle,NB_MOD_Client *client){
  if(client->socket==0) return(0);
#if defined(WIN32)
  closesocket(client->socket);
#else
  close(client->socket);
#endif
  client->socket=0;
  return(0);
  }

/*
*  command() method
*
*    <node>[(<args>)][:<text>]
*
*  The intial version of this is hard coded to generate a packet for
*  debugging the server.  Will add reverse translation (assertion to packet)
*  function later.
*/
static int clientCommand(nbCELL context,void *skillHandle,NB_MOD_Client *client,nbCELL arglist,char *text){
  char packet[NB_BUFSIZE],*packetCursor=packet;
  int len,sent;
  char *packetLenP,*trapLenP;

  if(client->trace) nbLogMsg(context,0,'T',"clientCommand() text=[%s]",text);

  *packetCursor=0x30; packetCursor++;
  *packetCursor=0x82; packetCursor++; // 2 byte packet length follows
  packetLenP=packetCursor;
  packetCursor+=2;     // will come back and insert length
  *packetCursor=0x02; packetCursor++;
  *packetCursor=0x01; packetCursor++;
  *packetCursor=0x00; packetCursor++; // version number - snmpV1
  *packetCursor=0x04; packetCursor++; // String
  *packetCursor=0x06; packetCursor++; // len=6
  strcpy(packetCursor,"public");      // community string
  packetCursor+=strlen(packetCursor);         // 2013-01-14 eat - VID 964-0.8.13-3 FP but replaced strchr(packetCursor,0)
  *packetCursor=0xA4; packetCursor++; // 0xA4 for V1 trap
  *packetCursor=0x82; packetCursor++; // 2 byte packet length follows
  trapLenP=packetCursor;
  packetCursor+=2;     // will come back and insert length
  *packetCursor=0x06; packetCursor++; // OID   
  *packetCursor=0x09; packetCursor++; // len=9
  *packetCursor=0x2b; packetCursor++; 
  *packetCursor=0x06; packetCursor++; 
  *packetCursor=0x01; packetCursor++; 
  *packetCursor=0x04; packetCursor++; 
  *packetCursor=0x01; packetCursor++; 
  *packetCursor=0xaf; packetCursor++; 
  *packetCursor=0x55; packetCursor++; 
  *packetCursor=0x81; packetCursor++; 
  *packetCursor=0x0d; packetCursor++; 
  
  *packetCursor=0x40; packetCursor++; // address
  *packetCursor=0x04; packetCursor++; // len=4 
  *packetCursor=0x81; packetCursor++;  
  *packetCursor=0xac; packetCursor++;  
  *packetCursor=0x0d; packetCursor++;  
  *packetCursor=0x05; packetCursor++;  

  *packetCursor=0x02; packetCursor++; // integer
  *packetCursor=0x01; packetCursor++; // len=1 
  *packetCursor=0x06; packetCursor++; // trap generic type 6
  *packetCursor=0x02; packetCursor++; // integer
  *packetCursor=0x01; packetCursor++; // len=1 
  *packetCursor=0x00; packetCursor++; // trap specific type 1
  *packetCursor=0x43; packetCursor++; // uptime
  *packetCursor=0x04; packetCursor++; // len=4 
  *packetCursor=0xf6; packetCursor++; 
  *packetCursor=0x3c; packetCursor++; 
  *packetCursor=0xe5; packetCursor++; 
  *packetCursor=0xa5; packetCursor++; 
  
  if(translateText2VarbindingList(context,client,&packetCursor,text)<0) return(1); 
  len=packetCursor-trapLenP-2;
  *trapLenP=len>>8;
  *(trapLenP+1)=len&0xff;
  len=packetCursor-packetLenP-2;
  *packetLenP=len>>8;
  *(packetLenP+1)=len&0xff;
  len=packetCursor-packet;
  sent=send(client->socket,packet,len,0);
  if(sent<0){
    perror("send failed");
    return(1);
    }
  return(0);
  }

/*
*  destroy() method
*
*    undefine <node>
*/
static int clientDestroy(nbCELL context,void *skillHandle,NB_MOD_Client *client){
  nbLogMsg(context,0,'T',"clientDestroy called");
  if(client->socket!=0) clientDisable(context,skillHandle,client);
  nbFree(client,sizeof(NB_MOD_Client));
  return(0);
  }

#if defined(_WINDOWS)
_declspec (dllexport)
#endif
extern void *clientBind(nbCELL context,void *moduleHandle,nbCELL skill,nbCELL arglist,char *text){
  nbSkillSetMethod(context,skill,NB_NODE_CONSTRUCT,clientConstruct);
  nbSkillSetMethod(context,skill,NB_NODE_DISABLE,clientDisable);
  nbSkillSetMethod(context,skill,NB_NODE_ENABLE,clientEnable);
  nbSkillSetMethod(context,skill,NB_NODE_COMMAND,clientCommand);
  nbSkillSetMethod(context,skill,NB_NODE_DESTROY,clientDestroy);
  return(NULL);
  }

