/*
* Copyright (C) 2014 Ed Trettevik <eat@nodebrain.org>
*
* NodeBrain is free software; you can modify and/or redistribute it under the
* terms of either the MIT License (Expat) or the following NodeBrain License.
*
* Permission to use and redistribute with or without fee, in source and binary
* forms, with or without modification, is granted free of charge to any person
* obtaining a copy of this software and included documentation, provided that
* the above copyright notice, this permission notice, and the following
* disclaimer are retained with source files and reproduced in documention
* included with source and binary distributions. 
*
* Unless required by applicable law or agreed to in writing, this software is
* distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
* KIND, either express or implied.
*
*=============================================================================
* Program:  NodeBrain API Test Suite
*
* File:     lib/test/eStatePublisher.c 
*
* Title:    API Test - A node that publishes the value of specified cells.
*
* Category: Example - Short illustrations using sparse C code formatting.
*
* Purpose:
*
*   This file exercises NodeBrain API functions that implement a node that
*   accesses specified cells and reports their names and values.
*
* See NodeBrain Library manual for a description of each API function.
*
* Description:
*
*   In this example, we create a node for publishing the names and values of 
*   specified cells.  Here we simply create a string and write it to the log.
*   In a real application the state might be reported to an external system
*   via a protocol of choice.
*
*   Our node skill only implements an assertion method.  If one were to add
*   a publishing method more interesting that writing to the log, a construct
*   method would be appropriate for initialization the communication, and a
*   node knowledge structure would be included to manage the communication.
*
*   The node skill created here is "publisher" and we define a "Publish" node
*   with this skill.
*
*     define Publish node publisher;
*
*   Assertions to a publisher node are of the following form.
*
*     Publish(<nodeTerm>,<topic>,cell1,cell2,...)
*
*   Example:
*
*     Publish(foo,"BigA",a,b,c)
*
*
*   If a=1, b=2, and c=3, the output will be as follows.
*
*     Publishing==> _.foo/"BigA",a=1,b=2,c=3;
*   
*=============================================================================
* Change History:
*
* Date       Name/Change
* ---------- -----------------------------------------------------------------
* 2014-12-22 Ed Trettevik - introduced in version 0.9.04
*=============================================================================
*/
#include <nb/nb.h>

#define TEST( TITLE ) nbLogPut( context, "\nTEST: line %5d - %s\n", __LINE__, TITLE )


/*
*  Publisher assertion handler
*
*    publisher(node,rule,term2,...);
*
*    topic - only the value is used from the topic argument
*    term1 - term to publish with value; e.g. a=7
*    term2 - second term to publish; e.g. b="happy day"
*
*  Example:
*
*    publisher(foo,BigA,a,b,c);
*
*    If a, b, and c are 3, "horse", and 27 respectively, we output:
*
*       foo/BigA: a=3,b="horse",c=27;
*
*  Note:
*
*    The argument cells do not have to be terms.  They can be more interesting formulas.
*/
static int publisherAssert( nbCELL context, void *skillHandle, void *knowledgeHandle, nbCELL arglist, nbCELL value )
{
        int size=4096;
        char buffer[size];
        char *cursor=buffer;
	nbSET argSet;
	nbCELL cell,defCell,contextCell;
        int cellType;
        char comma = ' ';

	argSet = nbListOpen( context, arglist );
	cell = nbListGetCell( context, &argSet );
	if(!cell || ( cellType = nbCellGetType( context, cell ) ) != NB_TYPE_TERM ||
		( defCell = nbTermGetDefinition( context, cell ) ) == NULL ||
		( cellType = nbCellGetType( context, defCell ) ) != NB_TYPE_NODE )
	{
		nbLogMsg( context, 0, 'T', "Node %s assertion expects a node as the first argument.",nbNodeGetName( context ) );
		return( 0 );
	}
       	size = nbCellGetName( context, cell, &cursor, size );  // get the value of the first argument cell as the topic
	if ( size > 1 ) *cursor='/', cursor++;
	size--;
	contextCell = cell;
	nbCellDrop( context, cell );
	cell = nbListGetCell( context, &argSet );
	if(!cell)
	{
		nbLogMsg( context, 0, 'T', "Node %s assertion expects a cell with a topical value as the second argument.",nbNodeGetName( context ) );
		return( 0 );
	}
       	size = nbCellGetValueName( context, cell, &cursor, size );  // get the value of the first argument cell as the topic
	if ( size > 1 ) *cursor=':', cursor++;
	size--;
	nbCellDrop( context, cell );
	while ( ( cell = nbListGetCell( context, &argSet ) ) )	// step through argument cells, starting with second argument
	{
		if( size > 1 ) *cursor=comma, cursor++;
		size--;
                size = nbCellGetName( contextCell, cell, &cursor, size ); // get name relative to argument node
                if( size > 1 ) *cursor='=', cursor++;
                size--;
        	size = nbCellGetValueName( contextCell, cell, &cursor, size ); // get name relative to argument node
		comma = ',';
	}
	if ( size > 1 ) *cursor=';', cursor++;
	size--;
	if ( size > 0 ) *cursor=0, cursor++;
	size--;
	if ( size > 0 ) nbLogPut( context, "Publishing==> %s\n", buffer );
	else nbLogPut( context, "Node %s assertion creates string %d characters too long for buffer",nbNodeGetName( context ), size );
	return ( 0 ); 
}

static void *publisherBind( nbCELL context, void *moduleHandle, nbCELL skill, nbCELL arglist, char *text )
{
	nbSkillSetMethod( context, skill, NB_NODE_ASSERT, publisherAssert );	// registers publisherAssert as what is executed when term() is asserted, where term is a publisher node
	return ( NULL );
}

int main( int argc, char *argv[] )
{
	nbCELL context;

	context = nbStart( argc, argv );

	TEST( "nbSkillDeclare - declare a new skill called publisher" );
	nbSkillDeclare( context, publisherBind, NULL, "", "publisher", NULL, "" );// "publisher" is the skill name, and is associated with publisherBind 

	TEST( "nbCmd - define a node that uses the skill" );
	nbCmd( context, "define Publish node publisher;", NB_CMDOPT_ECHO );	// creates a node named Publish with publisher skill provided by nbSkillDeclare & publicherBind

	TEST( "nbCmd - create a node with cells to publish and a rule to publish them" );
	nbCmd( context, "foo. define a cell 1;", NB_CMDOPT_ECHO );
	nbCmd( context, "foo. define b cell 2;", NB_CMDOPT_ECHO );
	nbCmd( context, "foo. define c cell 3;", NB_CMDOPT_ECHO );
        nbCmd( context, "foo. define BigA if(a>20) Publish(foo,\"BigA\",a,b,c);", NB_CMDOPT_ECHO );
        
	TEST( "Alert without expecting anything to publish" );
	nbCmd( context, "foo. alert a=1,b=2,c=3;", NB_CMDOPT_ECHO );
	TEST( "Alert with expectation that we will publish" );
	nbCmd( context, "foo. alert a=21,b=3,c=40;", NB_CMDOPT_ECHO );
	TEST( "Alert without expecting anything to publish" );
	nbCmd( context, "foo. alert a=1,b=2,c=3;", NB_CMDOPT_ECHO );
	TEST( "Alert with expectation that we will publish" );
	nbCmd( context, "foo. alert a=79,b=7,c=9;", NB_CMDOPT_ECHO );

	return ( nbStop( context ) );
}
