/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.dfs.DistributedFileSystem;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.Trash;
import org.apache.hadoop.ipc.RPC;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.util.ToolBase;

public class FsShell
extends ToolBase {
    protected FileSystem fs;
    private Trash trash;

    public void init() throws IOException {
        this.conf.setQuietMode(true);
        this.fs = FileSystem.get(this.conf);
        this.trash = new Trash(this.conf);
    }

    private void copyBytes(InputStream in, OutputStream out) throws IOException {
        PrintStream ps = out instanceof PrintStream ? (PrintStream)out : null;
        byte[] buf = new byte[this.conf.getInt("io.file.buffer.size", 4096)];
        int bytesRead = in.read(buf);
        while (bytesRead >= 0) {
            out.write(buf, 0, bytesRead);
            if (ps != null && ps.checkError()) {
                throw new IOException("Unable to write to output stream.");
            }
            bytesRead = in.read(buf);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyFromStdin(Path dst) throws IOException {
        if (this.fs.isDirectory(dst)) {
            throw new IOException("When source is stdin, destination must be a file.");
        }
        if (this.fs.exists(dst)) {
            throw new IOException("Target " + dst.toString() + " already exists.");
        }
        FSDataOutputStream out = this.fs.create(dst);
        try {
            this.copyBytes(System.in, out);
        }
        finally {
            out.close();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void printToStdout(Path src) throws IOException {
        if (this.fs.isDirectory(src)) {
            throw new IOException("Source must be a file.");
        }
        FSDataInputStream in = this.fs.open(src);
        try {
            this.copyBytes(in, System.out);
        }
        finally {
            in.close();
        }
    }

    void copyFromLocal(Path src, String dstf) throws IOException {
        if (src.toString().equals("-")) {
            this.copyFromStdin(new Path(dstf));
        } else {
            this.fs.copyFromLocalFile(src, new Path(dstf));
        }
    }

    void moveFromLocal(Path src, String dstf) throws IOException {
        this.fs.moveFromLocalFile(src, new Path(dstf));
    }

    void copyToLocal(String[] argv, int pos) throws IOException {
        if (argv.length - pos < 2 || argv.length - pos == 2 && argv[pos].equalsIgnoreCase("-crc")) {
            System.err.println("Usage: -get [-crc] <src> <dst>");
            System.exit(-1);
        }
        boolean copyCrc = false;
        if ("-crc".equalsIgnoreCase(argv[pos])) {
            copyCrc = true;
        }
        int n = ++pos;
        String srcf = argv[n];
        int n2 = ++pos;
        ++pos;
        String dstf = argv[n2];
        if (dstf.equals("-")) {
            if (copyCrc) {
                System.err.println("-crc option is not valid when destination is stdout.");
            }
            this.cat(srcf);
        } else {
            Path[] srcs = this.fs.globPaths(new Path(srcf));
            if (srcs.length > 1 && !new File(dstf).isDirectory()) {
                throw new IOException("When copying multiple files, destination should be a directory.");
            }
            Path dst = new Path(dstf);
            for (int i = 0; i < srcs.length; ++i) {
                ((DistributedFileSystem)this.fs).copyToLocalFile(srcs[i], dst, copyCrc);
            }
        }
    }

    void copyMergeToLocal(String srcf, Path dst) throws IOException {
        this.copyMergeToLocal(srcf, dst, false);
    }

    void copyMergeToLocal(String srcf, Path dst, boolean endline) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        for (int i = 0; i < srcs.length; ++i) {
            if (endline) {
                FileUtil.copyMerge(this.fs, srcs[i], FileSystem.getLocal(this.conf), dst, false, this.conf, "\n");
                continue;
            }
            FileUtil.copyMerge(this.fs, srcs[i], FileSystem.getLocal(this.conf), dst, false, this.conf, null);
        }
    }

    void moveToLocal(String srcf, Path dst) throws IOException {
        System.err.println("Option '-moveToLocal' is not implemented yet.");
    }

    void cat(String srcf) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        for (int i = 0; i < srcs.length; ++i) {
            this.printToStdout(srcs[i]);
        }
    }

    private void setReplication(String[] cmd, int pos) throws IOException {
        if (cmd.length - pos < 2 || cmd.length - pos == 2 && cmd[pos].equalsIgnoreCase("-R")) {
            System.err.println("Usage: [-R] <repvalue> <path>");
            System.exit(-1);
        }
        boolean recursive = false;
        short rep = 3;
        if ("-R".equalsIgnoreCase(cmd[pos])) {
            recursive = true;
            ++pos;
        }
        try {
            rep = Short.parseShort(cmd[pos]);
            ++pos;
        }
        catch (NumberFormatException e) {
            System.err.println("Cannot set replication to: " + cmd[pos]);
            System.exit(-1);
        }
        this.setReplication(rep, cmd[pos], recursive);
    }

    public void setReplication(short newRep, String srcf, boolean recursive) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        for (int i = 0; i < srcs.length; ++i) {
            this.setReplication(newRep, srcs[i], recursive);
        }
    }

    private void setReplication(short newRep, Path src, boolean recursive) throws IOException {
        if (!this.fs.isDirectory(src)) {
            this.setFileReplication(src, newRep);
            return;
        }
        Path[] items = this.fs.listPaths(src);
        if (items == null) {
            throw new IOException("Could not get listing for " + src);
        }
        for (int i = 0; i < items.length; ++i) {
            Path cur = items[i];
            if (!this.fs.isDirectory(cur)) {
                this.setFileReplication(cur, newRep);
                continue;
            }
            if (!recursive) continue;
            this.setReplication(newRep, cur, recursive);
        }
    }

    private void setFileReplication(Path file, short newRep) throws IOException {
        if (this.fs.setReplication(file, newRep)) {
            System.out.println("Replication " + newRep + " set: " + file);
        } else {
            System.err.println("Could not set replication for: " + file);
        }
    }

    public void ls(String srcf, boolean recursive) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        boolean printHeader = srcs.length == 1;
        for (int i = 0; i < srcs.length; ++i) {
            this.ls(srcs[i], recursive, printHeader);
        }
    }

    private void ls(Path src, boolean recursive, boolean printHeader) throws IOException {
        Path[] items = this.fs.listPaths(src);
        if (items == null) {
            throw new IOException("Could not get listing for " + src);
        }
        if (!recursive && printHeader) {
            System.out.println("Found " + items.length + " items");
        }
        for (int i = 0; i < items.length; ++i) {
            Path cur = items[i];
            System.out.println(cur + "\t" + (this.fs.isDirectory(cur) ? "<dir>" : "<r " + this.fs.getReplication(cur) + ">\t" + this.fs.getLength(cur)));
            if (!recursive || !this.fs.isDirectory(cur)) continue;
            this.ls(cur, recursive, printHeader);
        }
    }

    public void du(String src) throws IOException {
        Path[] items = this.fs.listPaths(this.fs.globPaths(new Path(src)));
        if (items == null) {
            throw new IOException("Could not get listing for " + src);
        }
        System.out.println("Found " + items.length + " items");
        for (int i = 0; i < items.length; ++i) {
            Path cur = items[i];
            System.out.println(cur + "\t" + this.fs.getContentLength(cur));
        }
    }

    public void dus(String src) throws IOException {
        Path[] paths = this.fs.globPaths(new Path(src));
        if (paths == null || paths.length == 0) {
            throw new IOException("dus: No match: " + src);
        }
        for (int i = 0; i < paths.length; ++i) {
            Path[] items = this.fs.listPaths(paths[i]);
            if (items == null) continue;
            long totalSize = 0L;
            for (int j = 0; j < items.length; ++j) {
                totalSize += this.fs.getContentLength(items[j]);
            }
            String pathStr = paths[i].toString();
            System.out.println(("".equals(pathStr) ? "." : pathStr) + "\t" + totalSize);
        }
    }

    public void mkdir(String src) throws IOException {
        Path f = new Path(src);
        if (!this.fs.mkdirs(f)) {
            throw new IOException("Mkdirs failed to create " + src);
        }
    }

    public void rename(String srcf, String dstf) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        Path dst = new Path(dstf);
        if (srcs.length > 1 && !this.fs.isDirectory(dst)) {
            throw new IOException("When moving multiple files, destination should be a directory.");
        }
        for (int i = 0; i < srcs.length; ++i) {
            if (!this.fs.rename(srcs[i], dst)) {
                throw new IOException("Rename failed " + srcs[i]);
            }
            System.out.println("Renamed " + srcs[i] + " to " + dstf);
        }
    }

    private int rename(String[] argv, Configuration conf) throws IOException {
        Path dst;
        int i = 0;
        int exitCode = 0;
        String cmd = argv[i++];
        String dest = argv[argv.length - 1];
        if (argv.length > 3 && !this.fs.isDirectory(dst = new Path(dest))) {
            throw new IOException("When moving multiple files, destination " + dest + " should be a directory.");
        }
        while (i < argv.length - 1) {
            try {
                this.rename(argv[i], dest);
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
            ++i;
        }
        return exitCode;
    }

    public void copy(String srcf, String dstf, Configuration conf) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        Path dst = new Path(dstf);
        if (srcs.length > 1 && !this.fs.isDirectory(dst)) {
            throw new IOException("When copying multiple files, destination should be a directory.");
        }
        for (int i = 0; i < srcs.length; ++i) {
            FileUtil.copy(this.fs, srcs[i], this.fs, dst, false, conf);
        }
    }

    private int copy(String[] argv, Configuration conf) throws IOException {
        Path dst;
        int i = 0;
        int exitCode = 0;
        String cmd = argv[i++];
        String dest = argv[argv.length - 1];
        if (argv.length > 3 && !this.fs.isDirectory(dst = new Path(dest))) {
            throw new IOException("When copying multiple files, destination " + dest + " should be a directory.");
        }
        while (i < argv.length - 1) {
            try {
                this.copy(argv[i], dest, conf);
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
            ++i;
        }
        return exitCode;
    }

    public void delete(String srcf, boolean recursive) throws IOException {
        Path[] srcs = this.fs.globPaths(new Path(srcf));
        for (int i = 0; i < srcs.length; ++i) {
            this.delete(srcs[i], recursive);
        }
    }

    private void delete(Path src, boolean recursive) throws IOException {
        if (this.fs.isDirectory(src) && !recursive) {
            throw new IOException("Cannot remove directory \"" + src + "\", use -rmr instead");
        }
        if (this.trash.moveToTrash(src)) {
            System.out.println("Moved to trash: " + src);
            return;
        }
        if (!this.fs.delete(src)) {
            throw new IOException("Delete failed " + src);
        }
        System.out.println("Deleted " + src);
    }

    private void expunge() throws IOException {
        this.trash.expunge();
        this.trash.checkpoint();
    }

    public static String byteDesc(long len) {
        double val = 0.0;
        String ending = "";
        if (len < 0x100000L) {
            val = 1.0 * (double)len / 1024.0;
            ending = " k";
        } else if (len < 0x40000000L) {
            val = 1.0 * (double)len / 1048576.0;
            ending = " MB";
        } else if (len < 0x2000000000L) {
            val = 1.0 * (double)len / 1.073741824E9;
            ending = " GB";
        } else if (len < 0x4000000000000L) {
            val = 1.0 * (double)len / 1.099511627776E12;
            ending = " TB";
        } else {
            val = 1.0 * (double)len / 1.125899906842624E15;
            ending = " PB";
        }
        return FsShell.limitDecimal(val, 2) + ending;
    }

    public static String limitDecimal(double d, int placesAfterDecimal) {
        String strVal = Double.toString(d);
        int decpt = strVal.indexOf(".");
        if (decpt >= 0) {
            strVal = strVal.substring(0, Math.min(strVal.length(), decpt + 1 + placesAfterDecimal));
        }
        return strVal;
    }

    private void printHelp(String cmd) {
        String summary = "hadoop fs is the command to execute fs commands. The full syntax is: \n\nhadoop fs [-fs <local | file system URI>] [-conf <configuration file>]\n\t[-D <property=value>] [-ls <path>] [-lsr <path>] [-du <path>]\n\t[-dus <path>] [-mv <src> <dst>] [-cp <src> <dst>] [-rm <src>]\n\t[-rmr <src>] [-put <localsrc> <dst>] [-copyFromLocal <localsrc> <dst>]\n\t[-moveFromLocal <localsrc> <dst>] [-get <src> <localdst>]\n\t[-getmerge <src> <localdst> [addnl]] [-cat <src>]\n\t[-copyToLocal <src><localdst>] [-moveToLocal <src> <localdst>]\n\t[-mkdir <path>] [-report] [-setrep [-R] <rep> <path/file>]\n[-help [cmd]]\n";
        String conf = "-conf <configuration file>:  Specify an application configuration file.";
        String D = "-D <property=value>:  Use value for given property.";
        String fs = "-fs [local | <file system URI>]: \tSpecify the file system to use.\n\t\tIf not specified, the current configuration is used, \n\t\ttaken from the following, in increasing precedence: \n\t\t\thadoop-default.xml inside the hadoop jar file \n\t\t\thadoop-default.xml in $HADOOP_CONF_DIR \n\t\t\thadoop-site.xml in $HADOOP_CONF_DIR \n\t\t'local' means use the local file system as your DFS. \n\t\t<file system URI> specifies a particular file system to \n\t\tcontact. This argument is optional but if used must appear\n\t\tappear first on the command line.  Exactly one additional\n\t\targument must be specified. \n";
        String ls = "-ls <path>: \tList the contents that match the specified file pattern. If\n\t\tpath is not specified, the contents of /user/<currentUser>\n\t\twill be listed. Directory entries are of the form \n\t\t\tdirName (full path) <dir> \n\t\tand file entries are of the form \n\t\t\tfileName(full path) <r n> size \n\t\twhere n is the number of replicas specified for the file \n\t\tand size is the size of the file, in bytes.\n";
        String lsr = "-lsr <path>: \tRecursively list the contents that match the specified\n\t\tfile pattern.  Behaves very similarly to hadoop fs -ls,\n\t\texcept that the data is shown for all the entries in the\n\t\tsubtree.\n";
        String du = "-du <path>: \tShow the amount of space, in bytes, used by the files that \n\t\tmatch the specified file pattern.  Equivalent to the unix\n\t\tcommand \"du -sb <path>/*\" in case of a directory, \n\t\tand to \"du -b <path>\" in case of a file.\n\t\tThe output is in the form \n\t\t\tname(full path) size (in bytes)\n";
        String dus = "-dus <path>: \tShow the amount of space, in bytes, used by the files that \n\t\tmatch the specified file pattern.  Equivalent to the unix\n\t\tcommand \"du -sb\"  The output is in the form \n\t\t\tname(full path) size (in bytes)\n";
        String mv = "-mv <src> <dst>:   Move files that match the specified file pattern <src>\n\t\tto a destination <dst>.  When moving multiple files, the \n\t\tdestination must be a directory. \n";
        String cp = "-cp <src> <dst>:   Copy files that match the file pattern <src> to a \n\t\tdestination.  When copying multiple files, the destination\n\t\tmust be a directory. \n";
        String rm = "-rm <src>: \tDelete all files that match the specified file pattern.\n\t\tEquivlent to the Unix command \"rm <src>\"\n";
        String rmr = "-rmr <src>: \tRemove all directories which match the specified file \n\t\tpattern. Equivlent to the Unix command \"rm -rf <src>\"\n";
        String put = "-put <localsrc> <dst>: \tCopy a single file from the local file system \n\t\tinto fs. \n";
        String copyFromLocal = "-copyFromLocal <localsrc> <dst>:  Identical to the -put command.\n";
        String moveFromLocal = "-moveFromLocal <localsrc> <dst>:  Same as -put, except that the source is\n\t\tdeleted after it's copied.\n";
        String get = "-get <src> <localdst>:  Copy files that match the file pattern <src> \n\t\tto the local name.  <src> is kept.  When copying mutiple, \n\t\tfiles, the destination must be a directory. \n";
        String getmerge = "-getmerge <src> <localdst>:  Get all the files in the directories that \n\t\tmatch the source file pattern and merge and sort them to only\n\t\tone file on local fs. <src> is kept.\n";
        String cat = "-cat <src>: \tFetch all files that match the file pattern <src> \n\t\tand display their content on stdout.\n";
        String copyToLocal = "-copyToLocal <src> <localdst>:  Identical to the -get command.\n";
        String moveToLocal = "-moveToLocal <src> <localdst>:  Not implemented yet \n";
        String mkdir = "-mkdir <path>: \tCreate a directory in specified location. \n";
        String setrep = "-setrep [-R] <rep> <path/file>:  Set the replication level of a file. \n\t\tThe -R flag requests a recursive change of replication level \n\t\tfor an entire tree.\n";
        String help = "-help [cmd]: \tDisplays help for given command or all commands if none\n\t\tis specified.\n";
        if ("fs".equals(cmd)) {
            System.out.println(fs);
        } else if ("conf".equals(cmd)) {
            System.out.println(conf);
        } else if ("D".equals(cmd)) {
            System.out.println(D);
        } else if ("ls".equals(cmd)) {
            System.out.println(ls);
        } else if ("lsr".equals(cmd)) {
            System.out.println(lsr);
        } else if ("du".equals(cmd)) {
            System.out.println(du);
        } else if ("dus".equals(cmd)) {
            System.out.println(dus);
        } else if ("rm".equals(cmd)) {
            System.out.println(rm);
        } else if ("rmr".equals(cmd)) {
            System.out.println(rmr);
        } else if ("mkdir".equals(cmd)) {
            System.out.println(mkdir);
        } else if ("mv".equals(cmd)) {
            System.out.println(mv);
        } else if ("cp".equals(cmd)) {
            System.out.println(cp);
        } else if ("put".equals(cmd)) {
            System.out.println(put);
        } else if ("copyFromLocal".equals(cmd)) {
            System.out.println(copyFromLocal);
        } else if ("moveFromLocal".equals(cmd)) {
            System.out.println(moveFromLocal);
        } else if ("get".equals(cmd)) {
            System.out.println(get);
        } else if ("getmerge".equals(cmd)) {
            System.out.println(getmerge);
        } else if ("copyToLocal".equals(cmd)) {
            System.out.println(copyToLocal);
        } else if ("moveToLocal".equals(cmd)) {
            System.out.println(moveToLocal);
        } else if ("cat".equals(cmd)) {
            System.out.println(cat);
        } else if ("get".equals(cmd)) {
            System.out.println(get);
        } else if ("setrep".equals(cmd)) {
            System.out.println(setrep);
        } else if ("help".equals(cmd)) {
            System.out.println(help);
        } else {
            System.out.println(summary);
            System.out.println(fs);
            System.out.println(ls);
            System.out.println(lsr);
            System.out.println(du);
            System.out.println(dus);
            System.out.println(mv);
            System.out.println(cp);
            System.out.println(rm);
            System.out.println(rmr);
            System.out.println(put);
            System.out.println(copyFromLocal);
            System.out.println(moveFromLocal);
            System.out.println(get);
            System.out.println(getmerge);
            System.out.println(cat);
            System.out.println(copyToLocal);
            System.out.println(moveToLocal);
            System.out.println(mkdir);
            System.out.println(setrep);
            System.out.println(help);
        }
    }

    private int doall(String cmd, String[] argv, Configuration conf, int startindex) {
        int exitCode = 0;
        for (int i = startindex; i < argv.length; ++i) {
            try {
                if ("-cat".equals(cmd)) {
                    this.cat(argv[i]);
                    continue;
                }
                if ("-mkdir".equals(cmd)) {
                    this.mkdir(argv[i]);
                    continue;
                }
                if ("-rm".equals(cmd)) {
                    this.delete(argv[i], false);
                    continue;
                }
                if ("-rmr".equals(cmd)) {
                    this.delete(argv[i], true);
                    continue;
                }
                if ("-du".equals(cmd)) {
                    this.du(argv[i]);
                    continue;
                }
                if ("-dus".equals(cmd)) {
                    this.dus(argv[i]);
                    continue;
                }
                if ("-ls".equals(cmd)) {
                    this.ls(argv[i], false);
                    continue;
                }
                if (!"-lsr".equals(cmd)) continue;
                this.ls(argv[i], true);
                continue;
            }
            catch (RemoteException e) {
                exitCode = -1;
                try {
                    String[] content = e.getLocalizedMessage().split("\n");
                    System.err.println(cmd.substring(1) + ": " + content[0]);
                }
                catch (Exception ex) {
                    System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
                }
                continue;
            }
            catch (IOException e) {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
            }
        }
        return exitCode;
    }

    public void printUsage(String cmd) {
        if ("-fs".equals(cmd)) {
            System.err.println("Usage: java FsShell [-fs <local | file system URI>]");
        } else if ("-conf".equals(cmd)) {
            System.err.println("Usage: java FsShell [-conf <configuration file>]");
        } else if ("-D".equals(cmd)) {
            System.err.println("Usage: java FsShell [-D <[property=value>]");
        } else if ("-ls".equals(cmd) || "-lsr".equals(cmd) || "-du".equals(cmd) || "-dus".equals(cmd) || "-rm".equals(cmd) || "-rmr".equals(cmd) || "-mkdir".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <path>]");
        } else if ("-mv".equals(cmd) || "-cp".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <src> <dst>]");
        } else if ("-put".equals(cmd) || "-copyFromLocal".equals(cmd) || "-moveFromLocal".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <localsrc> <dst>]");
        } else if ("-get".equals(cmd) || "-copyToLocal".equals(cmd) || "-moveToLocal".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " [-crc] <src> <localdst>]");
        } else if ("-cat".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <src>]");
        } else if ("-get".equals(cmd)) {
            System.err.println("Usage: java FsShell [" + cmd + " <src> <localdst> [addnl]]");
        } else if ("-setrep".equals(cmd)) {
            System.err.println("Usage: java FsShell [-setrep [-R] <rep> <path/file>]");
        } else {
            System.err.println("Usage: java FsShell");
            System.err.println("           [-fs <local | file system URI>]");
            System.err.println("           [-conf <configuration file>]");
            System.err.println("           [-D <[property=value>]");
            System.err.println("           [-ls <path>]");
            System.err.println("           [-lsr <path>]");
            System.err.println("           [-du <path>]");
            System.err.println("           [-dus <path>]");
            System.err.println("           [-mv <src> <dst>]");
            System.err.println("           [-cp <src> <dst>]");
            System.err.println("           [-rm <path>]");
            System.err.println("           [-rmr <path>]");
            System.err.println("           [-expunge]");
            System.err.println("           [-put <localsrc> <dst>]");
            System.err.println("           [-copyFromLocal <localsrc> <dst>]");
            System.err.println("           [-moveFromLocal <localsrc> <dst>]");
            System.err.println("           [-get [-crc] <src> <localdst>]");
            System.err.println("           [-getmerge <src> <localdst> [addnl]]");
            System.err.println("           [-cat <src>]");
            System.err.println("           [-copyToLocal [-crc] <src> <localdst>]");
            System.err.println("           [-moveToLocal [-crc] <src> <localdst>]");
            System.err.println("           [-mkdir <path>]");
            System.err.println("           [-setrep [-R] <rep> <path/file>]");
            System.err.println("           [-help [cmd]]");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int run(String[] argv) throws Exception {
        String cmd;
        if (argv.length < 1) {
            this.printUsage("");
            return -1;
        }
        int exitCode = -1;
        int i = 0;
        if ("-put".equals(cmd = argv[i++]) || "-copyFromLocal".equals(cmd) || "-moveFromLocal".equals(cmd)) {
            if (argv.length != 3) {
                this.printUsage(cmd);
                return exitCode;
            }
        } else if ("-get".equals(cmd) || "-copyToLocal".equals(cmd) || "-moveToLocal".equals(cmd)) {
            if (argv.length < 3) {
                this.printUsage(cmd);
                return exitCode;
            }
        } else if ("-mv".equals(cmd) || "-cp".equals(cmd)) {
            if (argv.length < 3) {
                this.printUsage(cmd);
                return exitCode;
            }
        } else if (("-rm".equals(cmd) || "-rmr".equals(cmd) || "-cat".equals(cmd) || "-mkdir".equals(cmd)) && argv.length < 2) {
            this.printUsage(cmd);
            return exitCode;
        }
        try {
            this.init();
        }
        catch (RPC.VersionMismatch v) {
            System.err.println("Version Mismatch between client and server... command aborted.");
            return exitCode;
        }
        catch (IOException e) {
            System.err.println("Bad connection to FS. command aborted.");
            return exitCode;
        }
        exitCode = 0;
        try {
            if ("-put".equals(cmd) || "-copyFromLocal".equals(cmd)) {
                this.copyFromLocal(new Path(argv[i++]), argv[i++]);
            } else if ("-moveFromLocal".equals(cmd)) {
                this.moveFromLocal(new Path(argv[i++]), argv[i++]);
            } else if ("-get".equals(cmd) || "-copyToLocal".equals(cmd)) {
                this.copyToLocal(argv, i);
            } else if ("-getmerge".equals(cmd)) {
                if (argv.length > i + 2) {
                    this.copyMergeToLocal(argv[i++], new Path(argv[i++]), Boolean.parseBoolean(argv[i++]));
                } else {
                    this.copyMergeToLocal(argv[i++], new Path(argv[i++]));
                }
            } else if ("-cat".equals(cmd)) {
                exitCode = this.doall(cmd, argv, this.conf, i);
            } else if ("-moveToLocal".equals(cmd)) {
                this.moveToLocal(argv[i++], new Path(argv[i++]));
            } else if ("-setrep".equals(cmd)) {
                this.setReplication(argv, i);
            } else if ("-ls".equals(cmd)) {
                if (i < argv.length) {
                    exitCode = this.doall(cmd, argv, this.conf, i);
                } else {
                    this.ls(".", false);
                }
            } else if ("-lsr".equals(cmd)) {
                if (i < argv.length) {
                    exitCode = this.doall(cmd, argv, this.conf, i);
                } else {
                    this.ls(".", true);
                }
            } else if ("-mv".equals(cmd)) {
                exitCode = this.rename(argv, this.conf);
            } else if ("-cp".equals(cmd)) {
                exitCode = this.copy(argv, this.conf);
            } else if ("-rm".equals(cmd)) {
                exitCode = this.doall(cmd, argv, this.conf, i);
            } else if ("-rmr".equals(cmd)) {
                exitCode = this.doall(cmd, argv, this.conf, i);
            } else if ("-expunge".equals(cmd)) {
                this.expunge();
            } else if ("-du".equals(cmd)) {
                if (i < argv.length) {
                    exitCode = this.doall(cmd, argv, this.conf, i);
                } else {
                    this.du("");
                }
            } else if ("-dus".equals(cmd)) {
                if (i < argv.length) {
                    exitCode = this.doall(cmd, argv, this.conf, i);
                } else {
                    this.dus("");
                }
            } else if ("-mkdir".equals(cmd)) {
                exitCode = this.doall(cmd, argv, this.conf, i);
            } else if ("-help".equals(cmd)) {
                if (i < argv.length) {
                    this.printHelp(argv[i]);
                } else {
                    this.printHelp("");
                }
            } else {
                exitCode = -1;
                System.err.println(cmd.substring(1) + ": Unknown command");
                this.printUsage("");
            }
        }
        catch (RemoteException e) {
            exitCode = -1;
            try {
                String[] content = e.getLocalizedMessage().split("\n");
                System.err.println(cmd.substring(1) + ": " + content[0]);
            }
            catch (Exception ex) {
                System.err.println(cmd.substring(1) + ": " + ex.getLocalizedMessage());
            }
        }
        catch (IOException e) {
            exitCode = -1;
            System.err.println(cmd.substring(1) + ": " + e.getLocalizedMessage());
        }
        finally {
            this.fs.close();
        }
        return exitCode;
    }

    public static void main(String[] argv) throws Exception {
        int res = new FsShell().doMain(new Configuration(), argv);
        System.exit(res);
    }
}

