/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.fs;

import java.io.FileNotFoundException;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.zip.CRC32;
import java.util.zip.Checksum;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.ChecksumException;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FSInputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.FilterFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.StringUtils;

public abstract class ChecksumFileSystem
extends FilterFileSystem {
    private static final byte[] CHECKSUM_VERSION = new byte[]{99, 114, 99, 0};
    private static final PathFilter DEFAULT_FILTER = new PathFilter(){

        public boolean accept(Path file) {
            return !ChecksumFileSystem.isChecksumFile(file);
        }
    };

    public static double getApproxChkSumLength(long size) {
        return 0.01f * (float)size;
    }

    public ChecksumFileSystem(FileSystem fs) {
        super(fs);
    }

    public FileSystem getRawFileSystem() {
        return this.fs;
    }

    public Path getChecksumFile(Path file) {
        return new Path(file.getParent(), "." + file.getName() + ".crc");
    }

    public static boolean isChecksumFile(Path file) {
        String name = file.getName();
        return name.startsWith(".") && name.endsWith(".crc");
    }

    public long getChecksumFileLength(Path file, long fileSize) {
        return FSOutputSummer.getChecksumLength(fileSize, this.getBytesPerSum());
    }

    public int getBytesPerSum() {
        return this.getConf().getInt("io.bytes.per.checksum", 512);
    }

    private int getSumBufferSize(int bytesPerSum, int bufferSize) {
        int defaultBufferSize = this.getConf().getInt("io.file.buffer.size", 4096);
        int proportionalBufferSize = bufferSize / bytesPerSum;
        return Math.max(bytesPerSum, Math.max(proportionalBufferSize, defaultBufferSize));
    }

    public FSDataInputStream open(Path f, int bufferSize) throws IOException {
        if (!this.exists(f)) {
            throw new FileNotFoundException(f.toString());
        }
        return new FSDataInputStream((FSInputStream)new FSInputChecker(this, f, bufferSize), bufferSize);
    }

    public FSDataOutputStream create(Path f, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
        if (this.exists(f) && !overwrite) {
            throw new IOException("File already exists:" + f);
        }
        Path parent = f.getParent();
        if (parent != null && !this.mkdirs(parent)) {
            throw new IOException("Mkdirs failed to create " + parent);
        }
        return new FSDataOutputStream((OutputStream)new FSOutputSummer(this, f, overwrite, bufferSize, replication, blockSize, progress), bufferSize);
    }

    public boolean setReplication(Path src, short replication) throws IOException {
        boolean value = this.fs.setReplication(src, replication);
        if (!value) {
            return false;
        }
        Path checkFile = this.getChecksumFile(src);
        if (this.exists(checkFile)) {
            this.fs.setReplication(checkFile, replication);
        }
        return true;
    }

    public boolean rename(Path src, Path dst) throws IOException {
        if (this.fs.isDirectory(src)) {
            return this.fs.rename(src, dst);
        }
        boolean value = this.fs.rename(src, dst);
        if (!value) {
            return false;
        }
        Path checkFile = this.getChecksumFile(src);
        if (this.fs.exists(checkFile)) {
            value = this.fs.isDirectory(dst) ? this.fs.rename(checkFile, dst) : this.fs.rename(checkFile, this.getChecksumFile(dst));
        }
        return value;
    }

    public boolean delete(Path f) throws IOException {
        if (this.fs.isDirectory(f)) {
            return this.fs.delete(f);
        }
        Path checkFile = this.getChecksumFile(f);
        if (this.fs.exists(checkFile)) {
            this.fs.delete(checkFile);
        }
        return this.fs.delete(f);
    }

    public Path[] listPaths(Path[] files) throws IOException {
        return this.fs.listPaths(files, DEFAULT_FILTER);
    }

    public Path[] listPaths(Path f) throws IOException {
        return this.fs.listPaths(f, DEFAULT_FILTER);
    }

    public boolean mkdirs(Path f) throws IOException {
        return this.fs.mkdirs(f);
    }

    public void lock(Path f, boolean shared) throws IOException {
        if (this.fs.isDirectory(f)) {
            this.fs.lock(f, shared);
        } else {
            Path checkFile = this.getChecksumFile(f);
            if (this.fs.exists(checkFile)) {
                this.fs.lock(checkFile, shared);
            }
            this.fs.lock(f, shared);
        }
    }

    public void release(Path f) throws IOException {
        if (this.fs.isDirectory(f)) {
            this.fs.release(f);
        } else {
            Path checkFile = this.getChecksumFile(f);
            if (this.fs.exists(checkFile)) {
                this.fs.release(this.getChecksumFile(f));
            }
            this.fs.release(f);
        }
    }

    public void copyFromLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        FileSystem localFs = ChecksumFileSystem.getNamed("file:///", this.getConf());
        FileUtil.copy(localFs, src, this, dst, delSrc, this.getConf());
    }

    public void copyToLocalFile(boolean delSrc, Path src, Path dst) throws IOException {
        FileSystem localFs = ChecksumFileSystem.getNamed("file:///", this.getConf());
        FileUtil.copy(this, src, localFs, dst, delSrc, this.getConf());
    }

    public void copyToLocalFile(Path src, Path dst, boolean copyCrc) throws IOException {
        if (!this.fs.isDirectory(src)) {
            this.fs.copyToLocalFile(src, dst);
            FileSystem localFs = ChecksumFileSystem.getNamed("file:///", this.getConf());
            if (localFs instanceof ChecksumFileSystem) {
                localFs = ((ChecksumFileSystem)localFs).getRawFileSystem();
            }
            if (localFs.isDirectory(dst)) {
                dst = new Path(dst, src.getName());
            }
            if (localFs.exists(dst = this.getChecksumFile(dst))) {
                localFs.delete(dst);
            }
            Path checksumFile = this.getChecksumFile(src);
            if (copyCrc && this.fs.exists(checksumFile)) {
                this.fs.copyToLocalFile(checksumFile, dst);
            }
        } else {
            Path[] srcs;
            for (Path srcFile : srcs = this.listPaths(src)) {
                this.copyToLocalFile(srcFile, new Path(dst, srcFile.getName()), copyCrc);
            }
        }
    }

    public Path startLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        return tmpLocalFile;
    }

    public void completeLocalOutput(Path fsOutputFile, Path tmpLocalFile) throws IOException {
        this.moveFromLocalFile(tmpLocalFile, fsOutputFile);
    }

    public boolean reportChecksumFailure(Path f, FSDataInputStream in, long inPos, FSDataInputStream sums, long sumsPos) {
        return false;
    }

    private static class FSOutputSummer
    extends FilterOutputStream {
        private FSDataOutputStream sums;
        private Checksum sum = new CRC32();
        private int inSum;
        private int bytesPerSum;
        private static final float CHKSUM_AS_FRACTION = 0.01f;

        public FSOutputSummer(ChecksumFileSystem fs, Path file, boolean overwrite, short replication, long blockSize, Configuration conf) throws IOException {
            this(fs, file, overwrite, conf.getInt("io.file.buffer.size", 4096), replication, blockSize, null);
        }

        public FSOutputSummer(ChecksumFileSystem fs, Path file, boolean overwrite, int bufferSize, short replication, long blockSize, Progressable progress) throws IOException {
            super(fs.getRawFileSystem().create(file, overwrite, 1, replication, blockSize, progress));
            this.bytesPerSum = fs.getBytesPerSum();
            int sumBufferSize = fs.getSumBufferSize(this.bytesPerSum, bufferSize);
            this.sums = fs.getRawFileSystem().create(fs.getChecksumFile(file), true, sumBufferSize, replication, blockSize);
            this.sums.write(CHECKSUM_VERSION, 0, CHECKSUM_VERSION.length);
            this.sums.writeInt(this.bytesPerSum);
        }

        public void write(byte[] b, int off, int len) throws IOException {
            int summed = 0;
            while (summed < len) {
                int inBuf = len - summed;
                int goal = this.bytesPerSum - this.inSum;
                int toSum = inBuf <= goal ? inBuf : goal;
                this.sum.update(b, off + summed, toSum);
                summed += toSum;
                this.inSum += toSum;
                if (this.inSum != this.bytesPerSum) continue;
                this.writeSum();
            }
            this.out.write(b, off, len);
        }

        private void writeSum() throws IOException {
            if (this.inSum != 0) {
                this.sums.writeInt((int)this.sum.getValue());
                this.sum.reset();
                this.inSum = 0;
            }
        }

        public void close() throws IOException {
            this.writeSum();
            if (this.sums != null) {
                this.sums.close();
            }
            this.out.close();
        }

        public static long getChecksumLength(long size, int bytesPerSum) {
            return ((long)Math.ceil((float)size / (float)bytesPerSum) + 1L) * 4L + (long)CHECKSUM_VERSION.length;
        }
    }

    private static class FSInputChecker
    extends FSInputStream {
        public static final Log LOG = LogFactory.getLog((String)"org.apache.hadoop.fs.FSInputChecker");
        private ChecksumFileSystem fs;
        private Path file;
        private FSDataInputStream datas;
        private FSDataInputStream sums;
        private Checksum sum = new CRC32();
        private int inSum;
        private static final int HEADER_LENGTH = 8;
        private int bytesPerSum = 1;

        public FSInputChecker(ChecksumFileSystem fs, Path file) throws IOException {
            this(fs, file, fs.getConf().getInt("io.file.buffer.size", 4096));
        }

        public FSInputChecker(ChecksumFileSystem fs, Path file, int bufferSize) throws IOException {
            this.datas = fs.getRawFileSystem().open(file, 1);
            this.fs = fs;
            this.file = file;
            Path sumFile = fs.getChecksumFile(file);
            try {
                int sumBufferSize = fs.getSumBufferSize(fs.getBytesPerSum(), bufferSize);
                this.sums = fs.getRawFileSystem().open(sumFile, sumBufferSize);
                byte[] version = new byte[CHECKSUM_VERSION.length];
                this.sums.readFully(version);
                if (!Arrays.equals(version, CHECKSUM_VERSION)) {
                    throw new IOException("Not a checksum file: " + sumFile);
                }
                this.bytesPerSum = this.sums.readInt();
            }
            catch (FileNotFoundException e) {
                this.stopSumming();
            }
            catch (IOException e) {
                LOG.warn((Object)("Problem opening checksum file: " + file + ".  Ignoring exception: " + StringUtils.stringifyException(e)));
                this.stopSumming();
            }
        }

        private long getChecksumFilePos(long dataPos) {
            return 8L + 4L * (dataPos / (long)this.bytesPerSum);
        }

        public void seek(long desired) throws IOException {
            long checksumBoundary = desired / (long)this.bytesPerSum * (long)this.bytesPerSum;
            if (checksumBoundary != this.getPos()) {
                this.datas.seek(checksumBoundary);
                if (this.sums != null) {
                    this.sums.seek(this.getChecksumFilePos(checksumBoundary));
                }
            }
            if (this.sums != null) {
                this.sum.reset();
                this.inSum = 0;
            }
            int delta = (int)(desired - checksumBoundary);
            this.readBuffer(new byte[delta], 0, delta);
        }

        public int read() throws IOException {
            byte[] b = new byte[1];
            this.readBuffer(b, 0, 1);
            return b[0] & 0xFF;
        }

        public int read(byte[] b) throws IOException {
            return this.read(b, 0, b.length);
        }

        public int read(byte[] b, int off, int len) throws IOException {
            long curPos = this.getPos();
            long endPos = (long)len + curPos / (long)this.bytesPerSum * (long)this.bytesPerSum;
            return this.readBuffer(b, off, (int)(endPos - curPos));
        }

        private int readBuffer(byte[] b, int off, int len) throws IOException {
            int read;
            boolean retry;
            int retriesLeft = 3;
            long oldPos = this.getPos();
            do {
                --retriesLeft;
                retry = false;
                read = 0;
                boolean endOfFile = false;
                while (read < len && !endOfFile) {
                    int count = this.datas.read(b, off + read, len - read);
                    if (count < 0) {
                        endOfFile = true;
                        continue;
                    }
                    read += count;
                }
                if (this.sums == null || read == 0) continue;
                long oldSumsPos = this.sums.getPos();
                try {
                    int summed = 0;
                    while (summed < read) {
                        int inBuf = read - summed;
                        int goal = this.bytesPerSum - this.inSum;
                        int toSum = inBuf <= goal ? inBuf : goal;
                        try {
                            this.sum.update(b, off + summed, toSum);
                        }
                        catch (ArrayIndexOutOfBoundsException e) {
                            throw new RuntimeException("Summer buffer overflow b.len=" + b.length + ", off=" + off + ", summed=" + summed + ", read=" + read + ", bytesPerSum=" + this.bytesPerSum + ", inSum=" + this.inSum, e);
                        }
                        summed += toSum;
                        this.inSum += toSum;
                        if (this.inSum == this.bytesPerSum) {
                            this.verifySum(read - (summed - this.bytesPerSum));
                            continue;
                        }
                        if (read != summed || !endOfFile) continue;
                        this.verifySum(read - read / this.bytesPerSum * this.bytesPerSum);
                    }
                }
                catch (ChecksumException ce) {
                    LOG.info((Object)("Found checksum error: " + StringUtils.stringifyException(ce)));
                    long errPos = ce.getPos();
                    boolean shouldRetry = this.fs.reportChecksumFailure(this.file, this.datas, errPos, this.sums, errPos / (long)this.bytesPerSum);
                    if (!shouldRetry || retriesLeft == 0) {
                        throw ce;
                    }
                    if (this.seekToNewSource(oldPos)) {
                        retry = true;
                        this.seek(oldPos);
                        continue;
                    }
                    throw ce;
                }
            } while (retry);
            return read == 0 ? -1 : read;
        }

        private void verifySum(int delta) throws IOException {
            int crc;
            try {
                crc = this.sums.readInt();
            }
            catch (IOException e) {
                LOG.warn((Object)("Problem reading checksum file: " + e + ". Ignoring."));
                this.stopSumming();
                return;
            }
            int sumValue = (int)this.sum.getValue();
            this.sum.reset();
            this.inSum = 0;
            if (crc != sumValue) {
                long pos = this.getPos() - (long)delta;
                throw new ChecksumException("Checksum error: " + this.file + " at " + pos, pos);
            }
        }

        public long getPos() throws IOException {
            return this.datas.getPos();
        }

        public int read(long position, byte[] buffer, int offset, int length) throws IOException {
            return this.datas.read(position, buffer, offset, length);
        }

        public void readFully(long position, byte[] buffer, int offset, int length) throws IOException {
            this.datas.readFully(position, buffer, offset, length);
        }

        public void readFully(long position, byte[] buffer) throws IOException {
            this.datas.readFully(position, buffer);
        }

        public void close() throws IOException {
            this.datas.close();
            this.stopSumming();
        }

        private void stopSumming() {
            if (this.sums != null) {
                try {
                    this.sums.close();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                this.sums = null;
                this.bytesPerSum = 1;
            }
        }

        public int available() throws IOException {
            return this.datas.available();
        }

        public boolean markSupported() {
            return this.datas.markSupported();
        }

        public synchronized void mark(int readlimit) {
            this.datas.mark(readlimit);
        }

        public synchronized void reset() throws IOException {
            this.datas.reset();
        }

        public long skip(long n) throws IOException {
            return this.datas.skip(n);
        }

        public boolean seekToNewSource(long targetPos) throws IOException {
            boolean newDataSource = this.datas.seekToNewSource(targetPos);
            return this.sums.seekToNewSource(this.getChecksumFilePos(targetPos)) || newDataSource;
        }
    }
}

