/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.io;

import java.io.EOFException;
import java.io.IOException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.LocalFileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.DataInputBuffer;
import org.apache.hadoop.io.DataOutputBuffer;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.io.Writable;
import org.apache.hadoop.io.WritableComparable;
import org.apache.hadoop.io.WritableComparator;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;

public class MapFile {
    public static final String INDEX_FILE_NAME = "index";
    public static final String DATA_FILE_NAME = "data";

    protected MapFile() {
    }

    public static void rename(FileSystem fs, String oldName, String newName) throws IOException {
        Path oldDir = new Path(oldName);
        Path newDir = new Path(newName);
        if (!fs.rename(oldDir, newDir)) {
            throw new IOException("Could not rename " + oldDir + " to " + newDir);
        }
    }

    public static void delete(FileSystem fs, String name) throws IOException {
        Path dir = new Path(name);
        Path data = new Path(dir, DATA_FILE_NAME);
        Path index = new Path(dir, INDEX_FILE_NAME);
        fs.delete(data);
        fs.delete(index);
        fs.delete(dir);
    }

    public static long fix(FileSystem fs, Path dir, Class keyClass, Class valueClass, boolean dryrun, Configuration conf) throws Exception {
        String dr = dryrun ? "[DRY RUN ] " : "";
        Path data = new Path(dir, DATA_FILE_NAME);
        Path index = new Path(dir, INDEX_FILE_NAME);
        int indexInterval = 128;
        if (!fs.exists(data)) {
            throw new Exception(dr + "Missing data file in " + dir + ", impossible to fix this.");
        }
        if (fs.exists(index)) {
            return -1L;
        }
        SequenceFile.Reader dataReader = new SequenceFile.Reader(fs, data, conf);
        if (!dataReader.getKeyClass().equals(keyClass)) {
            throw new Exception(dr + "Wrong key class in " + dir + ", expected" + keyClass.getName() + ", got " + dataReader.getKeyClass().getName());
        }
        if (!dataReader.getValueClass().equals(valueClass)) {
            throw new Exception(dr + "Wrong value class in " + dir + ", expected" + valueClass.getName() + ", got " + dataReader.getValueClass().getName());
        }
        long cnt = 0L;
        Writable key = (Writable)ReflectionUtils.newInstance(keyClass, conf);
        Writable value = (Writable)ReflectionUtils.newInstance(valueClass, conf);
        SequenceFile.Writer indexWriter = null;
        if (!dryrun) {
            indexWriter = SequenceFile.createWriter(fs, conf, index, keyClass, LongWritable.class);
        }
        try {
            long pos = 0L;
            LongWritable position = new LongWritable();
            while (dataReader.next(key, value)) {
                if (++cnt % (long)indexInterval == 0L) {
                    position.set(pos);
                    if (!dryrun) {
                        indexWriter.append(key, position);
                    }
                }
                pos = dataReader.getPosition();
            }
        }
        catch (Throwable t) {
            // empty catch block
        }
        dataReader.close();
        if (!dryrun) {
            indexWriter.close();
        }
        return cnt;
    }

    public static void main(String[] args) throws Exception {
        String usage = "Usage: MapFile inFile outFile";
        if (args.length != 2) {
            System.err.println(usage);
            System.exit(-1);
        }
        String in = args[0];
        String out = args[1];
        Configuration conf = new Configuration();
        int ioFileBufferSize = conf.getInt("io.file.buffer.size", 4096);
        LocalFileSystem fs = FileSystem.getLocal(conf);
        Reader reader = new Reader(fs, in, conf);
        Writer writer = new Writer(conf, (FileSystem)fs, out, reader.getKeyClass(), reader.getValueClass());
        WritableComparable key = (WritableComparable)ReflectionUtils.newInstance(reader.getKeyClass(), conf);
        Writable value = (Writable)ReflectionUtils.newInstance(reader.getValueClass(), conf);
        while (reader.next(key, value)) {
            writer.append(key, value);
        }
        writer.close();
    }

    public static class Reader {
        private int INDEX_SKIP = 0;
        private WritableComparator comparator;
        private WritableComparable nextKey;
        private long seekPosition = -1L;
        private int seekIndex = -1;
        private long firstPosition;
        private WritableComparable getKey;
        private SequenceFile.Reader data;
        private SequenceFile.Reader index;
        private boolean indexClosed = false;
        private int count = -1;
        private WritableComparable[] keys;
        private long[] positions;

        public Class getKeyClass() {
            return this.data.getKeyClass();
        }

        public Class getValueClass() {
            return this.data.getValueClass();
        }

        public Reader(FileSystem fs, String dirName, Configuration conf) throws IOException {
            this(fs, dirName, null, conf);
            this.INDEX_SKIP = conf.getInt("io.map.index.skip", 0);
        }

        public Reader(FileSystem fs, String dirName, WritableComparator comparator, Configuration conf) throws IOException {
            Path dir = new Path(dirName);
            Path dataFile = new Path(dir, MapFile.DATA_FILE_NAME);
            Path indexFile = new Path(dir, MapFile.INDEX_FILE_NAME);
            this.data = new SequenceFile.Reader(fs, dataFile, conf);
            this.firstPosition = this.data.getPosition();
            this.comparator = comparator == null ? WritableComparator.get(this.data.getKeyClass()) : comparator;
            this.getKey = this.comparator.newKey();
            this.index = new SequenceFile.Reader(fs, indexFile, conf);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void readIndex() throws IOException {
            if (this.keys != null) {
                return;
            }
            this.count = 0;
            this.keys = new WritableComparable[1024];
            this.positions = new long[1024];
            try {
                WritableComparable k;
                int skip = this.INDEX_SKIP;
                LongWritable position = new LongWritable();
                WritableComparable lastKey = null;
                while (this.index.next(k = this.comparator.newKey(), position)) {
                    if (lastKey != null && this.comparator.compare(lastKey, k) > 0) {
                        throw new IOException("key out of order: " + k + " after " + lastKey);
                    }
                    lastKey = k;
                    if (skip > 0) {
                        --skip;
                        continue;
                    }
                    skip = this.INDEX_SKIP;
                    if (this.count == this.keys.length) {
                        int newLength = this.keys.length * 3 / 2;
                        WritableComparable[] newKeys = new WritableComparable[newLength];
                        long[] newPositions = new long[newLength];
                        System.arraycopy(this.keys, 0, newKeys, 0, this.count);
                        System.arraycopy(this.positions, 0, newPositions, 0, this.count);
                        this.keys = newKeys;
                        this.positions = newPositions;
                    }
                    this.keys[this.count] = k;
                    this.positions[this.count] = position.get();
                    ++this.count;
                }
            }
            catch (EOFException e) {
                SequenceFile.LOG.warn((Object)("Unexpected EOF reading " + this.index + " at entry #" + this.count + ".  Ignoring."));
            }
            finally {
                this.indexClosed = true;
                this.index.close();
            }
        }

        public synchronized void reset() throws IOException {
            this.data.seek(this.firstPosition);
        }

        public synchronized WritableComparable midKey() throws IOException {
            this.readIndex();
            int pos = (this.count - 1) / 2;
            if (pos < 0) {
                throw new IOException("MapFile empty");
            }
            return this.keys[pos];
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void finalKey(WritableComparable key) throws IOException {
            long originalPosition = this.data.getPosition();
            try {
                this.readIndex();
                if (this.count > 0) {
                    this.data.seek(this.positions[this.count - 1]);
                } else {
                    this.reset();
                }
                while (this.data.next(key)) {
                }
            }
            finally {
                this.data.seek(originalPosition);
            }
        }

        public synchronized boolean seek(WritableComparable key) throws IOException {
            return this.seekInternal(key) == 0;
        }

        private synchronized int seekInternal(WritableComparable key) throws IOException {
            this.readIndex();
            if (this.seekIndex == -1 || this.seekIndex + 1 >= this.count || this.comparator.compare(key, this.keys[this.seekIndex + 1]) >= 0 || this.comparator.compare(key, this.nextKey) < 0) {
                this.seekIndex = this.binarySearch(key);
                if (this.seekIndex < 0) {
                    this.seekIndex = -this.seekIndex - 2;
                }
                this.seekPosition = this.seekIndex == -1 ? this.firstPosition : this.positions[this.seekIndex];
            }
            this.data.seek(this.seekPosition);
            if (this.nextKey == null) {
                this.nextKey = this.comparator.newKey();
            }
            while (this.data.next(this.nextKey)) {
                int c = this.comparator.compare(key, this.nextKey);
                if (c > 0) continue;
                return c;
            }
            return 1;
        }

        private int binarySearch(WritableComparable key) {
            int low = 0;
            int high = this.count - 1;
            while (low <= high) {
                int mid = low + high >> 1;
                WritableComparable midVal = this.keys[mid];
                int cmp = this.comparator.compare(midVal, key);
                if (cmp < 0) {
                    low = mid + 1;
                    continue;
                }
                if (cmp > 0) {
                    high = mid - 1;
                    continue;
                }
                return mid;
            }
            return -(low + 1);
        }

        public synchronized boolean next(WritableComparable key, Writable val) throws IOException {
            return this.data.next(key, val);
        }

        public synchronized Writable get(WritableComparable key, Writable val) throws IOException {
            if (this.seek(key)) {
                this.data.getCurrentValue(val);
                return val;
            }
            return null;
        }

        public synchronized WritableComparable getClosest(WritableComparable key, Writable val) throws IOException {
            if (this.seekInternal(key) > 0) {
                return null;
            }
            this.data.getCurrentValue(val);
            return this.nextKey;
        }

        public synchronized void close() throws IOException {
            if (!this.indexClosed) {
                this.index.close();
            }
            this.data.close();
        }
    }

    public static class Writer {
        private SequenceFile.Writer data;
        private SequenceFile.Writer index;
        private int indexInterval = 128;
        private long size;
        private LongWritable position = new LongWritable();
        private WritableComparator comparator;
        private DataInputBuffer inBuf = new DataInputBuffer();
        private DataOutputBuffer outBuf = new DataOutputBuffer();
        private WritableComparable lastKey;

        public Writer(Configuration conf, FileSystem fs, String dirName, Class keyClass, Class valClass) throws IOException {
            this(conf, fs, dirName, WritableComparator.get(keyClass), valClass, SequenceFile.getCompressionType(conf));
        }

        public Writer(Configuration conf, FileSystem fs, String dirName, Class keyClass, Class valClass, SequenceFile.CompressionType compress, Progressable progress) throws IOException {
            this(conf, fs, dirName, WritableComparator.get(keyClass), valClass, compress, progress);
        }

        public Writer(Configuration conf, FileSystem fs, String dirName, Class keyClass, Class valClass, SequenceFile.CompressionType compress) throws IOException {
            this(conf, fs, dirName, WritableComparator.get(keyClass), valClass, compress);
        }

        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass) throws IOException {
            this(conf, fs, dirName, comparator, valClass, SequenceFile.getCompressionType(conf));
        }

        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass, SequenceFile.CompressionType compress) throws IOException {
            this(conf, fs, dirName, comparator, valClass, compress, null);
        }

        public Writer(Configuration conf, FileSystem fs, String dirName, WritableComparator comparator, Class valClass, SequenceFile.CompressionType compress, Progressable progress) throws IOException {
            this.comparator = comparator;
            this.lastKey = comparator.newKey();
            Path dir = new Path(dirName);
            if (!fs.mkdirs(dir)) {
                throw new IOException("Mkdirs failed to create directory " + dir.toString());
            }
            Path dataFile = new Path(dir, MapFile.DATA_FILE_NAME);
            Path indexFile = new Path(dir, MapFile.INDEX_FILE_NAME);
            Class keyClass = comparator.getKeyClass();
            this.data = SequenceFile.createWriter(fs, conf, dataFile, keyClass, valClass, compress, progress);
            this.index = SequenceFile.createWriter(fs, conf, indexFile, keyClass, LongWritable.class, SequenceFile.CompressionType.BLOCK, progress);
        }

        public int getIndexInterval() {
            return this.indexInterval;
        }

        public void setIndexInterval(int interval) {
            this.indexInterval = interval;
        }

        public synchronized void close() throws IOException {
            this.data.close();
            this.index.close();
        }

        public synchronized void append(WritableComparable key, Writable val) throws IOException {
            this.checkKey(key);
            if (this.size % (long)this.indexInterval == 0L) {
                this.position.set(this.data.getLength());
                this.index.append(key, this.position);
            }
            this.data.append(key, val);
            ++this.size;
        }

        private void checkKey(WritableComparable key) throws IOException {
            if (this.size != 0L && this.comparator.compare(this.lastKey, key) > 0) {
                throw new IOException("key out of order: " + key + " after " + this.lastKey);
            }
            this.outBuf.reset();
            key.write(this.outBuf);
            this.inBuf.reset(this.outBuf.getData(), this.outBuf.getLength());
            this.lastKey.readFields(this.inBuf);
        }
    }
}

