/*
 * Decompiled with CFR 0.152.
 */
package com.eucalyptus.auth.policy;

import com.eucalyptus.auth.Debugging;
import com.eucalyptus.auth.PolicyParseException;
import com.eucalyptus.auth.entities.AuthorizationEntity;
import com.eucalyptus.auth.entities.ConditionEntity;
import com.eucalyptus.auth.entities.PolicyEntity;
import com.eucalyptus.auth.entities.PrincipalEntity;
import com.eucalyptus.auth.entities.StatementEntity;
import com.eucalyptus.auth.json.JsonUtils;
import com.eucalyptus.auth.policy.PolicySpec;
import com.eucalyptus.auth.policy.condition.ConditionOp;
import com.eucalyptus.auth.policy.condition.Conditions;
import com.eucalyptus.auth.policy.condition.NullConditionOp;
import com.eucalyptus.auth.policy.ern.Ern;
import com.eucalyptus.auth.policy.key.Key;
import com.eucalyptus.auth.policy.key.Keys;
import com.eucalyptus.auth.policy.key.QuotaKey;
import com.eucalyptus.auth.principal.Authorization;
import com.eucalyptus.auth.principal.Principal;
import com.eucalyptus.util.Pair;
import com.google.common.base.Optional;
import com.google.common.base.Strings;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import net.sf.json.JSONException;
import net.sf.json.JSONObject;
import org.apache.log4j.Logger;

public class PolicyParser {
    public static final int MAX_POLICY_SIZE = 16384;
    private static final Logger LOG = Logger.getLogger(PolicyParser.class);
    private final PolicyAttachmentType attachmentType;

    public static PolicyParser getInstance() {
        return new PolicyParser(PolicyAttachmentType.Identity);
    }

    public static PolicyParser getResourceInstance() {
        return new PolicyParser(PolicyAttachmentType.Resource);
    }

    private PolicyParser(PolicyAttachmentType attachmentType) {
        this.attachmentType = attachmentType;
    }

    public PolicyEntity parse(String policy) throws PolicyParseException {
        if (policy == null) {
            throw new PolicyParseException("Empty input policy text");
        }
        if (policy.length() > 16384) {
            throw new PolicyParseException("Policy size is too large");
        }
        try {
            JSONObject policyJsonObj = JSONObject.fromObject((Object)policy);
            String version = JsonUtils.getByType(String.class, policyJsonObj, "Version");
            List<StatementEntity> statements = this.parseStatements(policyJsonObj);
            return new PolicyEntity(version, policy, statements);
        }
        catch (JSONException e) {
            Debugging.logError(LOG, e, "Syntax error in input policy");
            throw new PolicyParseException((Throwable)e);
        }
    }

    private List<StatementEntity> parseStatements(JSONObject policy) throws JSONException {
        List<JSONObject> objs = JsonUtils.getRequiredArrayByType(JSONObject.class, policy, "Statement");
        ArrayList statements = Lists.newArrayList();
        for (JSONObject o : objs) {
            statements.add(this.parseStatement(o));
        }
        return statements;
    }

    private StatementEntity parseStatement(JSONObject statement) throws JSONException {
        String sid = JsonUtils.getByType(String.class, statement, "Sid");
        JsonUtils.checkRequired(statement, "Effect");
        String effect = JsonUtils.getByType(String.class, statement, "Effect");
        this.checkEffect(effect);
        PrincipalEntity principal = this.parsePrincipal(statement);
        List<AuthorizationEntity> authorizations = this.parseAuthorizations(statement, effect);
        List<ConditionEntity> conditions = this.parseConditions(statement, effect);
        return new StatementEntity(sid, principal, authorizations, conditions);
    }

    private PrincipalEntity parsePrincipal(JSONObject statement) {
        String principalElement = JsonUtils.checkBinaryOption(statement, "Principal", "NotPrincipal", this.attachmentType.isPrincipalRequired());
        JSONObject principal = JsonUtils.getByType(JSONObject.class, statement, principalElement);
        if (principal == null) {
            return null;
        }
        String principalType = JsonUtils.checkBinaryOption(principal, Principal.PrincipalType.AWS.name(), Principal.PrincipalType.Service.name());
        List<String> values = JsonUtils.parseStringOrStringList(principal, principalType);
        if (values.size() < 1 && this.attachmentType.isPrincipalRequired()) {
            throw new JSONException("Empty principal values");
        }
        boolean notPrincipal = "NotPrincipal".equals(principalElement);
        return new PrincipalEntity(notPrincipal, Principal.PrincipalType.valueOf((String)principalType), Sets.newHashSet(values));
    }

    private List<AuthorizationEntity> parseAuthorizations(JSONObject statement, String effect) throws JSONException {
        String actionElement = JsonUtils.checkBinaryOption(statement, "Action", "NotAction");
        List<String> actions = JsonUtils.parseStringOrStringList(statement, actionElement);
        if (actions.size() < 1) {
            throw new JSONException("Empty action values");
        }
        String resourceElement = JsonUtils.checkBinaryOption(statement, "Resource", "NotResource", this.attachmentType.isResourceRequired());
        List<String> resources = JsonUtils.parseStringOrStringList(statement, resourceElement);
        if (resources.size() < 1 && this.attachmentType.isResourceRequired()) {
            throw new JSONException("Empty resource values");
        }
        return this.decomposeStatement(effect, actionElement, actions, resourceElement, resources);
    }

    private List<AuthorizationEntity> decomposeStatement(String effect, String actionElement, List<String> actions, String resourceElement, List<String> resources) {
        HashMultimap actionMap = HashMultimap.create();
        for (String action : actions) {
            action = this.normalize(action);
            String vendor = this.checkAction(action);
            actionMap.put((Object)vendor, (Object)action);
        }
        HashMultimap resourceMap = HashMultimap.create();
        for (String resource : resources) {
            Ern ern = Ern.parse((String)resource);
            resourceMap.put((Object)Pair.lopair((Object)Strings.emptyToNull((String)ern.getNamespace()), (Object)ern.getResourceType()), (Object)ern.getResourceName());
        }
        boolean notAction = "NotAction".equals(actionElement);
        boolean notResource = "NotResource".equals(resourceElement);
        ArrayList results = Lists.newArrayList();
        for (Map.Entry actionSetEntry : actionMap.asMap().entrySet()) {
            String vendor = (String)actionSetEntry.getKey();
            Set actionSet = (Set)actionSetEntry.getValue();
            boolean added = false;
            for (Map.Entry resourceSetEntry : resourceMap.asMap().entrySet()) {
                Optional accountIdOrName = (Optional)((Pair)resourceSetEntry.getKey()).getLeft();
                String type = (String)((Pair)resourceSetEntry.getKey()).getRight();
                Set resourceSet = (Set)resourceSetEntry.getValue();
                if (!"*".equals(vendor) && !"*".equals(type) && !PolicySpec.isPermittedResourceVendor((String)vendor, (String)PolicySpec.vendor((String)type))) continue;
                results.add(new AuthorizationEntity(Authorization.EffectType.valueOf((String)effect), (String)accountIdOrName.orNull(), type, actionSet, notAction, resourceSet, notResource));
                added = true;
            }
            if (added) continue;
            results.add(new AuthorizationEntity(Authorization.EffectType.valueOf((String)effect), actionSet, notAction));
        }
        return results;
    }

    private List<ConditionEntity> parseConditions(JSONObject statement, String effect) throws JSONException {
        JSONObject condsObj = JsonUtils.getByType(JSONObject.class, statement, "Condition");
        boolean isQuota = Authorization.EffectType.Limit.name().equals(effect);
        ArrayList results = Lists.newArrayList();
        if (condsObj != null) {
            for (Object t : condsObj.keySet()) {
                String type = (String)t;
                Class<? extends ConditionOp> typeClass = this.checkConditionType(type);
                JSONObject paramsObj = JsonUtils.getByType(JSONObject.class, condsObj, type);
                for (Object k : paramsObj.keySet()) {
                    String key = (String)k;
                    HashSet values = Sets.newHashSet();
                    values.addAll(JsonUtils.parseStringOrStringList(paramsObj, key));
                    key = this.normalize(key);
                    this.checkConditionKeyAndValues(key, values, typeClass, isQuota);
                    results.add(new ConditionEntity(type, key, values));
                }
            }
        }
        return results;
    }

    private String checkAction(String action) throws JSONException {
        Matcher matcher = PolicySpec.ACTION_PATTERN.matcher(action);
        if (!matcher.matches()) {
            throw new JSONException("'" + action + "' is not a valid action");
        }
        if ("*".equals(action)) {
            return "*";
        }
        return matcher.group(1);
    }

    private Class<? extends ConditionOp> checkConditionType(String type) throws JSONException {
        if (type == null) {
            throw new JSONException("Empty condition type");
        }
        Class typeClass = Conditions.getConditionOpClass((String)type);
        if (typeClass == null) {
            throw new JSONException("Condition type '" + type + "' is not supported");
        }
        return typeClass;
    }

    private void checkConditionKeyAndValues(String key, Set<String> values, Class<? extends ConditionOp> typeClass, boolean isQuota) throws JSONException {
        if (key == null) {
            throw new JSONException("Empty key name");
        }
        Class keyClass = Keys.getKeyClass((String)key);
        if (keyClass == null) {
            throw new JSONException("Condition key '" + key + "' is not supported");
        }
        if (isQuota && !QuotaKey.class.isAssignableFrom(keyClass)) {
            throw new JSONException("Quota statement can only use quota keys.'" + key + "' is invalid.");
        }
        Key keyObj = Keys.getKeyInstance((Class)keyClass);
        if (!NullConditionOp.class.equals(typeClass)) {
            keyObj.validateConditionType(typeClass);
        }
        if (values.size() < 1) {
            throw new JSONException("No value for key '" + key + "'");
        }
        if (isQuota && values.size() > 1) {
            throw new JSONException("Quota key '" + key + "' can only have one value");
        }
        if (!NullConditionOp.class.equals(typeClass)) {
            for (String v : values) {
                keyObj.validateValueType(v);
            }
        }
    }

    private void checkEffect(String effect) throws JSONException {
        if (effect == null) {
            throw new JSONException("Effect can not be empty");
        }
        if (!PolicySpec.EFFECTS.contains(effect)) {
            throw new JSONException("Invalid Effect value: " + effect);
        }
    }

    private String normalize(String value) {
        return value != null ? value.trim().toLowerCase() : null;
    }

    private static enum PolicyAttachmentType {
        Identity(true, false),
        Resource(false, true);

        private final boolean requireResource;
        private final boolean requirePrincipal;

        private PolicyAttachmentType(boolean requireResource, boolean requirePrincipal) {
            this.requireResource = requireResource;
            this.requirePrincipal = requirePrincipal;
        }

        public boolean isResourceRequired() {
            return this.requireResource;
        }

        public boolean isPrincipalRequired() {
            return this.requirePrincipal;
        }
    }
}

