/*
 * Decompiled with CFR 0.152.
 */
package com.eucalyptus.auth.ldap;

import com.eucalyptus.auth.Debugging;
import com.eucalyptus.auth.LicParseException;
import com.eucalyptus.auth.json.JsonUtils;
import com.eucalyptus.auth.ldap.LdapIntegrationConfiguration;
import com.eucalyptus.auth.ldap.Selection;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import java.util.HashSet;
import java.util.Set;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import net.sf.json.JSONException;
import net.sf.json.JSONObject;
import org.apache.log4j.Logger;

public class LicParser {
    private static final Logger LOG = Logger.getLogger(LicParser.class);
    private static final String COMMENT = "_comment";
    private static final String LDAP_URL_PREFIX = "ldap://";
    public static final String LDAP_AUTH_METHOD_SIMPLE = "simple";
    public static final String LDAP_AUTH_METHOD_SASL_DIGEST_MD5 = "DIGEST-MD5";
    public static final String LDAP_AUTH_METHOD_SASL_GSSAPI = "GSSAPI";
    private static final Set<String> LDAP_AUTH_METHODS = Sets.newHashSet();
    private static LicParser instance;

    public static LicParser getInstance() {
        if (instance == null) {
            instance = new LicParser();
        }
        return instance;
    }

    public LdapIntegrationConfiguration parse(String licText) throws LicParseException {
        if (licText == null) {
            throw new LicParseException("Empty configuration");
        }
        try {
            JSONObject licJson = JSONObject.fromObject((Object)licText);
            LdapIntegrationConfiguration lic = new LdapIntegrationConfiguration();
            this.parseSyncConfig(licJson, lic);
            if (lic.isSyncEnabled()) {
                this.parseLdapService(licJson, lic);
                this.parseAccounts(licJson, lic);
                this.parseGroups(licJson, lic);
                this.parseUsers(licJson, lic);
            }
            return lic;
        }
        catch (JSONException e) {
            Debugging.logError(LOG, e, "Syntax error in input policy");
            throw new LicParseException("Syntax error in configuration", (Throwable)e);
        }
    }

    private void parseLdapService(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject ldapServiceObj = JsonUtils.getRequiredByType(JSONObject.class, licJson, "ldap-service");
        lic.setServerUrl(this.validateServerUrl(JsonUtils.getRequiredByType(String.class, ldapServiceObj, "server-url")));
        lic.setAuthMethod(this.validateAuthMethod(JsonUtils.getRequiredByType(String.class, ldapServiceObj, "auth-method"), false));
        lic.setAuthPrincipal(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, ldapServiceObj, "auth-principal")));
        lic.setAuthCredentials(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, ldapServiceObj, "auth-credentials")));
        lic.setUseSsl("true".equalsIgnoreCase(JsonUtils.getRequiredByType(String.class, ldapServiceObj, "use-ssl")));
        lic.setIgnoreSslCertValidation("true".equalsIgnoreCase(JsonUtils.getByType(String.class, ldapServiceObj, "ignore-ssl-cert-validation")));
        lic.setKrb5Conf(this.validateKrb5Conf(lic.getAuthMethod(), JsonUtils.getByType(String.class, ldapServiceObj, "krb5-conf")));
        lic.setUserAuthMethod(this.validateAuthMethod(JsonUtils.getByType(String.class, ldapServiceObj, "user-auth-method"), true));
    }

    private String validateKrb5Conf(String authMethod, String krb5Conf) throws JSONException {
        if (LDAP_AUTH_METHOD_SASL_GSSAPI.equals(authMethod) && Strings.isNullOrEmpty((String)krb5Conf)) {
            throw new JSONException("krb5.conf must be specified for GSSAPI/KerberosV5");
        }
        return krb5Conf;
    }

    private String validateServerUrl(String url) throws JSONException {
        if (Strings.isNullOrEmpty((String)url) || !url.startsWith(LDAP_URL_PREFIX)) {
            throw new JSONException("Invalid server url " + url);
        }
        return url;
    }

    private String validateAuthMethod(String method, boolean allowEmpty) throws JSONException {
        if (!allowEmpty && Strings.isNullOrEmpty((String)method) || !Strings.isNullOrEmpty((String)method) && !LDAP_AUTH_METHODS.contains(method)) {
            throw new JSONException("Unsupported LDAP authentication method " + (method != null ? method : "null"));
        }
        return method;
    }

    private String validateNonEmpty(String value) throws JSONException {
        if (Strings.isNullOrEmpty((String)value)) {
            throw new JSONException("Empty value is not allowed for LIC element");
        }
        return value;
    }

    private void parseAccounts(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        String which = JsonUtils.checkBinaryOption(licJson, "accounting-groups", "groups-partition");
        if ("accounting-groups".equals(which)) {
            lic.setHasAccountingGroups(true);
            this.parseAccountingGroups(licJson, lic);
        } else {
            lic.setHasAccountingGroups(false);
            this.parseGroupsPartition(licJson, lic);
        }
    }

    private void parseGroupsPartition(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject groupsPartition = JsonUtils.getByType(JSONObject.class, licJson, "groups-partition");
        for (Object t : groupsPartition.keySet()) {
            String partitionName = (String)t;
            if (partitionName.equalsIgnoreCase(COMMENT)) continue;
            HashSet groupSet = Sets.newHashSet();
            groupSet.addAll(JsonUtils.getArrayByType(String.class, groupsPartition, partitionName));
            lic.getGroupsPartition().put(partitionName, groupSet);
        }
        if (lic.getGroupsPartition().size() < 1) {
            throw new JSONException("Expecting more than 1 group partition");
        }
    }

    private void parseAccountingGroups(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject accountingGroups = JsonUtils.getByType(JSONObject.class, licJson, "accounting-groups");
        lic.setAccountingGroupBaseDn(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, accountingGroups, "base-dn")));
        lic.setAccountingGroupsSelection(this.parseSelection(JsonUtils.getByType(JSONObject.class, accountingGroups, "selection")));
        lic.setAccountingGroupIdAttribute(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, accountingGroups, "id-attribute")));
        lic.setGroupsAttribute(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, accountingGroups, "member-attribute")));
        lic.setMembersItemType(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, accountingGroups, "member-item-type")));
    }

    private void parseGroups(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject groups = JsonUtils.getRequiredByType(JSONObject.class, licJson, "groups");
        lic.setGroupBaseDn(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, groups, "base-dn")));
        lic.setGroupsSelection(this.parseSelection(JsonUtils.getByType(JSONObject.class, groups, "selection")));
        lic.setGroupIdAttribute(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, groups, "id-attribute")));
        lic.setUsersAttribute(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, groups, "member-attribute")));
        lic.setMembersItemType(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, groups, "member-item-type")));
    }

    private void parseUsers(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject users = JsonUtils.getRequiredByType(JSONObject.class, licJson, "users");
        lic.setUserBaseDn(this.validateNonEmpty(JsonUtils.getRequiredByType(String.class, users, "base-dn")));
        lic.setUsersSelection(this.parseSelection(JsonUtils.getByType(JSONObject.class, users, "selection")));
        lic.setUserIdAttribute(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, users, "id-attribute")));
        lic.setUserSaslIdAttribute(LicParser.toLowerCaseIfNotNull(JsonUtils.getByType(String.class, users, "sasl-id-attribute")));
        this.parseUserInfoMap(JsonUtils.getByType(JSONObject.class, users, "user-info-attributes"), lic);
    }

    private void parseUserInfoMap(JSONObject map, LdapIntegrationConfiguration lic) throws JSONException {
        if (map == null) {
            return;
        }
        for (Object m : map.keySet()) {
            String attr = (String)m;
            if (attr.equalsIgnoreCase(COMMENT)) continue;
            String name = JsonUtils.getByType(String.class, map, attr);
            lic.getUserInfoAttributes().put(attr, name);
        }
    }

    private Selection parseSelection(JSONObject obj) throws JSONException {
        if (obj == null) {
            return null;
        }
        Selection selection = new Selection();
        selection.setSearchFilter(JsonUtils.getRequiredByType(String.class, obj, "filter"));
        if (selection.getSearchFilter() == null) {
            throw new JSONException("Empty search filter is not allowed");
        }
        selection.getSelected().addAll(JsonUtils.getArrayByType(String.class, obj, "select"));
        this.validateDnSet(selection.getSelected());
        selection.getNotSelected().addAll(JsonUtils.getArrayByType(String.class, obj, "not-select"));
        this.validateDnSet(selection.getSelected());
        selection.getSelected().removeAll(selection.getNotSelected());
        return selection;
    }

    private void validateDnSet(Set<String> selected) throws JSONException {
        try {
            for (String dn : selected) {
                new LdapName(dn);
            }
        }
        catch (InvalidNameException e) {
            throw new JSONException("Invalid DN name", (Throwable)e);
        }
    }

    private void parseSyncConfig(JSONObject licJson, LdapIntegrationConfiguration lic) throws JSONException {
        JSONObject sync = JsonUtils.getRequiredByType(JSONObject.class, licJson, "sync");
        lic.setEnableSync("true".equalsIgnoreCase(JsonUtils.getRequiredByType(String.class, sync, "enable")));
        if (lic.isSyncEnabled()) {
            lic.setAutoSync("true".equalsIgnoreCase(JsonUtils.getRequiredByType(String.class, sync, "auto")));
            try {
                lic.setSyncInterval(Long.parseLong(JsonUtils.getRequiredByType(String.class, sync, "interval")));
            }
            catch (NumberFormatException e) {
                throw new JSONException("Invalid sync interval value");
            }
            lic.setCleanDeletion("true".equalsIgnoreCase(JsonUtils.getByType(String.class, sync, "clean-deletion")));
        }
    }

    private static String toLowerCaseIfNotNull(String value) {
        if (value != null) {
            return value.toLowerCase();
        }
        return value;
    }

    static {
        LDAP_AUTH_METHODS.add(LDAP_AUTH_METHOD_SIMPLE);
        LDAP_AUTH_METHODS.add(LDAP_AUTH_METHOD_SASL_DIGEST_MD5);
        LDAP_AUTH_METHODS.add(LDAP_AUTH_METHOD_SASL_GSSAPI);
    }
}

