/*
 * Decompiled with CFR 0.152.
 */
package com.eucalyptus.auth.ldap;

import com.eucalyptus.auth.Accounts;
import com.eucalyptus.auth.AuthException;
import com.eucalyptus.auth.LdapException;
import com.eucalyptus.auth.ldap.LdapClient;
import com.eucalyptus.auth.ldap.LdapIntegrationConfiguration;
import com.eucalyptus.auth.ldap.Selection;
import com.eucalyptus.auth.principal.Account;
import com.eucalyptus.auth.principal.Group;
import com.eucalyptus.auth.principal.User;
import com.eucalyptus.bootstrap.Bootstrap;
import com.eucalyptus.bootstrap.Hosts;
import com.eucalyptus.event.ClockTick;
import com.eucalyptus.event.Event;
import com.eucalyptus.event.EventListener;
import com.eucalyptus.event.ListenerRegistry;
import com.eucalyptus.event.SystemClock;
import com.eucalyptus.system.Threads;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import javax.naming.InvalidNameException;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.LdapName;
import org.apache.log4j.Logger;

public class LdapSync {
    private static final Logger LOG = Logger.getLogger(LdapSync.class);
    private static final boolean VERBOSE = true;
    private static final String LDAP_SYNC_THREAD = "LDAP sync";
    private static final LdapIntegrationConfiguration DEFAULT_LIC;
    private static LdapIntegrationConfiguration lic;
    private static boolean inSync;
    private static long timeTillNextSync;
    private static final ClockTickListener TIMER_LISTENER;

    public static synchronized boolean inSync() {
        return inSync;
    }

    public static synchronized void start() {
        if (lic.isSyncEnabled()) {
            if (lic.isAutoSync()) {
                ListenerRegistry.getInstance().register(ClockTick.class, (EventListener)TIMER_LISTENER);
            }
            LdapSync.startSync();
        }
    }

    public static synchronized LdapIntegrationConfiguration getLic() {
        return lic;
    }

    public static synchronized void setLic(LdapIntegrationConfiguration config) {
        LOG.debug((Object)("A new LIC is being set: " + config));
        lic = config;
        if (Bootstrap.isFinished().booleanValue()) {
            if (lic.isSyncEnabled()) {
                if (lic.isAutoSync()) {
                    ListenerRegistry.getInstance().register(ClockTick.class, (EventListener)TIMER_LISTENER);
                }
                LdapSync.startSync();
            } else {
                ListenerRegistry.getInstance().deregister(ClockTick.class, (EventListener)TIMER_LISTENER);
            }
        }
    }

    public static synchronized void forceSync() {
        if (lic.isSyncEnabled()) {
            LdapSync.startSync();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static synchronized boolean check() {
        if (!lic.isSyncEnabled()) {
            return true;
        }
        try (LdapClient ldap = null;){
            ldap = LdapClient.authenticateClient(lic);
            boolean bl = true;
            return bl;
        }
    }

    public static synchronized void authenticate(User user, String password) throws LdapException {
        if (!lic.isSyncEnabled()) {
            throw new LdapException("LDAP sync is not enabled");
        }
        try (LdapClient ldap = null;){
            String login = null;
            if ("simple".equals(lic.getRealUserAuthMethod())) {
                login = user.getInfo("dn");
            } else {
                login = user.getInfo("saslid");
                if (Strings.isNullOrEmpty((String)login)) {
                    login = user.getName();
                }
            }
            if (Strings.isNullOrEmpty((String)login)) {
                throw new LdapException("Invalid login user");
            }
            ldap = LdapClient.authenticateUser(lic, login, password);
        }
    }

    public static synchronized boolean enabled() {
        return lic.isSyncEnabled();
    }

    private static synchronized boolean getAndSetSync(boolean newValue) {
        boolean old = inSync;
        inSync = newValue;
        return old;
    }

    private static synchronized void periodicSync() {
        if (lic.isSyncEnabled() && lic.isAutoSync() && (timeTillNextSync -= SystemClock.getRate()) <= 0L) {
            LdapSync.startSync();
        }
    }

    private static void startSync() {
        LOG.debug((Object)"A new sync initiated.");
        timeTillNextSync = lic.getSyncInterval();
        if (!LdapSync.getAndSetSync(true)) {
            Threads.newThread((Runnable)new Runnable(){

                @Override
                public void run() {
                    LOG.debug((Object)"Sync started");
                    LdapSync.sync(lic);
                    LdapSync.getAndSetSync(false);
                    LOG.debug((Object)"Sync ended");
                }
            }, (String)LDAP_SYNC_THREAD).start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void sync(LdapIntegrationConfiguration lic) {
        Map<Object, Object> accountingGroups = Maps.newHashMap();
        HashMap groupDnToId = Maps.newHashMap();
        HashMap groups = Maps.newHashMap();
        HashMap userDnToId = Maps.newHashMap();
        HashMap users = Maps.newHashMap();
        try (LdapClient ldap = null;){
            ldap = LdapClient.authenticateClient(lic);
            LdapSync.loadLdapUsers(ldap, lic, userDnToId, users);
            LdapSync.loadLdapGroups(ldap, lic, userDnToId, groupDnToId, groups);
            if (lic.hasAccountingGroups()) {
                LdapSync.loadLdapAccountingGroups(ldap, lic, groupDnToId, accountingGroups);
            } else {
                accountingGroups = lic.getGroupsPartition();
            }
        }
        LOG.debug((Object)("Sync remote accounts: " + accountingGroups));
        LOG.debug((Object)("Sync remote groups: " + groups));
        LOG.debug((Object)("Sync remote users: " + users));
        LdapSync.checkConflictingIdentities(accountingGroups, groups, users);
        LdapSync.rebuildLocalAuthDatabase(lic, accountingGroups, groups, users);
    }

    private static void checkConflictingIdentities(Map<String, Set<String>> accountingGroups, Map<String, Set<String>> groups, Map<String, Map<String, String>> users) {
        if (accountingGroups.containsKey("eucalyptus")) {
            LOG.error((Object)"Account eucalyptus is reserved for Eucalyptus only. Sync will skip this account from LDAP.");
            accountingGroups.remove("eucalyptus");
        }
        if (users.containsKey("admin")) {
            LOG.error((Object)"User admin is reserved for Eucalyptus only. Sync will skip this user from LDAP.");
            users.remove("admin");
        }
        for (String group : groups.keySet()) {
            if (!group.startsWith("_")) continue;
            LOG.error((Object)("Group name starting with _ is reserved for Eucalyptus only. Sync will skip this group " + group));
            groups.remove(group);
        }
    }

    private static void rebuildLocalAuthDatabase(LdapIntegrationConfiguration lic, Map<String, Set<String>> accountingGroups, Map<String, Set<String>> groups, Map<String, Map<String, String>> users) {
        try {
            Set<String> oldAccountSet = LdapSync.getLocalAccountSet();
            for (Map.Entry<String, Set<String>> entry : accountingGroups.entrySet()) {
                String accountName = entry.getKey();
                Set<String> accountMembers = entry.getValue();
                if (oldAccountSet.contains(accountName)) {
                    oldAccountSet.remove(accountName);
                    LdapSync.updateAccount(lic, accountName, accountMembers, groups, users);
                    continue;
                }
                LdapSync.addNewAccount(accountName, accountMembers, groups, users);
            }
            if (lic.isCleanDeletion()) {
                LdapSync.removeObsoleteAccounts(oldAccountSet);
            }
        }
        catch (Exception e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.error((Object)"Error in rebuilding local auth database", (Throwable)e);
        }
    }

    private static void addNewAccount(String accountName, Set<String> accountMembers, Map<String, Set<String>> groups, Map<String, Map<String, String>> users) {
        LOG.debug((Object)("Adding new account " + accountName));
        try {
            Account account = Accounts.addAccount((String)accountName);
            account.addUser("admin", "/", true, null);
            for (String user : LdapSync.getAccountUserSet(accountMembers, groups)) {
                try {
                    LOG.debug((Object)("Adding new user " + user));
                    Map<String, String> info = users.get(user);
                    if (info == null) {
                        LOG.warn((Object)("Empty user info for user " + user));
                    }
                    account.addUser(user, "/", true, info);
                }
                catch (AuthException e) {
                    LOG.error((Object)e, (Throwable)e);
                    LOG.warn((Object)("Failed add new user " + user), (Throwable)e);
                }
            }
            for (String group : accountMembers) {
                Group dbGroup = null;
                try {
                    LOG.debug((Object)("Adding new group " + group));
                    dbGroup = account.addGroup(group, "/");
                    Set<String> groupUsers = groups.get(group);
                    if (groupUsers == null) {
                        LOG.error((Object)("Empty user set for group " + group));
                        continue;
                    }
                    for (String user : groupUsers) {
                        LOG.debug((Object)("Adding " + user + " to group " + group));
                        dbGroup.addUserByName(user);
                    }
                }
                catch (AuthException e) {
                    LOG.error((Object)e, (Throwable)e);
                    LOG.warn((Object)("Failed to add new group " + group + " in " + accountName), (Throwable)e);
                }
            }
        }
        catch (AuthException e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.error((Object)("Failed to add new account " + accountName), (Throwable)e);
        }
    }

    private static Set<String> getAccountUserSet(Set<String> members, Map<String, Set<String>> groups) {
        HashSet userSet = Sets.newHashSet();
        for (String member : members) {
            Set<String> groupMembers = groups.get(member);
            if (groupMembers == null) {
                LOG.error((Object)("Empty user set of group " + member));
                continue;
            }
            userSet.addAll(groupMembers);
        }
        return userSet;
    }

    private static void updateAccount(LdapIntegrationConfiguration lic, String accountName, Set<String> accountMembers, Map<String, Set<String>> groups, Map<String, Map<String, String>> users) {
        LOG.debug((Object)("Updating account " + accountName));
        Account account = null;
        try {
            account = Accounts.lookupAccountByName((String)accountName);
            Set<String> newUserSet = LdapSync.getAccountUserSet(accountMembers, groups);
            Set<String> oldUserSet = LdapSync.getLocalUserSet(account);
            for (String user : newUserSet) {
                if (oldUserSet.contains(user)) {
                    oldUserSet.remove(user);
                    try {
                        LdapSync.updateUser(account, user, users.get(user));
                    }
                    catch (AuthException e) {
                        LOG.error((Object)e, (Throwable)e);
                        LOG.warn((Object)("Failed to update user " + user + " in " + accountName), (Throwable)e);
                    }
                    continue;
                }
                try {
                    LdapSync.addNewUser(account, user, users.get(user));
                }
                catch (AuthException e) {
                    LOG.error((Object)e, (Throwable)e);
                    LOG.warn((Object)("Failed to add new user " + user + " in " + accountName), (Throwable)e);
                }
            }
            if (lic.isCleanDeletion()) {
                LdapSync.removeObsoleteUsers(account, oldUserSet);
            }
            Set<String> oldGroupSet = LdapSync.getLocalGroupSet(account);
            for (String group : accountMembers) {
                if (oldGroupSet.contains(group)) {
                    oldGroupSet.remove(group);
                    LdapSync.updateGroup(account, group, groups.get(group));
                    continue;
                }
                LdapSync.addNewGroup(account, group, groups.get(group));
            }
            if (lic.isCleanDeletion()) {
                LdapSync.removeObsoleteGroups(account, oldGroupSet);
            }
        }
        catch (AuthException e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.error((Object)("Failed to update account " + accountName), (Throwable)e);
        }
    }

    private static void removeObsoleteGroups(Account account, Set<String> oldGroupSet) {
        for (String group : oldGroupSet) {
            try {
                account.deleteGroup(group, true);
            }
            catch (AuthException e) {
                LOG.error((Object)e, (Throwable)e);
                LOG.warn((Object)("Failed to delete group " + group + " in " + account.getName()), (Throwable)e);
            }
        }
    }

    private static void addNewGroup(Account account, String group, Set<String> users) {
        LOG.debug((Object)("Adding new group " + group + " in account " + account.getName()));
        if (users == null) {
            LOG.error((Object)("Empty new user set of group " + group));
            return;
        }
        try {
            Group g = account.addGroup(group, "/");
            for (String user : users) {
                LOG.debug((Object)("Adding " + user + " to " + group));
                g.addUserByName(user);
            }
        }
        catch (AuthException e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.warn((Object)("Failed to add new group " + group + " in " + account.getName()), (Throwable)e);
        }
    }

    private static void updateGroup(Account account, String group, Set<String> users) {
        LOG.debug((Object)("Updating group " + group + " in account " + account.getName()));
        if (users == null) {
            LOG.error((Object)("Empty new user set of group " + group));
            return;
        }
        try {
            HashSet localUserSet = Sets.newHashSet();
            Group g = account.lookupGroupByName(group);
            for (User u : g.getUsers()) {
                localUserSet.add(u.getName());
            }
            for (String user : users) {
                if (localUserSet.contains(user)) {
                    localUserSet.remove(user);
                    continue;
                }
                LOG.debug((Object)("Adding " + user + " to " + g.getName()));
                g.addUserByName(user);
            }
            for (String user : localUserSet) {
                LOG.debug((Object)("Removing " + user + " from " + g.getName()));
                g.removeUserByName(user);
            }
        }
        catch (AuthException e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.warn((Object)("Failed to update group " + group + " in " + account.getName()), (Throwable)e);
        }
    }

    private static Set<String> getLocalGroupSet(Account account) throws AuthException {
        HashSet groupSet = Sets.newHashSet();
        for (Group group : account.getGroups()) {
            groupSet.add(group.getName());
        }
        return groupSet;
    }

    private static void removeObsoleteUsers(Account account, Set<String> oldUserSet) {
        oldUserSet.remove("admin");
        LOG.debug((Object)("Removing obsolete users: " + oldUserSet + ", in account " + account.getName()));
        for (String user : oldUserSet) {
            try {
                account.deleteUser(user, true, true);
            }
            catch (AuthException e) {
                LOG.error((Object)e, (Throwable)e);
                LOG.warn((Object)("Failed to delete user " + user + " in " + account.getName()));
            }
        }
    }

    private static void addNewUser(Account account, String user, Map<String, String> info) throws AuthException {
        LOG.debug((Object)("Adding new user " + user + " in account " + account.getName()));
        if (info == null) {
            LOG.warn((Object)("Empty user info for user " + user));
        }
        account.addUser(user, "/", true, info);
    }

    private static void updateUser(Account account, String user, Map<String, String> map) throws AuthException {
        LOG.debug((Object)("Updating user " + user + " in account " + account.getName()));
        if (map == null) {
            LOG.error((Object)("Empty info map of user " + user));
        } else {
            account.lookupUserByName(user).setInfo(map);
        }
    }

    private static Set<String> getLocalUserSet(Account account) throws AuthException {
        HashSet userSet = Sets.newHashSet();
        for (User user : account.getUsers()) {
            userSet.add(user.getName());
        }
        return userSet;
    }

    private static void removeObsoleteAccounts(Set<String> oldAccountSet) {
        oldAccountSet.remove("eucalyptus");
        LOG.debug((Object)("Removing obsolete accounts: " + oldAccountSet));
        for (String account : oldAccountSet) {
            try {
                Accounts.deleteAccount((String)account, (boolean)false, (boolean)true);
            }
            catch (AuthException e) {
                if ("Can not delete system account".equals(e.getMessage())) continue;
                LOG.error((Object)e, (Throwable)e);
                LOG.warn((Object)("Failed to delete account " + account), (Throwable)e);
            }
        }
    }

    private static Set<String> getLocalAccountSet() throws AuthException {
        HashSet accountSet = Sets.newHashSet();
        for (Account account : Accounts.listAllAccounts()) {
            accountSet.add(account.getName());
        }
        return accountSet;
    }

    private static String parseIdFromDn(String dn) {
        if (Strings.isNullOrEmpty((String)dn)) {
            return null;
        }
        try {
            LdapName ln = new LdapName(dn);
            if (ln.size() > 0) {
                return (String)ln.getRdn(ln.size() - 1).getValue();
            }
        }
        catch (InvalidNameException e) {
            LOG.error((Object)e, (Throwable)e);
            LOG.warn((Object)("Invalid DN " + dn), (Throwable)e);
        }
        return null;
    }

    private static String getId(String dn, String idAttrName, Attributes attrs) throws NamingException {
        String id = null;
        id = Strings.isNullOrEmpty((String)idAttrName) ? LdapSync.parseIdFromDn(dn) : LdapSync.getAttrWithNullCheck(attrs, idAttrName);
        if (Strings.isNullOrEmpty((String)id)) {
            throw new NamingException("Empty ID for " + attrs);
        }
        return id.toLowerCase();
    }

    private static Set<String> getMembers(String memberAttrName, Attributes attrs, Map<String, String> dnToId) throws NamingException {
        HashSet members = Sets.newHashSet();
        String memberItemType = lic.getMembersItemType();
        String memberId = null;
        String memberDn = null;
        Attribute membersAttr = attrs.get(memberAttrName);
        if (membersAttr != null) {
            NamingEnumeration<?> names = membersAttr.getAll();
            while (names.hasMore()) {
                if ("identity".equals(memberItemType)) {
                    memberId = LdapSync.sanitizeUserGroupId((String)names.next());
                } else {
                    memberDn = (String)names.next();
                    memberId = dnToId.get(memberDn.toLowerCase());
                }
                if (Strings.isNullOrEmpty((String)memberId)) {
                    LOG.warn((Object)("Can not map member DN " + memberDn + " to ID for " + attrs + ". Check corresponding selection section in your LIC."));
                    continue;
                }
                members.add(memberId.toLowerCase());
            }
        }
        return members;
    }

    private static void retrieveSelection(LdapClient ldap, String baseDn, Selection selection, String[] attrNames, LdapEntryProcessor processor) throws LdapException {
        LOG.debug((Object)("Search users by: baseDn=" + baseDn + ", attributes=" + attrNames + ", selection=" + selection));
        try {
            NamingEnumeration<SearchResult> results = ldap.search(baseDn, selection.getSearchFilter(), attrNames);
            while (results.hasMore()) {
                SearchResult res = results.next();
                try {
                    if (selection.getNotSelected().contains(res.getNameInNamespace())) continue;
                    processor.processLdapEntry(res.getNameInNamespace().toLowerCase(), res.getAttributes());
                }
                catch (NamingException e) {
                    LOG.debug((Object)("Failed to retrieve entry " + res));
                    LOG.error((Object)e, (Throwable)e);
                }
            }
            for (String dn : selection.getSelected()) {
                Attributes attrs = null;
                try {
                    attrs = ldap.getContext().getAttributes(dn, attrNames);
                    processor.processLdapEntry(dn.toLowerCase(), attrs);
                }
                catch (NamingException e) {
                    LOG.debug((Object)("Failed to retrieve entry " + attrs));
                    LOG.error((Object)e, (Throwable)e);
                }
            }
        }
        catch (NamingException e) {
            LOG.error((Object)e, (Throwable)e);
            throw new LdapException((Throwable)e);
        }
    }

    private static void loadLdapAccountingGroups(LdapClient ldap, final LdapIntegrationConfiguration lic, final Map<String, String> groupDnToId, final Map<String, Set<String>> accountingGroups) throws LdapException {
        LOG.debug((Object)"Loading accounting groups from LDAP/AD");
        HashSet attrNames = Sets.newHashSet();
        attrNames.add(lic.getGroupsAttribute());
        if (!Strings.isNullOrEmpty((String)lic.getAccountingGroupIdAttribute())) {
            attrNames.add(lic.getAccountingGroupIdAttribute());
        }
        LOG.debug((Object)("Attributes to load for accounting groups: " + attrNames));
        LdapSync.retrieveSelection(ldap, lic.getAccountingGroupBaseDn(), lic.getAccountingGroupsSelection(), attrNames.toArray(new String[0]), new LdapEntryProcessor(){

            @Override
            public void processLdapEntry(String dn, Attributes attrs) throws NamingException {
                LOG.debug((Object)("Retrieved accounting group: " + dn + " -> " + attrs));
                accountingGroups.put(LdapSync.sanitizeAccountId(LdapSync.getId(dn, lic.getAccountingGroupIdAttribute(), attrs)), LdapSync.getMembers(lic.getGroupsAttribute(), attrs, groupDnToId));
            }
        });
    }

    private static void loadLdapGroups(LdapClient ldap, final LdapIntegrationConfiguration lic, final Map<String, String> userDnToId, final Map<String, String> groupDnToId, final Map<String, Set<String>> groups) throws LdapException {
        LOG.debug((Object)"Loading groups from LDAP/AD");
        HashSet attrNames = Sets.newHashSet();
        attrNames.add(lic.getUsersAttribute());
        if (!Strings.isNullOrEmpty((String)lic.getGroupIdAttribute())) {
            attrNames.add(lic.getGroupIdAttribute());
        }
        LOG.debug((Object)("Attributes to load for groups: " + attrNames));
        LdapSync.retrieveSelection(ldap, lic.getGroupBaseDn(), lic.getGroupsSelection(), attrNames.toArray(new String[0]), new LdapEntryProcessor(){

            @Override
            public void processLdapEntry(String dn, Attributes attrs) throws NamingException {
                LOG.debug((Object)("Retrieved group: " + dn + " -> " + attrs));
                String id = LdapSync.sanitizeUserGroupId(LdapSync.getId(dn, lic.getGroupIdAttribute(), attrs));
                groupDnToId.put(dn, id);
                groups.put(id, LdapSync.getMembers(lic.getUsersAttribute(), attrs, userDnToId));
            }
        });
    }

    private static void loadLdapUsers(LdapClient ldap, final LdapIntegrationConfiguration lic, final Map<String, String> userDnToId, final Map<String, Map<String, String>> users) throws LdapException {
        LOG.debug((Object)"Loading users from LDAP/AD");
        HashSet attrNames = Sets.newHashSet();
        attrNames.addAll(lic.getUserInfoAttributes().keySet());
        if (!Strings.isNullOrEmpty((String)lic.getUserIdAttribute())) {
            attrNames.add(lic.getUserIdAttribute());
        }
        if (!Strings.isNullOrEmpty((String)lic.getUserSaslIdAttribute())) {
            attrNames.add(lic.getUserSaslIdAttribute());
        }
        LOG.debug((Object)("Attributes to load for users: " + attrNames));
        LdapSync.retrieveSelection(ldap, lic.getUserBaseDn(), lic.getUsersSelection(), attrNames.toArray(new String[0]), new LdapEntryProcessor(){

            @Override
            public void processLdapEntry(String dn, Attributes attrs) throws NamingException {
                LOG.debug((Object)("Retrieved user: " + dn + " -> " + attrs));
                String id = LdapSync.sanitizeUserGroupId(LdapSync.getId(dn, lic.getUserIdAttribute(), attrs));
                userDnToId.put(dn, id);
                HashMap infoMap = Maps.newHashMap();
                for (String attrName : lic.getUserInfoAttributes().keySet()) {
                    String infoKey = lic.getUserInfoAttributes().get(attrName);
                    String infoVal = LdapSync.getAttrWithNullCheck(attrs, attrName);
                    if (infoVal == null) continue;
                    infoMap.put(infoKey, infoVal);
                }
                infoMap.put("dn", dn);
                if (!Strings.isNullOrEmpty((String)lic.getUserSaslIdAttribute())) {
                    infoMap.put("saslid", LdapSync.getAttrWithNullCheck(attrs, lic.getUserSaslIdAttribute()));
                }
                users.put(id, infoMap);
            }
        });
    }

    private static String getAttrWithNullCheck(Attributes attrs, String attrName) throws NamingException {
        Attribute attr = attrs.get(attrName);
        if (attr != null) {
            return ((String)attr.get()).toLowerCase();
        }
        return null;
    }

    private static String sanitizeUserGroupId(String id) {
        if (id != null) {
            return id.replaceAll("[^a-zA-Z0-9+=,.@\\-_ ]", "-").replaceAll("-{2,}", "-");
        }
        return id;
    }

    private static String sanitizeAccountId(String id) {
        if (id != null) {
            return id.replaceAll("[^a-z0-9\\-]", "-").replaceAll("-{2,}", "-");
        }
        return id;
    }

    static {
        lic = DEFAULT_LIC = new LdapIntegrationConfiguration();
        inSync = false;
        TIMER_LISTENER = new ClockTickListener();
    }

    private static class ClockTickListener
    implements EventListener<Event> {
        private ClockTickListener() {
        }

        public void fireEvent(Event event) {
            if (Bootstrap.isOperational().booleanValue() && Hosts.isCoordinator() && event instanceof ClockTick) {
                LdapSync.periodicSync();
            }
        }
    }

    private static interface LdapEntryProcessor {
        public void processLdapEntry(String var1, Attributes var2) throws NamingException;
    }
}

