//
// C++ Implementation: powermanager
//
// Author: Oliver Groß <z.o.gross@gmx.de>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
#include <QtGui>
#include "powermanager.h"
#include "common.h"
#include "csettingsdialog.h"
#include "batteryicon.h"
#include "qtimermessagebox.h"

namespace qbat {
	using namespace std;
	
	CPowerManager::CPowerManager(QObject * parent) :
		QObject(parent),
		m_Timer(-1),
		m_DataTimer(-1),
		m_ACPlug(false),
		m_CriticalHandled(false),
		m_MeanRealativeCharge(-1)
	{
		readSettings();
		m_ContextMenu.addAction(tr("&Settings"), this, SLOT(showSettings()))->setEnabled(CBatteryIcon::sysfsDir().exists());
		m_ContextMenu.addAction(tr("&About"), this, SLOT(showAbout()));
		m_ContextMenu.addSeparator();
		m_ContextMenu.addAction(tr("&Quit"), qApp, SLOT(quit()));
		
		m_DefaultIcon = new QSystemTrayIcon(this);
		m_DefaultIcon->setIcon(QIcon(PATH_ICON_QBAT_MONO));
		m_DefaultIcon->setContextMenu(&m_ContextMenu);
		
		if (CBatteryIcon::sysfsDir().exists()) {
			m_DataTimer = startTimer(15000);
			m_Timer = startTimer(m_Settings.pollingRate);
		}
		updateSupplies();
	}
	
	CPowerManager::~CPowerManager() {
		killTimer(m_Timer);
		killTimer(m_DataTimer);
	}
	
	inline void CPowerManager::readSettings() {
		QSettings settingsFile(STRING_ORGANIZATION, STRING_APPNAME);
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_MAIN);
		m_Settings.pollingRate = settingsFile.value(STRING_SETTINGS_POLLINGRATE, m_Settings.pollingRate).toUInt();
		m_Settings.showBalloon = settingsFile.value(STRING_SETTINGS_SHOWBALLOON, m_Settings.showBalloon).toBool();
		settingsFile.endGroup();
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_ICONS);
		m_Settings.oldIconStyle = settingsFile.value(STRING_SETTINGS_OLDSTYLE, m_Settings.oldIconStyle).toBool();
		m_Settings.showChargeLabel = settingsFile.value(STRING_SETTINGS_SHOWLABEL, m_Settings.showChargeLabel).toBool();
		
		settingsFile.beginWriteArray(STRING_SETTINGS_COLORS);
		for (int i = 0; i < INT_COUNT_COLORS; i++) {
			settingsFile.setArrayIndex(i);
			m_Settings.colors[i] = settingsFile.value(STRING_SETTINGS_VALUE, m_Settings.colors[i]).value<QRgb>();
		}
		settingsFile.endArray();
		
		settingsFile.endGroup();
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_CRITICAL);
		m_Settings.handleCritical = settingsFile.value(STRING_SETTINGS_HANDLE, m_Settings.handleCritical).toBool();
		m_Settings.criticalCapacity = settingsFile.value(STRING_SETTINGS_CRITICALCAP, m_Settings.criticalCapacity).toUInt();
		m_Settings.executeCommand = settingsFile.value(STRING_SETTINGS_EXECCOMMAND, m_Settings.executeCommand).toBool();
		m_Settings.criticalCommand = settingsFile.value(STRING_SETTINGS_COMMAND, m_Settings.criticalCommand).toString();
		m_Settings.confirmCommand = settingsFile.value(STRING_SETTINGS_CONFIRMCOMMAND, m_Settings.confirmCommand).toBool();
		m_Settings.confirmWithTimeout = settingsFile.value(STRING_SETTINGS_CONFIRMTIMEOUT, m_Settings.confirmWithTimeout).toBool();
		m_Settings.timeoutValue = settingsFile.value(STRING_SETTINGS_TIMEOUTVALUE, m_Settings.timeoutValue).toUInt();
		settingsFile.endGroup();
	}
	
	inline void CPowerManager::writeSettings() {
		QSettings settingsFile(STRING_ORGANIZATION, STRING_APPNAME);
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_MAIN);
		settingsFile.setValue(STRING_SETTINGS_POLLINGRATE, m_Settings.pollingRate);
		settingsFile.setValue(STRING_SETTINGS_SHOWBALLOON, m_Settings.showBalloon);
		settingsFile.endGroup();
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_ICONS);
		settingsFile.setValue(STRING_SETTINGS_OLDSTYLE, m_Settings.oldIconStyle);
		settingsFile.setValue(STRING_SETTINGS_SHOWLABEL, m_Settings.showChargeLabel);
		
		settingsFile.beginWriteArray(STRING_SETTINGS_COLORS, INT_COUNT_COLORS);
		for (int i = 0; i < INT_COUNT_COLORS; i++) {
			settingsFile.setArrayIndex(i);
			settingsFile.setValue(STRING_SETTINGS_VALUE, m_Settings.colors[i]);
		}
		settingsFile.endArray();
		
		settingsFile.endGroup();
		
		settingsFile.beginGroup(STRING_SETTINGS_GRP_CRITICAL);
		settingsFile.setValue(STRING_SETTINGS_HANDLE, m_Settings.handleCritical);
		settingsFile.setValue(STRING_SETTINGS_CRITICALCAP, m_Settings.criticalCapacity);
		settingsFile.setValue(STRING_SETTINGS_EXECCOMMAND, m_Settings.executeCommand);
		settingsFile.setValue(STRING_SETTINGS_COMMAND, m_Settings.criticalCommand);
		settingsFile.setValue(STRING_SETTINGS_CONFIRMCOMMAND, m_Settings.confirmCommand);
		settingsFile.setValue(STRING_SETTINGS_CONFIRMTIMEOUT, m_Settings.confirmWithTimeout);
		settingsFile.setValue(STRING_SETTINGS_TIMEOUTVALUE, m_Settings.timeoutValue);
		settingsFile.endGroup();
	}
	
	void CPowerManager::timerEvent(QTimerEvent * event) {
		if (event->timerId() == m_Timer)
			updateSupplies();
		else if (event->timerId() == m_DataTimer) {
			updateBatteryData();
			updateMergedData();
			checkCritical();
		}
	}
	
	void CPowerManager::updateSupplies() {
		if (CBatteryIcon::sysfsDir().exists()) {
			bool oldAC = m_ACPlug;
			m_ACPlug = false;
			
			QStringList powerSupplies = CBatteryIcon::sysfsDir().entryList(QDir::Dirs | QDir::NoDotAndDotDot);
			
			QList<CBatteryIcon *> newBatteryIcons;
			CBatteryIcon * currentBatteryIcon;
			
			foreach(QString i, powerSupplies) {
				// this is possibly not very reliable
				if (CBatteryIcon::sysfsDir().exists(i + "/online")) {
					if (readIntSysFile(CBatteryIcon::sysfsDir().filePath(i + "/online").toAscii().constData()) == 1)
						m_ACPlug = true;
				}
				else {
					if (m_BatteryIcons.contains(i))
						currentBatteryIcon = m_BatteryIcons.take(i);
					else {
						currentBatteryIcon = new CBatteryIcon(&m_Settings, i, this);
						currentBatteryIcon->setContextMenu(&m_ContextMenu);
						currentBatteryIcon->updateData();
						currentBatteryIcon->updateIcon();
						currentBatteryIcon->updateToolTip();
					}
					newBatteryIcons << currentBatteryIcon;
				}
			}
			
			foreach(CBatteryIcon * i, m_BatteryIcons)
				delete i;
			
			m_BatteryIcons.clear();
			
			foreach(CBatteryIcon * i, newBatteryIcons) {
				m_BatteryIcons.insert(i->data().name, i);
				i->setVisible(true);
			}
			
			if (oldAC != m_ACPlug)
				updateBatteryData();
			
			updateMergedData();
			
			m_DefaultIcon->setToolTip(tr("QBat - %1").arg(m_ACPlug ? tr("AC adapter plugged in") : tr("AC adapter unplugged")));
			m_DefaultIcon->setVisible(m_BatteryIcons.isEmpty());
			
			checkCritical();
		}
		else {
			m_DefaultIcon->setToolTip(tr("QBat - %1").arg(tr("no information available")));
			m_DefaultIcon->setVisible(true);
		}
		
	}
	
	void CPowerManager::updateMergedData() {
		int relCount = 0;
		int relSum = 0;
		foreach (CBatteryIcon * battery, m_BatteryIcons) {
			if (battery->data().relativeCharge >= 0) {
				relSum += battery->data().relativeCharge;
				relCount++;
			}
		}
		
		m_MeanRealativeCharge = relCount ? qRound(relSum / relCount) : -1;
	}
	
	void CPowerManager::checkCritical() {
		if (
			m_ACPlug || m_CriticalHandled ||
			m_MeanRealativeCharge == -1 || !m_Settings.handleCritical ||
			m_MeanRealativeCharge > m_Settings.criticalCapacity
		) {
			m_CriticalHandled = false;
			return;
		}
		
		QString msgTitle = (m_Settings.executeCommand && m_Settings.confirmWithTimeout) ?
			tr("QBat - critical battery capacity (will automatically choose ok on timeout)"):
			tr("QBat - critical battery capacity");
		QString msgText = (m_Settings.executeCommand && m_Settings.confirmCommand) ?
			tr("WARNING: The attached battery(s) reached the critical mark.\n"
			"Click cancel and please make sure to save and shut down soon or provide another source of power\n"
			"or:\n"
			"Click ok to execute:\n%1").arg(m_Settings.criticalCommand) :
			tr("WARNING: The attached battery(s) reached the critical mark.\n"
			"Please make sure to save and shut down soon or provide another source of power.");
		if (m_Settings.executeCommand && (!m_Settings.confirmCommand || (QTimerMessageBox::warning(NULL, msgTitle, msgText,
			(m_Settings.confirmWithTimeout) ? m_Settings.timeoutValue : -1, QMessageBox::Ok | QMessageBox::Cancel))))
			QProcess::startDetached(m_Settings.criticalCommand);
		else {
			if (m_Settings.showBalloon)
				m_BatteryIcons.begin().value()->showMessage(msgTitle, msgText, QSystemTrayIcon::Warning, 7000);
			else
				QMessageBox::warning(NULL, msgTitle, msgText);
		}
		m_CriticalHandled = true;
	}
	
	void CPowerManager::updateBatteryData() {
		foreach (CBatteryIcon * battery, m_BatteryIcons)
			battery->updateData();
	}
	
	void CPowerManager::restartTimer() {
		killTimer(m_Timer);
		m_Timer = startTimer(m_Settings.pollingRate);
		killTimer(m_DataTimer);
		m_DataTimer = startTimer(15000);
	}
	
	void CPowerManager::showSettings() {
		CSettingsDialog dialog;
		
		if (dialog.execute(&m_Settings)) {
			foreach (CBatteryIcon * i, m_BatteryIcons) {
				delete i;
			}
			
			m_BatteryIcons.clear();
			m_MeanRealativeCharge = -1;
			
			updateSupplies();
			restartTimer();
			writeSettings();
		}
	}
	
	void CPowerManager::showAbout() {
		if (m_Settings.showBalloon) {
			if (m_DefaultIcon->isVisible())
				m_DefaultIcon->showMessage(tr("About QBat"), STRING_FULLNAME + "\nv" + QString(STRING_VERSION),
					QSystemTrayIcon::Information, 4000);
			else
				m_BatteryIcons.begin().value()->showMessage(tr("About QBat"), STRING_FULLNAME + "\nv" + QString(STRING_VERSION),
					QSystemTrayIcon::Information, 4000);
		}
		else {
			QMessageBox aboutBox;
			
			aboutBox.setWindowIcon(QIcon(PATH_ICON_QBAT));
			aboutBox.setIconPixmap(QPixmap(PATH_ICON_QBAT));
			aboutBox.setWindowTitle(tr("About QBat"));
			aboutBox.setText(STRING_FULLNAME + "\nv" + QString(STRING_VERSION));
			aboutBox.setStandardButtons(QMessageBox::Ok);
			
			aboutBox.exec();
		}
	}
}
