//
// C++ Implementation: batteryicon
//
// Author: Oliver Groß <z.o.gross@gmx.de>, (C) 2008
//
// Copyright: See COPYING file that comes with this distribution
//
#include "batteryicon.h"
#include "common.h"
#include <QPainter>
#include <QMessageBox>
#include <QApplication>

namespace qbat {
	QDir CBatteryIcon::m_SysfsDir(PATH_SYSFS_DIR);
	
	CBatteryIcon::CBatteryIcon(Settings * settings, QString batteryName, QObject * parent) :
		QSystemTrayIcon(parent),
		m_Settings(settings)
	{
		m_Data.name = batteryName;
		
		if (m_Settings->oldIconStyle) {
			m_Icon = QPixmap(28, 28);
			m_Icon.fill(Qt::transparent);
			updateIcon();
		}
		else {
			m_Icon.load(PATH_ICON_BATTERY);
			setIcon(m_Icon);
		}
		
		connect(this, SIGNAL(activated(QSystemTrayIcon::ActivationReason)),
			this, SLOT(handleClicks(QSystemTrayIcon::ActivationReason)));
	}
	
	CBatteryIcon::~CBatteryIcon() {
	}
	
	void CBatteryIcon::updateIcon() {
		QPainter painter;
		if (m_Settings->oldIconStyle) {
			painter.begin(&m_Icon);
			
			if (m_Data.currentCapacity != m_Data.fullCapacity || m_Data.status != STATE_BATTERY_FULL) {
				painter.setPen(QColor(m_Settings->colors[INT_COLOR_PEN_BASE + m_Data.status]));
				
				painter.setBrush(QColor(m_Settings->colors[INT_COLOR_BRUSH_MAIN]));
				painter.drawRect(0, 4, 27, 23);
				
				int chargedPixels = (int)(22 * m_Data.relativeCharge / 100.0);
				
				painter.fillRect(1, 5 + 22 - chargedPixels, 26, chargedPixels, QColor(m_Settings->colors[INT_COLOR_BRUSH_BASE + m_Data.status]));
				
				painter.setBrush(QColor(m_Settings->colors[INT_COLOR_BRUSH_POLE_BASE + m_Data.status]));
			}
			else {
				painter.setPen(QColor(m_Settings->colors[INT_COLOR_PEN_FULL]));
				painter.setBrush(QColor(m_Settings->colors[INT_COLOR_BRUSH_FULL]));
				painter.drawRect(0, 4, 27, 23);
				
				painter.setBrush(QColor(m_Settings->colors[INT_COLOR_BRUSH_POLE_FULL]));
			}
			painter.drawRect(7, 0, 13, 4);
			
			if (m_Settings->showChargeLabel) {
				painter.setBrush(Qt::NoBrush);
				
				((QFont&)painter.font()).setPixelSize((m_Data.relativeCharge < 100) ? 15 : 12);
				
				painter.setRenderHint(QPainter::TextAntialiasing);
				((QFont&)painter.font()).setBold(true);
				
				painter.drawText(1, 9, 26, 16, Qt::AlignHCenter, (m_Data.relativeCharge == -1) ? QString('?') : QString::number(m_Data.relativeCharge));
			}
			
			painter.end();
			setIcon(m_Icon);
		}
		else {
			QPixmap newIcon = m_Icon;
			painter.begin(&newIcon);
			
			int chargedPixels = INT_CHARGE_HEIGHT;
			if (m_Data.currentCapacity != m_Data.fullCapacity || m_Data.status != STATE_BATTERY_FULL)
				chargedPixels = (int)(INT_CHARGE_HEIGHT * m_Data.relativeCharge / 100.0);
				
			painter.fillRect(
				INT_CHARGE_XOFFSET,
				INT_CHARGE_YOFFSET + INT_CHARGE_HEIGHT - chargedPixels,
				INT_CHARGE_WIDTH,
				chargedPixels,
				QColor(m_Settings->colors[INT_COLOR_BRUSH_BASE + m_Data.status])
				);
			
			QString overlayStr = (m_Data.relativeCharge == -1) ? QString('?') : QString::number(m_Data.relativeCharge);
			int oldWidth = m_Icon.width();
			
			// overlay
			if (m_Settings->showChargeLabel) {
				QFont font = QApplication::font();
				font.setBold(true);
				
				float pointSize = font.pointSizeF();
				QFontMetrics fm(font);
				int w = fm.width(overlayStr);
				if( w > (oldWidth - 2) ) {
					pointSize *= float(oldWidth - 2) / float(w);
					font.setPointSizeF(pointSize);
				}
				painter.setFont(font);
				
				fm = QFontMetrics(font);
				QRect boundingRect = fm.boundingRect(overlayStr);
				boundingRect.setHeight(qMin(boundingRect.height(), INT_CHARGE_HEIGHT));
				boundingRect.adjust(0, 0, boundingRect.width() % 2, boundingRect.height() % 2);
				boundingRect.moveTo((oldWidth - boundingRect.width()) / 2, INT_CHARGE_YOFFSET + INT_CHARGE_HEIGHT / 2 - boundingRect.height() / 2);
				painter.setOpacity(0.7);
				painter.fillRect(boundingRect, QColor(m_Settings->colors[INT_COLOR_BRUSH_LABEL]));
				
				painter.setBrush(Qt::NoBrush);
				painter.setPen(QColor(m_Settings->colors[INT_COLOR_PEN_BASE + m_Data.status]));
				
				boundingRect.adjust(0, 1, 0, 1);
				painter.drawText(boundingRect, Qt::AlignCenter, overlayStr);
			}
			painter.end();
			setIcon(newIcon);
		}
	}
	
	void CBatteryIcon::updateToolTip() {
		QString newToolTip = m_Data.name;
		
		if (m_Data.relativeCharge != -1)
			newToolTip = tr("%1: %2%").arg(newToolTip).arg(m_Data.relativeCharge);
		
		newToolTip += '\n';
		
		switch (m_Data.status) {
		case STATE_BATTERY_DISCHARGING:
			newToolTip += tr("status: %1").arg(tr("discharging"));
			if (m_Data.rate) {
				newToolTip += '\n';
				qreal remainingTime  = (qreal)(m_Data.currentCapacity) / m_Data.rate;
				int remainingHours   = (int)remainingTime;
				int remainungMinutes = (int)(remainingTime * 60) % 60;
				newToolTip += tr("remaining time: %1:%2").arg(remainingHours, 2, 10, QChar('0')).arg(remainungMinutes, 2, 10, QChar('0'));
			}
			break;
		case STATE_BATTERY_CHARGING:
			newToolTip += tr("status: %1").arg(tr("charging"));
			if (m_Data.rate && m_Data.fullCapacity) {
				newToolTip += '\n';
				qreal remainingTime  = (qreal)(m_Data.fullCapacity - m_Data.currentCapacity) / m_Data.rate;
				int remainingHours   = (int)remainingTime;
				int remainungMinutes = (int)(remainingTime * 60) % 60;
				newToolTip += tr("remaining time: %1:%2").arg(remainingHours, 2, 10, QChar('0')).arg(remainungMinutes, 2, 10, QChar('0'));
			}
			break;
		case STATE_BATTERY_FULL:
			newToolTip += tr("status: %1").arg(tr("full"));
			break;
		default:
			newToolTip += tr("status: %1").arg(tr("unknown"));
			break;
		}
		newToolTip += '\n';
		
		if (m_Data.energyUnits) {
			if ((m_Data.rate) && (m_Data.status != STATE_BATTERY_FULL)) {
				double rateW = qRound(m_Data.rate / 100000.0) / 10.0;
				if (m_Data.voltage) {
					double rateA = qRound((m_Data.rate / m_Data.voltage) / 1000.0) / 10.0;
					newToolTip += tr("current rate: %1W / %2A").arg(rateW).arg(rateA) + '\n';
				}
				else
					newToolTip += tr("current rate: %1W").arg(rateW) + '\n';
			}
			
			newToolTip += tr("current capacity: %1mWh").arg(m_Data.currentCapacity / 1000);
			
			if (m_Data.fullCapacity)
				newToolTip += '\n' + tr("last full capacity: %1mWh").arg(m_Data.fullCapacity / 1000);
			
			if (m_Data.designCapacity)
				newToolTip += '\n' + tr("design capacity: %1mWh").arg(m_Data.designCapacity / 1000);
		}
		else {
			if ((m_Data.rate) && (m_Data.status != STATE_BATTERY_FULL)) {
				double rateA = m_Data.rate / 100000.0;
				if (m_Data.voltage) {
					double rateW = qRound(rateA * m_Data.voltage / 100.0) / 10.0;
					newToolTip += tr("current rate: %1W / %2A").arg(rateW).arg(qRound(rateA) / 10.0) + '\n';
				}
				else
					newToolTip += tr("current rate: %1A").arg(qRound(rateA) / 10.0) + '\n';
			}
			
			newToolTip += tr("current capacity: %1mAh").arg(m_Data.currentCapacity / 1000);
			
			if (m_Data.fullCapacity)
				newToolTip += '\n' + tr("last full capacity: %1mAh").arg(m_Data.fullCapacity / 1000);
			
			if (m_Data.designCapacity)
				newToolTip += '\n' + tr("design capacity: %1mAh").arg(m_Data.designCapacity / 1000);
		}
		
		if (m_Data.designCapacity && m_Data.fullCapacity < m_Data.designCapacity)
				newToolTip += '\n' + tr("capacity loss: %1%").arg((int)((1 - (double)m_Data.fullCapacity / m_Data.designCapacity) * 100));
		
		setMessage(newToolTip);
	}
	
	void CBatteryIcon::updateData(int currentCapacity, int fullCapacity, int designCapacity, int rate, int voltage, int status, bool energyUnits) {
		m_Data.energyUnits = energyUnits;
		
		bool noupdate = true;
		bool doUpdateIcon = false;
		
		if (rate != m_Data.rate) {
			noupdate = false;
			m_Data.rate = rate;
		}
		
		if (voltage != m_Data.voltage) {
			noupdate = false;
			m_Data.voltage = voltage;
		}
		
		if (status != m_Data.status) {
			noupdate = false;
			doUpdateIcon = true;
			m_Data.status = status;
		}
		
		if (currentCapacity != m_Data.currentCapacity) {
			noupdate = false;
			
			// calculate rate if it is missing
			if (!m_Data.rate && m_Data.currentCapacity)
				m_Data.rate = qRound(abs(currentCapacity - m_Data.currentCapacity) * (3600 / 15.0));
			
			m_Data.currentCapacity = currentCapacity;
		}
		
		if (fullCapacity != m_Data.fullCapacity) {
			noupdate = false;
			m_Data.fullCapacity = fullCapacity;
		}
		
		if (designCapacity != m_Data.designCapacity) {
			noupdate = false;
			m_Data.designCapacity = designCapacity;
		}
		
		if (noupdate)
			return;
		
		int newRelativeCharge = calcRelativeDef(currentCapacity, fullCapacity);
		
		if (newRelativeCharge != m_Data.relativeCharge) {
			m_Data.relativeCharge = newRelativeCharge;
			doUpdateIcon = true;
		}
		
		if (isVisible()) {
			if (doUpdateIcon)
				updateIcon();
			
			updateToolTip();
		}
	}
	
	void CBatteryIcon::updateData() {
		int currentCapacity = 0;
		int fullCapacity = 0;
		int designCapacity = 0;
		int rate = 0;
		int voltage = 0;
		int status = 0;
		bool energyUnits = m_SysfsDir.exists(m_Data.name + CAPTION_NOW(CAPTION_ENERGY));
		
		rate = readIntSysFile(m_SysfsDir.filePath(m_Data.name + "/current_now").toAscii().constData());
		voltage = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_NOW(CAPTION_VOLTAGE)).toAscii().constData()) / 10000;
		
		char buffer[BUF_SIZE];
		readStringFromFile(buffer, m_SysfsDir.filePath(m_Data.name + "/status").toAscii().constData());
		status = toStatusInt(buffer);
		
		if (energyUnits) {
			fullCapacity    = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_FULL(CAPTION_ENERGY)).toAscii().constData());
			designCapacity  = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_DESIGN(CAPTION_ENERGY)).toAscii().constData());
			currentCapacity = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_NOW(CAPTION_ENERGY)).toAscii().constData());
		}
		else {
			fullCapacity    = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_FULL(CAPTION_CHARGE)).toAscii().constData());
			designCapacity  = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_DESIGN(CAPTION_CHARGE)).toAscii().constData());
			currentCapacity = readIntSysFile(m_SysfsDir.filePath(m_Data.name + CAPTION_NOW(CAPTION_CHARGE)).toAscii().constData());
		}
		
		updateData(currentCapacity, fullCapacity, designCapacity, rate, voltage, status, energyUnits);
	}
	
	void CBatteryIcon::setMessage(QString value) {
		m_Message = value;
		setToolTip(tr("QBat - %1").arg(m_Message));
	}
	
	void CBatteryIcon::handleClicks(QSystemTrayIcon::ActivationReason reason) {
		switch (reason) {
			case Trigger:
				if (supportsMessages())
					showMessage(tr("QBat - Information"), m_Message);
				else
					QMessageBox::information(NULL, tr("QBat - Information"), m_Message);
				break;
			default:
				break;
		}
	}
}
