/***************************************************************************//**
    @file l502api.h
    Файл содержит все необходимые описания типов, констант и функций для работы
    с модулем L502 из пользовательской программы.
    @date 11.03.2012
    @author Borisov Alexey <borisov@lcard.ru>
 ******************************************************************************/

#ifndef L502_API_H
#define L502_API_H

#ifdef __cplusplus
extern "C" {
#endif

#include "l502api_compat.h"
#include "x502api.h"


/***************************************************************************//**
    @addtogroup func_open
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Получение списка серийных номеров модулей L502.

    Функция возвращает список номеров всех найденных модулей L502, независимо от
    того, открыты они сейчас или нет.

    Если нужен список только тех модулей, которые не открыты (то есть
    только тех, с которыми можно установить соединение), то для этого можно
    передать в функцию флаг #X502_GETDEVS_FLAGS_ONLY_NOT_OPENED.

 @param[in] serials   Массив размером size*#X502_SERIAL_SIZE байт, в который
                      будут сохранены серийные номера найденных модулей.
                      Может быть NULL, если size=0, а devcnt!=NULL, в случае,
                      если нужно только получить количество модулей в системе.
 @param[in] size      Определяет, сколько максимально серийных номеров может
                      быть сохранено в массив serial. Будут сохранены только
                      первые size серийных номеров.
                      Может быть 0, если serials=NULL
 @param[in] flags     Флаги из #t_x502_getdevs_flags, определяющие поведение
                      функции.
 @param[out] devcnt   Если devcnt!=NULL, то в данную переменную сохраняется
                      общее число найденных модулей L502
                      (может быть больше size).
 @return              Если <0 - код ошибки, иначе количество сохраненных
                      серийных номеров в массиве serials (всегда <= size)
*******************************************************************************/
X502_EXPORT(int32_t) L502_GetSerialList(char serials[][X502_SERIAL_SIZE], uint32_t size,
                                        uint32_t flags, uint32_t *devcnt);

/***************************************************************************//**
    @brief Открытие модуля L502 по его серийному номеру.

    Функция устанавливает связь с модулем L502 по его серийному номеру.
    После успешного выполнения  этой функции, пользователь получает эксклюзивный
    доступ к модулю через описатель модуля. До закрытия связи с помощью
    X502_Close() никто другой установить связь с модулем не сможет
    (будет возвращена ошибка #X502_ERR_DEVICE_ACCESS_DENIED).

    Если в качестве серийного номера передан NULL или пустая строка, то будет
    установлена связь с первым найденным модулем, с которым получится успешно
    ее установить.
    Если в системе нет ни одного модуля, то будет возвращена ошибка
    #X502_ERR_DEVICE_NOT_FOUND. Если в системе присутствуют модули L502, но
    соединение ни с одним из них установить не удалось, то будет возвращена
    ошибка, полученная при попытке установить соединение с последним
    найденным модулем.

    После завершения работы с устройством соединение должно быть закрыто с
    помощью X502_Close().

    @param[in] hnd     Описатель устройства.
    @param[in] serial  Указатель на строку с серийным номером открываемого
                       модуля или NULL.
    @return            Код ошибки.
*******************************************************************************/
X502_EXPORT(int32_t) L502_Open(t_x502_hnd hnd, const char *serial);


/** @} */


/***************************************************************************//**
    @addtogroup func_devrec
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Получить список записей, соответствующих подключенным модулям L502

    Функция находит все подключенные модули L502 и инициализирует
    записи о каждом найденном устройстве и сохраняет их в переданный список
    (если не нулевой).
    Возвращенные в списке записи должны быть очищены после использования
    с помощью X502_FreeDevRecordList() (также в случае повторного
    вызов L502_GetDevRecordsList() с тем же массивом записей, записи, полученные
    при предыдущем вызове, должны быть сперва очищены).

 @param[in] list      Массив для сохранения записей о найденных устройствах.
                      Должен содержать место для сохранения не менее size записей.
                      Может быть NULL, если size=0, а devcnt!=NULL, в случае,
                      если нужно только получить количество модулей в системе.
 @param[in] size      Определяет, сколько максимально записей может
                      быть сохранено в массив list. Будут сохранены только
                      первые size записей, если устройств найденно больше.
 @param[in] flags     Флаги из #t_x502_getdevs_flags, определяющие поведение
                      функции.
 @param[out] devcnt   Если не нулевой указатель, то в данную переменную сохраняется
                      общее число найденных модулей L502 (может быть больше size).
 @return              Если <0 --- код ошибки, иначе количество сохраненных
                      записей о найденных устройствах (всегда <= size).
                      Именно на этот размер нужно сделать в дальнейшем
                      X502_FreeDevRecordList() для освобождения памяти,
                      выделенной под информацию, на которую ссылается запись.
 ******************************************************************************/
X502_EXPORT(int32_t) L502_GetDevRecordsList(t_x502_devrec *list, uint32_t size,
                                             uint32_t flags, uint32_t *devcnt) ;
/** @} */




/***************************************************************************//**
    @addtogroup func_misc
    @{
*******************************************************************************/

/**************************************************************************//**
  @brief Получить версию драйвера модуля L502.

  Функция возвращает версию драйвера, установленного для
  указанного открытого устройства.
  Версия возвращается в виде 32-битного числа.
  Строковое представление возвращенной версии - четыре числа,
  старшее соответствует старшему байту, младшее - младшему.

  Старший байт - мажорная версия, второй по старшинству байт - минорная,
  третий - ревизия, четвертый - номер сборки (не используется - всегда 0).

  Это та версия, которая отображается в диспетчере устройств в Windows или
  с помощью modinfo в Linux.

  Данная функция доступна только для устройств с интерфейсом PCI/PCI-Express (L502)

  @param[in]  hnd     Описатель модуля.
  @param[out] ver     32-битное число, представляющее собой версию драйвера
  @return             Код ошибки.
  *****************************************************************************/
X502_EXPORT(int32_t) L502_GetDriverVersion(t_x502_hnd hnd, uint32_t* ver);

/** @} */



#ifdef __cplusplus
}
#endif

#endif

