/*
 * common part
 */
#ifndef LTRAPI_H_
#define LTRAPI_H_

#include "ltrapitypes.h"
#ifndef LTRAPI_DISABLE_COMPAT_DEFS
    #include "ltrapi_compat_defs.h"
#endif

/*================================================================================================*/
#ifdef _WIN32
    #ifdef LTRAPIWIN_EXPORTS
      #define LTRAPIWIN_DllExport(type)   __declspec(dllexport) type APIENTRY
    #else
      #define LTRAPIWIN_DllExport(type)   __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTRAPIWIN_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTRAPIWIN_DllExport(type) type
#endif


/***************************************************************************//**
  @addtogroup const_list Константы и перечисления.
  @{
  *****************************************************************************/

/** @brief Коды ошибок.

   Коды ошибок, которые могут возвращать функции библиотеки ltrapi.
   Также включают общие коды ошибок для модулей. */
enum en_LTR_ERRORS {
    LTR_OK                            =  0,   /**< Операция выполнена без ошибок */
    LTR_ERROR_UNKNOWN                 = -1,   /**< Неизвестная ошибка.*/
    LTR_ERROR_PARAMETERS              = -2,   /**< Неверно задан один из параметров функции */
    /** @cond obsoleted */
    LTR_ERROR_PARAMETRS               = LTR_ERROR_PARAMETERS,
    /** @endcond */
    LTR_ERROR_MEMORY_ALLOC            = -3,   /**< Ошибка динамического выделения памяти.*/
    LTR_ERROR_OPEN_CHANNEL            = -4,   /**< Ошибка инициализации канала обмена с ltrd */
    LTR_ERROR_OPEN_SOCKET             = -5,   /**< Ошибка подключения к ltrd */
    LTR_ERROR_CHANNEL_CLOSED          = -6,   /**< Канал обмена с ltrd не был создан или закрыт */
    LTR_ERROR_SEND                    = -7,   /**< Ошибка передачи данных в ltrd */
    LTR_ERROR_RECV                    = -8,   /**< Ошибка приема данных от ltrd */
    LTR_ERROR_EXECUTE                 = -9,   /**< Ошибка обмена с крейт-контроллером */
    LTR_WARNING_MODULE_IN_USE         = -10,  /**< Предупреждение: уже создано активное соединение с данным модулем */

    LTR_ERROR_NOT_CTRL_CHANNEL        = -11,  /**< Данная операция доступна только для управляющего соединения */
    LTR_ERROR_SRV_INVALID_CMD         = -12,  /**< Команда не поддерживается ltrd */
    LTR_ERROR_SRV_INVALID_CMD_PARAMS  = -13,  /**< ltrd не поддерживает указанные параметры команды */
    LTR_ERROR_INVALID_CRATE           = -14,  /**< Указанный крейт не найден */
    LTR_ERROR_EMPTY_SLOT              = -15,  /**< В указанном слоте отсутствует модуль */
    LTR_ERROR_UNSUP_CMD_FOR_SRV_CTL   = -16,  /**< Команда не поддерживается управляющим соединением */
    LTR_ERROR_INVALID_IP_ENTRY        = -17,  /**< Неверная запись сетевого адреса крейта */
    LTR_ERROR_NOT_IMPLEMENTED         = -18,  /**< Данная возможность не реализована */
    LTR_ERROR_CONNECTION_CLOSED       = -19,  /**< Соединение было закрыто службой ltrd */
    LTR_ERROR_LTRD_UNKNOWN_RETCODE    = -20,  /**< Неизвестный код ошибки службы ltrd */
    LTR_ERROR_LTRD_CMD_FAILED         = -21,  /**< Ошибка выполнения управляющей команды ltrd */
    LTR_ERROR_INVALID_CON_SLOT_NUM    = -22,  /**< Указан неверный номер слота при открытии соединения */

    LTR_ERROR_INVALID_MODULE_DESCR    = -40,  /**< Неверный описатель модуля */
    LTR_ERROR_INVALID_MODULE_SLOT     = -41,  /**< Указан неверный слот для модуля */
    LTR_ERROR_INVALID_MODULE_ID       = -42,  /**< Неверный ID-модуля в ответе на сброс */
    LTR_ERROR_NO_RESET_RESPONSE       = -43,  /**< Нет ответа на сброс модуля */
    LTR_ERROR_SEND_INSUFFICIENT_DATA  = -44,  /**< Передано меньше слов в модуль, чем запрашивалось */
    LTR_ERROR_RECV_INSUFFICIENT_DATA  = -45,  /**< Принято меньше слов от модуля, чем запрашивалось */
    LTR_ERROR_NO_CMD_RESPONSE         = -46,  /**< Нет ответа на переданную команду */
    LTR_ERROR_INVALID_CMD_RESPONSE    = -47,  /**< Пришел неверный ответ на команду */
    LTR_ERROR_INVALID_RESP_PARITY     = -48,  /**< Ошибка четности в принятом ответе на команду */
    LTR_ERROR_INVALID_CMD_PARITY      = -49,  /**< Ошибка четности в переданной команде */
    LTR_ERROR_UNSUP_BY_FIRM_VER       = -50,  /**< Возможность не поддерживается данной версией прошивки */
    LTR_ERROR_MODULE_STARTED          = -51,  /**< Операция не допустима при запущенном сборе данных */
    LTR_ERROR_MODULE_STOPPED          = -52,  /**< Сбор данных остановлен */
    LTR_ERROR_RECV_OVERFLOW           = -53,  /**< Произошло переполнение буфера службы ltrd при приеме данных */
    LTR_ERROR_FIRM_FILE_OPEN          = -54,  /**< Ошибка открытия файла прошивки */
    LTR_ERROR_FIRM_FILE_READ          = -55,  /**< Ошибка чтения файла прошивки */
    LTR_ERROR_FIRM_FILE_FORMAT        = -56,  /**< Ошибка формата файла прошивки */
    LTR_ERROR_FPGA_LOAD_READY_TOUT    = -57,  /**< Превышен таймаут ожидания готовности ПЛИС к загрузке*/
    LTR_ERROR_FPGA_LOAD_DONE_TOUT     = -58,  /**< Превышен таймаут ожидания перехода ПЛИС в рабочий режим */
    LTR_ERROR_FPGA_IS_NOT_LOADED      = -59,  /**< Прошивка ПЛИС не загружена */
    LTR_ERROR_FLASH_INVALID_ADDR      = -60,  /**< Неверный адрес flash-памяти */
    LTR_ERROR_FLASH_WAIT_RDY_TOUT     = -61,  /**< Превышен таймаут ожидания завершения записи или стирания Flash-памяти */
    LTR_ERROR_FIRSTFRAME_NOTFOUND     = -62,  /**< Не найдено начало кадра данных в потоке от модуля */
    LTR_ERROR_CARDSCONFIG_UNSUPPORTED = -63,  /**< Крейт не поддерживает сохранение конфигурации модулей */
    LTR_ERROR_FLASH_OP_FAILED         = -64,  /**< Ошибка выполнения операции с flash-памятью */
    LTR_ERROR_FLASH_NOT_PRESENT       = -65,  /**< Flash-память не обнаружена */
    LTR_ERROR_FLASH_UNSUPPORTED_ID    = -66,  /**< Обнаружен неподдерживаемый тип flash-памяти */
    LTR_ERROR_FLASH_UNALIGNED_ADDR    = -67,  /**< Невыровненный адрес flash-памяти */
    LTR_ERROR_FLASH_VERIFY            = -68,  /**< Ошибка при проверке записанных данных во flash-память */
    LTR_ERROR_FLASH_UNSUP_PAGE_SIZE   = -69,  /**< Установлен неподдерживаемый размер страницы flash-памяти */
    LTR_ERROR_FLASH_INFO_NOT_PRESENT  = -70,  /**< Отсутствует информация о модуле во flash-памяти */
    LTR_ERROR_FLASH_INFO_UNSUP_FORMAT = -71,  /**< Неподдерживаемый формат информации о модуле во flash-памяти */
    LTR_ERROR_FLASH_SET_PROTECTION    = -72,  /**< Не удалось установить защиту flash-памяти */
    LTR_ERROR_FPGA_NO_POWER           = -73,  /**< Нет питания микросхемы ПЛИС */
    LTR_ERROR_FPGA_INVALID_STATE      = -74,  /**< Недействительное состояние загрузки ПЛИС */
    LTR_ERROR_FPGA_ENABLE             = -75,  /**< Не удалось перевести ПЛИС в разрешенное состояние */
    LTR_ERROR_FPGA_AUTOLOAD_TOUT      = -76,  /**< Истекло время ожидания автоматической загрузки ПЛИС */
    LTR_ERROR_PROCDATA_UNALIGNED      = -77,  /**< Обрабатываемые данные не выравнены на границу кадра */
    LTR_ERROR_PROCDATA_CNTR           = -78,  /**< Ошибка счетчика в обрабатываемых данных */
    LTR_ERROR_PROCDATA_CHNUM          = -79,  /**< Неверный номер канала в обрабатываемых данных */
    LTR_ERROR_PROCDATA_WORD_SEQ       = -80,  /**< Неверная последовательность слов в обрабатываемых данных */
    LTR_ERROR_FLASH_INFO_CRC          = -81,  /**< Неверная контрольная сумма в записанной информации о модуле */
    LTR_ERROR_PROCDATA_UNEXP_CMD      = -82,  /**< Обнаружена неожиданная команда в потоке данных */
};


/** @cond kd_extension */
#define LTR_OPENINFLG_REOPEN (0x01ul)
#define LTR_OPENOUTFLG_REOPEN (0x01ul)
/** @endcond */


/** @brief Режим подключения пользовательских выводов процессора крейта.

   Данные константы задают режим подключения конкретной ножки процессора, которые
   могут использоваться в некоторых крейтах при написании пользовательской прошивки
   процессора. Для штатной прошивки не имеют значения.
   */
enum en_LTR_UserIoCfg {
    LTR_USERIO_DIGIN1   = 1,    /**< Ножка является входом и подключена к DIGIN1 */
    LTR_USERIO_DIGIN2   = 2,    /**< Ножка является входом и подключена к DIGIN2 */
    LTR_USERIO_DIGOUT   = 0,    /**< Ножка является выходом */
    LTR_USERIO_DEFAULT  = LTR_USERIO_DIGOUT /**< Значение по умолчанию */
};

/** @brief Режим работы выходов DIGOUTx крейта.

    Данные значения определяют, какой сигнал будет выставлен на определенном
    выходе DIGOUT разъема синхронизации крейта.
  */
enum en_LTR_DigOutCfg {
    LTR_DIGOUT_CONST0   = 0, /**< Постоянный уровень логического "0" */
    LTR_DIGOUT_CONST1   = 1, /**< Постоянный уровень логической "1" */
    LTR_DIGOUT_USERIO0  = 2, /**< Выход подключен к ножке userio0 */
    LTR_DIGOUT_USERIO1  = 3, /**< Выход подключен к ножке userio1 */
    LTR_DIGOUT_DIGIN1   = 4, /**< Выход подключен ко входу DIGIN1 */
    LTR_DIGOUT_DIGIN2   = 5, /**< Выход подключен ко входу DIGIN2 */
    LTR_DIGOUT_START    = 6, /**< На выход подаются импульсы, соответствующие моментам генерации метки "СТАРТ" */
    LTR_DIGOUT_SECOND   = 7, /**< На выход подаются импульсы, соответствующие моментам генерации метки "СЕКУНДА" */
    LTR_DIGOUT_IRIG     = 8, /**< Контроль сигналов точного времени IRIG (digout1: готовность, digout2: секунда) */
    LTR_DIGOUT_DEFAULT  = LTR_DIGOUT_CONST0 /**< Значение по умолчанию */
};

/** @brief Режим генерации синхрометок.

    Данные значения используются для указания условия, по которому будут генерироваться
    метки "СТАРТ" или "СЕКУНДА".
*/
enum en_LTR_MarkMode {
    LTR_MARK_OFF                = 0, /**< Генерация метки отключена */
    LTR_MARK_EXT_DIGIN1_RISE    = 1, /**< Метка генерируется по фронту сигнала на входе DIGIN1 */
    LTR_MARK_EXT_DIGIN1_FALL    = 2, /**< Метка генерируется по спаду сигнала на входе DIGIN1 */
    LTR_MARK_EXT_DIGIN2_RISE    = 3, /**< Метка генерируется по фронту сигнала на входе DIGIN2 */
    LTR_MARK_EXT_DIGIN2_FALL    = 4, /**< Метка генерируется по спаду сигнала на входе DIGIN2 */

    /** Внутренняя генерация метки крейт-контроллером. Для метки "СТАРТ" генерация выполняется
        однократно при вызове LTR_MakeStartMark(). Для метки "СЕКУНДА" --- по таймеру крейта
        (раз в секунду), начиная с вызова LTR_StartSecondMark(). */
    LTR_MARK_INTERNAL           = 5,

    /** @cond kd_extension */
    /* Источник метки - вход NAMUR (только для LTR-E-7/15) */
    LTR_MARK_NAMUR1_LO2HI       = 8,    /* по сигналу NAMUR1 (START_IN), возрастание тока */
    LTR_MARK_NAMUR1_HI2LO       = 9,    /* по сигналу NAMUR1 (START_IN), спад тока */
    LTR_MARK_NAMUR2_LO2HI       = 10,   /* по сигналу NAMUR2 (M1S_IN), возрастание тока */
    LTR_MARK_NAMUR2_HI2LO       = 11,   /* по сигналу NAMUR2 (M1S_IN), спад тока */
    /** @endcond */

    /* Источник метки - декодер сигналов точного времени IRIG-B006
       IRIG может использоваться только для меток "СЕКУНДА", для "СТАРТ" игнорируется */
    LTR_MARK_SEC_IRIGB_DIGIN1   = 16,   /**< Источником метки является декодер
                                             сигналов точного времени IRIG-B006.
                                             Может использоваться только для
                                             секундных меток.
                                             В качестве входного сигнала используется
                                             сигнал со входа DIGIN1 */
    LTR_MARK_SEC_IRIGB_nDIGIN1  = 17,   /**< Аналогично #LTR_MARK_SEC_IRIGB_DIGIN1, но
                                             в качестве входного сигнала используется
                                             инвертированный сигнал со входа DIGIN1 */
    LTR_MARK_SEC_IRIGB_DIGIN2   = 18,   /**< Аналогично #LTR_MARK_SEC_IRIGB_DIGIN1, но
                                             в качестве входного сигнала используется
                                             сигнал со входа DIGIN2 */
    LTR_MARK_SEC_IRIGB_nDIGIN2  = 19,   /**< Аналогично #LTR_MARK_SEC_IRIGB_DIGIN1, но
                                             в качестве входного сигнала используется
                                             инвертированный сигнал со входа DIGIN2 */
};

/** Уровень вывода журнала службой ltrd. */
enum en_LTR_LogLevel {
    LTR_LOGLVL_ERR_FATAL    = 0, /**< Фатальные ошибки */
    LTR_LOGLVL_ERR          = 1, /**< Ошибки */
    LTR_LOGLVL_WARN         = 2, /**< Предупреждения */
    LTR_LOGLVL_INFO         = 3, /**< Информационные сообщения */
    LTR_LOGLVL_DETAIL       = 4, /**< Детали */
    LTR_LOGLVL_DBG_HIGH     = 5, /**< Отладочные сообщения повышенного уровня важности */
    LTR_LOGLVL_DBG_MED      = 6, /**< Отладочные сообщения среднего уровня важности */
    LTR_LOGLVL_DBG_LOW      = 7  /**< Отладочные сообщения низкого уровня важности */
};

/** @brief Флаги функции получения информации о подключенных крейтах.

    Данные флаги могут упралять работой функции LTR_GetCratesEx(). В функцию в
    качестве параметра flags передается набор данных флагов, объединенных через
    битовое "ИЛИ". */
enum en_LTR_GetCratesFlags {
    /** Флаг указывает, что функция должна возвратить только список крейтов,
       с которыми службой установлено рабочее соединение, в котором возможно
       управление модулями крейта. Так как на текущий момент у крейта в один
       момент времени только один интерфейс (заданный в настройках) позволяет
       работать с модулями, то при данном флаге не будет возвращаться несколько 
       записей, связанных с одним крейтом, при одновременном подключении крейта по
       нескольким интерфейсам.

       Данный флаг доступен начиная с версии ltrd 2.1.5.0 и ltrapi 1.31.1.*/
    LTR_GETCRATES_FLAGS_WORKMODE_ONLY   = 0x1
};

/** @cond kd_extension */
enum en_LTR_CrateThermNum {
    LTR_CRATE_THERM_LOWER = 0, /**< Нижний датчик температуры */
    LTR_CRATE_THERM_UPPER = 1  /**< Верхний датчик температуры */
};
/** @endcond */


/** @brief Настраиваемые параметры службы ltrd.

    Список параметров настройки, управляющих работой службы ltrd.
    Данные параметры настраиваются через конфигурационный файл или
    через LTR_SetServerParameter(). В описании каждого кода указывается,
    каким образом задается значение параметра. */
enum en_LTRD_Params {
    /** Интервал опроса крейта, для проверки, рабочее ли соединение с крейтом
        по интерфейсу Ethernet и обнаружения отключения крейта.
        Параметр типа DWORD задает время в мс.*/
    LTRD_PARAM_ETH_CRATE_POLL_TIME      = 0x100,
    /** Таймаут на установление соединения службы с крейтом по Ethernet.
           Параметр типа DWORD задает время в мс. */
    LTRD_PARAM_ETH_CRATE_CON_TOUT       = 0x101,
    /** Таймаут на ответ на управляющую команду крейту по Ethernet.
        Параметр типа DWORD задает время в мс. */
    LTRD_PARAM_ETH_CRATE_CTLCMD_TOUT    = 0x102,
    /** Интервал проверки адресов хоста для запуска автоподключения.
        Параметр типа DWORD задает время в мс. */
    LTRD_PARAM_ETH_INTF_CHECK_TIME      = 0x103,
    /** Время, через которое будет выполнено повторное подключения к крейту
        по Ethernet в случае возникновения ошибки, если для данной IP записи
        установлен флаг #LTR_CRATE_IP_FLAG_RECONNECT.

        Параметр доступен начиная с версии ltrd 2.1.5.0 и ltrapi 1.31.1.*/
    LTRD_PARAM_ETH_CRATE_RECONNECT_TIME = 0x104
};


/** @} */

#pragma pack(4)

/***************************************************************************//**
  @addtogroup type_list Типы данных.
  @{
  *****************************************************************************/

/** @brief Описатель соединения.

    Данная структура содержит всю информацию о соединении со службой ltrd.
    Данная структура используется в большинстве функций в качестве первого параметра.
    Часть полей (saddr, sport, csn, cc) предназначены для заполнения пользователем перед
    установкой соединения для задания параметров соединения и после этого уже не изменяются.
    Часть параметров (flags, tmark) предназначены только для чтения и отображают дополнительную
    информацию, которая обновляется при приеме данных через LTR_Recv() или в других случаях.
*/
typedef struct {
    /** IP-адрес машины, на которой запущена служба ltrd, в
       [32-битном формате](@ref sect_ip_addr_format).
       По умолчанию устанавливается в #LTRD_ADDR_DEFAULT равный
       #LTRD_ADDR_LOCAL, что соответствует случаю, когда служба запущена на той
       же машине, что и пользовательская программа. */
    DWORD saddr;
    /** Номер TCP порта, который будет использоваться при подключении к службе
        ltrd. По умолчанию используется #LTRD_PORT_DEFAULT. */
    WORD  sport;
    /** Серийный номер крейта, которому должно соответствовать соединение.
        Если задана пустая строка (как по умолчанию), то будет установлено
        соединение с первым крейтом из текущего списка активных крейтов ltrd.
        В таком случае после установки соединение это поле будет изменено на
        реальный серийный номер крейта, с которым установлено соединение.
        Если записана строка #LTR_CSN_SERVER_CONTROL, то будет установлено
        управляющее соединение со службой ltrd, которое не связано ни с одним крейтом
        и может быть установлено при отсутствии активных крейтов. */
    CHAR  csn[LTR_CRATE_SERIAL_SIZE];
    /** Тип канала для соединения с ltrd. Задается одной из констант из
        #en_LTR_CC_ChNum. Указывает, является ли данное соединение управляющим
        (#LTR_CC_CHNUM_CONTROL) или это соединение с модулем в заданном
        слоте (#LTR_CC_CHNUM_MODULE1 .. #LTR_CC_CHNUM_MODULE16).
        Может также в редких случаях при необходимости объединяться с флагами
        #en_LTR_CC_Iface и #en_LTR_CC_Flags для указания дополнительных параметров
        соединения. */
    WORD  cc;
    /** Флаги состояния соединения. Объединенный через логическое ИЛИ набор
        значений из #en_LTR_ChStateFlags.
        Данное поле предназначено только для чтения и не должно явно изменяться
        пользователем. */
    DWORD flags;
    /** Последнее значение синхрометок, принятое для данного
        соединения.
        Данное поле обновляется при выполнении LTR_Recv(), если во время
        приема были обнаружены синхрометки.
        Данное поле предназначено только для чтения пользователем и
        не должно явно изменяться пользователем. */
    DWORD tmark;
    /** Непрозрачный указатель на структуру с параметрами, необходимыми для
        обеспечения обмена. Пользователь никак не должен использовать данное поле. */
    LPVOID Internal;
} TLTR;

/** @brief Конфигурация линий разъема синхронизации.

    Данная структура используется для настройки функций выходов DIGOUT на разъеме
    синхронизации SYNC крейтов LTR-EU, LTR-CEU и LTR-CU.

    Также данная структура позволяет настроить функции специальных пользовательских
    ножек процессора BlackFin для крейтов LTR-EU, однако последняя функция нужна
    только при написании пользовательской прошивки. */

typedef struct {
    /** Настройка функций пользовательских ножек процессора. Каждый элемент
       соответствует своей ножке userio0 - userio3 и задает ее режим одним
       из значений #en_LTR_UserIoCfg.
       (для LTR-EU userio0 --- PF1 (ревизия платы 0) или PF0 (в ревизии 1+), userio1 --- PG13,
        userio2 --- PF3 (только ревизия 1+), userio3 --- резерв).
       В штатной прошивке крейта данные настройки не оказывают влияние на работу и
       должны быть установлены в значение #LTR_USERIO_DEFAULT. */
    WORD        userio[4];

    /** Настройка функций выходов DIGOUT на разъеме синхронизации крейта.
        Каждый элемент массива задает режим соответствующего выхода
        (элемент 0 --- DIGOUT1, а 1 --- DIGOUT2) одним из значений из
        #en_LTR_DigOutCfg. Кроме того, для использования этих функций выходы
        должны быть разрешены через digout_en. */
    WORD        digout[2];
    /** Разрешение выходов DIGOUT1 и DIGOUT2 на разъеме синхронизации
        крейта (0 --- запрещены, 1 --- разрешены). Все выходные линии разрешаются
        или запрещаются одновременно. Если выходы разрешены,
        то на них будет сформирован сигнал в соответствии с настроенной
        функцией, указанной в соответствующем элементе массива digout.
        Если запрещены, то выходы в третьем состоянии, независимо от
        настроенных функций в массиве digout. */
    WORD        digout_en;
} TLTR_CONFIG;


/** @cond kd_extension */
typedef struct {
    WORD size;
    BYTE autorun_ison;
} TLTR_SETTINGS;
/** @endcond */


/** @} */

#pragma pack()


/*================================================================================================*/
#ifdef __cplusplus
extern "C" {
#endif

/***************************************************************************//**
    @addtogroup func_init Функции инициализации и работы с соединением
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Инициализация описателя соединения

    Функция инициализирует поля структуры описателя соединения значениями по умолчанию.
    Эта функция должна вызываться первой для каждой структуры #TLTR перед вызовом
    остальных функций.
    @param[in] hnd      Описатель соединения.
    @return             [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Init(TLTR *hnd);

/***************************************************************************//**
    @brief Открытие соединения

    Функция устанавливает клиентское соединение в соответствие с установленными
    полями [saddr](@ref TLTR::saddr), [sport](@ref TLTR::sport),
    [csn](@ref TLTR::csn) и [cc](@ref TLTR::cc) описателя модуля.
    О типах соединений подробнее описано в главе @ref sect_con_types

    После завершения работы необходимо закрыть соединение с помощью LTR_Close().

    Если функция возвращает ошибку, то в некоторых случаях соединение может
    оставаться открытым (например, для ошибки #LTR_WARNING_MODULE_IN_USE), поэтому
    даже в случае возвращения данной функцией ошибки, все равно следует
    вызывать LTR_Close().

    Все возвращенные ошибки (включая #LTR_WARNING_MODULE_IN_USE) рекомендуется
    интерпретировать как признак того, что с соединением нельзя работать и
    единственной допустимой функцией, которую можно и нужно вызвать следующей,
    является LTR_Close().


    @param[in] hnd      Описатель соединения.
    @return             [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Open(TLTR *hnd);

/***************************************************************************//**
   @brief Открытие управляющего соединения со службой ltrd

   Данная функция устанавливает управляющее соединение со службой ltrd.
   Данное соединение может быть установлено даже когда нет ни одного подключенного
   крейта и позволяет выполнять команды, управляющие работой службы (за исключением команд
   [управления крейтами](@ref func_cratectl), которые требует открытия
   управляющего соединения с конкретным крейтом).

   По завершению работы с соединением необходимо его закрыть, вызвав LTR_Close().

   Действие функции аналогично правильному заполнению полей структуры #TLTR
   с использованием в качестве серийного номера строки #LTR_CSN_SERVER_CONTROL и
   вызову LTR_Open() и служит для удобства, чтобы не зполнять поля вручную.

   Тоже можно выполнить и заполнив правильно поля структуры #TLTR  и вызвав
   LTR_Open(). Данная функция введена для удобства, чтобы не заполнять поля
   вручную.

   Функция доступна в ltrapi версии 1.31.0 или выше.

   @param[in] hsrv        Описатель управляющего соединения.
   @param[in] ltrd_addr   IP-адрес машины, на которой запущена служба ltrd, в
                          [32-битном формате](@ref sect_ip_addr_format).
                          Для использования значения по умолчанию может быть
                          передано значение #LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port   Номер TCP порта, который будет использоваться при
                          подключении к службе ltrd.
                          Для использования порта по умолчанию может быть
                          передано значение #LTRD_PORT_DEFAULT.
    @return              [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_OpenSvcControl(TLTR *hsrv, DWORD ltrd_addr, WORD ltrd_port);

/***************************************************************************//**
   @brief Открытие управляющего соединения с крейтом

   Данная функция устанавливает управляющее соединение с крейтом по его
    серийному номеру (или с первым крейтом, если серийный номер не задан).


   Функция позволяет явно указать интерфейс, по которому установлено соединение
   между службой ltrd и крейтом на случай, если крейт подключен одновременно
   по двум интерфейсам (например в случае, если крейт настроен и подключен
   по Ethernet, но также и подключен по USB в режиме настройки). В большинстве
   случаев достаточно указать в качестве интерфейса #LTR_CRATE_IFACE_UNKNOWN,
   что означает, что крейт может быть подключен по любому интерфейсу. При этом,
   если крейт подключен по двум интерфейсам, то открытое соединение будет
   связано с крейтом c использованием интерфейса, который является рабочим
   для крейта, т.е. через который можно обмениваться данными с модулями крейта,
   что и необходимо для большинства программ.


   По завершению работы с соединением необходимо его закрыть, вызвав LTR_Close().

   Действие функции аналогично правильному заполнению полей структуры #TLTR и
   вызову LTR_Open(). Данная функция введена для удобства, чтобы не заполнять поля 
   вручную.

   Функция доступна в ltrapi версии 1.31.0 или выше.

   @param[in] hcrate      Описатель соединения.
   @param[in] ltrd_addr   IP-адрес машины, на которой запущена служба ltrd, в
                          [32-битном формате](@ref sect_ip_addr_format).
                          Для использования значения по умолчанию может быть
                          передано значение #LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port   Номер TCP порта, который будет использоваться при
                          подключении к службе ltrd.
                          Для использования порта по умолчанию может быть
                          передано значение #LTRD_PORT_DEFAULT.
   @param[in] crate_iface Значение из #en_LTR_CrateIface, указывающая интерфейс,
                          по которому должен быть подключен крейт. Если задано
                          значение #LTR_CRATE_IFACE_UNKNOWN, то крейт может быть
                          подключен по любому интерфейсу.
   @param[in] crate_sn    Строка с серийным номером крейта, с которым нужно
                          установить соединение. Если передана пустая строка, то
                          будет установлено соединение с первым подключенным крейтом
    @return               [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_OpenCrate(TLTR *hcrate, DWORD ltrd_addr, WORD ltrd_port,
                                       INT crate_iface, const char *crate_sn);

/***************************************************************************//**
    @brief Открытие соединения с заданным таймаутом

    Данная функция полностью аналогична LTR_Open(), но позволяет явно указать
    максимальное время установления соединения с ltrd.

    @param[in] hnd      Описатель соединения.
    @param[in] timeout  Время в мс на открытие соединения. Если за заданное время
                        не завершится соединение с ltrd, то функция завершится
                        с ошибкой.
    @return             [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_OpenEx(TLTR *hnd, DWORD timeout);

/***************************************************************************//**
    @brief Закрытие соединения

    Функция закрывает ранее открытое с помощью функций LTR_Open(),
    LTR_OpenEx(), LTR_OpenCrate() или LTR_OpenSvcControl() соединение.
    При любом возвращенном значении после вызова этой функции соответствующий описатель
    соединения уже нельзя использовать без открытия нового соединения.

    @param[in] hnd      Описатель соединения.
    @return             [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Close(TLTR *hnd);

/***************************************************************************//**
   @brief Проверка, открыто ли соединение

   Функция проверяет, открыто ли в данный момент клиентом соединение, соответствующее
   указанному описателю. Если соединение открыто, функция возвращает #LTR_OK,
    если закрыто --- код ошибки #LTR_ERROR_CHANNEL_CLOSED. Данная функция не проверяет
    каким-либо образом, действительно ли сейчас это соединение, а результат ее определяется
    только последовательностью вызова функций --- было ли успешно открыто и не закрыто
    данное соединение клиентом.
   @param[in] hnd       Описатель соединения.
   @return              [Код ошибки](@ref #en_LTR_ERRORS) (#LTR_OK, если соединение установлено).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_IsOpened(TLTR *hnd);
/** @} */


/***************************************************************************//**
    @addtogroup func_geninfo Функции информационного характера
    @{

    Набор функций, позволяющих получить общую информацию о службе ltrd, подключенных
    крейтах и статистике обмена. Эти функции могут быть использованы
    с любым управляющим соединением.
*******************************************************************************/


/***************************************************************************//**
    @brief Получение версии службы ltrd

    Функция возвращает номер версии службы ltrd, с которой установлено управляющее
    соединение.

    В текстовом виде версия ltrd состоит из четырех чисел, разделенных точками.
    Возвращаемое данной функцией число представляет собой 32-битное значение,
    каждый байт которого соответствует одной части номера версии, отделяемой
    в текстовой записи точкой, причем старший байт соответствует главному номеру
    (первому числу). Например, 0x02010403 соответствует версии 2.1.4.3.

    Версии службы ltrd начинаются с 2.0.0.0 и выше, в то время как версии
    со старшим числом 1 возвращались только ранее используемой программой 
    LTR Server.

    @param[in] hsrv      Описатель управляющего соединения.
    @param[out] version  В данной переменной возвращается версия службы ltrd в
                         формате, описанном выше.
    @return              [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetServerVersion(TLTR *hsrv, DWORD *version);

/***************************************************************************//**
    @brief Получение серийных номеров подключенных крейтов

    Функция возвращает список серийных номеров крейтов, с которыми установлено
    соединение службой ltrd (список активных крейтов). Полученные серийные номера 
    можно использовать  для установления управляющего соединения с крейтами 
    (для их конфигурации,  получения списка модулей и т.п.) или соединений с модулями.

    Данная функция может вернуть максимум #LTR_CRATES_MAX серийных номеров (если
    подключено крейтов больше, то будет возвращено только первые #LTR_CRATES_MAX).
    Если крейтов может быть больше #LTR_CRATES_MAX, то можно использовать
    функцию LTR_GetCratesEx(), в которой нет ограничения на количество
    возвращаемых номеров крейтов.

    Функция возвращает всегда разные серийные номера, т.е. если крейт подключен
    и по USB (в режиме настройки) и по Ethernet (в рабочем режиме), то функция
    заполнит только один элемент в выходном массиве.

    @param[in] hsrv   Описатель управляющего соединения.
    @param[out] csn   Двумерный массив размером [#LTR_CRATES_MAX][#LTR_CRATE_SERIAL_SIZE]
                      байт.
                      В данный массив будут сохранены серийные номера подключенных
                      крейтов (каждая строка соответствует своему серийному номеру
                      крейта).
                      Всегда заполняются все #LTR_CRATES_MAX серийных номеров.
                      Если крейтов меньше #LTR_CRATES_MAX, то на месте
                      серийных номеров после последнего действительного серийного
                      номера будет записана пустая строка (один символ с нулевым
                      кодом).
    @return [Код ошибки](@ref #en_LTR_ERRORS)
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetCrates(TLTR *hsrv, BYTE *csn);

/***************************************************************************//**
    @brief Получение информации о подключенных крейтах

    Данная функция позволяет получить список серийных номеров крейтов, с которыми
    установлено соединение службой ltrd (список активных крейтов)
    с дополнительной информацией о этих крейтах.

    В отличие от LTR_GetCrates() функция не ограничивает количество подключенных
    крейтов, информация о которых может быть возвращена.

    Также в отличие от LTR_GetCrates(), если один крейт подключен сразу
    по нескольким интерфейсам, то данная функция вернет две записи о крейте,
    т.е. его серийный номер будет указан дважды в массиве serials, но при этом
    в соответствующих  записях с информацией о крейте будет указан разный
    интерфейс подключения (может быть изменено флагами).

    Функцию можно использовать с нулевым значением max_crates для получения количества
    подключенных крейтов, чтобы затем создать массивы нужных размеров и вызвать
    функцию второй раз уже для получения информации о всех крейтах.

    Данная функция поддерживается только службой ltrd.
    Функция доступна в ltrapi версии 1.31.0 или выше.

    @param[in] hsrv         Описатель управляющего соединения.
    @param[in] max_crates   Максимальное количество крейтов, информация о которых
                            может быть возвращена функцией.
    @param[in] flags        Флаги из #en_LTR_GetCratesFlags, управляющие работой
                            функции.
    @param[out] crates_found В данном параметре возвращается общее количество подключенных
                            крейтов. Это значение может быть больше max_crates.
                            Если данная информация не нужна, то может быть передан
                            нулевой указатель.
    @param[out] crates_returned В данной переменной возвращается количество
                            действительных серийных номеров, записанных в массив serials.
                            Данное возвращенное значение не может быть больше
                            значения, переданного в параметре max_crates.
                            Если max_crates равно нулю, то в качестве данного параметра
                            можно передать нулевой указатель.
    @param[out] serials     Массив для сохранения серийных номеров подключенных
                            крейтов. Должен быть достаточного размера
                            для сохранения max_crates серийных номеров
                            (каждый серийный номер --- строка из #LTR_CRATE_SERIAL_SIZE
                            символов). По завершению функции будут заполнены
                            первые crates_returned серийных номеров.
                            Если max_crates равно нулю, то в качестве данного параметра 
                            может быть передан нулевой указатель.
   @param[out] info_list    Массив для сохранения информации о подключенных крейтах.
                            Должен быть достаточного размера для сохранения
                            max_crates структур #TLTR_CRATE_INFO.
                            Каждый элемент соответствует серийному номеру из
                            serials с тем же номером. Если информация о крейтах
                            не нужна, то может быть передан нулевой указатель.

   @return                  [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT)    LTR_GetCratesEx(TLTR *hsrv, DWORD max_crates, DWORD flags,
                                            DWORD *crates_found, DWORD *crates_returned,
                                            CHAR serials[][LTR_CRATE_SERIAL_SIZE],
                                            TLTR_CRATE_INFO *info_list);

/***************************************************************************//**
  @brief Получение описания крейта

  Функция позволяет получить структуру типа #TLTR_CRATE_DESCR с подробным
  описанием заданного крейта.
  Может быть получено описание любого подключенного крейта через одно управляющее
  соединение --- отдельное управляющее соединение с крейтом устанавливать не нужно.

  Данная функция поддерживается только службой ltrd (не поддерживается ранее
  используемой программой LTR Server).

   @param[in]  hsrv        Описатель управляющего соединения.
   @param[in]  crate_iface Интерфейс подключения крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  crate_sn    Серийный номер крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[out] descr       Структура, в которую будет сохранено описание крейта
   @param[in]  size        Размер структуры, переданной в качестве параметра
                           descr.
   @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetCrateDescr(TLTR *hsrv, INT crate_iface, const char *crate_sn,
                                           TLTR_CRATE_DESCR *descr, DWORD size);
/***************************************************************************//**
  @brief Получение статистики по крейту

  Функция возвращает дополнительные параметры статистики, которую ведет служба
  ltrd, связанные с заданным крейтом, в виде структуры #TLTR_CRATE_STATISTIC.

  Данная функция поддерживается только службой ltrd (не поддерживается ранее
  используемой программой LTR Server).

   @param[in]  hsrv        Описатель управляющего соединения.
   @param[in]  crate_iface Интерфейс подключения крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  crate_sn    Серийный номер крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[out] stat        Структура, в которую будут сохранены параметры
                           статистики по крейту.
   @param[in]  size        Размер структуры, переданной в качестве параметра
                           stat.
   @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetCrateStatistic(TLTR *hsrv, INT crate_iface,
                                               const char *crate_sn,
                                               TLTR_CRATE_STATISTIC *stat, DWORD size);

/***************************************************************************//**
  @brief Получение статистики по модулю

  Функция возвращает дополнительные параметры статистики, которую ведет служба
  ltrd, связанные с заданным модулем, в виде структуры #TLTR_MODULE_STATISTIC.

  Данная функция поддерживается только службой ltrd (не поддерживается ранее
  используемой программой LTR Server).

   @param[in]  hsrv        Описатель управляющего соединения.
   @param[in]  crate_iface Интерфейс подключения крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  crate_sn    Серийный номер крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  module_slot Номер слота модуля (от #LTR_CC_CHNUM_MODULE1 до
                            #LTR_CC_CHNUM_MODULE16).
   @param[out] stat        Структура, в которую будут сохранены параметры
                           статистики по крейту.
   @param[in]  size        Размер структуры, переданной в качестве параметра
                           stat.
   @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetModuleStatistic(TLTR *hsrv, INT crate_iface,
                                                const char *crate_sn, INT module_slot,
                                                TLTR_MODULE_STATISTIC *stat, DWORD size);

/** @} */

/***************************************************************************//**
    @addtogroup func_cratectl Функции управления крейтами
    @{

    Набор функций, реализующих общие команды управления крейтами LTR.
    Эти функции могут использоваться только с
    [управляющим соединением с крейтом](@ref sec_con_crate).
*******************************************************************************/

/***************************************************************************//**
    @brief Получение списка модулей в крейте

    Функция позволяет получить список идентификаторов модулей, которые установлены в крейте.

    На вход функции необходимо подать массив из #LTR_MODULES_PER_CRATE_MAX элементов,
    который функция заполнит нужными значениями.
    
    Каждый элемент соответствует своему слоту в крейте (mid[0] --- идентификатор
    модуля, вставленного в первый слот, mid[15] --- в 16-ый) и при успешном выполнении
    функции устанавливается в одно из значений из #en_LTR_MIDs. 
    Если в данном слоте  не вставлен модуль или слот вообще отсутствует в крейте, 
    то будет установлено значение #LTR_MID_EMPTY.
    Если модуль был обнаружен, но его тип еще не определен, то будет возвращено
    значение #LTR_MID_IDENTIFYING.

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @param[out] mid        Указатель на массив из #LTR_MODULES_PER_CRATE_MAX
                           элементов, в котором при успешном выполнении будут
                           возвращены идентификаторы установленных модулей.
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetCrateModules(TLTR *hcrate, WORD *mid);


/***************************************************************************//**
    @brief Получение информации о типе и интерфейсе подключения крейта

    Функция заполняет структуру #TLTR_CRATE_INFO информацией о крейте, с которым
    установлено управляющее соединение.

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @param[out] CrateInfo  В данной структуре при успехе возвращается информация
                            о крейте.
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetCrateInfo(TLTR *hcrate, TLTR_CRATE_INFO *CrateInfo);

/***************************************************************************//**
    @brief Конфигурация линий разъема синхронизации крейта

    Функция устанавливает конфигурацию линий разъема синхронизации SYNC в
    соответствии с параметрами, заданными в структуре #TLTR_CONFIG.
    Данная функция применима только для крейтов с наличием данного разъема
    (LTR-EU, LTR-CU, LTR-CEU).

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @param[in] conf        Конфигурация линий разъема синхронизации крейта.
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Config(TLTR *hcrate, const TLTR_CONFIG *conf);

/***************************************************************************//**
    @brief Настройка генерации метки "СТАРТ"

    Функция устанавливает режим генерации метки "СТАРТ" крейтом.
    Данная функция работает только с крейтами, поддерживающими стандартный
    механизм генерации  синхрометок (LTR-EU, LTR-CEU, LTR-CU).
    Крейт может генерировать метку как от внешнего события, так и внутреннего
    --- по команде с ПК.

    В зависимости от значения режима поведение функции несколько отличается.

    Если задан режим #LTR_MARK_INTERNAL, то при выполнении этой функции крейт
    генерирует одиночную метку "СТАРТ", после чего не генерирует метки до
    следующего вызова данной функции. Т.е. при внутренней генерации метки
    "СТАРТ" необходимо вызывать данную функцию каждый раз, когда метка должна быть
    сгенерирована.

    Остальные режимы соответствуют внешней генерации метки. В них данная
    функция просто настраивает крейт, чтобы он ожидал заданного в режиме события
    и по каждому такому событию генерировал метку.
    
    Чтобы отключить внешнюю генерацию меток, можно вызвать данную функцию
    с указанием режима #LTR_MARK_OFF.

    В случае закрытия соединения с крейтом без отключения генерации меток, крейт
    все равно будет продолжать генерацию до явного отключения.

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @param[in] mode        Режим генерации метки "СТАРТ" ---  значение из
                           #en_LTR_MarkMode.
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_MakeStartMark(TLTR *hcrate, INT mode);


/***************************************************************************//**
    @brief Запуск генерации меток "СЕКУНДА"

    Функция запускает генерацию секундных меток в указанном режиме.
    Данная функция работает только с крейтами, поддерживающими стандартный
    механизм генерации  синхрометок (LTR-EU, LTR-CEU, LTR-CU).

    Если задан режим #LTR_MARK_INTERNAL, то после вызова данной функции крейт
    начинает генерировать секундную метку раз в секунду (от внутреннего таймера).
    В остальных режимах крейт ожидает внешнего события и генерирует секундную
    метку по каждому событию.

    Останов генерации выполняется с помощью функции LTR_StopSecondMark().

    В случае закрытия соединения с крейтом без отключения генерации секундных
    меток, крейт все равно будет продолжать генерацию до явного отключения.

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @param[in] mode        Режим генерации меток "СЕКУНДА".
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_StartSecondMark(TLTR *hcrate, INT mode);
/***************************************************************************//**
    @brief Останов генерации меток "СЕКУНДА"

    При вызове данной функции крейт прекращает генерацию меток секунда, которая
    была запущена ранее с помощью LTR_StartSecondMark()

    @param[in] hcrate      Описатель управляющего соединения с крейтом.
    @return                [Код ошибки](@ref #en_LTR_ERRORS).
******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_StopSecondMark(TLTR *hcrate);

/** @cond kd_extension */
LTRAPIWIN_DllExport(INT) LTR_PutSettings(TLTR *hcrate, const TLTR_SETTINGS *settings);
/** @endcond */

/** @cond low_level */
/* Передача произвольного запроса крейту */
LTRAPIWIN_DllExport(INT) LTR_CrateCustomCtlReq(TLTR *hcrate, DWORD req, DWORD param,
                                               const BYTE *snd_buf, DWORD snd_size,
                                               BYTE *rcv_buf, DWORD rcv_size, DWORD *recvd_size);
LTRAPIWIN_DllExport(INT) LTR_CrateGetArray(TLTR *hcrate, DWORD address, BYTE *buf, DWORD size);
LTRAPIWIN_DllExport(INT) LTR_CratePutArray(TLTR *hcrate, DWORD address, const BYTE *buf, DWORD size);
/** @endcond */

/** @cond not_supported */
/* Получение сырых данных из крейта (если был установлен CC_RAW_DATA_FLAG) */
LTRAPIWIN_DllExport(INT) LTR_GetCrateRawData(TLTR *hcrate, DWORD *data, DWORD *tmark,
                                             DWORD size, DWORD timeout);
/** @endcond */
/** @} */


/***************************************************************************//**
    @addtogroup func_srvctl Функции управления службой ltrd
    @{

    Набор дополнительных функций управляющего соединения, связанные с управлением
    работой службы ltrd.
*******************************************************************************/

/***************************************************************************//**
    @brief Сброс указанного модуля.

    Данная функция позволяет сбросить любой модуль через управляющее соединение.
    По выполнению этой функции в службе выполняются следующие действия:
        - разрываются все клиентские соединения, установленные с данным модулем
        - сбрасывается вся статистика, накопленная ltrd по данному модулю
        - посылается команда сброса модуля для возвращения модуля в исходное
          состояние

    Данная функция может быть полезна в случае, если осталось незакрытое клиентское
    соединение с модулем, которое не дает установить новое рабочее соединение
    (возвращается ошибка #LTR_WARNING_MODULE_IN_USE). Вызов этой функции позволяет
    сбросить не закрытое соединение.

    Также функция может быть полезна, если программа закрылась не завершив корректно
    работу с модулем, например, оставив запущенным сбор данных. Сброс модуля
    позволяет остановить передачу от модуля ненужных данных, переведя его в начальное
    состояние.

    Данная функция поддерживается только службой ltrd (не поддерживается ранее
    используемой программой LTR Server).

   @param[in]  hsrv        Описатель управляющего соединения.
   @param[in]  crate_iface Интерфейс подключения крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  crate_sn    Серийный номер крейта (аналогично одноименному
                           параметру LTR_OpenCrate()).
   @param[in]  module_slot Номер слота модуля (от #LTR_CC_CHNUM_MODULE1 до
                            #LTR_CC_CHNUM_MODULE16).
   @param[in]  flags       Дополнительные флаги, управляющие работой функции.
                           Сейчас не используются, поэтому поле всегда должно
                           быть равно нулю.
   @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_ResetModule(TLTR *hsrv, INT crate_iface,
                                         const char *crate_sn, INT module_slot,
                                         DWORD flags);

/***************************************************************************//**
    @brief Установка уровня журнала

    Функция устанавливает, какого уровня сообщения будут выводится в журнал
    службы ltrd. Все сообщения с менее важным уровнем выводиться не будут.

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  level       Устанавливаемый уровень вывода в журнал --- значение
                           из #en_LTR_LogLevel.
    @param[in]  permanent   Если FALSE, то изменения касаются только текущего
                            запуска службы ltrd.                            
                            Если TRUE --- изменения сохраняются в настройках и
                            будут учитываться после перезапуска службы.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_SetLogLevel(TLTR *hsrv, INT level, BOOL permanent);

/***************************************************************************//**
    @brief Получение текущего уровня журнала

    Функция возвращает установленный уровень вывода в журнал службы ltrd.

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[out] level       В данной переменной возвращается установленный уровень
                            журнала --- значение из #en_LTR_LogLevel.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetLogLevel(TLTR *hsrv, INT *level);



/***************************************************************************//**
    @brief Установка параметров работы службы ltrd

    Функция выполняет установку одного параметра из настроек работы службы ltrd.
    Формат и смысл передаваемого значения определяется кодом параметра и описан
    в описании каждого параметра.
    Устанавливаемый параметр немедленно применяется и сохраняется в настройках
    службы ltrd.

    Данная функция поддерживается только службой ltrd (не поддерживается ранее
    используемой программой LTR Server).

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  param       Код параметра --- значение из #en_LTRD_Params.
    @param[in]  val         Указатель на данные с устанавливаемым значением.
                            Формат данных может зависеть от параметра и указан
                            в описании параметра.
    @param[in]  size        Размер данных, передаваемых в качестве значения
                            параметра (на которые указывает val).
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_SetServerParameter(TLTR *hsrv, DWORD param, void *val, DWORD size);


/***************************************************************************//**
    @brief Чтение параметров работы службы ltrd

    Функция выполняет чтение одного параметра из настроек работы службы ltrd.
    Формат и смысл передаваемого значения определяется кодом параметра и описан
    в описании каждого параметра.

    Данная функция поддерживается только службой ltrd (не поддерживается ранее
    используемой программой LTR Server).

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  param       Код параметра --- значение из #en_LTRD_Params
    @param[in]  val         Указатель на данные, куда будет сохранено значение
                            параметра.
                            Формат данных может зависеть от параметра и указан
                            в описании параметра.
    @param[in]  size        Размер массива, на который указывает переменная val.
                            Разные параметры могут требовать разного размера для
                            сохранения значения.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetServerParameter(TLTR *hsrv, DWORD param, void *val, DWORD *size);

/***************************************************************************//**
    @brief Перезапуск службы ltrd

    При успешном выполнении данной команды служба ltrd закрывает все клиентские
    соединения и все соединения с крейтами, и начинает работу сначала, заново
    считывая свои настройки.
    
    Соответственно, и текущее управляющее соединение, через которое передана
    команда, становится недействительным ---  единственной функцией, которая 
    может и должна быть вызвана следующей,  является LTR_Close().

    @param[in]  hsrv        Описатель управляющего соединения.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_ServerRestart(TLTR *hsrv);

/***************************************************************************//**
    @brief Останов службы ltrd

    При успешном выполнении данной команды служба ltrd закрывает все клиентские
    соединения и все соединения с крейтами и завершает свою работу.

    Соответственно, и текущее управляющее соединение, через которое передана
    команда, становится недействительным ---  единственной функцией, которая 
    может и должна быть вызвана следующей,  является LTR_Close().


    @param[in]  hsrv        Описатель управляющего соединения.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_ServerShutdown(TLTR *hsrv);




/** @} */

/***************************************************************************//**
    @addtogroup func_ip Функции управления подключением крейтов по Ethernet
    @{

    Данные функции используются с управляющим соединением. Они позволяют управлять
    списком записей с IP-адресами крейтов службы ltrd, а также выполнять команды для 
    установления и закрытия соединения между ltrd и крейтами по IP-адресам 
    с использованием записей из этого списка.
*******************************************************************************/


/***************************************************************************//**
    @brief Получение списка записей с IP-адресами крейтов

    Функция возвращает список записей с IP-адресами крейтов из настроек службы
    ltrd (как с установленным соединением с крейтом, так и не установленным).
    Также вместе с записью возвращается и состояние подключения крейта, 
    соответствующего данной записи.

    Функция позволяет вернуть не весь список записей, а только тех, адреса которых
    удовлетворяют заданному фильтру --- возвращаются только те IP адреса,
    которые относятся к подсети, заданной стандартным образом с помощью
    IP-адреса (ip_net) и маски подсети (ip_mask). Т.е. если задан адрес
    "192.168.1.0" и маска "255.255.255.0", то будут возвращены записи с
    IP-адресами вида "192.168.1.x", где x - любое значение от 0 до 255.

    Если нужно получить все записи, то в качестве адреса и маски можно передать
    нулевые значения.

    Для получения информации о состоянии конкретной записи можно
    задать полностью требуемый IP-адрес, а маску задать равной "255.255.255.255"

    Для получения произвольного количества IP-записей функцию можно вызвать сперва
    с нулевым значением max_entries, чтобы получить число записей в entries_found,
    а затем уже выделить массив для приема нужного количества записей и вызвать
    функцию повторно уже для получения информации по этим записям.


    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  max_entries Максимальное количество записей, которое может принять
                            массив info_array
    @param[in]  ip_net      IP-адрес сети, использующийся для фильтрации возвращаемых
                            записей, в [32-битном формате](@ref sect_ip_addr_format).
    @param[in]  ip_mask     Маска подсети для фильтрации возвращаемых записей
                            в [32-битном формате](@ref sect_ip_addr_format).
    @param[out] entries_found В данной переменной возвращается общее количество
                            найденных записей, удовлетворяющих условию фильтра.
                            Это значение может быть больше max_entries.
    @param[out] entries_returned Количество записей, которое было возвращено
                            в массиве info_array. При успешным выполнении это
                            значение равно наименьшему из max_entries и
                            entries_found.
    @param[out] info_array  Массив структур #TLTR_CRATE_IP_ENTRY, в котором будут
                            возвращены найденные записи с IP-адресами. Данный
                            массив должен быть достаточного размера для сохранения
                            max_entries записей. Если в max_entries передано
                            нулевое значение, то в качестве данного параметра
                            может быть передан нулевой указатель.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetListOfIPCrates(TLTR *hsrv, DWORD max_entries, DWORD ip_net,
    DWORD ip_mask, DWORD *entries_found, DWORD *entries_returned, TLTR_CRATE_IP_ENTRY *info_array);

/***************************************************************************//**
    @brief Добавление записи с IP-адресом крейта

    Функция добавляет запись с указанным параметром в список записей IP-адресов
    крейтов службы ltd.

    Чтобы установить соединение с крейтом по Ethernet (с использованием
    LTR_ConnectIPCrate()) его IP-адрес должен быть уже в списке записей.

    Если запись уже есть в списке, то данная функция только изменяет флаги
    записи (аналогично LTR_SetIPCrateFlags()).

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  ip_addr     IP-адрес крейта в
                            [32-битном формате](@ref sect_ip_addr_format).
    @param[in]  flags       Набор флагов, связанных с добавляемой записью.
                            Объединение по ИЛИ значений из #en_LTR_CrateIpFlags.
    @param[in]  permanent   Если FALSE, то изменения касаются только текущего
                            запуска службы ltrd.
                            Если TRUE --- изменения сохраняются в настройках и
                            будут учитываться после перезапуска службы.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_AddIPCrate(TLTR *hsrv, DWORD ip_addr, DWORD flags, BOOL permanent);

/***************************************************************************//**
    @brief Удаление записи с IP-адресом крейта

    Функция удаляет запись с указанным IP-адресом из списка IP-адресов крейтов
    службы ltrd.

    При этом с соответствующим крейтом не должно быть установлено соединение
    (состояние записи должно быть отлично от #LTR_CRATE_IP_STATUS_ONLINE или
    #LTR_CRATE_IP_STATUS_CONNECTING), в противном случае функция вернет ошибку.
    Для удаления записи с подключенным крейтом сперва нужно разорвать соединение
    с крейтом через LTR_DisconnectIPCrate().

    Если указанная запись отсутствует в списке, то функция никак не повлияет
    на список адресов и сразу завершится без ошибки.

    @param[in]  hsrv        Описатель управляющего соединения
    @param[in]  ip_addr     IP-адрес крейта в
                            [32-битном формате](@ref sect_ip_addr_format), запись
                            с которым должна быть удалена.
    @param[in]  permanent   Если FALSE, то изменения касаются только текущего
                            запуска службы ltrd.
                            Если TRUE --- изменения сохраняются в настройках и
                            будут учитываться после перезапуска службы.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_DeleteIPCrate(TLTR *hsrv, DWORD ip_addr, BOOL permanent);

/***************************************************************************//**
    @brief  Установление соединения с крейтом по IP-адресу

    Функция является командой службе ltrd, указывающей, что служба должна установить 
    соединение  с крейтом с заданным IP-адресом по Ethernet интерфейсу.
    Запись с заданным адресом должна быть в списке записей с IP-адресами крейтов
    службы (этот список можно получить с помощью LTR_GetListOfIPCrates()), иначе
    функция вернет ошибку.

    Завершение этой функции говорит о том, что служба приняла команду и начала
    подключение к крейту, но подключение еще может быть не завершено.
    При этом состояние соединения с крейтом для записи изменяется на
    #LTR_CRATE_IP_STATUS_CONNECTING.

    По завершению подключения это состояние изменится на
    #LTR_CRATE_IP_STATUS_ONLINE при успешном подключении или на
    #LTR_CRATE_IP_STATUS_ERROR при ошибке, что и является признаком завершения
    операции. Получить текущее состояние подключения можно получив информацию о записи
    с помощью LTR_GetListOfIPCrates(),

    Также при успешном завершении подключения крейт появится в списке активных
    крейтов, который можно получить с помощью LTR_GetCrates() или LTR_GetCratesEx().

    Если при вызове данной функции крейт уже подключен (состояние подключения
    #LTR_CRATE_IP_STATUS_ONLINE) или идет процесс подключения (состояние
    #LTR_CRATE_IP_STATUS_CONNECTING), то функция просто завершится успешно,
    не выполняя никаких действий.

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  ip_addr     IP-адрес крейта в
                            [32-битном формате](@ref sect_ip_addr_format),
                            с которым нужно установить соединение.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_ConnectIPCrate(TLTR *hsrv, DWORD ip_addr);


/***************************************************************************//**
    @brief  Разрыв соединения с крейтом по IP-адресу

    При выполнении данной функции служба разрывает соединение с крейтом, подключенным
    по Ethernet, с указанным адресом. Крейт исчезает из списка активных крейтов
    и состояние подключения для записи с данным IP-адресом изменяется на
    #LTR_CRATE_IP_STATUS_OFFLINE.

    Запись с таким адресом должна быть среди списка записей с IP-адресами крейтов
    службы ltrd, иначе функция вернет ошибку.

    Если запись с указанным адресом присутствует, но нет активного соединения
    (состояние подключение отлично от #LTR_CRATE_IP_STATUS_ONLINE или
    #LTR_CRATE_IP_STATUS_CONNECTING), то функция завершается успешно,
    не выполняя каких-либо действий.

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  ip_addr     IP-адрес крейта в
                            [32-битном формате](@ref sect_ip_addr_format),
                            соединение с которым нужно завершить.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_DisconnectIPCrate(TLTR *hsrv, DWORD ip_addr);


/***************************************************************************//**
    @brief  Установление соединения со всеми крейтами с признаком автоподключения

    Функция дает команду службе ltrd установить соединение через интерфейс Ethernet
    со всеми крейтами, для записей с IP-адресами которых установлен флаг
    #LTR_CRATE_IP_FLAG_AUTOCONNECT.

    При этом запуск подключения осуществляется только для тех записей, для которых
    нет активного соединения, т.е. на записи с текущим состоянием
    #LTR_CRATE_IP_STATUS_ONLINE или #LTR_CRATE_IP_STATUS_CONNECTING вызов функции
    влияния не оказывает.

    Как и в случае с LTR_ConnectIPCrate() завершение работы функции означает только
    то, что процесс подключения запущен, а о завершении подключения можно узнать
    при необходимости по изменению состояния подключения соответствующих записей
    (которое можно узнать с помощью LTR_GetListOfIPCrates().

    Если нет ни одной записи с установленным флагом  #LTR_CRATE_IP_FLAG_AUTOCONNECT,
    по которой не установлено соединение с крейтом, то функция завершается успешно,
    не выполняя каких-либо действий.

    @param[in]  hsrv        Описатель управляющего соединения.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_ConnectAllAutoIPCrates(TLTR *hsrv);


/***************************************************************************//**
    @brief  Разрыв соединения со всеми крейтами, подключенными по Ethernet

    По данной команде служба ltrd закрывает все активные соединения с крейтами,
    подключенными через интерфейс Ethernet. Все крейты будут удалены из списка
    активных крейтов и все IP-записи изменят состояние подключения на
    #LTR_CRATE_IP_STATUS_OFFLINE.

    Если на момент вызова функции не было ни одного крейта, подключенного по
    Ethernet, то функция завершается успешно, не выполняя каких-либо действий.
    @param[in]  hsrv        Описатель управляющего соединения.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_DisconnectAllIPCrates(TLTR *hsrv);

/***************************************************************************//**
    @brief  Установка флагов для записи с IP-адресом крейта

    Функция изменяет значение флагов, связанных с уже присутствующей записью
    с IP-адресом, на указанное значение.
    Запись с заданным IP-адресом должна присутствовать в списке записей службы
    ltrd, в противном случае функция завершится с ошибкой.

    @param[in]  hsrv        Описатель управляющего соединения.
    @param[in]  ip_addr     IP-адрес крейта в
                            [32-битном формате](@ref sect_ip_addr_format), для
                            записи с которым следует изменить флаги.
    @param[in]  flags       Новый набор флагов, связанных с записью.
                            Объединение по ИЛИ значений из #en_LTR_CrateIpFlags.
    @param[in]  permanent   Если FALSE, то изменения касаются только текущего
                            запуска службы ltrd.
                            Если TRUE --- изменения сохраняются в настройках и
                            будут учитываться после перезапуска службы.
    @return                 [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_SetIPCrateFlags(TLTR *hsrv, DWORD ip_addr, DWORD flags,
                                             BOOL permanent);


/** @} */

/***************************************************************************//**
    @addtogroup func_modules Функции для обмена данными с модулями
    @{

    Данные функции используются для передачи и приема данных модулям. Как правило
    пользователю нет необходимости использовать данные функции напрямую, т.к. для
    работы с модулями используются функции из специализированных библиотек для
    работы с конкретным модулем.

    Эти функции могут использоваться только для
    [соединений с модулями](@ref sec_con_module) и неприменимы
    для управляющих соединений.
*******************************************************************************/

/***************************************************************************//**
    @brief  Прием данных от модуля

    Функция принимает данные от модуля в 32-битных словах в специальном формате
    слов LTR. Кроме того, функция анализирует информацию о пришедших 
    [синхрометках](@ref sect_marks)   и о переполнении буфера в службе ltrd,
    обновляя значения поля #TLTR::flags  описателя модуля и заполняя массив tmark.

    Функция возвращает управление либо когда примет запрошенное количество слов,
    либо по истечении таймаута. При этом реально принятое количество слов можно
    узнать по возвращенному значению.

    @param[in]  hmodule  Описатель соединения с модулем.
    @param[out] data     Массив, в который будут сохранены принятые слова. Должен
                         быть размером на size 32-битных слов.
    @param[out] tmark    Указатель на массив размером на size 32-битных слов,
                         в который будут сохранены значения [синхрометок](@ref sect_marks),
                         соответствующие принятым данным. Т.е. принятому слову
                         data[i] соответствует элемент tmark[i], указывающий
                         количество меток "СТАРТ" и "СЕКУНДА", соответствующих
                         этому слову. Формат данных слов описан в разделе, посвященному
                         [синхрометкам](@ref sect_marks).
                         Если синхрометки не используются, то можно передать
                         в качестве параметра NULL.
   @param[in]  size      Запрашиваемое количество 32-битных слов, которое нужно
                         принять от модуля.
   @param[in]  timeout   Таймаут на выполнение операции в миллисекундах. 
                         Значение 0 означает значение таймаута по умолчанию. Если в течение
                         заданного времени не будет принято запрашиваемое количество
                         слов, то функция все равно вернет управление, возвратив
                         в качестве результата реально принятое количество слов.
   @return               Значение меньше нуля соответствует [коду ошибки](@ref #en_LTR_ERRORS).
                         Значение больше или равное нулю соответствует количеству реально
                         принятых и сохраненных в массив data слов.
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Recv(TLTR *hmodule, DWORD *data, DWORD *tmark, DWORD size, DWORD timeout);
/***************************************************************************//**
    @brief  Передача данных в модуль

    Функция посылает данные модулю в 32-битных словах в специальном формате
    слов LTR.

    Функция возвращает управление либо когда все данные будут записаны в буфер на
    передачу, либо по истечению таймаута.
    При этом реальное количество слов, записанное в буфер на передачу, можно
    узнать по возвращенному значению. Т.е. возвращенное значение гарантирует,
    что данное количество слов было поставлено на передачу, но эти данные еще
    могут не дойти до модуля на момент завершения функции.

   @param[in]  hmodule   Описатель соединения с модулем.
   @param[in]  data      Массив, содержащий данные в виде size 32-битных слов,
                         которые нужно передать в модуль.
   @param[in]  size      Количество слов, которое нужно послать модулю.
   @param[in]  timeout   Таймаут на выполнение операции в миллисекундах.
                         Значение 0 означает значение таймаута по умолчанию. Если в течение
                         заданного времени в буфере на передачу не будет места
                         для записи запрашиваемого количества слов, то функция
                         все равно вернет управление, возвратив в качестве результата
                         реально количество слов, записанных в буфер.                         
   @return               Значение меньше нуля соответствует [коду ошибки](@ref #en_LTR_ERRORS).
                         Значение больше или равное нулю соответствует количеству реально
                         записанных в буфер на передачу слов, которые поставлены
                         на отправку в модуль.
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_Send(TLTR *hmodule, const DWORD *data, DWORD size, DWORD timeout);

/***************************************************************************//**
    @brief Чтение времени последней секундной метки

    Данная функция возвращает значение времени, соответствующее последнему обнаруженному
    приходу расширенной метки "СЕКУНДА" с абсолютным временем при приеме данных
    по данному соединению посредством LTR_Recv().

    Данная функция работает только в случае, если крейт поддерживает генерацию
    расширенной метки "СЕКУНДА", с указанием абсолютного времени, каким-либо
    способом. Например, с использованием сервера времени, который передает
    время по протоколу IRIG-B.

    Если метки "СЕКУНДА" с абсолютным временем принято не было, то будет
    возвращено значение 0.

    @param[in]  hmodule   Описатель соединения с модулем
    @param[out] unixtime  Абсолютное значение времени в секундах с 1 января 1970 года
                          (unixtime).
    @return               [Код ошибки](@ref #en_LTR_ERRORS).
*******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_GetLastUnixTimeMark(TLTR *hmodule, LONGLONG *unixtime);

/** @cond kd_extension */
/* Прием данных из крейта с чтением массива меток времени в формате 64-бит unixtime */
LTRAPIWIN_DllExport(INT) LTR_RecvEx(TLTR *ltr, DWORD *data, DWORD *tmark, DWORD size,
                                    DWORD timeout, LONGLONG *unixtime);
/** @endcond */

/** @} */


/***************************************************************************//**
    @addtogroup func_misc Вспомогательные функции
    @{
*******************************************************************************/
/***************************************************************************//**
   @brief Получение текстового сообщения об ошибке

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для ОС Windows или UTF-8 для ОС Linux. Функция поддерживает только
   только коды ошибок, определенные в данной библиотеке и 
   возвращаемые функциями данной библиотеки.
   Библиотеки для конкретных модулей могут иметь дополнительные коды ошибок и
   свои функции получения текстового описания ошибки, которые поддерживают 
   эти дополнительные коды ошибок.

   @param[in] err       Код ошибки.
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTRAPIWIN_DllExport(LPCSTR) LTR_GetErrorString(INT err);


/***************************************************************************//**
   @brief Установка таймаута по умолчанию для соединения

   Функция устанавливает таймаут по умолчанию на выполнения операций для указанного
   соединения. При открытии соединения этот таймаут равен
   #LTR_DEFAULT_SEND_RECV_TIMEOUT.

   Для управляющего соединения этот таймат определяет время на выполнения любой
   команды (от передачи запроса службе до приема ответа).
   Для соединения с модулем --- это таймаут, который используется
   в LTR_Recv() или LTR_Send(), если в них передано нулевое значение таймаута

   @param hnd        Описатель соединения.
   @param tout       Таймаут по умолчанию в мс.
   @return             [Код ошибки](@ref #en_LTR_ERRORS).
 ******************************************************************************/
LTRAPIWIN_DllExport(INT) LTR_SetTimeout(TLTR *hnd, DWORD tout);

/** @} */






/** @cond not_supported */

/* Управление приоритетом процесса ltrserver.exe */
LTRAPIWIN_DllExport(INT) LTR_SetServerProcessPriority(TLTR *ltr, DWORD Priority);
/* Чтение приоритета процесса ltrserver.exe */
LTRAPIWIN_DllExport(INT) LTR_GetServerProcessPriority(TLTR *ltr, DWORD *Priority);


/* Чтение режима поиска IP-крейтов в локальной сети */
LTRAPIWIN_DllExport(INT) LTR_GetIPCrateDiscoveryMode(TLTR *ltr, BOOL *enabled, BOOL *autoconnect);

/* Установка режима поиска IP-крейтов в локальной сети */
LTRAPIWIN_DllExport(INT) LTR_SetIPCrateDiscoveryMode(TLTR *ltr, BOOL enabled, BOOL autoconnect,
    BOOL permanent);

/** @endcond */



/** @cond low_level */
LTRAPIWIN_DllExport(INT) LTR_RawRecv(TLTR *ltr, BYTE* buf, DWORD size, DWORD tout);
LTRAPIWIN_DllExport(INT) LTR_SendShutdown(TLTR *ltr);

/* -- Функции для внутреннего применения -- */
LTRAPIWIN_DllExport(INT) LTR__GenericCtlFunc(TLTR *ltr, LPCVOID request_buf, DWORD request_size,
    LPVOID reply_buf, DWORD reply_size, INT ack_error_code, DWORD timeout);
/** @endcond */

#ifdef __cplusplus
}
#endif

#endif /*#ifndef LTRAPI_H_*/
