
#ifndef __LTR35API__
#define __LTR35API__

#ifdef _WIN32
    #ifdef LTR35API_EXPORTS
        #define LTR35API_DllExport(type)   __declspec(dllexport) type APIENTRY
    #else
        #define LTR35API_DllExport(type)   __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTR35API_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTR35API_DllExport(type) type
#endif


#include "ltrapi.h"

#ifdef __cplusplus
extern "C" {
#endif


/***************************************************************************//**
  @addtogroup const_list Константы и перечисления.
  @{
  *****************************************************************************/

/** Размер строки с именем модуля в структуре #TINFO_LTR35 */
#define LTR35_NAME_SIZE     8
/** Размер строки с серийным номером модуля в структуре #TINFO_LTR35 */
#define LTR35_SERIAL_SIZE   16

/** Количество каналов ЦАП */
#define LTR35_DAC_CHANNEL_CNT       8
/** Количество выходов для каждого канала ЦАП */
#define LTR35_DAC_CH_OUTPUTS_CNT    2
/** Максимальное количество цифровых линий */
#define LTR35_DOUT_LINES_MAX_CNT    16

/** Номер канала, соответствующий выводу на цифровые линии (если считать от 0) */
#define LTR35_CH_NUM_DIGOUTS        8

/** Максимальное количество отсчетов в странице ЦАП в режиме циклического автогенератора */
#define LTR35_MAX_POINTS_PER_PAGE   (4*1024*1024)

/** Количество арифметических синусоидальных генераторов в модуле */
#define LTR35_ARITH_SRC_CNT         4

/** Максимальное значение частоты преобразования ЦАП, которое можно установить */
#define LTR35_DAC_FREQ_MAX          192000
/** Минимальное значение частоты преобразования ЦАП, которое можно установить */
#define LTR35_DAC_FREQ_MIN          72000
/** Штатное значение частота преобразования ЦАП, для которого проверяются
    все метрологические характеристики модуля */
#define LTR35_DAC_FREQ_DEFAULT      192000



/** Максимальное значение коэф. затухания */
#define LTR35_ATTENUATION_MAX   119.5


/** Максимальный код ЦАП */
#define LTR35_DAC_CODE_MAX       0x7FFFFF
/** Код ЦАП, соответствующий максимальному значению диапазона в Вольтах */
#define LTR35_DAC_SCALE_CODE_MAX 0x600000


/** Устанавливаемое по умолчанию количество прочитанных отсчетов для выдачи
    периодического статуса в потоковом режиме */
#define LTR35_STREAM_STATUS_PERIOD_DEFAULT 1024
/** Максимальное значение количества отсчетов для выдачи
    периодического статуса в потоковом режиме */
#define LTR35_STREAM_STATUS_PERIOD_MAX     1024
/** Минимальное значение количества отсчетов для выдачи
    периодического статуса в потоковом режиме */
#define LTR35_STREAM_STATUS_PERIOD_MIN     8

/** Адрес, с которого начинается пользовательская область flash-памяти */
#define LTR35_FLASH_USERDATA_ADDR   0x100000
/** Размер пользовательской области flash-памяти */
#define LTR35_FLASH_USERDATA_SIZE   0x700000

/** Минимальный размер блока для стирания во flash-памяти модуля */
#define LTR35_FLASH_ERASE_BLOCK_SIZE   1024



/** @brief Коды ошибок, специфичные для LTR35.

   Коды ошибок, которые определены и используются только в ltr35api.
   Остальные коды ошибок, которые используются разными модулями, определены в ltrapi.h */
typedef enum {    
    LTR35_ERR_INVALID_SYNT_FREQ             = -10200, /**< Задана неподдерживаемая частота синтезатора */
    LTR35_ERR_PLL_NOT_LOCKED                = -10201, /**< Ошибка захвата PLL*/
    LTR35_ERR_INVALID_CH_SOURCE             = -10202, /**< Задано неверное значение источника данных для канала */
    LTR35_ERR_INVALID_CH_RANGE              = -10203, /**< Задано неверное значение диапазона канала */
    LTR35_ERR_INVALID_DATA_FORMAT           = -10204, /**< Задано неверное значение формата данных */
    LTR35_ERR_INVALID_MODE                  = -10205, /**< Задано неверное значение режима работы */
    LTR35_ERR_INVALID_DAC_RATE              = -10206, /**< Задано неверное значение скорости выдачи на ЦАП */
    LTR35_ERR_INVALID_SYNT_CNTRS            = -10207, /**< Задано недопустимое значение счетчиков синтезатора */
    LTR35_ERR_INVALID_ATTENUATION           = -10208, /**< Задано неверное значение коэффициента затухания канала ЦАП */
    LTR35_ERR_UNSUPPORTED_CONFIG            = -10209, /**< Заданная конфигурация ЦАП не поддерживается */
    LTR35_ERR_INVALID_STREAM_STATUS_PERIOD  = -10210, /**< Задано неверное количество отсчетов для выдачи статуса в потоковом режиме */
    LTR35_ERR_DAC_CH_NOT_PRESENT            = -10211, /**< Выбранный канал ЦАП отсутствует в данном модуле */
    LTR35_ERR_DAC_NO_SDRAM_CH_ENABLED       = -10212, /**< Не разрешен ни один канал ЦАП на вывод из SDRAM */
    LTR35_ERR_DAC_DATA_NOT_ALIGNED          = -10213, /**< Данные ЦАП не выравнены на кол-во разрешенных каналов */
    LTR35_ERR_NO_DATA_LOADED                = -10214, /**< Не было подгружено ни одного отсчета */
    LTR35_ERR_LTRD_UNSUP_STREAM_MODE        = -10215, /**< Данная версия ltrd/LtrServer не поддерживает потоковый режим LTR35 */
    LTR35_ERR_MODE_UNSUP_FUNC               = -10216, /**< Данная функция не поддерживается в установленном режиме */
    LTR35_ERR_INVALID_ARITH_GEN_NUM         = -10217, /**< Задано неверное значение номера арифметического генератора */    
} e_LTR35_ERROR_CODES;




/** @brief Флаги для управления цифровыми выходами.

    Флаги управления цифровыми выходами. Могут быть объединены через логическое
    “ИЛИ” со значениями цифровых выходов при подготовке данных на вывод с помощью
    LTR35_PrepareData() */
typedef enum {
    LTR35_DIGOUT_WORD_DIS_H = 0x00020000, /**< Запрещение (перевод в третье состояние)
                                              старшей половины цифровых выходов.
                                              Имеет значение только для LTR35-3. */
    LTR35_DIGOUT_WORD_DIS_L = 0x00010000  /**< Запрещение младшей половины
                                              цифровых выходов */
} e_LTR35_DOUTS_WORD_FLAGS;


/** @brief Флаги для подготовки данных.

    Флаги, управляющие работой функций LTR35_PrepareData() и LTR35_PrepareDacData() */
typedef enum {
    LTR35_PREP_FLAGS_VOLT   = 0x01, /**< Флаг указывает, что данные на входе
                                         заданны в вольтах и их нужно перевести в коды */
} e_LTR35_PREP_FLAGS;

/** @brief Флаги состояния модуля.

    Набор флагов, определяющих состояние модуля. Возвращаются функцией
    LTR35_GetStatus() */
typedef enum {
    LTR35_STATUS_FLAG_PLL_LOCK      = 0x0001, /**< Признак захвата PLL в момент
                                                    передачи статуса. Если равен
                                                   нулю, то модуль неработоспособен. */
    LTR35_STATUS_FLAG_PLL_LOCK_HOLD = 0x0002, /**< Признак, что захват PLL не
                                                   пропадал с момента предыдущей
                                                   передачи статуса. Должен быть
                                                   установлен во всех статусах,
                                                   кроме первого */
} e_LTR35_STATUS;


/** Формат данных для передачи данных модулю */
typedef enum {
    LTR35_FORMAT_24 = 0, /**< 24-битный формат. Один отсчет занимает два 32-битных слова LTR */
    LTR35_FORMAT_20 = 1  /**< 20-битный формат. Один отсчет занимает одно 32-битное слово LTR */
} e_LTR35_DATA_FORMAT;

/** Режим работы модуля */
typedef enum {
    LTR35_MODE_CYCLE    = 0, /**< Режим циклического автогенератора. Данные подкачиваются
                                  в буфер перед запуском выдачи, после чего выдаются
                                  по кругу без подкачки. */
    LTR35_MODE_STREAM   = 1  /**< Потоковый режим. Данные постоянно подкачиваются
                                  в очередь и выдаются при наличии на вывод */
} e_LTR35_MODE;

/** Используемый выход для канала ЦАП */
typedef enum {
    LTR35_DAC_OUT_FULL_RANGE = 0, /**< Выход 1:1 с диапазоном от -10 до +10 для
                                       LTR35-1 и от -2 до +20 для LTR35-2 */
    LTR35_DAC_OUT_DIV_RANGE  = 1  /**< Выход 1:5 для LTR35-1 или выход 1:10
                                       для LTR35-2 */
} e_LTR35_DAC_OUTPUT;

/** Источники сигнала для каналов ЦАП */
typedef enum {
    LTR35_CH_SRC_SDRAM = 0, /**< Сигнал берется из буфера в SDRAM модуля.
                                 При этом буфер работает циклически или в виде
                                 очереди в зависимости от режима */
    LTR35_CH_SRC_SIN1  = 1, /**< Синус  от первого арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_COS1  = 2, /**< Косинус  от первого арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_SIN2  = 3, /**< Синус  от второго арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_COS2  = 4, /**< Косинус  от второго арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_SIN3  = 5, /**< Синус  от третьего арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_COS3  = 6, /**< Косинус  от третьего арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_SIN4  = 7, /**< Синус  от четвертого арифметического синусоидального
                                 генератора */
    LTR35_CH_SRC_COS4  = 8 /**<  Косинус  от четвертого арифметического синусоидального
                                 генератора */
} e_LTR35_CH_SRC;

/** @brief Скорость выдачи данных.

    Определяет количество тактов частоты синтезатора, которое требуется для
    выдачи одного отсчета ЦАП (а также цифровых линий), т.е. отношение частоты
    преобразования ЦАП к частоте синтезатора */
typedef enum {
    LTR35_DAC_RATE_DOUBLE = 1, /**< Частота синтезатора, деленная на 384 */
    LTR35_DAC_RATE_QUAD   = 2  /**< Частота синтезатора, деленная на 192 */
} e_LTR35_RATE;

/** Флаги для записи во flash-память модуля */
typedef enum {
    /** Признак, что записываемая область памяти уже стерта и не требуется
        дополнительно стирать обновляемые сектора */
    LTR35_FLASH_WRITE_ALREDY_ERASED = 0x00001
} e_LTR35_FLASH_WRITE_FLAGS;

/** Модификации модуля LTR35 */
typedef enum {
    LTR35_MOD_UNKNOWN = 0, /**< Неизвестная (не поддерживаемая библиотекой) модификация */
    LTR35_MOD_1 = 1,  /**< LTR35-1 */
    LTR35_MOD_2 = 2,  /**< LTR35-2 */
    LTR35_MOD_3 = 3   /**< LTR35-3 */
} e_LTR35_MODIFICATION;



/** @} */

/***************************************************************************//**
  @addtogroup type_list Типы данных.
  @{
  *****************************************************************************/

#pragma pack(4)

/** @brief Калибровочные коэффициенты.

    Структура, хранящая калибровочные коэффициенты для одного выхода одного
    канала ЦАП. */
typedef struct {
    float   Offset; /**< Код смещения */
    float   Scale;  /**< Коэффициент шкалы */
} TLTR35_CBR_COEF;

/** @brief Описание выхода ЦАП.

    Структура описывает характеристики определенного выхода для данной модификации
    модуля LTR35 */
typedef struct {
    double AmpMax; /**< Максимальное пиковое значение амплитуды сигнала для данного выхода */
    double AmpMin; /**< Минимальное пиковое значение амплитуды сигнала для данного выхода */
    INT    CodeMax; /**< Код ЦАП, соответствующий максимальной амплитуде */
    INT    CodeMin; /**< Код ЦАП, соответствующий минимальной амплитуде */
    DWORD  Reserved[3]; /**< Резервные поля */
} TLTR35_DAC_OUT_DESCR;

/** @brief Информация о модуле.

    Структура, содержащая информацию о версиях прошивок микросхем модуля,
    информацию модификации модуля и информацию, считанную
    из Flash-памяти модуля (серийный номер, калибровочные коэффициенты).

    Все поля доступны после вызова LTR35_Open(). */
typedef struct {
    CHAR        Name[LTR35_NAME_SIZE];      /**< Название модуля
                                                (оканчивающаяся нулем ASCII-строка) */
    CHAR        Serial[LTR35_SERIAL_SIZE];  /**< Серийный номер модуля
                                                  (оканчивающаяся нулем ASCII-строка) */
    WORD        VerFPGA;                     /**< Версия прошивки ПЛИС модуля
                                                  (действительна только при ее наличии) */
    BYTE        VerPLD;                      /**< Версия прошивки PLD  */

    BYTE        Modification;               /**< Модификация модуля. Одно из значений из #e_LTR35_MODIFICATION */
    BYTE        DacChCnt;                   /**< Количество установленных каналов ЦАП */
    BYTE        DoutLineCnt;                /**< Количество линий цифрового вывода */

    /** Описание параметров выходов для данной модификации модуля */
    TLTR35_DAC_OUT_DESCR  DacOutDescr[LTR35_DAC_CH_OUTPUTS_CNT];
    DWORD       Reserved1[26]; /**< Резервные поля */
    /** Заводские калибровочные коэффициенты */
    TLTR35_CBR_COEF CbrCoef[LTR35_DAC_CHANNEL_CNT][LTR35_DAC_CH_OUTPUTS_CNT];
    /** Дополнительные резервные поля */
    DWORD       Reserved2[64*LTR35_DAC_CHANNEL_CNT*LTR35_DAC_CH_OUTPUTS_CNT];
} TINFO_LTR35;



/** @brief Настройки канала ЦАП.

    Структура, содержащая настройки одного канала ЦАП. */
typedef struct {
    BOOLEAN Enabled; /**< Разрешение выдачи сигнала для данного канала */
    BYTE    Output;  /**< Используемый выход для данного канала (значение из #e_LTR35_DAC_OUTPUT) */
    BYTE    Source;  /**< Источник данных для данного канала (значение из #e_LTR35_CH_SRC) */
    double  ArithAmp; /**< Амплитуда сигнала в режиме арифметического генератора */
    double  ArithOffs; /**< Смещение сигнала в режиме арифметического генератора */
    double  Attenuation; /**< Коэффициент затухания в dB (от 0 до 119.5 с шагом 0.5) */
    DWORD   Reserved[8]; /**< Резервные поля */
} TLTR35_CHANNEL_CONFIG;

/** @brief Настройки арифметического генератора.

    Структура, содержащая настройки одного арифметического генератора. */
typedef struct {
    double Phase; /**< Начальная фаза сигнала в радианах */
    double Delta; /**< Приращение фазы сигнала для каждого значения, выведенного на ЦАП,
                        в радианах */
    DWORD   Reserved[32]; /**< Резервные поля */
} TLTR35_ARITH_SRC_CONFIG;

/** @brief Настройки синтезатора.

    Для получения входной частоты ЦАП из опорной частоты 30 МГц в модуле
    используется синтезатор. Полученная на выходе синтезатора частота определяется
    3-мя коэффициентами, заданными в данной структуре, и рассчитывается по формуле
    \f$ f_{out}=\frac{f_{in}*b}{r*2*a}\f$, где \f$f_{in}=30 \f$ МГц.
    Результирующая частота преобразования ЦАП уже зависит от полученной
    входной частоты ЦАП с выхода синтезатора и от настроенной скорости выдачи
    (поле [DacRate](@ref TLTR35_CONFIG::DacRate) структуры
    [настроек модуля](@ref TLTR35::Cfg)).

    Как правило пользователю не требуется заполнять эти параметры вручную. Для
    их заполнения используется функция LTR35_FillFreq()
 */
typedef struct {
    WORD b; /**< Коэффициент b в настройках синтезатора */
    WORD r; /**< Коэффициент r в настройках синтезатора */
    BYTE a; /**< Коэффициент a в настройках синтезатора */
} TLTR35_SYNT_CONFIG;

/** @brief Настройки модуля.

    Структура содержит все настройки модуля, которые должен заполнить
    пользователь перед вызовом LTR35_Configure(). */
typedef struct {    
    /** Настройки каналов ЦАП. */
    TLTR35_CHANNEL_CONFIG    Ch[LTR35_DAC_CHANNEL_CNT];
    /** Настройки арифметических генероторов. */
    TLTR35_ARITH_SRC_CONFIG  ArithSrc[LTR35_ARITH_SRC_CNT];

    BYTE Mode; /**< Режим работы модуля (значение из #e_LTR35_MODE). */
    BYTE DataFmt; /**< Формат данных (значение из #e_LTR35_DATA_FORMAT). */

    BYTE DacRate; /**< Скорости выдачи (константа из #e_LTR35_RATE).
                       Как правило заполняется с помощью функции LTR35_FillFreq(). */
    TLTR35_SYNT_CONFIG Synt; /**< Настройки синтезатора.
                        Как правило заполняется с помощью функции LTR35_FillFreq().*/

    WORD StreamStatusPeriod; /**< Период передачи статусных слов. В потоковом
                                  режиме ([Mode](@ref TLTR35_CONFIG::Mode)
                                  = #LTR35_MODE_STREAM) статусное слово будет
                                  передаваться после вывода каждых
                                  StreamStatusPeriod слов из буфера.
                                  0 означает выбор значения по-умолчанию. */
    BYTE EchoEnable;        /**<  Разрешение передачи эхо-данных от одного
                                  выбранного канала. */
    BYTE EchoChannel;       /**<  При разрешенной передачи эхо-данных определяет
                                  номер канала, которому будут соответствовать
                                  эти данные */

    DWORD Reserved[63];   /**< Резервные поля (должны быть установлены в 0) */
} TLTR35_CONFIG;


/** @brief Параметры текущего состояния модуля.

    Структура, содержащая параметры модуля, которые пользователь должен использовать
    только для чтения, так как они изменяются только внутри функций ltr35api. */
typedef struct {
    BYTE   FpgaState; /**<  Текущее состояние ПЛИС. Одно из значений из e_LTR_FPGA_STATE */
    BYTE   Run;      /**< Признак, запущен ли сейчас вывод на ЦАП (в потоковом режиме
                          или в режиме циклического автогенератора) */
    double DacFreq; /**< Установленная частота ЦАП. Обновляется после
                         вызова LTR35_Configure(). */
    BYTE    EnabledChCnt; /**< Количество разрешенных каналов ЦАП. Обновляется после
                               вызова LTR35_Configure(). */
    BYTE    SDRAMChCnt; /**< Количество разрешенных каналов ЦАП, отсчеты для
                             которых берутся из буфера модуля. Обновляется после
                             вызова LTR35_Configure(). */
    BYTE    ArithChCnt; /**< Количество разрешенных каналов ЦАП, настроенных
                             на режим арифметического генератора. Обновляется после
                             вызова LTR35_Configure(). */

    DWORD   Reserved[32]; /**< Резервные поля */
} TLTR35_STATE;

/** @brief  Управляющая структура модуля.

    Хранит текущие настройки модуля, информацию о его состоянии,
    структуру канала связи. Передается в большинство функций
    библиотеки. Некоторые поля структуры доступны для изменения пользователем
    для настройки параметров модуля. Перед использованием требует инициализации
    с помощью функции LTR35_Init().
 */
typedef struct {
    INT size;            /**< Размер структуры. Заполняется в LTR35_Init(). */
    /** Структура, содержащая состояние клиентского соединения со службой ltrd.
        Не используется напрямую пользователем. */
    TLTR Channel;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
      используемыми исключительно библиотекой и недоступными для пользователя. */
    void* Internal;
    TLTR35_CONFIG Cfg;           /**< Настройки модуля. Заполняются пользователем
                                      перед вызовом LTR35_Configure() */
    /** Состояние модуля и рассчитанные параметры. Поля изменяются функциями
        библиотеки. Пользовательской программой могут использоваться
        только для чтения. */
    TLTR35_STATE  State;
    TINFO_LTR35   ModuleInfo; /**< Информация о модуле */
} TLTR35;
#pragma pack()


/** @} */

/***************************************************************************//**
    @addtogroup func_init Функции инициализации и работы с соединением с модулем.
    @{
*******************************************************************************/

/***************************************************************************//**
  @brief Инициализация описателя модуля.

  Функция инициализирует поля структуры описателя модуля значениями по-умолчанию.
  Эта функция должна вызываться для каждой структуры #TLTR35 перед вызовом
  остальных функций.
   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_Init(TLTR35 *hnd);

/***************************************************************************//**
   @brief Открытие соединения с модулем.

   Функция устанавливает соединение с модулем в соответствии с переданными
   параметрами, проверяет наличие модуля и считывает информацию о нем.
   Должна быть вызвана перед работой с модулем. После завершения работы
   необходимо закрыть соединение с помощью LTR35_Close().

   @param[in] hnd       Описатель модуля
   @param[in] ltrd_addr IP-адрес машины, на которой запущена служба ltrd в 32-битном
                        формате (описан в разделе "Формат задания IP-адресов"
                        @docref_ltrapi{руководства для библиотеки ltrapi}).
                        Если служба ltrd запущена на той же машине, что и программа,
                        вызывающая данную функцию, то в качестве адреса
                        можно передать LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port TCP-порт для подключения к службе ltrd. По умолчанию
                        используется LTRD_PORT_DEFAULT.
   @param[in] csn       Серийный номер крейта, в котором находится интересующий
                        модуль. Представляет собой оканчивающуюся нулем ASCII-строку.
                        Для соединения с первым найденным крейтом можно передать
                        пустую строку или нулевой указатель.
   @param[in] slot      Номер слота крейта, в котором установлен интересующий модуль.
                        Значение от LTR_CC_CHNUM_MODULE1 до LTR_CC_CHNUM_MODULE16.
   @return              Код ошибки
*******************************************************************************/
LTR35API_DllExport(INT) LTR35_Open(TLTR35 *hnd, DWORD ltrd_addr, WORD ltrd_port,
                                   const CHAR *csn, WORD slot);
/***************************************************************************//**
   @brief Закрытие соединения с модулем.

   Функция закрывает ранее открытое с помощью LTR35_Open() соединение. Должна
   вызываться после завершения работы с модулем. При любом возвращенном значении
   после вызова этой функции соответствующий описатель уже нельзя использовать
   для работы с модулем без открытия нового соединения.

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_Close (TLTR35 *hnd);

/***************************************************************************//**
   @brief Проверка, открыто ли соединение с модулем.

   Функция проверяет, открыто ли в данный момент соединение с модулем. Если
   соединение открыто, функция возвращает LTR_OK, если закрыто --- код ошибки
   LTR_ERROR_CHANNEL_CLOSED.
   @param[in] hnd       Описатель модуля
   @return              Код ошибки (LTR_OK, если соединение установлено)
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_IsOpened (TLTR35 *hnd);
/** @} */

/***************************************************************************//**
    @addtogroup func_config Функции для изменения настроек модуля
    @{
*******************************************************************************/


/***************************************************************************//**
   @brief Подбор коэффициентов для получения заданной частоты преобразования ЦАП.

   Функция подбирает нужные настройки (параметры синтезатора из
   [Synt](@ref TLTR35_CONFIG::Synt) и значение поля
   [DacRate](@ref TLTR35_CONFIG::DacRate)), чтобы  результирующая частота
   преобразования ЦАП была наиболее близка к указанной пользователем.
   При этом эта же частота определяет частоту вывода на цифровые линии.

   @param[in,out] cfg       Структура с настройками модуля, поля которой будут изменены,
                            чтобы полученная частота преобразования ЦАП была наиболее
                            близка к заданной. Поля, не влияющие на частоту
                            преобразования ЦАП, будут оставлены без изменений.
   @param[in]     freq      Требуемая частота преобразования ЦАП в Гц.
   @param[out]    fnd_freq  В данной переменной возвращается реально полученная
                            частота преобразования ЦАП, соответствующая подобранным
                            параметрам. Может быть передан нулевой указатель,
                            если это значение не интересует.
   @return                  Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FillFreq(TLTR35_CONFIG *cfg, double freq, double *fnd_freq);

/***************************************************************************//**
   @brief Запись настроек в модуль.

   Функция передает настройки, соответствующие значением полей настроек модуля
   из поля [Cfg](@ref TLTR35::Cfg) описателя модуля, в модуль.
   Должна вызываться перед началом передачи данных в модуль.

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_Configure(TLTR35 *hnd);
/** @} */



/***************************************************************************//**
    @addtogroup func_stream Функции для управления выдачей данных
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Передача данных ЦАП и цифровых выходов в модуль.

    Функция передает данные для вывода в модуль. Данные должны быть в
    специальном формате. Подготовить данные в нужном формате можно с помощью
    функций LTR35_PrepareDacData() или LTR35_PrepareData().

    Возвращение означает только то, что данные переданы в буфер сокета и поставлены
    на передачу, а не то, что они уже дошли до самого модуля или тем более выведены.

    Если места в буфере нет, то функция будет ожидать освобождения. Выход из функции
    осуществляется либо когда все данные будут записаны в буфер, либо когда
    истечет указанный таймаут.

    @param[in]  hnd     Описатель модуля.
    @param[in]  data    Массив со словами, которые необходимо передать модулю
    @param[in]  size    Количество передаваемых 32-битных слов.
    @param[in]  timeout Таймаут на выполнение функции в мс.
    @return             Если < 0 --- код ошибки.
                        Если >= 0 --- количество записанных слов.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_Send(TLTR35 *hnd, const DWORD *data,
                                    DWORD size, DWORD timeout);

/***************************************************************************//**
    @brief Подготовка данных для передачи в модуль.

    Функция принимает отсчеты ЦАП и цифровых линий и на основе этих данных
    формирует слова специального формата для дальнейшей передачи в модуль с
    помощью LTR35_Send().

    Данные от ЦАП принимаются в следующем порядке: 1-ый отсчет первого разрешенного
    канала с источником из буфера модуля, затем 1-ый отсчет второго такого канала
    и т.д., затем по второму отсчету каждого канала и т.д. Количество данных
    ЦАП на входе должно быть кратно количеству разрешенных каналов с источником
    #LTR35_CH_SRC_SDRAM (поле [SDRAMChCnt](@ref TLTR35_STATE::SDRAMChCnt)
    структуры состояния модуля [State](@ref TLTR35::State)). Данные для ЦАП могут
    быть либо в кодах ЦАП, либо в Вольтах (если передан флаг #e_LTR35_PREP_FLAGS).
    При этом калибровки применяются всегда аппаратно внутри модуля.

    Также функция принимает данные цифровых линий --- младшие биты соответствуют
    значениям цифровых линий, которые могут быть объеденены по "или" с флагами
    из #e_LTR35_DOUTS_WORD_FLAGS. На каждые [SDRAMChCnt](@ref TLTR35_STATE::SDRAMChCnt)
    отсчетов ЦАП в выходной буфер помещается одно слово цифровых линий.

    Формат выходных слов и количество слов на отсчет определяется заданным
    [форматом передаваемых данных](@ref TLTR35_CONFIG::DataFmt).
    Функция завершает работу как только закончится один из входных массивов
    или место в выходном буфере.

    @param[in]  hnd         Описатель модуля.
    @param[in]  dac_data    Массив отсчетов ЦАП. Может быть передан нулевой указатель,
                            если используются только цифровые линии.
    @param[in,out] dac_size На входе задает количество отсчетов в массиве dac_data,
                            на выходе возвращает, сколько отсчетов было реально
                            записано в выходной массив.
    @param[in] dout_data    Массив значений для вывода на цифровые линии.
                            Может быть передан нулевой указатель,
                            если используются только данные ЦАП.
    @param[in,out] dout_size На выходе задает количество значений в массиве dout_data,
                             на выходе возвращает, сколько значений было реально
                             записано в выходной массив.
    @param[in] flags         Набор флагов из #e_LTR35_PREP_FLAGS.
    @param[out] result       Результирующий массив в специальном формате для
                             передачи в модуль.
    @param[in,out] snd_size  На входе задает размер массива result, на выходе
                             возвращает, сколько всего 32-битных слов было записано
                             в массив result в ходе выполнения функции.
    @return                  Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_PrepareData(TLTR35 *hnd, const double *dac_data,
                                          DWORD *dac_size, const DWORD *dout_data,
                                          DWORD *dout_size,DWORD flags,
                                          DWORD *result, DWORD *snd_size);

/***************************************************************************//**
    @brief Подготовка данных ЦАП для передачи в модуль.

    Данная функция является упрощенным вариантом LTR35_PrepareData(), но
    подготавливает только данные ЦАП (без поддержки цифровых выходов).
    Выходной массив должен быть обязательно достаточного размера.

    @param[in]  hnd         Описатель модуля.
    @param[in]  dac_data    Массив отсчетов ЦАП. Формат аналогичен
                            LTR35_PrepareData().
    @param[in]  size        Количество отсчетов в массиве dac_data. Значение
                            должно быть кратно [SDRAMChCnt](@ref TLTR35_STATE::SDRAMChCnt).
    @param[in] flags        Набор флагов из #e_LTR35_PREP_FLAGS
    @param[out] result      Результирующий массив в специальном формате для
                            передачи в модуль. Массив должен быть размером
                            не меньше n*size, где n = 1 или 2 в зависимости от
                            [формата передаваемых данных](@ref TLTR35_CONFIG::DataFmt).
    @param[out] snd_size    Если не нулевой указатель, то в данной переменной
                            возвращается количество подготовленных слов для
                            передачи в модуль.
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_PrepareDacData(TLTR35 *hnd, const double *dac_data,
                                             DWORD size, DWORD flags, DWORD *result,
                                             DWORD *snd_size);
/***************************************************************************//**
    @brief Смена страницы вывода в режиме циклического автогенератора.

    При вызове данной функции начинается циклическая выдача данных с ранее
    загруженной страницы буфера модуля. Перед вызовом выводимые данные должны
    быть загружены с помощью LTR35_Send().
    При этом для записи становится доступна следующая страница.

    Функция доступна только в циклическом режиме
    ([Mode](@ref TLTR35_CONFIG::Mode) = #LTR35_MODE_CYCLE).

    Если при вызове этой функции выдача данных еще не была запущена (первый вызов после
    LTR35_Configure() или LTR35_Stop()), то по вызову функции будет запущена
    выдача отсчетов, записанных до этого в первую страницу циклического буфера.
    Последующий вызов LTR35_Send() будет записывать данные уже во вторую страницу.

    Если при вызове уже идет вывод циклической страницы, то при выводе последнего
    отсчета с этой страницы происходит смена страниц. Начинает выводится вторая страница,
    а ранее выводимая становится доступна для записи новых данных.

    Функция возвращает управления после получения подтверждения смены страницы.
    Так как для этого все ранее записанные данные должны быть переданы в модуль
    и кроме того, если уже идет выдача данных, то она должна дойти до последнего
    отсчета из выдаваемой страницы, то время ответа может быть значительным.
    Именно поэтому таймаут задается вручную через параметр функции.


    @param[in]  hnd         Описатель модуля.
    @param[in]  flags       Флаги (резерв --- должен передаваться 0).
    @param[in]  tout        Таймаут в мс на время выполнения функции (на время
                            ожидания ответа о завершении смены страницы).
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_SwitchCyclePage(TLTR35 *hnd, DWORD flags, DWORD tout);

/***************************************************************************//**
    @brief Запуск выдачи данных в потоковом режиме.

    При вызове данной функции начинается выдача данных на ЦАП в потоковом режиме.
    ([Mode](@ref TLTR35_CONFIG::Mode) = #LTR35_MODE_STREAM).

    При этом для реализации непрерывного вывода часть данных уже должна
    быть подкачана в буфер модуля с помощью LTR35_Send().

    @param[in]  hnd         Описатель модуля.
    @param[in]  flags       Флаги (резерв --- должен передаваться 0).
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_StreamStart(TLTR35 *hnd, DWORD flags);

/***************************************************************************//**
   @brief Останов выдачи данных.

   Функция завершает выдачу данных на ЦАП как в потоковом режиме, так и в
   режиме циклического автогенератора, а также останавливает каналы в режиме
   арифметического генератора. На выходах остаются последние выведенные значения.

   При этом следует учесть, что в потоковом режиме, если сильно заполнен
   буфер ltrd, то команда дойдет до модуля только когда все слова данных из
   буфера будут переданы в модуль, поэтому время выполнения может быть значительное.
   Для этого случая сделана функция LTR35_StopWithTout(), в которой явно можно
   указать время ожидания ответа на команду останова.

    @param[in]  hnd         Описатель модуля.
    @param[in]  flags       Флаги (резерв --- должен передаваться 0).
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_Stop(TLTR35 *hnd, DWORD flags);

/***************************************************************************//**
   @brief Останов выдачи данных с заданным временем ожидания ответа.

   Функция аналогична LTR35_Stop() за исключением того, что время ожидания
    ответа можно задать явно в мс. Служит для случая, если время ответа может
    быть существенно больше стандартного времени ответа на команду (например,
    в потоковом режиме, когда буфер ltrd заполнен данными на передачу).

    @param[in]  hnd         Описатель модуля.
    @param[in]  flags       Флаги (резерв --- должен передаваться 0).
    @param[in]  tout        Время ожидания ответа в мс
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_StopWithTout(TLTR35 *hnd, DWORD flags, DWORD tout);




/***************************************************************************//**
   @brief Изменение частоты для арифметического генератора.

    Функция позволяет изменить частоту арифметического генератора при уже
    запущенной выдаче данных без ее останова и повторного вызова LTR35_Configure().

    При этом частота задается также как и при настройке с помощью значения
    приращения фазы на каждый выводимый отсчет. При успешном выполнении функции
    соответствующим образом обновляется поле [Delta](@ref TLTR35_ARITH_SRC_CONFIG::Delta)
    настроек соответствующего арифметического генератора.

    @param[in]  hnd         Описатель модуля.
    @param[in]  gen_num     Номер арифметического генератора (от 0 до #LTR35_ARITH_SRC_CNT-1).
    @param[in]  delta       Новое значение приращения фазы на отсчет ЦАП в радианах.
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_SetArithSrcDelta(TLTR35 *hnd, BYTE gen_num, double delta);

/***************************************************************************//**
   @brief Изменение амплитуды и смещения арифметического сигнала.

    Функция позволяет изменить амплитуду и смещение сигнала канала ЦАП,
    источником которого является один из арифметических генераторов, при уже
    запущенной выдаче данных без ее останова и повторного вызова LTR35_Configure().

    При успешном выполнении соответствующим образом обновляются поля
    [ArithAmp](@ref TLTR35_CHANNEL_CONFIG::ArithAmp) и
    [ArithOffs](@ref TLTR35_CHANNEL_CONFIG::ArithOffs) настроек соответствующего
    канала.

    @param[in]  hnd         Описатель модуля.
    @param[in]  ch_num      Номер канала ЦАП.
    @param[in]  amp         Новое значение амплитуды в Вольтах.
    @param[in]  offset      Новое значение смещения в Вольтах.
    @return                 Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_SetArithAmp(TLTR35 *hnd, BYTE ch_num,
                                          double amp, double offset);



/***************************************************************************//**
   @brief Прием эхо-данных от модуля.

    Функция позволяет принять слова от модуля с эхо-данными, если один из
    каналов ЦАП был настроен на генерацию эхо-данных. Функция выделяет коды
    ЦАП в принятых словах и возвращает в указанном массиве.

    @param[in]  hnd     Описатель модуля.
    @param[in]  data    Массив, в который будут сохранены принятые отсчеты ЦАП.
                        Отсчеты возвращены в виде 32-битного знакового целого числа,
                        в котором биты старшего байта дублируют знаковый бит.
                        Должен быть размером на size 32-битных слов.
    @param[out] tmark   Указатель на массив размером на size 32-битных слов,
                        в который будут сохранены значения счетчиков синхрометок,
                        соответствующие принятым данным. Генерация меток
                        настраивается для крейта или специального модуля отдельно.
                        Синхрометки подробнее описаны в разделе "Синхрометки"
                        @docref_ltrapi{руководства для библиотеки ltrapi}.
                        Если синхрометки не используются, то можно передать
                        в качестве параметра нулевой указатель.
   @param[in]  size     Запрашиваемое количество 32-битных слов на прием.
   @param[in]  timeout  Таймаут на выполнение операции в миллисекундах. Если в течение
                        заданного времени не будет принято запрашиваемое количество
                        слов, то функция все равно вернет управление, возвратив
                        в качестве результата реально принятое количество слов
   @return              Значение меньше нуля соответствует коду ошибки. Значение
                        больше или равное нулю соответствует количеству реально
                        принятых и сохраненных в массив data слов.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_RecvEchoResp(TLTR35 *hnd, INT *data, DWORD *tmark, DWORD size,
                                           DWORD timeout);


/** @} */

/***************************************************************************//**
    @addtogroup func_misc Функции вспомогательного характера
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Получение сообщения об ошибке.

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для ОС Windows или UTF-8 для ОС Linux. Функция может обработать как ошибки
   из ltr35api, так и общие коды ошибок из ltrapi.

   @param[in] err       Код ошибки
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTR35API_DllExport(LPCSTR) LTR35_GetErrorString(INT err);

/***************************************************************************//**
   @brief Проверка, разрешена ли работа ПЛИС модуля.

   Функция проверяет, разрешена ли работа ПЛИС модуля. Для настройки и генерации
   данных ПЛИС должен быть всегда разрешен.

   @param[in] hnd      Описатель модуля.
   @param[out] enabled В случае успешного выполнения функции в этой переменной
                       возвращается FALSE, если ПЛИС запрещен, или TRUE в
                       противном случае.
   @return             Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FPGAIsEnabled(TLTR35 *hnd, BOOL *enabled);

/***************************************************************************//**
   @brief Разрешение работы ПЛИС модуля.

   Функция разрешает или запрещает работу ПЛИС модуля. Для настройки и генерации
   данных ПЛИС должен быть всегда разрешен. В LTR35_Open() выполняется разрешение
   работы ПЛИС, если была найдена прошивка ПЛИС в памяти модуля и она была
   успешно загружена, поэтому в штатной работе данная функция не используется.

   @param[in] hnd      Описатель модуля.
   @param[in] enable   Если FALSE --- запрет работы ПЛИС, иначе --- разрешение.
   @return             Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FPGAEnable(TLTR35 *hnd, BOOL enable);

/***************************************************************************//**
   @brief Получение информации о состоянии модуля.

   Функция позволяет получить набор флагов, описывающих состояние работы модуля.

   @param[in]  hnd      Описатель модуля.
   @param[out] status   Набор флагов из #e_LTR35_STATUS, объединенных по "или",
                        описывающих состояние модуля
   @return              Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_GetStatus(TLTR35 *hnd, DWORD *status);

/** @} */

/***************************************************************************//**
    @addtogroup func_flash Функции для работы с flash-памятью модуля
    @{
*******************************************************************************/



/***************************************************************************//**
   @brief Чтение данных из flash-памяти модуля

   Функция вычитывает данные, записанные во flash-памяти модуля по заданному
   адресу. Пользователю выделяется область памяти с адреса
   #LTR35_FLASH_USERDATA_ADDR размером #LTR35_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо прочитать данные
   @param[out] data    Массив на size байт, в который будут записаны прочитанные
                       из Flash-памяти данные
   @param[in] size     Количество данных в байтах, которое необходимо прочитать
   @return             Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FlashRead(TLTR35 *hnd, DWORD addr, BYTE *data, DWORD size);

/***************************************************************************//**
   @brief Запись данных во flash-память модуля

   Функция записывает данные во flash-памяти модуля по заданному
   адресу. Запись может выполняться и в не стертые области.
   Однако в стертые заранее с помощью LTR35_FlashErase() области запись выполняется
   быстрее, если указан флаг #LTR35_FLASH_WRITE_ALREDY_ERASED.

   Пользователю выделяется область памяти с адреса
   #LTR35_FLASH_USERDATA_ADDR размером #LTR35_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо выполнить запись
   @param[in] data     Массив из size байт с данными, которые будут записаны
   @param[in] size     Количество данных в байтах, которое необходимо записать
   @param[in] flags    Набор флагов из #e_LTR35_FLASH_WRITE_FLAGS, объединенных
                       по "или".
   @return             Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FlashWrite(TLTR35 *hnd, DWORD addr, const BYTE *data, DWORD size, DWORD flags);


/***************************************************************************//**
   @brief Стирание области flash-память модуля

   Функция стирает область во flash-память модуля по заданному
   адресу. Стирание необходимо выполнять перед записью данных.
   Стирание возможно только блоками, кратными #LTR35_FLASH_ERASE_BLOCK_SIZE байт.
   Пользователю выделяется область памяти с адреса
   #LTR35_FLASH_USERDATA_ADDR размером #LTR35_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо выполнить стирание
   @param[in] size     Размер стираемой области в байтах. Должен быть кратен
                       #LTR35_FLASH_ERASE_BLOCK_SIZE.
   @return             Код ошибки.
 ******************************************************************************/
LTR35API_DllExport(INT) LTR35_FlashErase(TLTR35 *hnd, DWORD addr, DWORD size);

/** @} */

#ifdef __cplusplus
}
#endif

#endif

