#ifndef LTR25API_H_
#define LTR25API_H_

#include "ltrapi.h"

#ifdef __cplusplus
extern "C" {
#endif


#ifdef _WIN32
    #ifdef LTR25API_EXPORTS
        #define LTR25API_DllExport(type) __declspec(dllexport) type APIENTRY
    #else
        #define LTR25API_DllExport(type) __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTR25API_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTR25API_DllExport(type) type
#endif


/***************************************************************************//**
  @addtogroup const_list Константы и перечисления.
  @{
  *****************************************************************************/

/** Количество каналов АЦП в одном модуле LTR25 */
#define LTR25_CHANNEL_CNT        8

/** Количество частот дискретизации. */
#define LTR25_FREQ_CNT           8

/** Количество частот, для которых сохраняются калибровочные коэффициенты */
#define LTR25_CBR_FREQ_CNT       2

/** Количество значений источника тока */
#define LTR25_I_SRC_VALUE_CNT    2

/** Размер поля с названием модуля. */
#define LTR25_NAME_SIZE          8

/** Размер поля с серийным номером модуля. */
#define LTR25_SERIAL_SIZE       16

/** Максимальное пиковое значение в Вольтах для диапазона измерения модуля */
#define LTR25_ADC_RANGE_PEAK    10

/** Код АЦП, соответствующее максимальному пиковому значению */
#define LTR25_ADC_SCALE_CODE_MAX 2000000000



/** Адрес, с которого начинается пользовательская область flash-памяти */
#define LTR25_FLASH_USERDATA_ADDR   0x0
/** Размер пользовательской области flash-памяти */
#define LTR25_FLASH_USERDATA_SIZE   0x100000
/** Минимальный размер блока для стирания flash-памяти. Все операции стирания
    должны быть кратны данному размеру */
#define LTR25_FLASH_ERASE_BLOCK_SIZE 4096




/** @brief Коды ошибок, специфичные для модуля LTR25.

   Коды ошибок, которые определены и используются только в ltr25api.
   Остальные коды ошибок, которые используются разными модулями, определены в ltrapi.h */
typedef enum {
    LTR25_ERR_FPGA_FIRM_TEMP_RANGE      = -10600, /**< Загружена прошивка ПЛИС для неверного температурного диапазона */
    LTR25_ERR_I2C_ACK_STATUS            = -10601, /**< Ошибка обмена при обращении к регистрам АЦП по интерфейсу I2C */
    LTR25_ERR_I2C_INVALID_RESP          = -10602, /**< Неверный ответ на команду при обращении к регистрам АЦП по интерфейсу I2C */
    LTR25_ERR_INVALID_FREQ_CODE         = -10603, /**< Неверно задан код частоты АЦП */
    LTR25_ERR_INVALID_DATA_FORMAT       = -10604, /**< Неверно задан формат данных АЦП */
    LTR25_ERR_INVALID_I_SRC_VALUE       = -10605, /**< Неверно задано значение источника тока */
    LTR25_ERR_CFG_UNSUP_CH_CNT          = -10606, /**< Для заданной частоты и формата не поддерживается заданное количество каналов АЦП */
    LTR25_ERR_NO_ENABLED_CH             = -10607, /**< Не был разрешен ни один канал АЦП */
    LTR25_ERR_ADC_PLL_NOT_LOCKED        = -10608, /**< Ошибка захвата PLL АЦП */
    LTR25_ERR_ADC_REG_CHECK             = -10609, /**< Ошибка проверки значения записанных регистров АЦП */
    LTR25_ERR_LOW_POW_MODE_NOT_CHANGED  = -10610, /**< Не удалось перевести АЦП из/в низкопотребляющее состояние */
    LTR25_ERR_LOW_POW_MODE              = -10611, /**< Модуль находится в низкопотребляющем режиме */
} e_LTR25_ERROR_CODES;


/** Коды частот сбора АЦП */
typedef enum {
    LTR25_FREQ_78K     = 0,     /**< 78.125 кГц */
    LTR25_FREQ_39K     = 1,     /**< 39.0625 кГц */
    LTR25_FREQ_19K     = 2,     /**< 19.53125 кГц */
    LTR25_FREQ_9K7     = 3,     /**< 9.765625 кГц */
    LTR25_FREQ_4K8     = 4,     /**< 4.8828125 кГц */
    LTR25_FREQ_2K4     = 5,    /**< 2.44140625 кГц */
    LTR25_FREQ_1K2     = 6,    /**< 1.220703125 кГц */
    LTR25_FREQ_610     = 7,    /**< 610.3515625 Гц */
} e_LTR25_FREQS;


/** Форматы данных от модуля */
typedef enum {
    LTR25_FORMAT_20 = 0, /**< 20-битный целочисленный (1 слово на отсчет) */
    LTR25_FORMAT_32 = 1, /**< 32-битный целочисленный (2 слова на отсчет) */
} e_LTR25_FORMATS;

/** Значения источника тока. */
typedef enum {
    LTR25_I_SRC_VALUE_2_86   = 0, /**< 2.86 мА. */
    LTR25_I_SRC_VALUE_10     = 1  /**< 10 мА. */
} e_LTR25_I_SOURCES;

/** @brief Флаги обработки данных.

    Флаги, управляющие работой функции LTR25_ProcessData() */
typedef enum {
    /** Признак, что нужно перевести коды АЦП в Вольты. Если данный флаг не указан,
        то будут возвращены коды АЦП. При этом код #LTR25_ADC_SCALE_CODE_MAX
        соответствует максимальному напряжению для установленного диапазона. */
    LTR25_PROC_FLAG_VOLT          = 0x0001,
    /** По умолчанию LTR25_ProcessData() предполагает, что ей на обработку
        передаются все принятые данные и проверяет непрерывность счетчика не только
        внутри переданного блока данных, но и между вызовами.
        Если обрабатываются не все данные или одни и те же данные обрабатываются
        повторно, то нужно указать данный флаг, чтобы счетчик проверялся только
        внутри обрабатываемого блока */
    LTR25_PROC_FLAG_NONCONT_DATA = 0x0100
} e_LTR25_PROC_FLAGS;



/** @brief Состояние входного канала.

    Возвращается LTR25_ProcessData() для каждого разрешенного канала и определяет,
    был ли обнаружен обрыв или КЗ для данного канала в обрабатываемом с помощью
    LTR25_ProcessData() блоке */
typedef enum {
    LTR25_CH_STATUS_OK          = 0, /**< Канал в рабочем состоянии */
    LTR25_CH_STATUS_SHORT       = 1, /**< Было обнаружено короткое замыкание */
    LTR25_CH_STATUS_OPEN        = 2, /**< Был обнаружен разрыв цепи */
} e_LTR25_CH_STATUS;


/** @} */

/***************************************************************************//**
  @addtogroup type_list Типы данных.
  @{
  *****************************************************************************/

#pragma pack(4)

/** @brief Калибровочные коэффициенты.

    Структура, хранящая калибровочные коэффициенты для одного канала
    и диапазона. */
typedef struct {
    float   Offset; /**< Код смещения */
    float   Scale;  /**< Коэффициент шкалы */
} TLTR25_CBR_COEF;



/** @brief Набор коэффициентов для коррекции АЧХ модуля */
typedef struct {
    /** Частота сигнала, для которой снято отношение амплитуд из FirCoef */
    double AfcFreq;
    /** Набор отношений измеренной амплитуды синусоидального сигнала
         к реальной амплитуде для макс. частоты дискретизации и частоты сигнала
         из AfcFreq для каждого канала */
    double FirCoef[LTR25_CHANNEL_CNT];
} TLTR25_AFC_COEFS;

/** @brief  Информация о модуле.

    Структура, содержащая информацию о версиях прошивок микросхем модуля и
    информацию, считанную из Flash-памяти модуля (серийный номер, калибровочные
    коэффициенты).

    Все поля заполняются при вызове LTR25_Open()
 */
typedef struct {
    /** Название модуля ("LTR25").  */
    CHAR        Name[LTR25_NAME_SIZE];
    /** Серийный номер модуля.      */
    CHAR        Serial[LTR25_SERIAL_SIZE];
    /** Версия прошивки ПЛИС */
    WORD        VerFPGA;
    /** Версия прошивки PLD */
    BYTE        VerPLD;
    /** Ревизия платы */
    BYTE        BoardRev;
    /** Признак, это индустриальный вариант модуля или нет */
    BOOL        Industrial;
    /** Зарезервированные поля. Всегда равны 0 */
    DWORD       Reserved[8];
    /** Калибровочные коэффициенты модуля. Считываются из Flash-памяти
        модуля при вызове LTR25_Open() или LTR25_GetConfig() и загружаются
        в ПЛИС для применения во время вызова LTR25_SetADC() */
    TLTR25_CBR_COEF CbrCoef [LTR25_CHANNEL_CNT] [LTR25_CBR_FREQ_CNT];
    /** Коэффициенты для коррекции АЧХ модуля */
    TLTR25_AFC_COEFS AfcCoef;
    /** Резервные поля */
    double     Reserved2[32*LTR25_CHANNEL_CNT-sizeof(TLTR25_AFC_COEFS)/sizeof(double)];
} TINFO_LTR25;

/** @brief Настройки канала АЦП.

    Структура, содержащая настройки одного канала АЦП. */
typedef struct {
    BOOL  Enabled; /**< Признак, разрешен ли сбор по данному каналу */
    DWORD Reserved[11];  /**< Резервные поля (не должны изменяться пользователем) */
} TLTR25_CHANNEL_CONFIG;

/** @brief Настройки модуля.

    Структура содержит все настройки модуля, которые должен заполнить
    пользователь перед вызовом LTR25_SetADC(). */
typedef struct {
    TLTR25_CHANNEL_CONFIG Ch[LTR25_CHANNEL_CNT]; /**< Настройки каналов АЦП */
    BYTE        FreqCode; /**< Код, задающий требуемую частоту сбора АЦП.
                               Одно из значений #e_LTR25_FREQS */
    BYTE        DataFmt;  /**< Формат, в котором будут передаваться отсчеты
                               АЦП от модуля. Одно из значений #e_LTR25_FORMATS.
                               Формат определяет также количество передаваемых
                               слов на один отсчет и влияет на максимальное
                               количество разрешенных каналов */
    BYTE        ISrcValue; /**< Используемое значение источника тока. Одно из
                                значений #e_LTR25_I_SOURCES */
    DWORD       Reserved[50]; /**< Резервные поля (не должны изменяться пользователем) */
} TLTR25_CONFIG;


/** @brief Параметры текущего состояния модуля.

    Структура, содержащая параметры модуля, которые пользователь должен использовать
    только для чтения, так как они изменяются только внутри функций ltr25api. */
typedef struct {
    BYTE   FpgaState; /**<  Текущее состояние ПЛИС. Одно из значений из e_LTR_FPGA_STATE */
    BYTE   EnabledChCnt; /**< Количество разрешенных каналов. Устанавливается
                                после вызова LTR25_SetADC() */
    BOOL   Run;         /**< Признак, запущен ли сбор данных */
    double AdcFreq;     /**< Установленная частота АЦП. Обновляется после
                            вызова LTR25_SetADC() */
    BOOL   LowPowMode; /**< Признак, находится ли модуль в состоянии низкого
                            потребления. В этом состояние нельзя выполнять настройку
                            АЦП или запуск сбора данных. Управление данным
                            режимом выполняется с помощью LTR25_SetLowPowMode() */
    DWORD   Reserved[31]; /**< Резервные поля */
} TLTR25_STATE;

/** @brief  Управляющая структура модуля.
 *
 *  Хранит текущие настройки модуля, информацию о
 *  его состоянии, структуру канала связи. Передается в большинство функций
 *  библиотеки. Некоторые поля структуры доступны для изменения пользователем
 *  для настройки параметров модуля. Перед использованием требует инициализации
 *  с помощью функции LTR25_Init().
 */
typedef struct {
    INT         Size; /**< Размер структуры. Заполняется в LTR25_Init(). */
    /** Структура, содержащая состояние клиентского соединения со службой ltrd.
        Не используется напрямую пользователем. */
    TLTR        Channel;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
      используемыми исключительно библиотекой и недоступными для пользователя. */
    PVOID       Internal;
    /** Настройки модуля. Заполняются пользователем перед вызовом LTR25_SetADC(). */
    TLTR25_CONFIG Cfg;
    /** Состояние модуля и рассчитанные параметры. Поля изменяются функциями
        библиотеки. Пользовательской программой могут использоваться
        только для чтения. */
    TLTR25_STATE  State;
    /** Информация о модуле */
    TINFO_LTR25 ModuleInfo;
} TLTR25;

#pragma pack()

/** @} */

/***************************************************************************//**
    @addtogroup func_init Функции инициализации и работы с соединением с модулем.
    @{
*******************************************************************************/

/***************************************************************************//**
  @brief Инициализация описателя модуля.

  Функция инициализирует поля структуры описателя модуля значениями по-умолчанию.
  Эта функция должна вызываться для каждой структуры #TLTR25 перед вызовом
  остальных функций.
   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_Init(TLTR25 *hnd);

/***************************************************************************//**
   @brief Открытие соединения с модулем.

   Функция устанавливает соединение с модулем в соответствии с переданными
   параметрами, проверяет наличие модуля и считывает информацию о нем.
   Должна быть вызвана перед работой с модулем. После завершения работы
   необходимо закрыть соединение с помощью LTR25_Close().

   @param[in] hnd       Описатель модуля
   @param[in] ltrd_addr  IP-адрес машины, на которой запущена служба ltrd в 32-битном
                        формате (описан в разделе "Формат задания IP-адресов"
                        @docref_ltrapi{руководства для библиотеки ltrapi}).
                        Если служба ltrd запущена на той же машине, что и программа,
                        вызывающая данную функцию, то в качестве адреса
                        можно передать LTRD_ADDR_DEFAULT.
   @param[in] ltrd_port TCP-порт для подключения к службе ltrd. По умолчанию
                        используется LTRD_PORT_DEFAULT.
   @param[in] csn       Серийный номер крейта, в котором находится интересующий
                        модуль. Представляет собой оканчивающуюся нулем ASCII-строку.
                        Для соединения с первым найденным крейтом можно передать
                        пустую строку или нулевой указатель.
   @param[in] slot      Номер слота крейта, в котором установлен интересующий модуль.
                        Значение от LTR_CC_CHNUM_MODULE1 до LTR_CC_CHNUM_MODULE16.
   @return              Код ошибки
*******************************************************************************/
LTR25API_DllExport(INT) LTR25_Open(TLTR25 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *csn, INT slot);

/** @cond kd_extension */
LTR25API_DllExport(INT) LTR25_OpenEx(TLTR25 *hnd, DWORD ltrd_addr, WORD ltrd_port,
    const CHAR *crate_sn, INT slot, DWORD in_flags, DWORD *out_flags);
/** @endcond */

/***************************************************************************//**
   @brief Закрытие соединения с модулем.

   Функция закрывает ранее открытое с помощью LTR25_Open() соединение. Должна
   вызываться после завершения работы с модулем. При любом возвращенном значении
   после вызова этой функции соответствующий описатель уже нельзя использовать
   для работы с модулем без открытия нового соединения.

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_Close(TLTR25 *hnd);

/***************************************************************************//**
   @brief Проверка, открыто ли соединение с модулем.

   Функция проверяет, открыто ли в данный момент соединение с модулем. Если
   соединение открыто, функция возвращает LTR_OK, если закрыто --- код ошибки
   LTR_ERROR_CHANNEL_CLOSED.
   @param[in] hnd       Описатель модуля
   @return              Код ошибки (LTR_OK, если соединение установлено)
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_IsOpened(TLTR25 *hnd);



/** @} */

/***************************************************************************//**
    @addtogroup func_config Функции для изменения настроек модуля
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Запись настроек в модуль.

   Функция передает настройки, соответствующие значением полей поля Cfg описателя
   модуля, в модуль. Должна вызываться перед запуском сбора данных с помощью
   LTR25_Start().

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_SetADC(TLTR25 *hnd);

/** @} */



/***************************************************************************//**
    @addtogroup func_stream Функции для управления сбором данных
    @{
*******************************************************************************/

/***************************************************************************//**
    @brief Запуск сбора данных.

    При вызове данной функции запускается сбор данных АЦП модулей. После успешного
    завершения этой функции запускается АЦП и модуль начинает передавать в ПК
    полученные отсчеты, которые необходимо вычитывать с помощью LTR25_Recv().
    При завершении измерения для остановки сбора данных необходимо вызвать
    LTR25_Stop().

    Хотя бы один из каналов АЦП должен быть до этого разрешен и модуль должен быть
    сконфигурирован с помощью LTR25_SetADC().

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_Start(TLTR25 *hnd);

/***************************************************************************//**
    @brief Останов сбора данных.

    При вызове данной функции модуль останавливает сбор и выдачу данных АЦП.

    При этом вычитываются и выбрасываются все переданные, но непрочитанные
    данные от модуля.

   @param[in] hnd       Описатель модуля
   @return              Код ошибки
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_Stop(TLTR25 *hnd);

/***************************************************************************//**
   @brief Прием данных от модуля.

   Функция принимает запрашиваемое число слов от модуля. Возвращаемые слова
   находятся в специальном формате, который включает в себя служебную информацию.
   Формат и количество слов на один отсчет определяется настройкой Cfg.DataFmt

   Для обработки принятых слов и получения значений АЦП используется функция
   LTR25_ProcessData().

   Функция возвращает управление либо когда примет запрошенное количество слов,
   либо по истечению таймаута. При этом реально принятое количество слов можно
   узнать по возвращенному значению.

   @param[in]  hnd      Описатель модуля.
   @param[out] data     Массив, в который будут сохранены принятые слова. Должен
                        быть размером на size 32-битных слов.
   @param[out] tmark    Указатель на массив размером на size 32-битных слов,
                        в который будут сохранены значения счетчиков синхрометок,
                        соответствующие принятым данным. Генерация меток
                        настраивается для крейта или специального модуля отдельно.
                        Синхрометки подробнее описаны в разделе "Синхрометки"
                        @docref_ltrapi{руководства для библиотеки ltrapi}.
                        Если синхрометки не используются, то можно передать
                        в качестве параметра нулевой указатель.
   @param[in]  size     Запрашиваемое количество 32-битных слов на прием.
   @param[in]  timeout  Таймаут на выполнение операции в миллисекундах. Если в течение
                        заданного времени не будет принято запрашиваемое количество
                        слов, то функция все равно вернет управление, возвратив
                        в качестве результата реально принятое количество слов
   @return              Значение меньше нуля соответствует коду ошибки. Значение
                        больше или равное нулю соответствует количеству реально
                        принятых и сохраненных в массив data слов.
*******************************************************************************/
LTR25API_DllExport(INT) LTR25_Recv(TLTR25 *hnd, DWORD *data, DWORD *tmark, DWORD size,
                                   DWORD timeout);


/****************************************************************************//**
   @brief Обработка принятых от модуля слов.

   Функция используется для обработки слов, принятых от модуля с помощью LTR25_Recv().
   Функция проверяет служебные поля принятых слов, извлекает полезную информацию
   с отсчетами и, при указании флага #LTR25_PROC_FLAG_VOLT, переводит отсчеты в
   Вольты.

   Функция предполагает, что передаваемые слова выровнены на начала кадра (первое
   слово первого разрешенного канала). Если это не так, то неполный кадр в начале
   будет отброшен и функция вернет ошибку LTR_ERROR_PROCDATA_UNALIGNED.

   Также функция анализирует признаки обрыва или КЗ в разрешенных каналах. При
   наличии подобного признака для хотя бы одного отсчета соответствующего
   канала устанавливается соответствующей статус в элементе массива ch_status.

   В отличие от модуля LTR24 в модуле LTR25 калибровка отсчетов и корректировка
   АЧХ выполняется внутри модуля аппаратно, поэтому в LTR25_ProcessData() нет
   подобных флагов.

   Функция проверяет целостность данных с помощью счетчика из служебной
   информации. По-умолчанию функция предполагает, что все принятые данные обрабатываются
   и только один раз, проверяя непрерывность счетчика и между вызовами функции.
   Если это условие не выполняется, то нужно передать флаг #LTR25_PROC_FLAG_NONCONT_DATA.

   @param[in] hnd      Описатель модуля.
   @param[in] src      Указатель на массив, содержащий слова, принятые от модуля
                       с помощью LTR25_Recv(), которые нужно обработать.
   @param[out] dest    Указатель на массив, в который будут сохранены обработанные
                       данные. Порядок следования соответствует порядку во входном
                       массиве (т.е. сперва первый отсчет первого разрешенного канала,
                        затем первый отсчет второго канала и т.д.).
   @param[in,out] size  На входе принимает размер массива src для обработки. На
                        выходе, при успешном завершении, возвращает количество
                        сохраненных отсчетов в массиве dest.
   @param[in]  flags   Флаги из #e_LTR25_PROC_FLAGS, управляющие работой функции.
                        Может быть объединено несколько флагов через логическое
                        ИЛИ.
   @param[out] ch_status  Массив размером на количество элементов, соответствующее
                        количеству разрешенных каналов. В каждом элменте сохраняется
                        статус канала (одно из значений #e_LTR25_CH_STATUS),
                        определяющий, были ли признаки КЗ или обрыва
                        в соответствующем канале. Может быть передан нулевой
                        указатель, если эта информация не нужна.
   @return              Код ошибки.
  *****************************************************************************/
LTR25API_DllExport(INT) LTR25_ProcessData(TLTR25 *hnd, const DWORD *src, double *dest,
                                    INT *size, DWORD flags, DWORD *ch_status);


/****************************************************************************//**
   @brief Поиск начала первого кадра.

   Функция находит в принятом массиве сырых данных от модуля индекс первого
   слова первого начала кадра. Может использоваться для выравнивания на
   начало кадра данных в случае произошедшего сбоя без останова сбора.

   Если в переданном массиве начало кадра не найдено, то функция вернет ошибку
   LTR_ERROR_FIRSTFRAME_NOTFOUND.

   @param[in] hnd      Описатель модуля.
   @param[in] data     Указатель на массив, содержащий слова, принятые от модуля
                       с помощью LTR25_Recv(), в которых ищется начало кадра.
   @param[in] size     Количество слов в массиве data
   @param[out] index   В этой переменной возвращается индекс элемента,
                       соответствующего началу первого кадра, если функция
                       завершилась успешно.
   @return             Код ошибки.
  *****************************************************************************/
LTR25API_DllExport(INT) LTR25_SearchFirstFrame(TLTR25 *hnd, const DWORD *data, DWORD size,
                                                DWORD *index);

/** @} */

/***************************************************************************//**
    @addtogroup func_misc Функции вспомогательного характера
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Получение сообщения об ошибке.

   Функция возвращает строку, соответствующую переданному коду ошибки, в кодировке
   CP1251 для ОС Windows или UTF-8 для ОС Linux. Функция может обработать как ошибки
   из ltr25api, так и общие коды ошибок из ltrapi.

   @param[in] err       Код ошибки
   @return              Указатель на строку, содержащую сообщение об ошибке.
 ******************************************************************************/
LTR25API_DllExport(LPCSTR) LTR25_GetErrorString(INT err);


/***************************************************************************//**
    @brief Чтение информации и калибровочных коэффициентов.

    Функция считывает информацию из flash-памяти модуля и обновляет поля ModuleInfo
    в управляющей структуре модуля. Так как данная операция уже выполняется при
    вызове LTR25_Open(), то как правило вызов данной функции не требуется.
    Однако данная функция может быть использована для восстановления измененных
    коэффициентов в ModuleInfo на заводские.


   @param[in] hnd      Описатель модуля.
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_GetConfig(TLTR25 *hnd);

/** @cond kd_extension */
LTR25API_DllExport(INT) LTR25_StoreConfig(TLTR25 *hnd, TLTR_CARD_START_MODE start_mode);
/** @endcond */


/***************************************************************************//**
    @brief Перевод модуля в режим низкого потребления.

    Функция переводит модуль в режим низкого потребления или переводит его
    из этого режима снова в рабочий. В режиме низкого потребления
    снято питание с АЦП и источники тока установлены 2.86 мА. Доступ
    к регистрам АЦП невозможен. Этот режим может быть использован для полного
    сброса АЦП, для чего в этом режиме модуль должен находится не менее 5 с.

   @param[in] hnd           Описатель модуля.
   @param[in] lowPowMode    Если FALSE --- перевод модуля в рабочий режим, иначе ---
                            в режим низкого потребления.
   @return                  Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_SetLowPowMode(TLTR25 *hnd, BOOL lowPowMode);





/***************************************************************************//**
   @brief Проверка, разрешена ли работа ПЛИС модуля.

   Функция проверяет, разрешена ли работа ПЛИС модуля. Для настройки и сбора
   данных ПЛИС должен быть всегда разрешен.

   @param[in] hnd      Описатель модуля.
   @param[out] enabled В случае успешного выполнения функции в этой переменной
                       возвращается FALSE, если ПЛИС запрещен, или TRUE в
                       противном случае.
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_FPGAIsEnabled(TLTR25 *hnd, BOOL *enabled);

/***************************************************************************//**
   @brief Разрешение работы ПЛИС модуля.

   Функция разрешает или запрещает работу ПЛИС модуля. Для настройки и сбора
   данных ПЛИС должен быть всегда разрешен. В LTR25_Open() выполняется разрешение
   работы ПЛИС, если была найдена прошивка ПЛИС в памяти модуля и она была
   успешно загружена, поэтому в штатной работе данная функция не используется.

   @param[in] hnd      Описатель модуля.
   @param[in] enable   Если FALSE --- запрет работы плис, иначе --- разрешение
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_FPGAEnable(TLTR25 *hnd, BOOL enable);

/** @} */


/***************************************************************************//**
    @addtogroup func_flash Функции для работы с flash-памятью модуля
    @{
*******************************************************************************/

/***************************************************************************//**
   @brief Чтение данных из flash-памяти модуля

   Функция вычитывает данные, записанные во flash-памяти модуля по заданному
   адресу. Пользователю выделяется область памяти с адреса
   #LTR25_FLASH_USERDATA_ADDR размером #LTR25_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо прочитать данные
   @param[out] data    Массив на size байт, в который будут записаны прочитанные
                       из Flash-памяти данные
   @param[in] size     Количество данных в байтах, которое необходимо прочитать
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_FlashRead(TLTR25 *hnd, DWORD addr, BYTE *data, DWORD size);

/***************************************************************************//**
   @brief Запись данных во flash-память модуля

   Функция записывает данные во flash-память модуля по заданному
   адресу. Записываемая область должна быть предварительно стерта с помощью
   LTR25_FlashErase(). Пользователю выделяется область памяти с адреса
   #LTR25_FLASH_USERDATA_ADDR размером #LTR25_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо выполнить запись
   @param[in] data     Массив из size байт с данными, которые будут записаны
   @param[in] size     Количество данных в байтах, которое необходимо записать
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_FlashWrite(TLTR25 *hnd, DWORD addr, const BYTE *data, DWORD size);

/***************************************************************************//**
   @brief Стирание области flash-память модуля

   Функция стирает область во flash-памяти модуля по заданному
   адресу. Стирание необходимо выполнять перед записью данных.
   Стирание возможно только блоками, кратными #LTR25_FLASH_ERASE_BLOCK_SIZE байт.
   Пользователю выделяется область памяти с адреса
   #LTR25_FLASH_USERDATA_ADDR размером #LTR25_FLASH_USERDATA_SIZE байт.

   @param[in] hnd      Описатель модуля.
   @param[in] addr     Адрес памяти, начиная с которого необходимо выполнить стирание
   @param[in] size     Размер стираемой области в байтах. Должен быть кратен
                       #LTR25_FLASH_ERASE_BLOCK_SIZE.
   @return             Код ошибки.
 ******************************************************************************/
LTR25API_DllExport(INT) LTR25_FlashErase(TLTR25 *hnd, DWORD addr, DWORD size);

/** @} */


#ifdef __cplusplus
}
#endif

#endif
