#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "ltr25api.h"

#include "adau1978.h"
#include "crc.h"
#include "devices/flash_dev_sst25.h"
#include "flash.h"
#include "ltrmodule.h"
#include "ltrmodule_fpga_autoload.h"
#include "ltrslot.h"
#include "ports/flash_iface_ltr.h"
#include "ltimer.h"


/*================================================================================================*/
#ifndef M_PI
#define M_PI 3.14159265359
#endif

#define ADC_PLL_LOCK_TOUT  2000
#define ADC_WAIT_STABLE_TIME  850

#define CMD_ROM_WR_EN                   (LTR010CMD_PROGR | 0x10)
#define CMD_STATUS_FPGA                 (LTR010CMD_INSTR | 0x23)
#define CMD_CH_CONTROL                  (LTR010CMD_INSTR | 0x00)
#define CMD_ADC_FREQ                    (LTR010CMD_INSTR | 0x01)
#define CMD_GO                          (LTR010CMD_INSTR | 0x02)
#define CMD_ADC                         (LTR010CMD_INSTR | 0x04)
#define CMD_K_ADDR                      (LTR010CMD_INSTR | 0x07)
#define CMD_K_DATA1                     (LTR010CMD_INSTR | 0x08)
#define CMD_K_DATA2                     (LTR010CMD_INSTR | 0x09)

#define CMD_RESP_ADC                    (LTR010CMD_INSTR | 0x04)
#define CMD_RESP_ADC_ERR                (LTR010CMD_INSTR | 0x05)

#define FORMAT20_CNTR_MOD               15
#define FORMAT32_CNTR_MOD               13

#define LTR25_COEF_ADDR_CBR                0
#define LTR25_COEF_ADDR_FILTER1            0x40
#define LTR25_COEF_ADDR_FILTER2            0x80
#define LTR25_COEF_ADDR_FILTER3            0xC0

#define CMD_BIT_K_ADDR_WR                  (1UL << 15)

#define FLASH_PROT_MASK          (SST25_STATUS_BP0 | SST25_STATUS_BP1 | SST25_STATUS_BP2 | \
    SST25_STATUS_BP3 | SST25_STATUS_BPL)
#define FLASH_PROT_DEFAULT       (SST25_STATUS_BP2 | SST25_STATUS_BP0 | SST25_STATUS_BPL)
#define FLASH_PROT_FPGA_UPDATE   (SST25_STATUS_BP0)
#define FLASH_PROT_INFO_UPDATE   (0)

/* количество успешных проверок захвата PLL перед принятием решения, что все ок */
#define PLL_LOCK_CHECK_CNT   3

/* Адрес дескриптора в EEPROM. */
#define LTR25_FLASH_ADDR_MODULE_INFO    0x1F0000
/* Признак нового формата информации во Flash-памяти */
#define LTR25_FLASH_INFO_SIGN           0xA55A1919
#define LTR25_FLASH_INFO_CRC_SIZE       2
#define LTR25_FLASH_INFO_HDR_SIZE       16
#define LTR25_FLASH_INFO_FORMAT         1
#define LTR25_FLASH_INFO_MIN_SIZE       sizeof(t_ltr25_flash_info)
#define LTR25_FLASH_INFO_MAX_SIZE       0x10000


#define ARRAY_LEN(a) (sizeof(a)/sizeof((a)[0]))

#define LTR25_MAKE_ADC_CMD(adc_msk, reg_addr, wr, data) LTR_MODULE_MAKE_CMD(CMD_ADC, \
        (((wr & 0x01) << 15) | ((adc_msk & 0x03) << 13) | ((reg_addr & 0x1F) << 8) | (data & 0xFF)))

#define LTR25_MAKE_ADC_FREQ_CMD(hnd, clk_en)  LTR_MODULE_MAKE_CMD(CMD_ADC_FREQ, \
    f_freq_params[hnd->Cfg.FreqCode].FreqCode | (clk_en ? (1<<8) : 0))

#define f_adau1978_reg_rd(hnd, adc_msk, reg, pval) f_adau1978_reg_op(hnd, adc_msk, reg, 0, 0, pval)
#define f_adau1978_reg_wr(hnd, adc_msk, reg, val) f_adau1978_reg_op(hnd, adc_msk, reg, 1, val, NULL)


/*================================================================================================*/
typedef struct {
    t_flash_iface flash;
    BOOLEAN cntr_lost;
    BYTE cntr_val;
    INT cur_freq_cfg;

    BOOL afc_cor_enabled;
    BOOL last_adc_freq_valid;
    double last_adc_freq;
    BOOL afc_last_valid[LTR25_CHANNEL_CNT];
    double afc_fir_k[LTR25_CHANNEL_CNT];    /* рассчитанный коэффициент для коррекции АЧХ */
    double afc_fir_last[LTR25_CHANNEL_CNT]; /* действительность последних значений */
} t_internal_params;

/* Информация о модуле (16 частот, коэффициенты АЧХ и измеренные значения источников тока). */
typedef struct {
    DWORD sign;
    DWORD size;
    DWORD format;
    DWORD flags;
    CHAR name[LTR25_NAME_SIZE];
    CHAR serial[LTR25_SERIAL_SIZE];
    TLTR25_CBR_COEF cbr[LTR25_CHANNEL_CNT][LTR25_FREQ_CNT];
} t_ltr25_flash_info;

#pragma pack(4)
struct LTR25Config {
    struct {
        CHAR name[LTR25_NAME_SIZE];
        CHAR serial[LTR25_SERIAL_SIZE];
        WORD ver_fpga;
        BYTE ver_pld;
        BYTE rev_board;
        BOOL is_industrial;
        struct Calibration {
            float offset;
            float gain;
        } calibr[LTR25_CHANNEL_CNT][LTR25_CBR_FREQ_CNT];
        struct AFC {
            double freq;
            double fir_coeffs[LTR25_CHANNEL_CNT];
        } afc;
    } info;
    struct {
        struct {
            BOOL is_enabled;
        } channels[LTR25_CHANNEL_CNT];
        BYTE sample_rate;
        BYTE resolution;
        BYTE current_source_out;
    } cfg;
    struct {
        BYTE fpga;
        BOOL is_running;
        BOOL ison_low_power;
    } state;
};
#pragma pack()


/*================================================================================================*/
static void conv_hltr25_to_ltr25cfg(const void *hcard, void *cfg);
static void conv_ltr25cfg_to_hltr25(const void *cfg, void *hcard);
static INT f_adau1978_reg_op(TLTR25 *hnd, BYTE adc_msk, BYTE reg, BYTE wr, BYTE wr_data,
    BYTE *rd_val);
static void f_calc_afc_k(TLTR25 *hnd);
static INT f_check_adc_config(TLTR25 *hnd, BYTE adc_msk);
static INT f_configure_adc(TLTR25 *hnd, BYTE adc_msk);
static INT f_fpga_status_cmd(TLTR25 *hnd, WORD data);
static INT f_load_coef(TLTR25 *hnd);
static INT f_set_flash_protection(TLTR25 *hnd, BYTE protect);

/*================================================================================================*/
/* Текстовые описания кодов ошибок. */
static const TLTR_ERROR_STRING_DEF f_err_tbl[] = {
    {LTR25_ERR_FPGA_FIRM_TEMP_RANGE,
        "Загружена прошивка ПЛИС для неверного температурного диапазона"},
    {LTR25_ERR_I2C_ACK_STATUS, "Ошибка обмена при обращении к регистрам АЦП по интерфейсу I2C"},
    {LTR25_ERR_I2C_INVALID_RESP,
        "Неверный ответ на команду при обращении к регистрам АЦП по интерфейсу I2C"},
    {LTR25_ERR_INVALID_FREQ_CODE, "Неверно задан код частоты АЦП"},
    {LTR25_ERR_INVALID_DATA_FORMAT, "Неверно задан формат данных АЦП"},
    {LTR25_ERR_INVALID_I_SRC_VALUE, "Неверно задано значение источника тока"},
    {LTR25_ERR_CFG_UNSUP_CH_CNT,
        "Для заданной частоты и формата не поддерживается заданное количество каналов АЦП"},
    {LTR25_ERR_NO_ENABLED_CH, "Не был разрешен ни один канал АЦП"},
    {LTR25_ERR_ADC_PLL_NOT_LOCKED, "Ошибка захвата PLL АЦП"},
    {LTR25_ERR_ADC_REG_CHECK, "Ошибка проверки значения записанных регистров АЦП"},
    {LTR25_ERR_LOW_POW_MODE_NOT_CHANGED,
        "Не удалось перевести АЦП из/в низкопотребляющее состояние"},
    {LTR25_ERR_LOW_POW_MODE, "Модуль находится в низкопотребляющем режиме"}
};

static const struct {
    double AdcFreq;
    BYTE FreqCode;
    BYTE AdcFs;                             /* Значение поля FS регистра АЦП */
    BYTE MaxCh20;
    BYTE MAXCh24;
    BYTE CoefIdx;
} f_freq_params[] = {
    {78.125e3,       1,  ADAU1987_SAI_CTRL0_FS_64_96, 6, 3, 0}, /**< 78.125 кГц */
    {39.0625e3,      3,  ADAU1987_SAI_CTRL0_FS_32_48, 8, 6, 1}, /**< 39.0625 кГц */
    {19.53125e3,     5,  ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}, /**< 19.53125 кГц */
    {9.765625e3,     7,  ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}, /**< 9.765625 кГц */
    {4.8828125e3,    9,  ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}, /**< 4.8828125 кГц */
    {2.44140625e3,   11, ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}, /**< 2.44140625 кГц */
    {1.220703125e3,  13, ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}, /**< 1.220703125 кГц */
    {0.6103515625e3, 15, ADAU1987_SAI_CTRL0_FS_32_48, 8, 8, 1}  /**< 610.3515625 Гц */
};

static const struct {
    BYTE addr;
    BYTE val;
} f_adc_regs[] = {
    {ADAU1978_REG_MPOWER, ADAU1978_REGBIT_MPOWER_PWUP},
    {ADAU1978_REG_BPOWER_SAI, ADAU1978_REGBIT_BPOWER_SAI_ADC_EN1 |
        ADAU1978_REGBIT_BPOWER_SAI_ADC_EN2 | ADAU1978_REGBIT_BPOWER_SAI_ADC_EN3 |
        ADAU1978_REGBIT_BPOWER_SAI_ADC_EN4 | ADAU1978_REGBIT_BPOWER_SAI_VREF_EN |
        ADAU1978_REGBIT_BPOWER_SAI_LDO_EN},
    {ADAU1978_REG_SAI_CTRL0, ADAU1987_SAI_CTRL0_SAI_TDM4 | ADAU1987_SAI_CTRL0_SDATA_FMT_LJ},
    {ADAU1978_REG_SAI_CTRL1, 0},
    {ADAU1978_REG_SAI_CMAP12, 0x10},
    {ADAU1978_REG_SAI_CMAP34, 0x32},
    {ADAU1978_REG_SAI_OVERTEMP, ADAU1978_REGBIT_SAI_OVERTEMP_DRV_C1 |
        ADAU1978_REGBIT_SAI_OVERTEMP_DRV_C2 | ADAU1978_REGBIT_SAI_OVERTEMP_DRV_C3 |
        ADAU1978_REGBIT_SAI_OVERTEMP_DRV_C4},
    {ADAU1978_REG_POSTADC_GAIN1, 0xA0},
    {ADAU1978_REG_POSTADC_GAIN2, 0xA0},
    {ADAU1978_REG_POSTADC_GAIN3, 0xA0},
    {ADAU1978_REG_POSTADC_GAIN4, 0xA0},
    {ADAU1978_REG_MISC, 0x02},
    {ADAU1978_REG_DC_HPF_CAL, 0x00},
    {ADAU1978_REG_PLLCTL, ADAU1978_REGBIT_PLLCTL_PLLMUTE | ADAU1978_REGBIT_PLLCTL_CLKS |
        ADAU1978_PLLCTL_MCS_256}
};


/*================================================================================================*/
/*------------------------------------------------------------------------------------------------*/
static void conv_hltr25_to_ltr25cfg(const void *hcard, void *cfg) {
    size_t ich;
    const TLTR25 *hltr25 = hcard;
    struct LTR25Config *pcfg = cfg;

    strncpy(pcfg->info.name, hltr25->ModuleInfo.Name, ARRAY_LEN(pcfg->info.name));
    strncpy(pcfg->info.serial, hltr25->ModuleInfo.Serial, ARRAY_LEN(pcfg->info.serial));
    pcfg->info.ver_fpga = hltr25->ModuleInfo.VerFPGA;
    pcfg->info.ver_pld = hltr25->ModuleInfo.VerPLD;
    pcfg->info.rev_board = hltr25->ModuleInfo.BoardRev;
    pcfg->info.is_industrial = hltr25->ModuleInfo.Industrial;
    pcfg->info.afc.freq = hltr25->ModuleInfo.AfcCoef.AfcFreq;

    pcfg->cfg.sample_rate = hltr25->Cfg.FreqCode;
    pcfg->cfg.resolution = hltr25->Cfg.DataFmt;
    pcfg->cfg.current_source_out = hltr25->Cfg.ISrcValue;

    pcfg->state.fpga = hltr25->State.FpgaState;
    pcfg->state.is_running = hltr25->State.Run;
    pcfg->state.ison_low_power = hltr25->State.LowPowMode;

    for (ich = 0; (ich < LTR25_CHANNEL_CNT); ich++) {
        size_t ifq;

        pcfg->cfg.channels[ich].is_enabled = hltr25->Cfg.Ch[ich].Enabled;
        pcfg->info.afc.fir_coeffs[ich] = hltr25->ModuleInfo.AfcCoef.FirCoef[ich];
        for (ifq = 0; (ifq < LTR25_CBR_FREQ_CNT); ifq++) {
            pcfg->info.calibr[ich][ifq].offset = hltr25->ModuleInfo.CbrCoef[ich][ifq].Offset;
            pcfg->info.calibr[ich][ifq].gain = hltr25->ModuleInfo.CbrCoef[ich][ifq].Scale;
        }
    }
}

/*------------------------------------------------------------------------------------------------*/
static void conv_ltr25cfg_to_hltr25(const void *cfg, void *hcard) {
    size_t ich;
    TLTR25 *hltr25 = hcard;
    const struct LTR25Config *pcfg = cfg;

    strncpy(hltr25->ModuleInfo.Name, pcfg->info.name, ARRAY_LEN(hltr25->ModuleInfo.Name));
    strncpy(hltr25->ModuleInfo.Serial, pcfg->info.serial, ARRAY_LEN(hltr25->ModuleInfo.Serial));
    hltr25->ModuleInfo.VerFPGA = pcfg->info.ver_fpga;
    hltr25->ModuleInfo.VerPLD = pcfg->info.ver_pld;
    hltr25->ModuleInfo.BoardRev = pcfg->info.rev_board;
    hltr25->ModuleInfo.Industrial = pcfg->info.is_industrial;
    hltr25->ModuleInfo.AfcCoef.AfcFreq = pcfg->info.afc.freq;

    hltr25->Cfg.FreqCode = pcfg->cfg.sample_rate;
    hltr25->Cfg.DataFmt = pcfg->cfg.resolution;
    hltr25->Cfg.ISrcValue = pcfg->cfg.current_source_out;

    hltr25->State.AdcFreq = f_freq_params[hltr25->Cfg.FreqCode].AdcFreq;
    hltr25->State.EnabledChCnt = 0;
    hltr25->State.FpgaState = pcfg->state.fpga;
    hltr25->State.Run = pcfg->state.is_running;
    hltr25->State.LowPowMode = pcfg->state.ison_low_power;

    for (ich = 0; (ich < LTR25_CHANNEL_CNT); ich++) {
        size_t ifq;

        hltr25->ModuleInfo.AfcCoef.FirCoef[ich] = pcfg->info.afc.fir_coeffs[ich];
        hltr25->Cfg.Ch[ich].Enabled = pcfg->cfg.channels[ich].is_enabled;
        if (hltr25->Cfg.Ch[ich].Enabled)
            hltr25->State.EnabledChCnt++;

        for (ifq = 0; (ifq < LTR25_CBR_FREQ_CNT); ifq++) {
            hltr25->ModuleInfo.CbrCoef[ich][ifq].Offset = pcfg->info.calibr[ich][ifq].offset;
            hltr25->ModuleInfo.CbrCoef[ich][ifq].Scale = pcfg->info.calibr[ich][ifq].gain;
        }
    }

    memset(hltr25->Internal, 0, sizeof(t_internal_params));
    f_calc_afc_k(hltr25);
}

/*------------------------------------------------------------------------------------------------*/
static INT f_adau1978_reg_op(TLTR25 *hnd, BYTE adc_msk, BYTE reg, BYTE wr, BYTE wr_data,
    BYTE *rd_val) {
    DWORD ack = 0;
    DWORD cmd = LTR25_MAKE_ADC_CMD(adc_msk, reg, wr, wr_data);
    INT err = LTR_OK;

    err = ltr_module_send_with_echo_resps(&hnd->Channel, &cmd, 1, &ack);
    if (err == LTR_OK) {
        ack = LTR_MODULE_CMD_DATA(ack);
        cmd = LTR_MODULE_CMD_DATA(cmd);
        if ((cmd & 0xFF00) != (ack & 0xFF00)) {
            err = LTR25_ERR_I2C_INVALID_RESP;
        } else if (rd_val != NULL) {
            *rd_val = ack & 0xFF;
        }
    } else if ((err == LTR_ERROR_INVALID_CMD_RESPONSE) &&
               (LTR_MODULE_GET_CMD_CODE(ack) == CMD_RESP_ADC_ERR)) {
        err = LTR25_ERR_I2C_ACK_STATUS;
    }

    return err;
}

/*------------------------------------------------------------------------------------------------*/
static void f_calc_afc_k(TLTR25 *hnd) {
    /* расчет коэффициентов фильтров */
    unsigned ch;
    t_internal_params *par = (t_internal_params *)hnd->Internal;
    double set_freq = hnd->State.AdcFreq;

    if (set_freq > 9000) {
        par->afc_cor_enabled = TRUE;
        /* если уже были рассчитаны коэффициенты для этой частоты, то ничего делать
         * не нужно, иначе - выполняем перерасчет
         */
        if (!par->last_adc_freq_valid || (fabs(par->last_adc_freq-set_freq) > 0.1)) {
            double fi, a, k1, h;
            fi = set_freq / 4;

            for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
                a = 1.0 / hnd->ModuleInfo.AfcCoef.FirCoef[ch];

                k1 = -0.5 + sqrt(0.25 - (1 -a * a)/
                    (2 - 2 * cos(2*M_PI*hnd->ModuleInfo.AfcCoef.AfcFreq/f_freq_params[0].AdcFreq)));
                h = sqrt((1+k1)*(1+k1) - 2*(1+k1)*k1*cos(2*M_PI*fi/f_freq_params[0].AdcFreq) +
                    k1*k1);

                par->afc_fir_k[ch] = -0.5 + sqrt(0.25 - 0.5*(1 - h*h));
            }

            par->last_adc_freq_valid = 1;
            par->last_adc_freq = set_freq;
        }
    } else {
         par->afc_cor_enabled = FALSE;
    }
}

/*------------------------------------------------------------------------------------------------*/
static INT f_check_adc_config(TLTR25 *hnd, BYTE adc_msk) {
    unsigned r;
    INT err = LTR_OK;

    for (r = 0; ((r < sizeof(f_adc_regs)/sizeof(f_adc_regs[0])) && (err == LTR_OK)); r++) {
        BYTE set_val = f_adc_regs[r].val;
        const BYTE addr = f_adc_regs[r].addr;
        BYTE rd_val;
        if (addr == ADAU1978_REG_SAI_CTRL0)
            set_val |=  f_freq_params[hnd->Cfg.FreqCode].AdcFs;

        err = f_adau1978_reg_rd(hnd, adc_msk, addr, &rd_val);
        if (err == LTR_OK) {
            if (addr == ADAU1978_REG_PLLCTL) {
                set_val |= ADAU1978_REGBIT_PLLCTL_PLLLOCK;
                if (!(rd_val & ADAU1978_REGBIT_PLLCTL_PLLLOCK))
                    err = LTR25_ERR_ADC_PLL_NOT_LOCKED;
            }
            if ((err == LTR_OK) && (set_val != rd_val))
                err = LTR25_ERR_ADC_REG_CHECK;
        }
    }


    return err;
}

/*------------------------------------------------------------------------------------------------*/
static INT f_configure_adc(TLTR25 *hnd, BYTE adc_msk) {
    unsigned r;
    INT err = LTR_OK;

    for (r = 0; ((r < sizeof(f_adc_regs)/sizeof(f_adc_regs[0])) && (err == LTR_OK)); r++) {
        BYTE val = f_adc_regs[r].val;
        const BYTE addr = f_adc_regs[r].addr;
        if (addr == ADAU1978_REG_SAI_CTRL0)
            val |=  f_freq_params[hnd->Cfg.FreqCode].AdcFs;

        err = f_adau1978_reg_wr(hnd, adc_msk, addr, val);
    }


    /* ожидание захвата PLL */
    if (err == LTR_OK) {
        t_ltimer lock_tmr;
        INT lock = 0;

        ltimer_set(&lock_tmr, LTIMER_MS_TO_CLOCK_TICKS(ADC_PLL_LOCK_TOUT));

        while ((err == LTR_OK) && (lock != PLL_LOCK_CHECK_CNT)) {
            BYTE wrd = 0;
            err = f_adau1978_reg_rd(hnd, adc_msk, ADAU1978_REG_PLLCTL, &wrd);
            if (wrd & ADAU1978_REGBIT_PLLCTL_PLLLOCK) {
                lock++;
            } else {
                lock = 0;
                if (ltimer_expired(&lock_tmr))
                    err = LTR25_ERR_ADC_PLL_NOT_LOCKED;
            }
        }
    }


    return err;
}

/*------------------------------------------------------------------------------------------------*/
static INT f_fpga_status_cmd(TLTR25 *hnd, WORD data) {
    DWORD cmd, ack;
    INT err;
    /* Получаем версию ПЛИС, если он был успешно загружен */
    cmd = LTR_MODULE_MAKE_CMD(CMD_STATUS_FPGA, data);
    err = ltr_module_send_with_echo_resps(&hnd->Channel, &cmd, 1, &ack);

    if (err == LTR_OK) {
        unsigned char industrial_firm;
        ack = LTR_MODULE_CMD_DATA(ack);

        hnd->ModuleInfo.VerFPGA = ack & 0x03FF;
        hnd->ModuleInfo.BoardRev = (ack >> 11) & 0x1F;
        industrial_firm  = (ack >> 10) & 1;
        hnd->State.LowPowMode = (ack >> 9) & 1;


        if (industrial_firm != hnd->ModuleInfo.Industrial)
            err = LTR25_ERR_FPGA_FIRM_TEMP_RANGE;
    }

    return err;
}

/*------------------------------------------------------------------------------------------------*/
static INT f_load_coef(TLTR25 *hnd) {
    unsigned ch;
    DWORD cmd[1+2*2*LTR25_CHANNEL_CNT];
    unsigned freq_idx = f_freq_params[hnd->Cfg.FreqCode].CoefIdx;
    unsigned put_pos = 0;
    INT err = LTR_OK;


    cmd[put_pos++] = LTR_MODULE_MAKE_CMD(CMD_K_ADDR, CMD_BIT_K_ADDR_WR | LTR25_COEF_ADDR_CBR);
    for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
        const double dval = hnd->ModuleInfo.CbrCoef[ch][freq_idx].Scale * 0x40000000;
        const INT val = (int)((dval >= 0) ? dval + 0.5 : dval - 0.5);

        cmd[put_pos++] = LTR_MODULE_MAKE_CMD(CMD_K_DATA1, (val >> 16) & 0xFFFF);
        cmd[put_pos++] = LTR_MODULE_MAKE_CMD(CMD_K_DATA2, val & 0xFFFF);
    }

    for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
        const double dval = hnd->ModuleInfo.CbrCoef[ch][freq_idx].Offset;
        const INT val = (int)((dval >= 0) ? dval + 0.5 : dval - 0.5);

        cmd[put_pos++] = LTR_MODULE_MAKE_CMD(CMD_K_DATA1, (val >> 16) & 0xFFFF);
        cmd[put_pos++] = LTR_MODULE_MAKE_CMD(CMD_K_DATA2, val & 0xFFFF);
    }

    err = ltr_module_send_cmd(&hnd->Channel, cmd, put_pos);


    return err;
}

/*------------------------------------------------------------------------------------------------*/
static INT f_set_flash_protection(TLTR25 *hnd, BYTE protect) {
    BYTE status;
    t_internal_params *params = (t_internal_params *)hnd->Internal;
    INT err = LTR_OK;


    err = flash_sst25_set_status(&params->flash, protect);

    if (err == LTR_OK)
        err = flash_sst25_get_status(&params->flash, &status);

    if ((err == LTR_OK) && ((status & FLASH_PROT_MASK) != (protect & FLASH_PROT_MASK)))
        err = LTR_ERROR_FLASH_SET_PROTECTION;

    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_AdcRegRead(TLTR25 *hnd, BYTE adc_msk, BYTE reg, BYTE *val) {
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK)
        err = f_adau1978_reg_rd(hnd, adc_msk, reg, val);
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_AdcRegWrite(TLTR25 *hnd, BYTE adc_msk, BYTE reg, BYTE val) {
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK)
        err = f_adau1978_reg_wr(hnd, adc_msk, reg,  val);
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Close(TLTR25 *hnd) {
    INT err = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;
    if (err == LTR_OK) {
        if (hnd->Internal != NULL) {
            t_internal_params *params = (t_internal_params *)hnd->Internal;
            flash_iface_ltr_close(&params->flash);
            free(hnd->Internal);
            hnd->Internal = NULL;
        }
        err = LTR_Close(&hnd->Channel);
    }
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FlashErase(TLTR25 *hnd, DWORD addr, DWORD size) {
    INT res = LTR25_IsOpened(hnd);
    if (res == LTR_OK) {
        t_internal_params *pars = (t_internal_params *)hnd->Internal;
        t_flash_errs flash_res = flash_iface_ltr_set_channel(&pars->flash, &hnd->Channel);
        if (!flash_res)
            flash_res = flash_sst25_erase(&pars->flash, addr, size);
        if (flash_res)
            res = flash_iface_ltr_conv_err(flash_res);
    }
    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FlashRead(TLTR25 *hnd, DWORD addr, BYTE *data, DWORD size) {
    INT res = LTR25_IsOpened(hnd);
    if (res == LTR_OK) {
        t_internal_params *pars = (t_internal_params*)hnd->Internal;
        t_flash_errs flash_res = flash_iface_ltr_set_channel(&pars->flash, &hnd->Channel);
        if (!flash_res)
            flash_res = flash_sst25_read(&pars->flash, addr, data, size);
        if (flash_res)
            res = flash_iface_ltr_conv_err(flash_res);
    }
    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FlashWrite(TLTR25 *hnd, DWORD addr, const BYTE *data, DWORD size) {
    INT res = LTR25_IsOpened(hnd);
    if (res == LTR_OK) {
        t_internal_params *pars = (t_internal_params *)hnd->Internal;
        t_flash_errs flash_res = flash_iface_ltr_set_channel(&pars->flash, &hnd->Channel);
        if (!flash_res)
            flash_res = flash_sst25_write(&pars->flash, addr, data, size, 0);
        if (flash_res)
            res = flash_iface_ltr_conv_err(flash_res);
    }
    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FPGAEnable(TLTR25 *hnd, BOOL enable) {
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK)
        err = ltrmodule_fpga_enable(&hnd->Channel, enable, &hnd->State.FpgaState);
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FPGAFirmwareWriteEnable(TLTR25 *hnd, BOOL en) {
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK) {
        DWORD cmd = CMD_ROM_WR_EN;
        err = ltr_module_send_cmd(&hnd->Channel, &cmd, 1);
        if (err == LTR_OK) {
            err = flash_iface_ltr_conv_err(f_set_flash_protection(hnd,
                en ? FLASH_PROT_FPGA_UPDATE : FLASH_PROT_DEFAULT));
            /* пишем произвольную команду, чтобы запретить изменение статусного регистра */
            if (!en) {
                err = ltrmodule_fpga_enable(&hnd->Channel,
                    ltrmodule_fpga_is_enabled(hnd->State.FpgaState), &hnd->State.FpgaState);
            }
        }
    }
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_FPGAIsEnabled(TLTR25 *hnd, BOOL *enabled) {
    INT err = LTR25_IsOpened(hnd);
    if ((err == LTR_OK) && (enabled == NULL))
        err = LTR_ERROR_PARAMETERS;
    if (err == LTR_OK)
        *enabled = ltrmodule_fpga_is_enabled(hnd->State.FpgaState);
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_GetConfig(TLTR25 *hnd) {
    INT err = LTR25_IsOpened(hnd);

    if (err == LTR_OK) {
        struct {
            DWORD sign;
            DWORD size;
            DWORD format;
        } hdr;
        t_internal_params *params = (t_internal_params *)hnd->Internal;
        err = flash_iface_ltr_conv_err(flash_iface_ltr_set_channel(&params->flash, &hnd->Channel));

        if (err == LTR_OK) {
            /* вначале читаем только минимальный заголовок,
             * чтобы определить сразу признак информации и узнать полный размер
             */
            err = flash_iface_ltr_conv_err(flash_read(&params->flash, LTR25_FLASH_ADDR_MODULE_INFO,
                (unsigned char *)&hdr, sizeof(hdr)));
        }

        if ((err == LTR_OK) && (hdr.sign != LTR25_FLASH_INFO_SIGN))
            err = LTR_ERROR_FLASH_INFO_NOT_PRESENT;
        if ((err == LTR_OK) && (hdr.format != LTR25_FLASH_INFO_FORMAT))
            err = LTR_ERROR_FLASH_INFO_UNSUP_FORMAT;
        if ((err == LTR_OK) && (hdr.size < LTR25_FLASH_INFO_MIN_SIZE))
            err = LTR_ERROR_FLASH_INFO_UNSUP_FORMAT;
        if (err == LTR_OK) {
            t_ltr25_flash_info *pinfo = malloc(hdr.size+LTR25_FLASH_INFO_CRC_SIZE);
            if (pinfo == NULL)
                err = LTR_ERROR_MEMORY_ALLOC;
            /* читаем оставшуюся информацию */
            if (err == LTR_OK) {
                memcpy(pinfo, &hdr, sizeof(hdr));
                err = flash_iface_ltr_conv_err(flash_read(&params->flash,
                    LTR25_FLASH_ADDR_MODULE_INFO+sizeof(hdr), ((BYTE *)pinfo) + sizeof(hdr),
                    hdr.size+LTR25_FLASH_INFO_CRC_SIZE-sizeof(hdr)));
            }

            /* рассчитываем CRC и сверяем со считанной из памяти */
            if (err == LTR_OK) {
                WORD crc, crc_ver;
                crc_ver = eval_crc16(0, (unsigned char *)pinfo, hdr.size);
                crc = ((BYTE *)pinfo)[hdr.size] | (((WORD)((BYTE *)pinfo)[hdr.size+1]) << 8);
                if (crc != crc_ver)
                    err = LTR_ERROR_FLASH_INFO_CRC;
            }

            if (err == LTR_OK) {
                unsigned ch, freq;
                memcpy(hnd->ModuleInfo.Name, pinfo->name, LTR25_NAME_SIZE);
                memcpy(hnd->ModuleInfo.Serial, pinfo->serial, LTR25_SERIAL_SIZE);

                for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
                    for (freq = 0; (freq < LTR25_CBR_FREQ_CNT); freq++) {
                        hnd->ModuleInfo.CbrCoef[ch][freq] = pinfo->cbr[ch][freq];
                    }
                }
            }

            free(pinfo);
        }
    } /*if (err == LTR_OK)*/
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(LPCSTR) LTR25_GetErrorString(INT err) {
    size_t i;
    for (i = 0; (i < sizeof(f_err_tbl) / sizeof(f_err_tbl[0])); i++) {
        if (f_err_tbl[i].code == err)
            return f_err_tbl[i].message;
    }
    return LTR_GetErrorString(err);
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Init(TLTR25 *hnd) {
    INT res = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;
    if (res == LTR_OK) {
        memset(hnd, 0, sizeof(*hnd));
        hnd->Size = sizeof(*hnd);
        hnd->Internal = NULL;
        res = LTR_Init(&hnd->Channel);
    }
    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_IsOpened(TLTR25 *hnd) {
     return (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_IsOpened(&hnd->Channel);
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Open(TLTR25 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *crate_sn,
    INT slot) {
    return LTR25_OpenEx(hnd, ltrd_addr, ltrd_port, crate_sn, slot, 0, NULL);
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_OpenEx(TLTR25 *hnd, DWORD ltrd_addr, WORD ltrd_port,
    const CHAR *crate_sn, INT slot, DWORD in_flags, DWORD *out_flags) {
    DWORD cmd;
    int fatal_err;
    INT warning;
    DWORD open_flags = 0;
    DWORD out_flg = 0;
    INT err = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;

    if (err == LTR_OK) {
        if (LTR25_IsOpened(hnd) == LTR_OK)
            LTR25_Close(hnd);
    }

    if (err == LTR_OK) {
        free(hnd->Internal);
        if ((hnd->Internal = calloc(1, sizeof(t_internal_params))) == NULL)
            err = LTR_ERROR_MEMORY_ALLOC;
    }

    if ((err == LTR_OK) && (in_flags & LTR_OPENINFLG_REOPEN)) {
        err = ltrslot_restore_config(hnd, ltrd_addr, ltrd_port, crate_sn, slot,
            sizeof(struct LTR25Config), conv_ltr25cfg_to_hltr25, &out_flg);
        if ((err == LTR_OK) && (out_flg & LTR_OPENOUTFLG_REOPEN))
                open_flags |= LTR_MOPEN_INFLAGS_DONT_RESET;
    }

    if (err == LTR_OK) {
        err = ltr_module_open(&hnd->Channel, ltrd_addr, ltrd_port, crate_sn, slot, LTR_MID_LTR25,
            &open_flags, &cmd, &warning);
        if (err == LTR_OK) {
            hnd->ModuleInfo.VerPLD = (cmd >> 1) & 0x1F;
            hnd->ModuleInfo.Industrial = (cmd & 1) ? TRUE : FALSE;
        }
    }

    if ((err == LTR_OK) && !(in_flags & LTR_OPENINFLG_REOPEN))
        err = ltrslot_stop(&hnd->Channel);

    if ((err == LTR_OK) && !(open_flags & LTR_MOPEN_OUTFLAGS_DONT_INIT)) {
        INT fpga_status_res;
        INT flash_res = 0;
        unsigned ch, freq;

        /* установка параметров, читаемых из flash-памяти в значения по умолчанию */
        for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
            for (freq = 0; (freq < LTR25_CBR_FREQ_CNT); freq++) {
                hnd->ModuleInfo.CbrCoef[ch][freq].Scale = 1.0;
                hnd->ModuleInfo.CbrCoef[ch][freq].Offset = 0.0;
            }
        }
        strcpy(hnd->ModuleInfo.Name, "LTR25");
        strcpy(hnd->ModuleInfo.Serial, "");

        memset(&hnd->State, 0, sizeof(hnd->State));

        fpga_status_res = ltrmodule_fpga_check_load(&hnd->Channel, &hnd->State.FpgaState);
        /* За исключением случая, когда не смогли дождаться готовности,
         * можем работать с flash-памятью
         */
        if (hnd->State.FpgaState != LTR_FPGA_STATE_LOAD_PROGRESS) {
            t_internal_params *params = (t_internal_params *)hnd->Internal;
            params->cur_freq_cfg = -1;

            flash_res = flash_iface_ltr_init(&params->flash, &hnd->Channel);
            if (flash_res == 0)
                flash_res = flash_sst25_set(&params->flash);

            if (flash_res == 0) {
                BYTE status;
                /* если разрешено изменение защиты страниц, то сразу
                 * устанавливаем защиту по-умолчанию и запрещаем изменение */
                flash_res = flash_sst25_get_status(&params->flash, &status);

                if ((flash_res == 0) && !(status & SST25_STATUS_BPL))
                    flash_res = f_set_flash_protection(hnd, FLASH_PROT_DEFAULT);
            }

            if (flash_res) flash_res = flash_iface_ltr_conv_err(flash_res);
            else           flash_res = LTR25_GetConfig(hnd);
        }

        if ((err == LTR_OK) && (fpga_status_res == LTR_OK)) {
            err = LTR25_FPGAEnable(hnd, TRUE);
            if (err != LTR_OK)
                err = LTR_ERROR_FPGA_ENABLE;
            /* чтение версии и проверка прошивки */
            if (err == LTR_OK)
                err = f_fpga_status_cmd(hnd, 0);
        }

        if (err == LTR_OK) {
            hnd->ModuleInfo.AfcCoef.AfcFreq = 19200;
            for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
                hnd->ModuleInfo.AfcCoef.FirCoef[ch] = 0.99515;
            }
        }


        if (err == LTR_OK)
            err = flash_res;
        if (err == LTR_OK)
            err = fpga_status_res;
    } /*if ((err == LTR_OK) && !(open_flags & LTR_MOPEN_OUTFLAGS_DONT_INIT))*/

    if (out_flags != NULL)
        *out_flags = out_flg;

    fatal_err = ((err != LTR_OK) && (err != LTR_ERROR_FPGA_LOAD_DONE_TOUT) &&
        (err != LTR_ERROR_FPGA_ENABLE) && (err != LTR_ERROR_FLASH_INFO_NOT_PRESENT) &&
        (err != LTR_ERROR_FLASH_INFO_UNSUP_FORMAT) && (err != LTR25_ERR_FPGA_FIRM_TEMP_RANGE));
    if (fatal_err)
        LTR25_Close(hnd);

    return (err == LTR_OK) ? warning : err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_ProcessData(TLTR25 *hnd, const DWORD *src, double *dest, INT *size,
    DWORD flags, DWORD *ch_status) {
    INT res = LTR25_IsOpened(hnd);
    if ((res == LTR_OK) && ((src == NULL) || (size == NULL) || (*size == 0)))
        res = LTR_ERROR_PARAMETERS;
    if (res == LTR_OK) {
        BYTE chan_list[LTR25_CHANNEL_CNT];  /* Список номеров каналов в кадре. */
        t_internal_params *params = (t_internal_params *)hnd->Internal;
        BYTE i, ch_num;
        BYTE frame_pos = 0;
        INT rem_size = *size;
        INT put_size = 0;
        INT wrd_err = LTR_OK;
        BYTE cntr = params->cntr_val;
        BOOLEAN cntr_lost = (flags & LTR25_PROC_FLAG_NONCONT_DATA) ? TRUE : params->cntr_lost;
        BYTE exp_wrd_num = 0;
        WORD prev_wrd_data = 0;

        if (ch_status != NULL)
            memset(ch_status, 0, hnd->State.EnabledChCnt*sizeof(ch_status[0]));


        for (i = 0, ch_num = 0; (i < LTR25_CHANNEL_CNT); i++) {
            if (hnd->Cfg.Ch[i].Enabled)
                chan_list[ch_num++] = i;
        }

        if (flags & LTR25_PROC_FLAG_NONCONT_DATA) {
            for (i = 0; (i < LTR25_CHANNEL_CNT); i++) {
                params->afc_last_valid[i] = FALSE;
            }
        }


        for (; (rem_size != 0); rem_size--, src++) {
            BOOLEAN drop_wrd = FALSE;
            INT wrd = 0;
            BYTE rcv_ch = (*src >> 4) & 7;
            DWORD cur_status = LTR25_CH_STATUS_OK;

            /* при несовпадении каналов выбрасываем весь кадр */
            if (rcv_ch != chan_list[frame_pos])
                wrd_err = (put_size == 0) ? LTR_ERROR_PROCDATA_UNALIGNED : LTR_ERROR_PROCDATA_CHNUM;

            if (hnd->Cfg.DataFmt == LTR25_FORMAT_20) {
                BYTE rcv_cntr = (*src >> 7) & 1;
                BYTE exp_cntr = (cntr == (FORMAT20_CNTR_MOD - 1));

                if (!cntr_lost && (exp_cntr != rcv_cntr)) {
                    res = LTR_ERROR_PROCDATA_CNTR;
                    cntr_lost = TRUE;
                }

                if (cntr_lost && (rcv_cntr != 0)) {
                    cntr_lost = FALSE;
                    cntr = 0;
                } else {
                    if (++cntr == FORMAT20_CNTR_MOD)
                        cntr = 0;
                }

                if (wrd_err == LTR_OK) {
                    wrd = ((*src & 0x0F) << 16) | ((*src >> 16) & 0xFFFF);
                    if (wrd == 0x07FFFF) {
                        cur_status = LTR25_CH_STATUS_OPEN;
                    } else if (wrd == 0x080000) {
                        cur_status = LTR25_CH_STATUS_SHORT;
                    } else {
                        wrd <<= 12;
                    }
                }
            } else if (hnd->Cfg.DataFmt == LTR25_FORMAT_32) {
                BYTE rcv_cntr = *src & 0x0F;
                BYTE rcv_wrd_num = (*src >> 7) & 1;

                if (cntr_lost) {
                    cntr = rcv_cntr;
                } else if (!cntr_lost && (cntr != rcv_cntr)) {
                    res = wrd_err = LTR_ERROR_PROCDATA_CNTR;
                    cntr = rcv_cntr;
                }

                if (++cntr == FORMAT32_CNTR_MOD)
                    cntr = 0;

                cntr_lost = FALSE;

                if (wrd_err == LTR_OK) {
                    /* проверка правильности следования слов (старший/младший) */
                    if (rcv_wrd_num != exp_wrd_num) {
                        wrd_err = (put_size == 0) ?
                            LTR_ERROR_PROCDATA_UNALIGNED : LTR_ERROR_PROCDATA_WORD_SEQ;
                    }
                }

                if (wrd_err == LTR_OK) {
                    if (!rcv_wrd_num) {
                        /* для первого слова сохраняем данные, чтобы объединить со вторым */
                        prev_wrd_data = (*src >> 16) & 0xFFFF;
                        drop_wrd = TRUE;
                    } else {
                        /* для второго - восстанавливаем весь 32-битный отсчет */
                        wrd = ((INT)prev_wrd_data << 16) | ((*src >> 16) & 0xFFFF);

                        if (wrd == 0x7FFFFFFF)           cur_status = LTR25_CH_STATUS_OPEN;
                        else if (wrd == (INT)0x80000000) cur_status = LTR25_CH_STATUS_SHORT;
                    }
                    exp_wrd_num ^= 1;
                }
            }

            /* при ошибке в слове выбрасываем весь соответствующий кадр и
             * будем искать начало следующего
             */
            if (wrd_err != LTR_OK) {
                put_size -= frame_pos;
                frame_pos = 0;
                exp_wrd_num = 0;
                if (res == LTR_OK)
                    res = wrd_err;

                for (i = 0; (i < LTR25_CHANNEL_CNT); i++) {
                    params->afc_last_valid[i] = FALSE;
                }
            } else if (!drop_wrd) {
                double val = 0;
                unsigned cur_ch = chan_list[frame_pos];

                if (cur_status == 0) {
                    double x_fir;

                    val = wrd;

                    if (params->afc_cor_enabled) {
                        x_fir = val;
                        if (params->afc_last_valid[cur_ch]) {
                            val = (val - params->afc_fir_last[cur_ch]) * params->afc_fir_k[cur_ch] +
                                x_fir;
                        }
                        params->afc_fir_last[cur_ch] = x_fir;
                        params->afc_last_valid[cur_ch] = TRUE;
                    }

                    if (flags & LTR25_PROC_FLAG_VOLT)
                        val = val * LTR25_ADC_RANGE_PEAK / LTR25_ADC_SCALE_CODE_MAX;
                } else {
                    params->afc_last_valid[cur_ch] = FALSE;
                }

                if (dest != NULL)
                    dest[put_size] = val;
                if ((cur_status != LTR25_CH_STATUS_OK) && (ch_status != NULL))
                    ch_status[frame_pos] = cur_status;

                put_size++;
                if (++frame_pos == ch_num)
                    frame_pos = 0;
            }
        } /*for (; (rem_size != 0); rem_size--, src++)*/


        if (!(flags & LTR25_PROC_FLAG_NONCONT_DATA)) {
            params->cntr_lost = cntr_lost;
            params->cntr_val = cntr;
        }

        *size = put_size;
    } /*if (res == LTR_OK)*/

    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Recv(TLTR25 *hnd, DWORD *data, DWORD *tmark, DWORD size,
    DWORD timeout) {
    /* Получаем данные АЦП */
    int res = LTR25_IsOpened(hnd);
    if (res == LTR_OK)
        res = LTR_Recv(&hnd->Channel, data, tmark, size, timeout);

    if ((res >= 0) && (hnd->Channel.flags & LTR_FLAG_RBUF_OVF))
        res = LTR_ERROR_RECV_OVERFLOW;

    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_SearchFirstFrame(TLTR25 *hnd, const DWORD *data, DWORD size,
    DWORD *frame_idx) {
    INT res = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR :
        ((data == NULL) || (size == 0) || (frame_idx == NULL)) ? LTR_ERROR_PARAMETERS : LTR_OK;

    if (res == LTR_OK) {
        INT i;
        INT idx_first = -1;
        INT first_ch = -1;

        for (i = 0; ((i < LTR25_CHANNEL_CNT) && (first_ch < 0)); i++) {
            if (hnd->Cfg.Ch[i].Enabled) {
                first_ch = i;
                break;
            }
        }

        if (first_ch < 0) {
            res = LTR25_ERR_NO_ENABLED_CH;
        } else {
            for (i = 0; ((i < (INT)size) && (idx_first < 0)); i++, data++) {
                BYTE rcv_ch  = (*data >> 4) & 7;
                BYTE rcv_wrd_num = (hnd->Cfg.DataFmt == LTR25_FORMAT_20) ? 0 : (*data >> 7) & 1;

                if ((rcv_ch == first_ch) && (rcv_wrd_num == 0))
                    idx_first = i;
            }

            if (idx_first >= 0) {
                *frame_idx = idx_first;
            } else {
                res = LTR_ERROR_FIRSTFRAME_NOTFOUND;
            }
        }
    }

    return res;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_SetADC(TLTR25 *hnd) {
    /* Настройка АЦП. Не влияет на фазирование, если только не меняется частота дискретизации. */
    DWORD ch_ctl = 0;
    BYTE ch_cnt = 0;
    INT err = LTR25_IsOpened(hnd);

    if ((err == LTR_OK) && hnd->State.LowPowMode)
        err = LTR25_ERR_LOW_POW_MODE;
    if ((err == LTR_OK) && hnd->State.Run)
        err = LTR_ERROR_MODULE_STARTED;
    if ((err == LTR_OK) && (hnd->Cfg.FreqCode >= LTR25_FREQ_CNT))
        err = LTR25_ERR_INVALID_FREQ_CODE;
    if (err == LTR_OK) {
        if ((hnd->Cfg.DataFmt != LTR25_FORMAT_20) && (hnd->Cfg.DataFmt != LTR25_FORMAT_32))
            err = LTR25_ERR_INVALID_DATA_FORMAT;
    }
    if (err == LTR_OK) {
        const int isok_isrc = ((hnd->Cfg.ISrcValue == LTR25_I_SRC_VALUE_2_86) ||
            (hnd->Cfg.ISrcValue == LTR25_I_SRC_VALUE_10));
        if (!isok_isrc)
            err = LTR25_ERR_INVALID_I_SRC_VALUE;
    }

    if (err == LTR_OK) {
        INT ch;
        for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
            if (hnd->Cfg.Ch[ch].Enabled) {
                ch_cnt++;
                ch_ctl |= (1 << ch);
            }
        }

        if (ch_cnt == 0) {
            err = LTR25_ERR_NO_ENABLED_CH;
        } else {
            const BYTE nch = (hnd->Cfg.DataFmt == LTR25_FORMAT_20) ?
                f_freq_params[hnd->Cfg.FreqCode].MaxCh20 : f_freq_params[hnd->Cfg.FreqCode].MAXCh24;
            if (ch_cnt > nch)
                err = LTR25_ERR_CFG_UNSUP_CH_CNT;
        }
    }

    if (err == LTR_OK) {
        DWORD cmds[3], acks[3];
        t_ltimer tmr;
        DWORD cmd_cnt = 0;
        t_internal_params *params = (t_internal_params *)hnd->Internal;
        BOOL new_freq = (params->cur_freq_cfg != hnd->Cfg.FreqCode);

        if (new_freq) {
            cmds[cmd_cnt++] = LTR25_MAKE_ADC_FREQ_CMD(hnd, 0);
            cmds[cmd_cnt++] = LTR25_MAKE_ADC_FREQ_CMD(hnd, 1);
        }
        cmds[cmd_cnt++] = LTR_MODULE_MAKE_CMD(CMD_CH_CONTROL,
            (hnd->Cfg.ISrcValue << 12) | (hnd->Cfg.DataFmt << 8) | ch_ctl);

        err = ltr_module_send_with_echo_resps(&hnd->Channel, cmds, cmd_cnt, acks);
        if (err == LTR_OK) {
            ltimer_set(&tmr, LTIMER_MS_TO_CLOCK_TICKS(ADC_WAIT_STABLE_TIME));

            if (new_freq) {
                if (err==LTR_OK)
                    err = f_configure_adc(hnd, 0x01);
                if (err==LTR_OK)
                    err = f_configure_adc(hnd, 0x02);
            }

            if (err == LTR_OK)
                err = f_load_coef(hnd);

            if (err == LTR_OK)
                err = f_check_adc_config(hnd, 0x01);
            if (err == LTR_OK)
                err = f_check_adc_config(hnd, 0x02);

            if ((err == LTR_OK) && new_freq)
                LTRAPI_SLEEP_MS(LTIMER_CLOCK_TICKS_TO_MS(ltimer_expiration(&tmr)));
        }


        if (err == LTR_OK) {
            params->cur_freq_cfg = hnd->Cfg.FreqCode;
            hnd->State.EnabledChCnt = ch_cnt;
            hnd->State.AdcFreq = f_freq_params[hnd->Cfg.FreqCode].AdcFreq;
        }
    }

    if (err == LTR_OK)
        f_calc_afc_k(hnd);

    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT)  LTR25_SetLowPowMode(TLTR25 *hnd, BOOL lowPowMode) {
    INT err = LTR25_IsOpened(hnd);
    if ((err == LTR_OK) && hnd->State.Run) {
        err = LTR_ERROR_MODULE_STARTED;
    }

    if (err == LTR_OK) {
        err = f_fpga_status_cmd(hnd, lowPowMode ? 0x01 : 0x02);
        if (err == LTR_OK) {
            t_internal_params * params = (t_internal_params *)hnd->Internal;
            params->cur_freq_cfg = -1;
        }
    }

    if (err == LTR_OK) {
        if (hnd->State.LowPowMode != lowPowMode)
            err = LTR25_ERR_LOW_POW_MODE_NOT_CHANGED;
    }
    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Start(TLTR25 *hnd) {
    /* Запуск АЦП */
    INT err = LTR25_IsOpened(hnd);
    if ((err == LTR_OK) && hnd->State.LowPowMode)
        err = LTR25_ERR_LOW_POW_MODE;
    if ((err == LTR_OK) && hnd->State.Run)
        err = LTR_ERROR_MODULE_STARTED;

    if (err == LTR_OK) {
        DWORD cmd, ack;
        cmd = LTR_MODULE_MAKE_CMD(CMD_GO, 1);
        err = ltr_module_send_with_echo_resps(&hnd->Channel, &cmd, 1, &ack);
    }

    if (err == LTR_OK) {
        unsigned ch;
        t_internal_params *params = (t_internal_params *)hnd->Internal;
        hnd->State.Run = 1;
        params->cntr_val = 0;
        params->cntr_lost = FALSE;

        for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
            params->afc_last_valid[ch] = FALSE;
        }
    }


    if (err == LTR_OK) {
        err = ltrslot_start_wconfig(hnd, &hnd->Channel, sizeof(struct LTR25Config), LTR_MID_LTR25,
            conv_hltr25_to_ltr25cfg);
    if (err == LTR_ERROR_CARDSCONFIG_UNSUPPORTED)
        err = LTR_OK;
    }

    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_Stop(TLTR25 *hnd) {
    /* Останов АЦП */
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK) {
        DWORD cmd = LTR_MODULE_MAKE_CMD(CMD_GO, 0);
        err = ltr_module_stop(&hnd->Channel, &cmd, 1, cmd, 0, 0, NULL);
    }

    if (err == LTR_OK)
        err = ltrslot_stop(&hnd->Channel);

    if (err == LTR_OK)
        hnd->State.Run = 0;

    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_StoreConfig(TLTR25 *hnd, TLTR_CARD_START_MODE start_mode) {
    INT err = LTR25_IsOpened(hnd);

    if (err == LTR_OK) {
        err = ltrslot_store_config(hnd, &hnd->Channel, sizeof(struct LTR25Config), LTR_MID_LTR25,
            start_mode, conv_hltr25_to_ltr25cfg);
    }

    return err;
}

/*------------------------------------------------------------------------------------------------*/
LTR25API_DllExport(INT) LTR25_WriteConfig(TLTR25 *hnd) {
    INT err = LTR25_IsOpened(hnd);
    if (err == LTR_OK) {
        BOOL en;
        err = LTR25_FPGAIsEnabled(hnd, &en);
        if (en && (err == LTR_OK))
            err = LTR25_FPGAEnable(hnd, FALSE);


        if (err == LTR_OK) {
            DWORD cmd = CMD_ROM_WR_EN;
            err = ltr_module_send_cmd(&hnd->Channel, &cmd, 1);
            if (err == LTR_OK) {
                t_ltr25_flash_info info, info_ver;
                unsigned ch, freq;
                WORD crc, crc_ver;
                t_internal_params *params = (t_internal_params *)hnd->Internal;
                t_flash_errs flash_res, dis_res;

                info.sign = LTR25_FLASH_INFO_SIGN;
                info.size = sizeof(info);
                info.format = LTR25_FLASH_INFO_FORMAT;
                info.flags = 0;
                memcpy(info.name, hnd->ModuleInfo.Name, LTR25_NAME_SIZE);
                memcpy(info.serial, hnd->ModuleInfo.Serial, LTR25_SERIAL_SIZE);

                for (ch = 0; (ch < LTR25_CHANNEL_CNT); ch++) {
                    for (freq = 0; (freq < LTR25_CBR_FREQ_CNT); freq++) {
                        info.cbr[ch][freq] = hnd->ModuleInfo.CbrCoef[ch][freq];
                    }
                }

                crc = eval_crc16(0, (BYTE *)&info, sizeof(info));


                flash_res = flash_iface_ltr_set_channel(&params->flash, &hnd->Channel);

                if (!flash_res)
                    flash_res = f_set_flash_protection(hnd, FLASH_PROT_INFO_UPDATE);
                if (!flash_res) {
                    flash_res = flash_erase(&params->flash, LTR25_FLASH_ADDR_MODULE_INFO,
                        LTR25_FLASH_INFO_MAX_SIZE);
                }
                if (!flash_res) {
                    flash_res = flash_write(&params->flash, LTR25_FLASH_ADDR_MODULE_INFO,
                        (unsigned char *)&info, sizeof(info), 0);
                }
                if (!flash_res) {
                    flash_res = flash_write(&params->flash,
                        LTR25_FLASH_ADDR_MODULE_INFO+sizeof(info), (unsigned char *)&crc,
                        sizeof(crc), 0);
                }

                dis_res = f_set_flash_protection(hnd, FLASH_PROT_DEFAULT);
                if (!flash_res) {
                    flash_res = dis_res;
                }

                if (!flash_res) {
                    flash_res = flash_read(&params->flash, LTR25_FLASH_ADDR_MODULE_INFO,
                        (unsigned char *)&info_ver, sizeof(info));
                }
                if (!flash_res) {
                    flash_res = flash_read(&params->flash,
                        LTR25_FLASH_ADDR_MODULE_INFO+sizeof(info), (unsigned char *)&crc_ver,
                        sizeof(crc_ver));
                }

                err = flash_iface_ltr_conv_err(flash_res);

                if (err == LTR_OK) {
                    if ((crc != crc_ver) || memcmp(&info, &info_ver, sizeof(info)))
                        err = LTR_ERROR_FLASH_VERIFY;
                }
            } /*if (err == LTR_OK)*/

            if (en) {
                INT en_err = LTR25_FPGAEnable(hnd, TRUE);
                if (err == LTR_OK)
                    err = en_err;
            }
        } /*if (err == LTR_OK)*/
    } /*if (err == LTR_OK)*/

    return err;
}

/*------------------------------------------------------------------------------------------------*/
#ifdef _WIN32
BOOL WINAPI DllMain(HINSTANCE hmod, DWORD reason, LPVOID resvd) {
    switch (reason) {
    case DLL_PROCESS_ATTACH:
    case DLL_THREAD_ATTACH:
    case DLL_THREAD_DETACH:
    case DLL_PROCESS_DETACH:
        break;
    }
    return TRUE;
}
#endif
