#ifndef LTR216API_H_
#define LTR216API_H_

#include "ltrapi.h"

#ifdef __cplusplus
extern "C" {
#endif


#ifdef _WIN32
    #ifdef LTR216API_EXPORTS
        #define LTR216API_DllExport(type) __declspec(dllexport) type APIENTRY
    #else
        #define LTR216API_DllExport(type) __declspec(dllimport) type APIENTRY
    #endif
#elif defined __GNUC__
    #define LTR216API_DllExport(type) __attribute__ ((visibility("default"))) type
#else
    #define LTR216API_DllExport(type) type
#endif

#define LTR216_ADC_CLOCK  (32000000)

#define LTR216_SYNC_FDIV_MAX   (0xFFFFFF)

#define LTR216_MAX_LCHANNEL (1024)

/** Код АЦП, соответствующее максимальному пиковому значению */
#define LTR216_ADC_SCALE_CODE_MAX 8000000



#define LTR216_RANGES_CNT   8

/** Максимальное значение параметра AdcODR при использовании SINC3 фильтра */
#define LTR216_SINC3_ODR_MAX_DIV   32767


typedef enum {
    LTR216_ERR_ADC_ID_CHECK             = -10700,    
    LTR216_ERR_ADC_RECV_SYNC_OVERRATE   = -10701,
    LTR216_ERR_ADC_RECV_INT_CYCLE_ERROR = -10702,
    LTR216_ERR_ADC_REGS_INTEGRITY       = -10703,
    LTR216_ERR_INVALID_ADC_SWMODE       = -10704,
    LTR216_ERR_INVALID_FILTER_TYPE      = -10705,
    LTR216_ERR_INVALID_ADC_ODR_CODE     = -10706,
    LTR216_ERR_INVALID_SYNC_FDIV        = -10707,
    LTR216_ERR_INVALID_LCH_CNT          = -10708,
} e_LTR216_ERROR_CODES;


typedef enum {
    LTR216_FILTER_SINC5_1   = 0,
    LTR216_FILTER_SINC3     = 1,
    LTR216_FILTER_ENH_50HZ  = 2
} e_LTR216_FILTER_TYPE;

typedef enum {
    LTR216_ADC_SWMODE_MULTICH_SYNC = 0,
    LTR216_ADC_SWMODE_SIGNLECH_CONT = 1
} e_LTR216_ADC_SWMODE;


/** Флаги, управляющие работой функции LTR216_ProcessData() */
typedef enum {
    /** Признак, что нужно перевести коды АЦП в Вольты. Если данный флаг не указан,
        то будут возвращены коды АЦП. При этом код #LTR25_ADC_SCALE_CODE_MAX
        соответствует максимальному напряжению для установленного диапазона. */
    LTR216_PROC_FLAG_VOLT          = 0x0001,
    /** По умолчанию LTR216_ProcessData() предполагает, что ей на обработку
        передаются все принятые данные и проверяет непрерывность счетчика не только
        внутри переданного блока данных, но и между вызовами.
        Если обрабатываются не все данные или одни и те же данные обрабатываются
        повторно, то нужно указать данный флаг, чтобы счетчик проверялся только
        внутри обрабатываемого блока */
    LTR216_PROC_FLAG_NONCONT_DATA = 0x0100
} e_LTR216_PROC_FLAGS;


typedef enum {
    LTR216_CH_STATUS_FLAG_OVERRANGE         = (1 << 0),
    LTR216_CH_STATUS_FLAG_LOAD_POW_EXCEEDED = (2 << 0)
} e_LTR216_CH_STATUS_FLAGS;


#pragma pack(4)


typedef struct {
    WORD VerFPGA;
} TINFO_LTR216;


typedef struct {
    BOOL   Run;         /**< Признак, запущен ли сбор данных */
    double AdcFreq;     /**< Установленная частота АЦП. Обновляется после
                            вызова LTR216_SetADC() */
} TLTR216_STATE;

typedef struct {
    DWORD SyncFreqDiv;
    DWORD CurrCode;
    DWORD AdcOdrCode;
    DWORD FilterType;
    DWORD Rx16Mode;
    DWORD AdcSwMode;
    DWORD LChCnt;                           // количество активных логических каналов
    DWORD LChTbl[LTR216_MAX_LCHANNEL];      // управляющая таблица с настройками логических каналов
    DWORD BgLChCnt;
    DWORD BgLChTbl[LTR216_MAX_LCHANNEL];
} TLTR216_CONFIG;


typedef struct {
    INT         Size; /**< Размер структуры. Заполняется в LTR25_Init(). */
    /** Структура, содержащая состояние клиентского соединения со службой ltrd.
        Не используется напрямую пользователем. */
    TLTR        Channel;
    /** Указатель на непрозрачную структуру с внутренними параметрами,
      используемыми исключительно библиотекой и недоступными для пользователя. */
    PVOID       Internal;

    TLTR216_CONFIG Cfg;
    TLTR216_STATE  State;
    /** Информация о модуле */
    TINFO_LTR216 ModuleInfo;
} TLTR216;

typedef struct {
    double Odr;
    double NotchFreq;
    double NotchDB;
    double reserved[3];
} TLTR216_FILTER_OUT_PARAMS;

#pragma pack()

/** @} */

LTR216API_DllExport(INT) LTR216_Init(TLTR216 *hnd);
LTR216API_DllExport(INT) LTR216_Open(TLTR216 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *csn, INT slot);
LTR216API_DllExport(INT) LTR216_Close(TLTR216 *hnd);
LTR216API_DllExport(INT) LTR216_IsOpened(TLTR216 *hnd);


LTR216API_DllExport(INT) LTR216_SetADC(TLTR216 *hnd);
LTR216API_DllExport(INT) LTR216_Start(TLTR216 *hnd);
LTR216API_DllExport(INT) LTR216_Stop(TLTR216 *hnd);


LTR216API_DllExport(INT) LTR216_Recv(TLTR216 *hnd, DWORD *data, DWORD *tmark, DWORD size, DWORD timeout);
LTR216API_DllExport(INT) LTR216_ProcessData(TLTR216 *hnd, const DWORD *src, double *dest,
                                            INT *size, DWORD flags, DWORD *ch_status);

LTR216API_DllExport(INT) LTR216_FindSyncFreqDiv(double adcFreq, DWORD *div, double *resultAdcFreq);
LTR216API_DllExport(INT) LTR216_GetFilterOutParams(DWORD adcSwMode, DWORD FilterType, DWORD AdcODRCode,
                                                   TLTR216_FILTER_OUT_PARAMS *filter_params);

LTR216API_DllExport(LPCSTR) LTR216_GetErrorString(INT err);

#ifdef __cplusplus
}
#endif

#endif
