#include "ltr216api.h"
#include "ad7176.h"
#include <string.h>
#include "ltrmodule.h"
#include "flash.h"
#include "lbitfield.h"
#include "ports/flash_iface_ltr.h"
#include "devices/flash_dev_sst25.h"
#include "ltimer.h"

#define CMD_ROM_WR_EN                   (LTR010CMD_PROGR | 0x10)
#define CMD_NOP                         (LTR010CMD_INSTR | 0x00)
#define CMD_CT_PRELOAD                  (LTR010CMD_INSTR | 0x01)
#define CMD_CT_CTL                      (LTR010CMD_INSTR | 0x02)
#define CMD_CT_IO                       (LTR010CMD_INSTR | 0x03)
#define CMD_CT_SIZE0                    (LTR010CMD_INSTR | 0x04)
#define CMD_CT_SIZE1                    (LTR010CMD_INSTR | 0x05)
#define CMD_F_DIV0                      (LTR010CMD_INSTR | 0x06)
#define CMD_F_DIV1                      (LTR010CMD_INSTR | 0x07)
#define CMD_GO                          (LTR010CMD_INSTR | 0x08)
#define CMD_CURR                        (LTR010CMD_INSTR | 0x09)
#define CMD_FPGA_VER                    (LTR010CMD_INSTR | 0x0F)
#define CMD_ADC_IO_DATA                 (LTR010CMD_INSTR | 0x10)

#define CMD_ERROR                       (LTR010CMD_INSTR | 0x0E)


#define LCH_MSK_OFFSET   (0x3FFFUL << 0)
#define LCH_MSK_SEC_TBL  (   0x1UL << 15)
#define LCH_MSK_GAIN     (   0x7UL << 16)
#define LCH_MSK_CHAN     (   0xFUL << 19)
#define LCH_MSK_SW       ( 0x7FFUL << 23)

#define LTABLE_CMD_BLOCK_SIZE 128

#define RESULT_ADC_SYNC_FREQ(div) ((double)LTR216_ADC_CLOCK/((div) + 1))


#define ADC_DATA_MSK_SEC_WRD            (0x1UL << 7)
#define ADC_DATA_MSK_CNTR               (0x7UL << 4)


#define ADC_DATA_FIRST_LOW_PTR          (0xFFUL << 24)
#define ADC_DATA_FIRST_BG_TABLE         (0x1UL  <<  2)
#define ADC_DATA_FIRST_HI_DATA          (0xFFUL << 16)
#define ADC_DATA_FIRST_HIGH_PTR         (0x3UL  <<  0)


#define ADC_DATA_SEC_FLAGS              (0xFUL << 0)
#define ADC_DATA_SEC_LOW_DATA           (0xFFFFUL << 16)

#define ADC_DATA_FLAG_OVERRANGE         (0x1UL << 0)
#define ADC_DATA_FLAG_REG_ERROR         (0x1UL << 1)
#define ADC_DATA_FLAG_LOAD_POW_EXC      (0x1UL << 2)




#define ADC_REGVAL_IFMODE(alt_sync)  (((alt_sync) ? AD7176_REGBIT_IFMODE_ALT_SYNC : 0) | AD7176_REGBIT_IFMODE_DATA_STAT)



typedef struct {
    t_flash_iface flash;
    WORD bg_tbl_ptr;
    BYTE cur_cntr;
    BOOLEAN cntr_lost;
    double odr;
} t_internal_params;

static const double f_int_ranges[LTR216_RANGES_CNT] = {
    9.77, 19.5, 39, 78.1, 156, 312, 625, 1250
};

static const struct {
    unsigned code;
    double single_odr;
    double multi_odr;
    double notch;
} filt_sinc5_pars[] = {
    { 0, 250000,    50000,    250000},
    { 1, 125000,    41667,    125000},
    { 2,  62500,    31250,     62500},
    { 3,  50000,    27778,     50000},
    { 4,  31250,    20833,     31250},
    { 5,  25000,    17857,     25000},
    { 6,  15625,    12500,     15625},
    { 7,  10000,    10000,     11905},
    { 8,   5000,     5000,      5435},
    { 9,   2500,     2500,      2604},
    {10,   1000,     1000,      1016},
    {11,    500,      500,       504},
    {12,  397.5,    397.5,       400},
    {13,    200,      200,    200.64},
    {14,  100.2,    100.2,     100.4},
    {15,  59.98,    59.98,     59.98},
    {16,  49.96,    49.96,     50.00},
    {17,     20,       20,     20.01},
    {18,  16.66,    16.66,     16.66},
    {19,     10,       10,        10},
    {20,      5,        5,         5},
};


static const struct {
    unsigned code;
    double odr;
    double db;
} filt_enh50_pars[] = {
    { 2,  27.27,  47},
    { 3,     25,  62},
    { 5,     20,  85},
    { 6,  16.67,  90},
};

/* Текстовые описания кодов ошибок. */
static const TLTR_ERROR_STRING_DEF f_err_tbl[] = {
    {LTR216_ERR_ADC_ID_CHECK, "Не удалось обнаржить микросхему АЦП"},
    {LTR216_ERR_ADC_RECV_SYNC_OVERRATE, "Частота синхронизации АЦП превысила частоту преобразования"},
    {LTR216_ERR_ADC_RECV_INT_CYCLE_ERROR, "Ошибка внутреннего цикла чтения данных с АЦП"},
    {LTR216_ERR_ADC_REGS_INTEGRITY, "Нарушена целостность регистров АЦП"},
    {LTR216_ERR_INVALID_ADC_SWMODE, "Задано неверное значение режима опроса каналов АЦП"},
    {LTR216_ERR_INVALID_FILTER_TYPE, "Задано неверное значение типа фильтра АЦП"},
    {LTR216_ERR_INVALID_ADC_ODR_CODE, "Задано неверное значение скорости преобразования АЦП"},
    {LTR216_ERR_INVALID_SYNC_FDIV, "Задано неверное значение делителя частоты синхронизации АЦП"},
    {LTR216_ERR_INVALID_LCH_CNT, "Задано неверное количество логических каналов"}
};



#define ADC_PUT_CMD_DATA(cmd, pos, data) do {    \
        cmd[pos++] = LTR_MODULE_MAKE_CMD_BYTES(CMD_ADC_IO_DATA, 0, (data)); \
        cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_NOP, 0); \
    } while(0)

#define ADC_PUT_CMD_START_RD(cmd, pos, addr)  ADC_PUT_CMD_DATA(cmd, pos,  \
    AD7176_REGBIT_COMMS_RW | ((addr) & AD7176_REGBIT_COMMS_RA))

#define ADC_PUT_CMD_START_WR(cmd, pos, addr)  ADC_PUT_CMD_DATA(cmd, pos,  \
    ((addr) & AD7176_REGBIT_COMMS_RA))


#define ADC_PUT_CMD_END(cmd, pos)  do { \
        cmd[pos++] = LTR_MODULE_MAKE_CMD_BYTES(CMD_ADC_IO_DATA, 1, 0); \
    } while(0)

static INT adc_check_transf_ack(const DWORD *ack, unsigned ack_cnt, BYTE *data) {
    INT err = LTR_OK;
    unsigned i;
    for (i = 0; (i < ack_cnt) && (err == LTR_OK); i++) {
        if (LTR_MODULE_GET_CMD_CODE(ack[i]) != CMD_ADC_IO_DATA) {
            err = LTR_ERROR_INVALID_CMD_RESPONSE;
        } else if ((data != NULL) && (i > 0) && (i != (ack_cnt -1))) {
            data[i-1] = LTR_MODULE_CMD_DATA(ack[i]) & 0xFF;
        }
    }
    return err;
}

static INT adc_check_ctio_ack(const DWORD *ack, unsigned ack_cnt, DWORD *data) {
    INT err = LTR_OK;
    unsigned i;
    for (i = 0; (i < ack_cnt) && (err == LTR_OK); i++) {
        if (LTR_MODULE_GET_CMD_CODE(ack[i]) != CMD_CT_IO) {
            err = LTR_ERROR_INVALID_CMD_RESPONSE;
        } else if (data != NULL) {
            WORD wrd = LTR_MODULE_CMD_DATA(ack[i]);
            if ((i & 1) == 0) {
                data[i/2] = wrd;
            } else {
                data[i/2] |= ((DWORD)wrd << 16);
            }
        }
    }
    return err;
}


static INT adc_read_reg16(TLTR216 *hnd, BYTE addr, WORD *data) {
    INT err;
    DWORD cmd[7], ack[4];
    BYTE  resp[2];
    unsigned pos = 0;

    ADC_PUT_CMD_START_RD(cmd, pos, addr);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_END(cmd, pos);

    err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
    if (err == LTR_OK) {
        err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, sizeof(ack)/sizeof(ack[0]));
    }
    if (err == LTR_OK) {
        err = adc_check_transf_ack(ack, sizeof(ack)/sizeof(ack[0]), resp);
    }
    if (err == LTR_OK) {
        if (data != NULL)
            *data = (resp[0] << 8) | resp[1];
    }
    return err;
}

static INT adc_read_reg32(TLTR216 *hnd, BYTE addr, DWORD *data) {
    INT err;
    DWORD cmd[11], ack[6];
    BYTE  resp[4];
    unsigned pos = 0;

    ADC_PUT_CMD_START_RD(cmd, pos, addr);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_END(cmd, pos);

    err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
    if (err == LTR_OK) {
        err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, sizeof(ack)/sizeof(ack[0]));
    }
    if (err == LTR_OK) {
        err = adc_check_transf_ack(ack, sizeof(ack)/sizeof(ack[0]), resp);
    }
    if (err == LTR_OK) {
        if (data != NULL)
            *data = (resp[0] << 24) | (resp[1] << 16) | (resp[2] << 8) | resp[3];
    }
    return err;
}

static INT adc_read_reg8(TLTR216 *hnd, BYTE addr, BYTE *data) {
    INT err;
    DWORD cmd[5], ack[3];
    BYTE  resp;
    unsigned pos = 0;

    ADC_PUT_CMD_START_RD(cmd, pos, addr);
    ADC_PUT_CMD_DATA(cmd, pos, 0xFF);
    ADC_PUT_CMD_END(cmd, pos);

    err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
    if (err == LTR_OK) {
        err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, sizeof(ack)/sizeof(ack[0]));
    }
    if (err == LTR_OK) {
        err = adc_check_transf_ack(ack, sizeof(ack)/sizeof(ack[0]), &resp);
    }
    if (err == LTR_OK) {
        *data = resp;
    }
    return err;
}

static INT adc_write_reg16(TLTR216 *hnd, BYTE addr, WORD data) {
    INT err;
    DWORD cmd[7], ack[4];
    unsigned pos = 0;

    ADC_PUT_CMD_START_WR(cmd, pos, addr);
    ADC_PUT_CMD_DATA(cmd, pos, (data >> 8) & 0xFF);
    ADC_PUT_CMD_DATA(cmd, pos, data & 0xFF);
    ADC_PUT_CMD_END(cmd, pos);

    err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
    if (err == LTR_OK)
        err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, sizeof(ack)/sizeof(ack[0]));
    if (err == LTR_OK)
        err = adc_check_transf_ack(ack, sizeof(ack)/sizeof(ack[0]), NULL);

    return err;
}

static INT adc_clear_rdy(TLTR216 *hnd) {
    INT err = LTR_OK;


    if (hnd->Cfg.AdcSwMode == LTR216_ADC_SWMODE_MULTICH_SYNC) {
        /* в режиме альтернативной синхронизации АЦП может выполнять преобразование
         * и его не сбросить через SYNC. переводим для сброса преобразования
         * в режим обычного SYNC, чтобы АЦП было в известном состоянии */
        err = adc_write_reg16(hnd, AD7176_REG_IFMODE, ADC_REGVAL_IFMODE(FALSE)
                              | AD7176_REGBIT_IFMODE_REG_CHECK);

        /* если RDY было установлено уже до этого, то сбрасываем его */
        if (err == LTR_OK) {
            err = adc_read_reg32(hnd, AD7176_REG_DATA, NULL);
        }

        /* Восстанавливаем режим альтернативной синхронизации */
        if (err == LTR_OK) {
            err = adc_write_reg16(hnd, AD7176_REG_IFMODE, ADC_REGVAL_IFMODE(TRUE)
                                  | AD7176_REGBIT_IFMODE_REG_CHECK);
        }


        /* При восстановлении альтернативной синхронизации АЦП само запускает
         * одно новое преобразование без SYNC.
         * Чтобы оно не помешало  - дожидаемся его завершения */
        if (err == LTR_OK) {
            BOOL rdy = 0;
            t_ltimer tmr;
            t_internal_params *params = (t_internal_params *)hnd->Internal;

            ltimer_set_ms(&tmr, (unsigned)(1000./params->odr + 200));
            while ((err == LTR_OK ) && !rdy && !ltimer_expired(&tmr)) {
                BYTE status;
                err = adc_read_reg8(hnd, AD7176_REG_STATUS, &status);
                rdy = (status & 0x80) == 0;
            }
        }
    }

    /* вычитываем значение АЦП, чтобы сбросить RDY, чтобы не помешал следующему
       старту */
    if (err == LTR_OK) {
        err = adc_read_reg32(hnd, AD7176_REG_DATA, NULL);
    }
    return err;
}


static INT ltable_load(TLTR216 *hnd, const DWORD *lch, DWORD lch_cnt, BOOL sec_tbl) {
    INT err;
    DWORD ct_cmd, ct_ack;
    ct_cmd = LTR_MODULE_MAKE_CMD(sec_tbl ? CMD_CT_SIZE1 : CMD_CT_SIZE0, lch_cnt - 1);
    err = ltr_module_send_with_echo_resps(&hnd->Channel, &ct_cmd, 1, &ct_ack);
    if (err == LTR_OK) {
        ct_cmd = LTR_MODULE_MAKE_CMD(CMD_CT_CTL, ((sec_tbl ? 1 : 0) << 15) | (1 << 14));
        err = ltr_module_send_with_echo_resps(&hnd->Channel, &ct_cmd, 1, &ct_ack);
    }
    if (err == LTR_OK) {
        DWORD i;
        DWORD rem_cnt = lch_cnt;
        while ((err == LTR_OK) && (rem_cnt != 0)) {
            DWORD cmd[LTABLE_CMD_BLOCK_SIZE], ack[LTABLE_CMD_BLOCK_SIZE/2];
            unsigned pos = 0;
            for (i = 0; (i < rem_cnt) && (pos < sizeof(cmd)/sizeof(cmd[0])); i++) {
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_CT_IO, (*lch & 0xFFFF));
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_NOP,   0);
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_CT_IO, ((*lch >> 16) & 0xFFFF));
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_NOP,   0);
                lch++;
                rem_cnt--;
            }
            err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
            if (err == LTR_OK)
                err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, pos/2);
            if (err == LTR_OK)
                err = adc_check_ctio_ack(ack, pos/2, NULL);
        }
    }
    return err;
}

#if 0
static INT ltable_read(TLTR216 *hnd, DWORD *lch, DWORD lch_cnt, BOOL sec_tbl) {
    INT err;
    DWORD ct_cmd, ct_ack;
    ct_cmd = LTR_MODULE_MAKE_CMD(CMD_CT_CTL, ((sec_tbl ? 1 : 0) << 15));
    err = ltr_module_send_with_echo_resps(&hnd->Channel, &ct_cmd, 1, &ct_ack);
    if (err == LTR_OK) {
        DWORD i;
        DWORD rem_cnt = lch_cnt;
        while ((err == LTR_OK) && (rem_cnt != 0)) {
            DWORD cmd[LTABLE_CMD_BLOCK_SIZE], ack[LTABLE_CMD_BLOCK_SIZE/2];
            unsigned pos = 0;
            for (i = 0; (i < rem_cnt) && (pos < sizeof(cmd)/sizeof(cmd[0])); i++) {
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_CT_IO, 0);
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_NOP,   0);
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_CT_IO, 0);
                cmd[pos++] = LTR_MODULE_MAKE_CMD(CMD_NOP,   0);
                rem_cnt--;
            }
            err = ltr_module_send_cmd(&hnd->Channel, cmd, pos);
            if (err == LTR_OK)
                err = ltr_module_recv_cmd_resp(&hnd->Channel, ack, pos/2);
            if (err == LTR_OK) {
                err = adc_check_ctio_ack(ack, pos/2, lch);
                lch += pos/4;
            }
        }
    }
    return err;
}
#endif




LTR216API_DllExport(INT) LTR216_Init(TLTR216 *hnd) {
    INT res = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;
    if (res == LTR_OK) {
        memset(hnd, 0, sizeof(*hnd));
        hnd->Size = sizeof(*hnd);
        hnd->Internal = NULL;
        res = LTR_Init(&hnd->Channel);
    }
    return res;
}

LTR216API_DllExport(INT) LTR216_IsOpened(TLTR216 *hnd) {
    return (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_IsOpened(&hnd->Channel);
}

LTR216API_DllExport(INT) LTR216_Open(TLTR216 *hnd, DWORD ltrd_addr, WORD ltrd_port, const CHAR *csn, INT slot) {    
    INT warning;
    DWORD i;
    DWORD cmd, ack;
    DWORD open_flags = 0;
    INT err = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;

    if (err == LTR_OK) {
        if (LTR216_IsOpened(hnd) == LTR_OK)
            LTR216_Close(hnd);
    }

    if (err == LTR_OK) {
        free(hnd->Internal);
        if ((hnd->Internal = calloc(1, sizeof(t_internal_params))) == NULL)
            err = LTR_ERROR_MEMORY_ALLOC;
    }

    if (err == LTR_OK) {
        err = ltr_module_open(&hnd->Channel, ltrd_addr, ltrd_port, csn, slot, LTR_MID_LTR216,
            &open_flags, &ack, &warning);
        if (err == LTR_OK) {
            hnd->State.Run = FALSE;

        }
    }

    if ((err == LTR_OK) && !(open_flags & LTR_MOPEN_OUTFLAGS_DONT_INIT)) {
        cmd = CMD_FPGA_VER;
        err = ltr_module_send_with_echo_resps(&hnd->Channel, &cmd, 1, &ack);
        if (err == LTR_OK) {
            hnd->ModuleInfo.VerFPGA = LTR_MODULE_CMD_DATA(ack);
        }

        if (err == LTR_OK) {
            WORD id;
            err = adc_read_reg16(hnd, AD7176_REG_ID, &id);
            if ((err == LTR_OK) && ((id & AD7176_ID_MASK) != AD7176_ID_VAL)) {
                err = LTR216_ERR_ADC_ID_CHECK;
            }

            if (err == LTR_OK) {
                err = adc_write_reg16(hnd, AD7176_REG_ADCMODE, AD7176_REGBIT_ADCMODE_REFEN |
                                      LBITFIELD_SET(AD7176_REGBIT_ADCMODE_MODE, AD7176_MODE_CONTINUOUS) |
                                      LBITFIELD_SET(AD7176_REGBIT_ADCMODE_CLOCKSEL, AD7176_CLOCKSEL_EXT_CLOCK));
            }

            if (err == LTR_OK) {
                err = adc_write_reg16(hnd, AD7176_REG_GPIOCON, AD7176_REGBIT_GPIOCON_MUX_IO |
                                      AD7176_REGBIT_GPIOCON_SYNC_EN | AD7176_REGBIT_GPIOCON_OP_EN0);
            }
            for (i = 0; (i < 4) && (err == LTR_OK); i++) {
                err = adc_write_reg16(hnd, AD7176_REG_CHMAP(i), ((i < 2) ? AD7176_REGBIT_CHMAP_CH_EN : 0) |
                                      LBITFIELD_SET(AD7176_REGBIT_CHMAP_SETUP_SEL, 0) |
                                      LBITFIELD_SET(AD7176_REGBIT_CHMAP_AINPOS, AD7176_AIN_1) |
                                      LBITFIELD_SET(AD7176_REGBIT_CHMAP_AINNEG, AD7176_AIN_0));
            }

            if (err == LTR_OK) {
                err = adc_write_reg16(hnd, AD7176_REG_SETUPCON(0), AD7176_REGBIT_SETUPCON_BIPOLAR |
                                      LBITFIELD_SET(AD7176_REGBIT_SETUPCON_REF_SEL, AD7176_REFSEL_INT_REF));
            }
        }

#if 0
        if (err == LTR_OK) {
            static DWORD lct[128], lct_resp[128];
            for (i = 0; i < sizeof(lct)/sizeof(lct[0]); i++) {
                lct[i] = 0x55 + i;
            }
            err = ltable_load(hnd, lct, sizeof(lct)/sizeof(lct[0]), FALSE);
            if (!err)
                err = ltable_read(hnd, lct_resp, sizeof(lct)/sizeof(lct[0]), FALSE);
            if (!err) {
                BOOL ok = TRUE;
                for (i = 0; (i < sizeof(lct)/sizeof(lct[0])) && ok; i++) {
                    ok = (lct[i] == lct_resp[i]);
                }
            }
        }
#endif

        if (err == LTR_OK) {
            t_internal_params *params = (t_internal_params *)hnd->Internal;
            INT flash_res = flash_iface_ltr_init(&params->flash, &hnd->Channel);
            if (flash_res == 0)
                flash_res = flash_sst25_set(&params->flash);
            if (flash_res == 0) {
                BYTE status;
                /* если разрешено изменение защиты страниц, то сразу
                 * устанавливаем защиту по-умолчанию и запрещаем изменение */
                flash_res = flash_sst25_get_status(&params->flash, &status);
                if ((flash_res == 0) && !(status & SST25_STATUS_BPL)) {
                    /** @todo */
                }
            }
        }
    }

    return (err == LTR_OK) ? warning : err;
}

LTR216API_DllExport(INT) LTR216_Close(TLTR216 *hnd) {
    INT err = (hnd == NULL) ? LTR_ERROR_INVALID_MODULE_DESCR : LTR_OK;
    if (err == LTR_OK) {
        if (hnd->Internal != NULL) {
            t_internal_params *params = (t_internal_params *)hnd->Internal;
            flash_iface_ltr_close(&params->flash);
            free(hnd->Internal);
            hnd->Internal = NULL;
        }
        err = LTR_Close(&hnd->Channel);
    }
    return err;
}

LTR216API_DllExport(INT) LTR216_SetADC(TLTR216 *hnd) {
    INT err = LTR216_IsOpened(hnd);

    if (err == LTR_OK) {
        TLTR216_FILTER_OUT_PARAMS filter_params;
        t_internal_params *params = (t_internal_params *)hnd->Internal;

        BOOL single_ch_mode = hnd->Cfg.AdcSwMode == LTR216_ADC_SWMODE_SIGNLECH_CONT;
        err = LTR216_GetFilterOutParams(hnd->Cfg.AdcSwMode, hnd->Cfg.FilterType, hnd->Cfg.AdcOdrCode, &filter_params);
        if (err == LTR_OK)
            params->odr = filter_params.Odr;


        if ((err == LTR_OK) && !single_ch_mode && (hnd->Cfg.SyncFreqDiv > LTR216_SYNC_FDIV_MAX))
            err = LTR216_ERR_INVALID_SYNC_FDIV;
        if ((err == LTR_OK) && single_ch_mode && (hnd->Cfg.LChCnt != 1))
            err = LTR216_ERR_INVALID_LCH_CNT;

        if (err == LTR_OK) {
            DWORD cmd[3], ack[3];
            DWORD FreqDiv = single_ch_mode ? 0xFFFFFF : hnd->Cfg.SyncFreqDiv;

            cmd[0] = LTR_MODULE_MAKE_CMD_BYTES(CMD_CURR, 1, hnd->Cfg.CurrCode);
            cmd[1] = LTR_MODULE_MAKE_CMD(CMD_F_DIV0, FreqDiv & 0xFFFF);
            cmd[2] = LTR_MODULE_MAKE_CMD(CMD_F_DIV1, (FreqDiv >> 16) & 0xFF);
            err = ltr_module_send_with_echo_resps(&hnd->Channel, cmd, 3, ack);
        }

        if (err == LTR_OK) {
            err = ltable_load(hnd, hnd->Cfg.LChTbl, hnd->Cfg.LChCnt, FALSE);
        }
        if (err == LTR_OK) {
            err = ltable_load(hnd, hnd->Cfg.BgLChTbl, hnd->Cfg.BgLChCnt, TRUE);
        }

        /* сперва сбрасываем бит REG_CHECK, чтобы сбросить ошибку по изменению регистров */
        if (err == LTR_OK) {
            err = adc_write_reg16(hnd, AD7176_REG_IFMODE, ADC_REGVAL_IFMODE(FALSE));
        }

        if (err == LTR_OK) {
            WORD filtr_reg;
            if (hnd->Cfg.FilterType == LTR216_FILTER_SINC5_1) {
                filtr_reg = LBITFIELD_SET(AD7176_REGBIT_FILTCON_ODR, hnd->Cfg.AdcOdrCode);
            } else if (hnd->Cfg.FilterType == LTR216_FILTER_ENH_50HZ) {
                filtr_reg = AD7176_REGBIT_FILTCON_ENHFILTEN | LBITFIELD_SET(AD7176_REGBIT_FILTCON_ENHFILT, hnd->Cfg.AdcOdrCode);
            } else {
                filtr_reg = AD7176_REGBIT_FILTCON_SINC3_MAP | (hnd->Cfg.AdcOdrCode & 0x7FFF);
            }
            err = adc_write_reg16(hnd, AD7176_REG_FILTCON(0), filtr_reg);
        }

        /* в одноканальном режиме отключаем второй канал мультиплексора АЦП, чтобы
            не сбрасывал фильтр */
        if (err == LTR_OK) {
            err = adc_write_reg16(hnd, AD7176_REG_CHMAP(1), (single_ch_mode ? 0 : AD7176_REGBIT_CHMAP_CH_EN) |
                              LBITFIELD_SET(AD7176_REGBIT_CHMAP_SETUP_SEL, 0) |
                              LBITFIELD_SET(AD7176_REGBIT_CHMAP_AINPOS, AD7176_AIN_1) |
                              LBITFIELD_SET(AD7176_REGBIT_CHMAP_AINNEG, AD7176_AIN_0));
        }

        /* снова его устанавливаем, чтобы проверять целостность регистров во время сбора */
        if (err == LTR_OK) {
            err = adc_write_reg16(hnd, AD7176_REG_IFMODE, ADC_REGVAL_IFMODE(!single_ch_mode) | AD7176_REGBIT_IFMODE_REG_CHECK);
        }


        if (err == LTR_OK) {
            err = adc_clear_rdy(hnd);
        }

        if (err == LTR_OK) {
            hnd->State.AdcFreq = single_ch_mode ? filter_params.Odr : RESULT_ADC_SYNC_FREQ(hnd->Cfg.SyncFreqDiv);
        }
    }

    return err;
}

LTR216API_DllExport(INT) LTR216_Start(TLTR216 *hnd) {
    INT err = LTR216_IsOpened(hnd);
    if ((err == LTR_OK) && hnd->State.Run)
        err = LTR_ERROR_MODULE_STARTED;
    if (err == LTR_OK) {
        DWORD cmd, ack;
        cmd = LTR_MODULE_MAKE_CMD(CMD_CT_PRELOAD, hnd->Cfg.AdcSwMode == LTR216_ADC_SWMODE_SIGNLECH_CONT ? 1 : 0);
        err = ltr_module_send_with_echo_resps(&hnd->Channel, &cmd, 1, &ack);
        if (err == LTR_OK) {
            cmd = LTR_MODULE_MAKE_CMD(CMD_GO, 1);
            err = ltr_module_send_cmd(&hnd->Channel, &cmd, 1);
        }
        if (err == LTR_OK) {
            hnd->State.Run = TRUE;
            t_internal_params *params = (t_internal_params *)hnd->Internal;
            params->cur_cntr = 0;
            params->cntr_lost = FALSE;
            params->bg_tbl_ptr = 0;
        }
    }
    return err;
}

LTR216API_DllExport(INT) LTR216_Stop(TLTR216 *hnd) {
    INT err = LTR216_IsOpened(hnd);
    if (err == LTR_OK) {
        DWORD cmd = LTR_MODULE_MAKE_CMD(CMD_GO, 0);
        err = ltr_module_stop(&hnd->Channel, &cmd, 1, cmd, 0, 0, NULL);
        if (err == LTR_OK) {
            err = adc_clear_rdy(hnd);
            hnd->State.Run = FALSE;
        }
    }
    return err;
}

LTR216API_DllExport(INT) LTR216_Recv(TLTR216 *hnd, DWORD *data, DWORD *tmark, DWORD size, DWORD timeout) {
    INT res = LTR216_IsOpened(hnd);
    if (res == LTR_OK)
        res = LTR_Recv(&hnd->Channel, data, tmark, size, timeout);
    if (res >= 0) {
        INT i;
        for (i = 0; i < res; i++) {
            if ((data[i] & LTR_MODULE_CMD_CODE_MSK) == CMD_ERROR) {
                WORD flags = LTR_MODULE_CMD_DATA(data[i]);
                if (flags & 2) {
                    res = LTR216_ERR_ADC_RECV_SYNC_OVERRATE;
                } else {
                    res = LTR216_ERR_ADC_RECV_INT_CYCLE_ERROR;
                }
            }
        }
    }

    if ((res >= 0) && (hnd->Channel.flags & LTR_FLAG_RBUF_OVF))
        res = LTR_ERROR_RECV_OVERFLOW;

    return res;
}


#define CHECK_WRD_CNTR(wrd, cntr, cntr_lost, wrd_err) do { \
    BYTE wrd_cntr = LBITFIELD_GET(ADC_DATA_MSK_CNTR, wrd); \
    if (cntr_lost) { \
        (cntr) = wrd_cntr; \
        (cntr_lost) = FALSE; \
    } else { \
        if ((cntr) != wrd_cntr) { \
            (wrd_err) = LTR_ERROR_PROCDATA_CNTR; \
            (cntr_lost) = TRUE; \
        } else { \
            if (++(cntr) == 7) \
                (cntr) = 0; \
        } \
    } \
} while(0)


LTR216API_DllExport(INT) LTR216_ProcessData(TLTR216 *hnd, const DWORD *src, double *dest,
                                            INT *size, DWORD flags, DWORD *ch_status) {
    INT err = LTR216_IsOpened(hnd);
    if (err == LTR_OK) {
        t_internal_params *params = (t_internal_params *)hnd->Internal;
        BYTE cntr = params->cur_cntr;
        WORD bg_table_pos = params->bg_tbl_ptr;
        WORD table_pos = 0;
        BOOLEAN table_bg = hnd->Cfg.LChTbl[table_pos] & LCH_MSK_SEC_TBL ? TRUE : FALSE;
        BOOLEAN cntr_lost = params->cntr_lost;       
        DWORD i;
        DWORD put_pos = 0;
        DWORD wrds_size = *size/2;
        BOOLEAN out = FALSE;

        if (ch_status != NULL) {
            /** @todo только для основных каналов */
            memset(ch_status, 0, sizeof(ch_status[0])*hnd->Cfg.LChCnt);
        }

        for (i = 0; (i < wrds_size) && !out; i++) {
            INT code = 0;
            BYTE  wrd_flags = 0;
            INT wrd_err = LTR_OK;
            DWORD wrd = *src++;

            wrd_err = LTR_MODULE_WRD_CHECK_DATA(wrd);
            if ((wrd_err == LTR_OK) && (wrd & ADC_DATA_MSK_SEC_WRD))
                wrd_err = LTR_ERROR_PROCDATA_WORD_SEQ;
            if (wrd_err == LTR_OK) {
                CHECK_WRD_CNTR(wrd, cntr, cntr_lost, wrd_err);
            }

            if (wrd_err == LTR_OK) {
                WORD wrd_table_ptr = LBITFIELD_GET(ADC_DATA_FIRST_LOW_PTR, wrd) |
                        (LBITFIELD_GET(ADC_DATA_FIRST_HIGH_PTR, wrd) << 8);
                BOOLEAN wrd_table_bg = wrd & ADC_DATA_FIRST_BG_TABLE ? TRUE : FALSE;

                code = LBITFIELD_GET(ADC_DATA_FIRST_HI_DATA, wrd) << 16;

                /* проверка второго слова на формат */
                wrd = *src++;
                wrd_err = LTR_MODULE_WRD_CHECK_DATA(wrd);
                if ((wrd_err == LTR_OK) && !(wrd & ADC_DATA_MSK_SEC_WRD)) {
                    wrd_err = LTR_ERROR_PROCDATA_WORD_SEQ;
                }
                if (wrd_err == LTR_OK) {
                    CHECK_WRD_CNTR(wrd, cntr, cntr_lost, wrd_err);
                }

                /* проверка правильности позиции в таблице */
                if ((wrd_err == LTR_OK) && ((wrd_table_bg != table_bg) ||
                                            (wrd_table_bg && (wrd_table_ptr != bg_table_pos)) ||
                                            (!wrd_table_bg && (wrd_table_ptr != table_pos)))) {
                    wrd_err = LTR_ERROR_PROCDATA_CHNUM;
                }

                if (wrd_err == LTR_OK) {
                    wrd_flags = LBITFIELD_GET(ADC_DATA_SEC_FLAGS, wrd);
                    code |= LBITFIELD_GET(ADC_DATA_SEC_LOW_DATA, wrd);
                }
            }

            if (wrd_err == LTR_OK) {
                if (wrd_flags & ADC_DATA_FLAG_REG_ERROR) {
                    err = LTR216_ERR_ADC_REGS_INTEGRITY;
                    out = TRUE;
                }

                if (!table_bg) {
                    if (wrd_flags & ADC_DATA_FLAG_OVERRANGE) {
                        if (ch_status != NULL)
                            ch_status[table_pos] |= LTR216_CH_STATUS_FLAG_OVERRANGE;
                    }
                    if (wrd_flags & ADC_DATA_FLAG_LOAD_POW_EXC) {
                        if (ch_status != NULL)
                            ch_status[table_pos] |= LTR216_CH_STATUS_FLAG_LOAD_POW_EXCEEDED;
                    }


                    double val = (code - 0x800000);

                    if (flags & LTR216_PROC_FLAG_VOLT)
                        val = (val * f_int_ranges[LBITFIELD_GET(LCH_MSK_GAIN, hnd->Cfg.LChTbl[table_pos])])/0x800000;
                    dest[put_pos++] = val;
                } else {
                    if (++bg_table_pos  == hnd->Cfg.BgLChCnt)
                        bg_table_pos = 0;
                }

                if (++table_pos == hnd->Cfg.LChCnt)
                    table_pos = 0;
                table_bg = hnd->Cfg.LChTbl[table_pos] & LCH_MSK_SEC_TBL ? TRUE : FALSE;
            } else {
                /** @todo поиск начала таблицы */
                err = wrd_err;
                out = TRUE;
            }
        }

        *size = put_pos;
        params->bg_tbl_ptr = bg_table_pos;
        params->cntr_lost = cntr_lost;
        params->cur_cntr = cntr;
    }

    return err;
}





LTR216API_DllExport(INT) LTR216_FindSyncFreqDiv(double adcSyncFreq, DWORD *div, double *resultAdcSyncFreq) {
    DWORD fnd_div = (DWORD)(LTR216_ADC_CLOCK/adcSyncFreq);
    if (fnd_div > 0)
        fnd_div--;
    if (fnd_div > LTR216_SYNC_FDIV_MAX)
        fnd_div = LTR216_SYNC_FDIV_MAX;
    if (div != NULL)
        *div = fnd_div;
    if (resultAdcSyncFreq != NULL)
        *resultAdcSyncFreq =RESULT_ADC_SYNC_FREQ(fnd_div);
    return LTR_OK;
}

LTR216API_DllExport(INT) LTR216_GetFilterOutParams(DWORD AdcSwMode, DWORD FilterType, DWORD AdcODRCode, TLTR216_FILTER_OUT_PARAMS *FilterParams) {
    INT err = FilterParams == NULL ? LTR_ERROR_PARAMETERS : LTR_OK;
    BOOL single_ch_mose = AdcSwMode == LTR216_ADC_SWMODE_SIGNLECH_CONT;

    if (err == LTR_OK) {
        memset(FilterParams, 0, sizeof(TLTR216_FILTER_OUT_PARAMS));

        if ((AdcSwMode != LTR216_ADC_SWMODE_MULTICH_SYNC) && (AdcSwMode != LTR216_ADC_SWMODE_SIGNLECH_CONT)) {
            err = LTR216_ERR_INVALID_ADC_SWMODE;
        }
    }

    if (err == LTR_OK) {
        if (FilterType == LTR216_FILTER_SINC5_1) {
            BOOL fnd = FALSE;
            unsigned i;
            for (i = 0; i < (sizeof(filt_sinc5_pars)/sizeof(filt_sinc5_pars[0])) && !fnd; i++) {
                if (AdcODRCode == filt_sinc5_pars[i].code) {
                   fnd = TRUE;
                   FilterParams->Odr =  single_ch_mose ? filt_sinc5_pars[i].single_odr : filt_sinc5_pars[i].multi_odr;
                   FilterParams->NotchFreq = filt_sinc5_pars[i].notch;
                }
            }

            if (!fnd) {
                err = LTR216_ERR_INVALID_ADC_ODR_CODE;
            }
        } else if (FilterType == LTR216_FILTER_ENH_50HZ) {
            unsigned i;
            BOOL fnd = FALSE;
            for (i = 0; i < (sizeof(filt_enh50_pars)/sizeof(filt_enh50_pars[0])) && !fnd; i++) {
                if (AdcODRCode == filt_enh50_pars[i].code) {
                   fnd = TRUE;
                   FilterParams->Odr = filt_enh50_pars[i].odr;
                   FilterParams->NotchFreq = 50;
                   FilterParams->NotchFreq = filt_enh50_pars[i].db;
                }
            }

            if (!fnd) {
                err = LTR216_ERR_INVALID_ADC_ODR_CODE;
            }
        } else if (FilterType == LTR216_FILTER_SINC3) {
            if ((AdcODRCode == 0) || (AdcODRCode > LTR216_SINC3_ODR_MAX_DIV)) {
                err = LTR216_ERR_INVALID_ADC_ODR_CODE;
            } else {
                double freq = 8000000./(32*AdcODRCode);
                FilterParams->Odr = freq / (single_ch_mose ? 1 : 3);
                FilterParams->NotchFreq = freq;
            }
        } else {
            err = LTR216_ERR_INVALID_FILTER_TYPE;
        }
    }
    return err;
}


LTR216API_DllExport(LPCSTR) LTR216_GetErrorString(INT err) {
    size_t i;
    for (i = 0; (i < sizeof(f_err_tbl) / sizeof(f_err_tbl[0])); i++) {
        if (f_err_tbl[i].code == err)
            return f_err_tbl[i].message;
    }
    return LTR_GetErrorString(err);
}
