/*
    Данный пример демонстрирует работу с модулем LTR25.
    По умолчанию идет работа с первым слотом первого крейта, но это можно изменить
        через параметры командной строки:
    ltr25_recv  slot crate_serial srvip
    где:
        slot         - номер слота (от 1 до 16)
        crate_serial - серийный номер крейта (если крейтов больше одного)
        srvip        - ip-адрес программы ltrd или LtrServer (если он запущен на другой машине)
    Параметры опциональны: можно указать только слот, слот и серийный номер крейта или все

    Пример принимает данные от первых трех каналов на максимальной частоте сбора.
    На экране отображается значение пик-пик (максимум - минимум) по принятому блоку
    по каждому каналу.

    Пользователь может изменить настройки на свои, поменяв заполнение полей
    структуры перед запуском сбора.

    Сбор идет до нажатия любой клавиши на Windows или  CTRL+C на Linux

    Сборка в VisualStudio:
    Для того чтобы собрать проект в Visual Studio, измените путь к заголовочным файлам
    (Проект (Project) -> Свойства (Properties) -> Свойства конфигурации (Configuration Properties)
    -> С/С++ -> Общие (General) -> Дополнительные каталоги включения (Additional Include Directories))
    на нужный в зависимаости от пути установки библиотек (ltr25api.h  и остальные заголовочные
    файлы должны находится в поддиректории ltr/include относительно указанного пути)
    и измените путь к .lib файлам на <путь установки библиотек>/lib/msvc
    (Проект (Project) -> Свойства (Properties) -> Свойства конфигурации (Configuration Properties) ->
    Компоновщик (Linker) -> Общие (General) -> Дополнительные каталоги библиотек (Additional Library Directories)).

    Внимание!: Если Вы собираете проект под Visual Studio и у Вас некорректно
    отображается вывод русских букв, то нужно изменить кодировку:
    выберите Файл (File) -> Дополнительные параметры сохранения (Advanced Save Options)...
    и в поле Кодировка (Encoding) выберите Юникод (UTF8, с сигнатурой)/Unicode (UTF-8 with signature)
    и сохраните изменения в файле.
*/

#include "ltr/include/ltr216api.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <locale.h>
#include <conio.h>
#else
#include <signal.h>
#include <unistd.h>
#endif


#define LTABLE_WORD(sw1, sw2, sw3, ch, gain, offs) ((((sw1) & 0xF) << 28) | \
    (((sw2) & 0xF) << 24) | \
    (((sw3) & 1) << 23) | \
    (((ch) & 0xF) << 19) | \
    (((gain) & 0x7) << 16) | \
    ((offs) & 0x3FFF))


#define LCH_CNT             (1)
/* количество отсчетов на канал, принмаемых за раз */
#define RECV_BLOCK_CH_SIZE  (4000/LCH_CNT)
/* таймаут на ожидание данных при приеме */
#define RECV_TOUT           (1000)


typedef struct {
    int slot;
    const char *serial;
    DWORD addr;
} t_open_param;


/* признак необходимости завершить сбор данных */
static int f_out = 0;

#ifndef _WIN32
/* Обработчик сигнала завершения для Linux */
static void f_abort_handler(int sig) {
    f_out = 1;
}
#endif


/* Разбор параметров командной строки. Если указано меньше, то используются
 * значения по умолчанию:
 * 1 параметр - номер слота (от 1 до 16)
 * 2 параметр - серийный номер крейта
 * 3 параметр - ip-адрес сервера */
static int f_get_params(int argc, char** argv, t_open_param* par) {
    int err = 0;
    par->slot = LTR_CC_CHNUM_MODULE1;
    par->serial = "";
    par->addr = LTRD_ADDR_DEFAULT;


    if (argc > 1)
        par->slot = atoi(argv[1]);
#if 0
    if (argc > 2)
        par->serial = argv[2];
    if (argc > 3) {
        int a[4],i;
        if (sscanf(argv[3], "%d.%d.%d.%d", &a[0], &a[1], &a[2], &a[3])!=4) {
            fprintf(stderr, "Неверный формат IP-адреса!!\n");
            err = -1;
        }

        for (i=0; (i < 4) && !err; i++) {
            if ((a[i]<0) || (a[i] > 255)) {
                fprintf(stderr, "Недействительный IP-адрес!!\n");
                err = -1;
            }
        }

        if (!err) {
            par->addr = (a[0] << 24) | (a[1]<<16) | (a[2]<<8) | a[3];
        }
    }
#endif
    return err;
}

int main(int argc, char** argv) {
    INT err = LTR_OK;
    TLTR216 hltr216;
    t_open_param par;

    int curr_code;

#ifndef _WIN32
    struct sigaction sa;
    memset(&sa, 0, sizeof(sa));
    /* В ОС Linux устанавливаем свой обработчик на сигнал закрытия,
       чтобы завершить сбор корректно */
    sa.sa_handler = f_abort_handler;
    sigaction(SIGTERM, &sa, NULL);
    sigaction(SIGINT, &sa, NULL);
    sigaction(SIGABRT, &sa, NULL);
#endif

#ifdef _WIN32
    /* для вывода русских букв в консоль для ОС Windows*/
    setlocale(LC_ALL, "");
#endif

    err = f_get_params(argc, argv, &par);

    if (argc < 3) {
        fprintf(stderr, "Не достаточно аргументов. Использование: ltr216test slot curr_code\n");
        err = LTR_ERROR_PARAMETERS;
    } else {
        if ((sscanf(argv[2], "%i", &curr_code) != 1) || (curr_code < 0) || (curr_code > 255)) {
            fprintf(stderr, "Неверно задан код потенциометра\n");
            err = LTR_ERROR_PARAMETERS;
        }
    }

    if (err==LTR_OK) {
        LTR216_Init(&hltr216);

        /* Устанавливаем соединение с модулем */
        err = LTR216_Open(&hltr216, par.addr, LTRD_PORT_DEFAULT, par.serial, par.slot);
        if (err!=LTR_OK) {
            fprintf(stderr, "Не удалось установить связь с модулем. Ошибка %d (%s)\n",
                    err, LTR216_GetErrorString(err));
        } else {
            INT close_err;
            DWORD i;
            double adcFreq;

            printf("Успешно установлена связь с модулем. Версия FPGA = %d\n", (int)hltr216.ModuleInfo.VerFPGA);

            hltr216.Cfg.CurrCode = curr_code;
            hltr216.Cfg.FilterType = LTR216_FILTER_SINC5_1;
            hltr216.Cfg.AdcOdrCode = 0;
            LTR216_FindSyncFreqDiv(40000, &hltr216.Cfg.SyncFreqDiv, &adcFreq);
#if LCH_CNT == 1
            hltr216.Cfg.AdcSwMode = LTR216_ADC_SWMODE_SIGNLECH_CONT;
#else
            hltr216.Cfg.AdcSwMode = LTR216_SWMODE_SIGNLECHANNEL_CONT;
#endif
            hltr216.Cfg.LChCnt = LCH_CNT;
#if LCH_CNT > 1
            for (i = 0; i < hltr216.Cfg.LChCnt - 1; i++) {
                hltr216.Cfg.LChTbl[i] = LTABLE_WORD(1, 0, 1, (i % 16), 7, 8192);
            }
            hltr216.Cfg.LChTbl[hltr216.Cfg.LChCnt - 1] = (1 << 15);
#else
            hltr216.Cfg.LChTbl[0] = LTABLE_WORD(1, 0, 1, 1, 7, 8192);
#endif

            hltr216.Cfg.BgLChCnt = 3;
            hltr216.Cfg.BgLChTbl[0] = LTABLE_WORD(1, 0, 1, 0, 7, 8192);
            hltr216.Cfg.BgLChTbl[1] = LTABLE_WORD(1, 0, 1, 1, 7, 8192);
            hltr216.Cfg.BgLChTbl[2] = LTABLE_WORD(1, 0, 1, 2, 7, 8192);
            err = LTR216_SetADC(&hltr216);
            if (err != LTR_OK) {
                fprintf(stderr, "Не удалось установить код потенциометра. Ошибка %d (%s)\n",
                        err, LTR216_GetErrorString(err));
            } else {
                printf("Настройки АЦП установленны успешно. Частота = %.2f Гц, код потенциометра: 0x%02X\n",
                       hltr216.State.AdcFreq, curr_code);
            }

            if (err==LTR_OK) {
                DWORD recvd_blocks=0;
                INT recv_data_cnt = RECV_BLOCK_CH_SIZE*hltr216.Cfg.LChCnt;
                /* В 24-битном формате каждому отсчету соответствует два слова от модуля,
                   а в 20-битном - одно */
                INT   recv_wrd_cnt = recv_data_cnt*2;
                DWORD  *rbuf = (DWORD*)malloc(recv_wrd_cnt*sizeof(rbuf[0]));
                double *data = (double *)malloc(recv_data_cnt*sizeof(data[0]));

                if ((rbuf==NULL) || (data==NULL)) {
                    fprintf(stderr, "Ошибка выделения памяти!\n");
                    err = LTR_ERROR_MEMORY_ALLOC;
                }

                if (err==LTR_OK) {
                    /* Запуск сбора данных */
                    err=LTR216_Start(&hltr216);
                    if (err!=LTR_OK) {
                        fprintf(stderr, "Не удалось запустить сбор данных! Ошибка %d (%s)\n",
                                err, LTR216_GetErrorString(err));
                    }
                }

                if (err==LTR_OK) {
                    printf("Сбор данных запущен. Для останова нажмите %s\n",
#ifdef _WIN32
                           "любую клавишу"
#else
                           "CTRL+C"
#endif
                           );
                    fflush(stdout);
                }


                /* ведем сбор данных до возникновения ошибки или до
                 * запроса на завершение */
                while (!f_out && (err==LTR_OK)) {
                    INT recvd;
                    /* В таймауте учитываем время сбора запрашиваемого числа отсчетов */
                    DWORD tout = RECV_TOUT + (DWORD)(1000.*RECV_BLOCK_CH_SIZE*hltr216.Cfg.LChCnt/hltr216.State.AdcFreq + 1);
                    DWORD ch_status[LCH_CNT];

                    /* Прием данных от модуля.  */
                    recvd = LTR216_Recv(&hltr216, rbuf, NULL, recv_wrd_cnt, tout);

                    /* Значение меньше нуля соответствуют коду ошибки */
                    if (recvd<0) {
                        err = recvd;
                        fprintf(stderr, "Ошибка приема данных. Ошибка %d:%s\n",
                                err, LTR216_GetErrorString(err));
                    } else if (recvd!=recv_wrd_cnt) {
                        fprintf(stderr, "Принято недостаточно данных. Запрашивали %d, приняли %d\n",
                                recv_wrd_cnt, recvd);
                        err = LTR_ERROR_RECV_INSUFFICIENT_DATA;
                    } else {
                        err = LTR216_ProcessData(&hltr216, rbuf, data, &recvd, LTR216_PROC_FLAG_VOLT, ch_status);
                        if (err != LTR_OK) {
                            fprintf(stderr, "Ошибка обработки данных. Ошибка %d:%s\n",
                                    err, LTR216_GetErrorString(err));
                        } else {
                            DWORD ch;
                            DWORD show_cnt = hltr216.Cfg.LChCnt;
                            if (show_cnt > 16)
                                show_cnt = 16;
                            recvd_blocks++;
                            printf("Принят блок %d (размер = %d):", recvd_blocks, recvd);
                            for (ch=0; (ch < show_cnt) ; ch++) {
                                printf("%8.2f", data[ch]);

                                if (ch==(show_cnt-1)) {
                                    printf("\n");
                                } else {
                                    printf(",  ");
                                }
                            }
                            fflush(stdout);
                        }
                    }

#ifdef _WIN32
                    /* проверка нажатия клавиши для выхода */
                    if (err==LTR_OK) {
                        if (_kbhit())
                            f_out = 1;
                    }
#endif
                } //while (!f_out && (err==LTR_OK))

                /* по завершению останавливаем сбор, если был запущен */
                if (hltr216.State.Run) {
                    INT stop_err = LTR216_Stop(&hltr216);
                    if (stop_err!=LTR_OK) {
                        fprintf(stderr, "Не удалось остановить сбор данных. Ошибка %d:%s\n",
                                stop_err, LTR216_GetErrorString(stop_err));
                        if (err==LTR_OK)
                            err = stop_err;
                    } else {
                        printf("Сбор остановлен успешно.\n");
                    }
                }

                free(rbuf);
                free(data);
            }



            /* закрываем связь с модулем */
            close_err = LTR216_Close(&hltr216);
            if (close_err!=LTR_OK) {
                fprintf(stderr, "Не удалось закрыть связь с модулем. Ошибка %d:%s\n",
                        close_err, LTR216_GetErrorString(close_err));
                if (err==LTR_OK)
                    err = close_err;
            } else {
                printf("Связь с модулем успешно закрыта.\n");
            }
        }
    }

    return err;
}
