#ifndef LBOOT_H_
#define LBOOT_H_

#include "lboot_config.h"
#include "stdint.h"

#include "lboot_prt.h"


/** коды ошибок программы загрузчика */
typedef enum {
    LBOOT_ERR_MIN_ID                  = -1,
    /** опция заданная в командной строке не поддерживается */
    LBOOT_ERR_INVALID_OPTION       = -1,
    /** недостаточное количество аргументов */
    LBOOT_ERR_INSUFFICIENT_ARGUMENTS  = -2,
    /** указан неверный интерфейс для передачи прошивки устройству */
    LBOOT_ERR_INVALID_INTERFACE       = -3,
    LBOOT_ERR_FIRMWARE_FILE_OPEN      = -4,
    LBOOT_ERR_SIGNATURE_FILE_OPEN     = -5,
    LBOOT_ERR_HASH_FILE_OPEN          = -6,
    LBOOT_ERR_MEMORY_ALLOC            = -7,
    LBOOT_ERR_ALLOC_RESOURCE          = -8,
    LBOOT_ERR_DEVICE_NOT_FOUND        = -9,
    LBOOT_ERR_IOCTRL_FAILED           = -10,
    LBOOT_ERR_INVALID_DEVICE          = -11,
    LBOOT_ERR_INVALID_SERIAL          = -12,
    LBOOT_ERR_INVALID_DEVMODE         = -13,
    /** загрузчик не поддерживает возможность загрузки, без запуска основной программы */
    LBOOT_ERR_UNSUP_FLAG_DONT_START   = -14,
    /** загрузчик не поддерживает возможность восстановления основной прошивки по команде */
    LBOOT_ERR_UNSUP_FLAG_RECOVERY     = -15,
    /** указан неверный фармат вывода в опции --output-fmt */
    LBOOT_ERR_INVALID_OUTPUT_FMT      = -16,
    /** загрузчик не поддерживает запись защищенного блока данных */
    LBOOT_ERR_UNSUP_WR_PROT_DATA      = -17,
    /** ошибка открытия файла с даннымы для записи в защищенный блок */
    LBOOT_ERR_PROT_DATA_FILE_OPEN     = -18
} t_lboot_pc_errs;


/** коды ошибок блока разбора OpenPGP записи в устройстве */
typedef enum {
    LBOOT_ERR_LCRYPT_OFFSET                      = -10000,
    LCRYPT_ERR_PGP_UNSUF_DATA                    = -10000,
    LCRYPT_ERR_PGP_SIGN_FORMAT                   = -10001,  /**< неверный формат PGP файла */
    LCRYPT_ERR_PGP_SIGN_UNSUPPORT_CRYPT_ALG      = -10002,  /**< неподдерживаемый алгоритм шифрования для подписи */
    LCRYPT_ERR_PGP_SIGN_UNSUPPORT_HASH_ALG       = -10003  /**< неподдерживаемый алгоритм вычисления hash-функции */
} t_lcrypt_errs;




/************   информация о устройстве ***********************************/

#define LBOOT_DEVNAME_SIZE            32
#define LBOOT_SERIAL_SIZE             32
#define LBOOT_SOFTVER_SIZE            32
#define LBOOT_REVISION_SIZE           16
#define LBOOT_IMPLEMENTATION_SIZE     16
#define LBOOT_SPECINFO_SIZE           64


#define LBOOT_DEVMODE_BOOT                0x1
#define LBOOT_DEVMODE_APPL                0x2

struct st_lboot_devinfo {
    char devname[LBOOT_DEVNAME_SIZE]; //название устройства
    char serial[LBOOT_SERIAL_SIZE];   //серийный номер
    char soft_ver[LBOOT_SOFTVER_SIZE]; //версия софта
    char brd_revision[LBOOT_REVISION_SIZE]; //аппаратная версия
    char brd_impl[LBOOT_IMPLEMENTATION_SIZE];
    char spec_info[LBOOT_SPECINFO_SIZE];
} ;
typedef struct st_lboot_devinfo t_lboot_devinfo;




typedef int (*t_lboot_intf_init)(void);
typedef int (*t_lboot_parse_option)(int opt);
typedef const char* (*t_lboot_get_opt_descr)(void);
typedef const char* (*t_lboot_get_err_str)(int err);
typedef int (*t_lboot_open)(const char* devname, const char* serial,
                          const void* params, t_lboot_devinfo* info, int* out);
typedef int (*t_lboot_get_devmode)(uint8_t* mode);
typedef int (*t_lboot_get_bootloader_version)(uint16_t* ver);
typedef int (*t_lboot_get_dev_flags)(uint32_t* flags);
typedef int (*t_lboot_get_features)(uint32_t* flags);
typedef int (*t_lboot_start_write)(t_lboot_startwr_info wr_info);
typedef int (*t_lboot_spec_cmd)(const t_lboot_speccmd_info* cmd_info, uint32_t size);
typedef int (*t_lboot_write)(const uint8_t* buf, uint32_t size);
typedef int (*t_lboot_write_sign)(const uint8_t* sign, uint32_t size);
typedef int (*t_lboot_close)(void);
typedef int (*t_lboot_app_to_boot)(void);

/** параметры интерфейса и его callback-функции */
typedef struct {
    const char* name;            /**< название интерфейса */
    const char* descr;           /**< описание интерфейса */
    /** максимальный размер блока, который можно записать за раз */
    uint32_t wr_block_size;
    /** код ошибки, с которого начинаются коды ошибок данного интерфейса */
    int err_start_code;
    /** последний код ошибки для данного интерфейса */
    int err_last_code;
    t_lboot_intf_init init;
    /** функция для разбора специфичных опций интерфейса (может быть NULL)*/
    t_lboot_parse_option opt_parse;
    /** получение строки с описанием специфичных опций интерфейса */
    t_lboot_get_opt_descr opt_descr;
    /** получение строки, соответствующей коду ошибки, специфичной для интерфейса */
    t_lboot_get_err_str   get_err_str;
    /** установить связ с модулем и получить информацию о модуле */
    t_lboot_open         open;
    t_lboot_get_devmode  get_devmode;
    t_lboot_get_bootloader_version get_bootldr_ver;
    t_lboot_get_dev_flags get_dev_flags;
    t_lboot_get_features  get_features;
    t_lboot_start_write  start_write;
    t_lboot_write        write;
    t_lboot_write_sign   write_sign;
    t_lboot_spec_cmd     speccmd;
    t_lboot_close        close;
    t_lboot_app_to_boot  app_to_boot;
} t_interface_info;



#define LBOOT_OPT_TFTP_IP_ADDR       0x12100
#define LBOOT_OPT_TFTP_PORT          0x12101
#define LBOOT_OPT_TFTP_BLOCK_SIZE    0x12102

/** опции для задания параметров соединения по пртротоколу tftp */
#define TFTP_OPTIONS  {"tftp-ip-addr",    required_argument,   0, LBOOT_OPT_TFTP_IP_ADDR}, \
                      {"tftp-port",    required_argument,   0, LBOOT_OPT_TFTP_PORT}, \
                      {"tftp-block-size", required_argument, 0, LBOOT_OPT_TFTP_BLOCK_SIZE},



#define LBOOT_OPT_USB_APP_BOOT_REQ      0x12200
#define LBOOT_OPT_USB_APP_LCOMP_ID      0x12201
#define LBOOT_OPT_USB_APP_DEV_ID        0x12202
#define LBOOT_OPT_USB_APP_LCOMP_DEVNAME 0x12203

/** опции для задания параметров соединения по пртротоколу tftp */
#define USB_OPTIONS  {"usb-app-boot-req",   required_argument,   0, LBOOT_OPT_USB_APP_BOOT_REQ}, \
                     {"usb-app-lcomp-id",  required_argument,   0, LBOOT_OPT_USB_APP_LCOMP_ID}, \
                     {"usb-app-dev-id",    required_argument,   0, LBOOT_OPT_USB_APP_DEV_ID}, \
                     {"usb-app-lcomp-devname", required_argument, 0, LBOOT_OPT_USB_APP_LCOMP_DEVNAME},

#endif
