#define _XOPEN_SOURCE 600
#define _LARGEFILE64_SOURCE 1
#include <fcntl.h>
#include <sys/file.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>
#include <string.h>

#include "fallocate_internal.h"

/** Helper: Return early, if we have either been successful or
 *  encountered an error that makes other attempts useless. */
static /*inline*/ int bail_out(int err)
{
	int eno = errno;
	/* We have been successful, return */
	if (err == 0)
		return -1;
	/* These errors indicate other attempts are futile.
	 * So exit as well */
	if (eno == EPERM ||
            eno == EFBIG ||
	    eno == EBADF ||
	    eno == ESPIPE ||
	    eno == EROFS)
		return eno;
	return 0;
}

/** fallocate function that attempts the syscall as well as the XFS/ocfs2 ioctl */
int linux_fallocate64(int fd, int mode, __off64_t start, __off64_t len)
{
	static char falloc_unsupp = 0;
	struct stat64 stbuf;
	if (!falloc_unsupp && mode != (int)FALLOC_FL_UNRESV) {
		int err = fallocate64(fd, mode, start, len);
		if (err && errno == ENOSYS)
			falloc_unsupp = 1;
		if (bail_out(err))
			return err;
	}
	if (fstat64(fd, &stbuf))
		return -1;
	return fallocate64_xfs_ocfs2(fd, mode,
				     start, len, &stbuf);
}

/** Rich interface, with various fallback options.
 * We prefer fallocate over posix_fallocate; glibc's 
 * posix_fallocate has the issue of falling back to writing 
 * zeros if the system call does not succeed, so this may 
 * end up being a slow operation. So we need to determine
 * whether it's still worth the effort if that happens
 * (or possibly we can ensure it does not happen by having
 *  control ov er what cluster filesystems are used).
 * Possibly, we may never want to call posix_fallocate()
 * on Linux.
 * See http://www.linux-mag.com/id/7272/2/
 */
int fallocate64_with_fallback(int fd, int mode, __off64_t start, 
			      __off64_t len, 
		   	      char fb_posix, char fb_sparse)
{
	static char falloc_unsupp2 = 0;
	struct stat64 stbuf;
	int err;
	if (!falloc_unsupp2 && mode != (int)FALLOC_FL_UNRESV) {
		err = fallocate64(fd, mode, start, len);
		/* Don't try again if kernel does not support syscall */
		if (err && errno == ENOSYS)
			falloc_unsupp2 = 1;	
		if (bail_out(err))
			return err;
	}
	if (fstat64(fd, &stbuf))
		return -1;
	err = fallocate64_xfs_ocfs2(fd, mode,
				    start, len, &stbuf);
	if (bail_out(err))
		return err;
	if (mode != 0 && mode != (int)FALLOC_FL_KEEP_SIZE) {
		errno = ENOTSUP;
		return -1;
	}
#ifdef HAVE_POSIX_FALLOCATE
	if (fb_posix && mode == 0)
		if (!posix_fallocate64(fd, start, len))
			return 0;
#else
	if (fb_posix & mode == 0)
		if (!fallocate64_zero(fd, start, len, &stbuf))
			return 0;
#endif
	if (fb_sparse)
		return fallocate64_sparse(fd, mode, start, len, &stbuf);
	/* Further fallback options here ... */

	/* Fall through ... */
	errno = ENOTSUP;
	return -1;
}
