/* Header file fallocate.h for libfallocate 
 * (c) garloff@suse.de, 8/2009
 * License: GNU LGPL v2 or later
 */

#ifndef _FALLOCATE_H
#define _FALLOCATE_H	1

#define _LARGEFILE64_SOURCE 1
#include <sys/stat.h>

#ifdef __cplusplus
extern "C" {
#endif

/** mode flags; default is 0 which means to extend the size of the file */
#define FALLOC_FL_ADJUST_SIZE	0x00 		/* Adjust file size */
#define FALLOC_FL_KEEP_SIZE	0x01 		/* Don't change file size */
#define FALLOC_FL_UNRESV 	0xdeadbeef	/* Only supported by xfs.ocfs2 ioctl */

/** Syscall wrapper for systems that lack it (glibc < 2.10).
 * This will tell the filesystem to reserve len bytes of space for 
 * the file with file descriptor fd starting at offset start.
 * mode should be set to one of the FALLOC_FL_ values. 
 */
extern int fallocate64(int fd, int mode, 
		       __off64_t start, __off64_t len);
#if _FILE_OFFSET_BITS == 64
#define fallocate(f,m,s,l) fallocate64(f,m,s,l)
#else
extern int fallocate(int fd, int mode, 
		     off_t start, off_t len);
#endif

/** More intelligent interface to fallocate, that will fallback to
 * the xfs/ocfs2 ioctl if fallocate failed with ENOSYS or EOPNOTSUPP
 * (and do a sparse write to adjust file length to mimic fallocate 
 *  if mode == FALLOC_FL_ADJUST_SIZE).
 */
extern int linux_fallocate64(int fd, int mode, 
			     __off64_t start, __off64_t len);
#define linux_fallocate(f,m,s,l) linux_fallocate64(f,m,s,l)

/** Feature rich interface with various fallback options;
 * it will try fallocate and the xfs/ocfs2 ioctl; if those
 * are not supported, two fallbacks can be tried:
 * - If fb_posix is set (and mode is FALLOC_FL_ADJUST_SIZE), 
 *   posix_fallocate() will be called, which will end up writing
 *   lots of zeros to the disk. (If glibc does not yet support
 *   posix_fallocate, the library will take care of writing
 *   zeros itself.)
 * - If fb_sparse is set (and mode is 0 or 1),
 *   a sparse write will be done at position start+len.
 *   For mode 0, 1 byte will be written, for mode 1, 0 bytes.
 *   The code checks the existing file size to not overwrite
 *   existing data.
 */
extern int fallocate64_with_fallback(int fd, int mode, 
				     __off64_t start, __off64_t len, 
				     char fb_posix, char fb_sparse);
#define fallocate_with_fallback(f,m,s,l,p,a) fallocate64_with_fallback(f,m,s,l,p,a)

#ifdef __cplusplus
}	/* extern "C" */
#endif

#endif	/* _FALLOCATE_H */

